import Checkbox from "@/Components/Checkbox";
import InputError from "@/Components/InputError";
import InputLabel from "@/Components/InputLabel";
import PrimaryButton from "@/Components/PrimaryButton";
import TextInput from "@/Components/TextInput";
import GuestLayout from "@/Layouts/GuestLayout";
import { Head, Link, useForm } from "@inertiajs/react";
import { Lock, Mail, Eye, EyeOff, Shield, Users, Star } from "lucide-react";
import { useState } from "react";

export default function Login({ status, canResetPassword }) {
    const { data, setData, post, processing, errors, reset } = useForm({
        email: "",
        password: "",
        remember: false,
    });

    const [showPassword, setShowPassword] = useState(false);

    const submit = (e) => {
        e.preventDefault();

        post(route("login"), {
            onFinish: () => reset("password"),
        });
    };

    return (
        <GuestLayout>
            <Head title="Log in" />

            <div className="w-full overflow-hidden bg-gradient-to-r from-blue-100 to-cyan-100 dark:bg-none border border-gray-300 dark:border-gray-600 px-8 py-8 shadow-xl sm:rounded-xl transition-colors duration-200">
                {/* Header */}
                <div className="text-center mb-8">
                    <div className="mx-auto w-12 h-12 bg-blue-100 dark:bg-blue-900/30 rounded-full flex items-center justify-center mb-4">
                        <Lock className="w-6 h-6 text-blue-600 dark:text-blue-400" />
                    </div>
                    <h2 className="text-2xl font-bold text-gray-900 dark:text-white mb-2">
                        Welcome back
                    </h2>
                    <p className="text-gray-600 dark:text-gray-400">
                        Sign in to your account to continue
                    </p>
                </div>

                {/* Status Message */}
                {status && (
                    <div className="mb-6 p-4 bg-green-50 dark:bg-green-900/20 border border-green-200 dark:border-green-700 rounded-lg">
                        <div className="flex items-center">
                            <Shield className="w-5 h-5 text-green-600 dark:text-green-400 mr-2" />
                            <span className="text-sm font-medium text-green-800 dark:text-green-200">
                                {status}
                            </span>
                        </div>
                    </div>
                )}

                {/* Social Proof */}
                <div className="mb-6 p-4 bg-blue-50 dark:bg-blue-900/20 border border-blue-200 dark:border-blue-700 rounded-lg">
                    <div className="flex items-center justify-between">
                        <div className="flex items-center">
                            <Users className="w-4 h-4 text-blue-600 dark:text-blue-400 mr-2" />
                            <span className="text-sm text-blue-700 dark:text-blue-300">
                                Join 10,000+ satisfied students
                            </span>
                        </div>
                        <div className="flex items-center">
                            {[...Array(5)].map((_, i) => (
                                <Star key={i} className="w-4 h-4 text-yellow-400 fill-current" />
                            ))}
                            <span className="text-sm text-blue-700 dark:text-blue-300 ml-1">
                                4.9/5
                            </span>
                        </div>
                    </div>
                </div>

                <form onSubmit={submit} className="space-y-6">
                    {/* Email Field */}
                    <div>
                        <InputLabel htmlFor="email" value="Email" className="text-gray-700 dark:text-gray-300" />
                        <div className="relative mt-1">
                            <div className="absolute inset-y-0 left-0 pl-3 flex items-center pointer-events-none">
                                <Mail className="h-5 w-5 text-gray-400" />
                            </div>
                            <TextInput
                                id="email"
                                type="email"
                                name="email"
                                value={data.email}
                                className="pl-10 w-full"
                                autoComplete="username"
                                isFocused={true}
                                onChange={(e) => setData("email", e.target.value)}
                                placeholder="Enter your email address"
                            />
                        </div>
                        <InputError message={errors.email} className="mt-2" />
                    </div>

                    {/* Password Field */}
                    <div>
                        <InputLabel htmlFor="password" value="Password" className="text-gray-700 dark:text-gray-300" />
                        <div className="relative mt-1">
                            <div className="absolute inset-y-0 left-0 pl-3 flex items-center pointer-events-none">
                                <Lock className="h-5 w-5 text-gray-400" />
                            </div>
                            <TextInput
                                id="password"
                                type={showPassword ? "text" : "password"}
                                name="password"
                                value={data.password}
                                className="pl-10 pr-10 w-full"
                                autoComplete="current-password"
                                onChange={(e) => setData("password", e.target.value)}
                                placeholder="Enter your password"
                            />
                            <button
                                type="button"
                                className="absolute inset-y-0 right-0 pr-3 flex items-center"
                                onClick={() => setShowPassword(!showPassword)}
                            >
                                {showPassword ? (
                                    <EyeOff className="h-5 w-5 text-gray-400 hover:text-gray-600" />
                                ) : (
                                    <Eye className="h-5 w-5 text-gray-400 hover:text-gray-600" />
                                )}
                            </button>
                        </div>
                        <InputError message={errors.password} className="mt-2" />
                    </div>

                    {/* Remember Me & Forgot Password */}
                    <div className="flex items-center justify-between">
                        <label className="flex items-center">
                            <Checkbox
                                name="remember"
                                checked={data.remember}
                                onChange={(e) => setData("remember", e.target.checked)}
                            />
                            <span className="ms-2 text-sm text-gray-600 dark:text-gray-400">
                                Remember me
                            </span>
                        </label>

                        {canResetPassword && (
                            <Link
                                href={route("password.request")}
                                className="text-sm text-blue-600 dark:text-blue-400 hover:text-blue-800 dark:hover:text-blue-300 underline transition-colors"
                            >
                                Forgot password?
                            </Link>
                        )}
                    </div>

                    {/* Submit Button */}
                    <PrimaryButton 
                        className="w-full py-3 text-base font-medium" 
                        disabled={processing}
                    >
                        {processing ? "Signing in..." : "Sign in"}
                    </PrimaryButton>

                    {/* Register Link */}
                    <div className="text-center">
                        <span className="text-sm text-gray-600 dark:text-gray-400">
                            Don't have an account?{" "}
                        </span>
                        <Link
                            href={route("register")}
                            className="text-sm text-blue-600 dark:text-blue-400 hover:text-blue-800 dark:hover:text-blue-300 font-medium underline transition-colors"
                        >
                            Create one now
                        </Link>
                    </div>
                </form>

                {/* Trust Indicators */}
                <div className="mt-8 pt-6 border-t border-gray-200 dark:border-gray-700">
                    <div className="flex items-center justify-center space-x-6 text-xs text-gray-500 dark:text-gray-400">
                        <div className="flex items-center">
                            <Shield className="w-3 h-3 mr-1" />
                            <span>Secure</span>
                        </div>
                        <div className="flex items-center">
                            <Users className="w-3 h-3 mr-1" />
                            <span>Trusted</span>
                        </div>
                        <div className="flex items-center">
                            <Star className="w-3 h-3 mr-1" />
                            <span>Rated</span>
                        </div>
                    </div>
                </div>
            </div>
        </GuestLayout>
    );
}
