import InputError from "@/Components/InputError";
import InputLabel from "@/Components/InputLabel";
import PrimaryButton from "@/Components/PrimaryButton";
import TextInput from "@/Components/TextInput";
import GuestLayout from "@/Layouts/GuestLayout";
import { Head, Link, useForm } from "@inertiajs/react";
import { useState, useEffect } from "react";

export default function WriterRegister({
    countries,
    subjects,
    citationStyles,
    educationLevels,
    languages,
    maxSubjects = 5,
    maxLanguages = 3,
    maxCitationStyles = 5,
}) {
    const { data, setData, post, processing, errors, reset } = useForm({
        name: "",
        email: "",
        password: "",
        password_confirmation: "",
        country_id: "",
        phone: "",
        education_level_id: "",
        subjects: [], // This will store an array of subject IDs
        languages: [], // This will store an array of language IDs
        citation_styles: [], // This will store an array of citation style IDs
        bio: "",
        experience_years: 0,
    });

    // State to track selected items
    const [selectedSubjects, setSelectedSubjects] = useState([]);
    const [selectedLanguages, setSelectedLanguages] = useState([]);
    const [selectedCitationStyles, setSelectedCitationStyles] = useState([]);

    // Error state
    const [subjectError, setSubjectError] = useState("");
    const [languageError, setLanguageError] = useState("");
    const [citationStyleError, setCitationStyleError] = useState("");

    // Update the form data whenever selected items change
    useEffect(() => {
        setData(
            "subjects",
            selectedSubjects.map((subject) => subject.id)
        );
    }, [selectedSubjects]);

    useEffect(() => {
        setData(
            "languages",
            selectedLanguages.map((language) => language.id)
        );
    }, [selectedLanguages]);

    useEffect(() => {
        setData(
            "citation_styles",
            selectedCitationStyles.map((style) => style.id)
        );
    }, [selectedCitationStyles]);

    // Auto-populate phone field with country dialing code
    const handleCountryChange = (e) => {
        const countryId = e.target.value;
        setData("country_id", countryId);
        if (countryId) {
            const selectedCountry = countries.find(
                (country) => country.id == countryId
            );
            if (selectedCountry && selectedCountry.dialing_code) {
                // Only set the dialing code if the phone field is empty or only contains a dialing code
                if (!data.phone || data.phone.startsWith("+")) {
                    setData("phone", selectedCountry.dialing_code);
                }
            }
        }
    };

    const handleSubjectChange = (e) => {
        const subjectId = e.target.value;
        if (subjectId === "") return;
        const selectedSubject = subjects.find(
            (subject) => subject.id == subjectId
        );

        // Check if already selected
        if (selectedSubjects.some((subject) => subject.id == subjectId)) {
            setSubjectError("This subject is already selected");
            return;
        }

        // Check if maximum number of subjects is reached
        if (selectedSubjects.length >= maxSubjects) {
            setSubjectError(
                `You can select a maximum of ${maxSubjects} subjects`
            );
            return;
        }

        setSelectedSubjects([...selectedSubjects, selectedSubject]);
        setSubjectError("");

        // Reset the select dropdown
        e.target.value = "";
    };

    const handleLanguageChange = (e) => {
        const languageId = e.target.value;
        if (languageId === "") return;
        const selectedLanguage = languages.find(
            (language) => language.id == languageId
        );

        // Check if already selected
        if (selectedLanguages.some((language) => language.id == languageId)) {
            setLanguageError("This language is already selected");
            return;
        }

        // Check if maximum number of languages is reached
        if (selectedLanguages.length >= maxLanguages) {
            setLanguageError(
                `You can select a maximum of ${maxLanguages} languages`
            );
            return;
        }

        setSelectedLanguages([...selectedLanguages, selectedLanguage]);
        setLanguageError("");

        // Reset the select dropdown
        e.target.value = "";
    };

    const handleCitationStyleChange = (e) => {
        const styleId = e.target.value;
        if (styleId === "") return;
        const selectedStyle = citationStyles.find(
            (style) => style.id == styleId
        );

        // Check if already selected
        if (selectedCitationStyles.some((style) => style.id == styleId)) {
            setCitationStyleError("This citation style is already selected");
            return;
        }

        // Check if maximum number of citation styles is reached
        if (selectedCitationStyles.length >= maxCitationStyles) {
            setCitationStyleError(
                `You can select a maximum of ${maxCitationStyles} citation styles`
            );
            return;
        }

        setSelectedCitationStyles([...selectedCitationStyles, selectedStyle]);
        setCitationStyleError("");

        // Reset the select dropdown
        e.target.value = "";
    };

    const removeSubject = (subjectToRemove) => {
        setSelectedSubjects(
            selectedSubjects.filter(
                (subject) => subject.id !== subjectToRemove.id
            )
        );
        setSubjectError("");
    };

    const removeLanguage = (languageToRemove) => {
        setSelectedLanguages(
            selectedLanguages.filter(
                (language) => language.id !== languageToRemove.id
            )
        );
        setLanguageError("");
    };

    const removeCitationStyle = (styleToRemove) => {
        setSelectedCitationStyles(
            selectedCitationStyles.filter(
                (style) => style.id !== styleToRemove.id
            )
        );
        setCitationStyleError("");
    };

    const submit = (e) => {
        e.preventDefault();

        // Validate that at least one subject is selected
        if (selectedSubjects.length === 0) {
            setSubjectError("Please select at least one subject");
            return;
        }

        // Validate that at least one language is selected
        if (selectedLanguages.length === 0) {
            setLanguageError("Please select at least one language");
            return;
        }

        // Validate that at least one citation style is selected
        if (selectedCitationStyles.length === 0) {
            setCitationStyleError("Please select at least one citation style");
            return;
        }

        post(route("writer.register"), {
            onFinish: () => reset("password", "password_confirmation"),
        });
    };

    // Add this near the top of your component
    console.log("Form errors:", errors);

    return (
        <GuestLayout>
            <Head title="Writer Registration" />
            
            {/* Hero Section */}
            <div className="text-center mb-8">
                <div className="inline-flex items-center justify-center w-16 h-16 bg-gradient-to-r from-blue-500 to-cyan-500 rounded-full mb-4">
                    <svg className="w-8 h-8 text-white" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                        <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M15.232 5.232l3.536 3.536m-2.036-5.036a2.5 2.5 0 113.536 3.536L6.5 21.036H3v-3.572L16.732 3.732z" />
                    </svg>
                </div>
                <h1 className="text-3xl font-bold text-gray-900 dark:text-white mb-2">
                    Join Our Writing Team
                </h1>
                <p className="text-gray-600 dark:text-gray-400 max-w-md mx-auto">
                    Become part of our community of professional writers. Complete your profile and start your journey with us.
                </p>
            </div>

            {/* Main Form Container */}
            <div className="bg-gradient-to-r from-blue-100 to-cyan-100 dark:bg-none border border-gray-300 dark:border-gray-600 rounded-2xl p-8 shadow-xl">
                <form onSubmit={submit} className="space-y-6">
                    {/* Personal Information Section */}
                    <div className="space-y-6">
                        <div className="border-b border-gray-200 dark:border-gray-700 pb-4">
                            <h2 className="text-xl font-semibold text-gray-900 dark:text-white mb-2">
                                Personal Information
                            </h2>
                            <p className="text-sm text-gray-600 dark:text-gray-400">
                                Tell us about yourself
                            </p>
                        </div>

                        <div className="grid grid-cols-1 md:grid-cols-2 gap-6">
                            <div>
                                <InputLabel htmlFor="name" value="Full Name" className="text-gray-700 dark:text-gray-300 font-medium" />
                                <TextInput
                                    id="name"
                                    name="name"
                                    value={data.name}
                                    className="mt-2 block w-full rounded-lg border-gray-300 dark:border-gray-600 dark:bg-gray-800 dark:text-gray-300 focus:border-blue-500 dark:focus:border-blue-400 focus:ring-blue-500 dark:focus:ring-blue-400"
                                    autoComplete="name"
                                    isFocused={true}
                                    onChange={(e) => setData("name", e.target.value)}
                                    required
                                />
                                <InputError message={errors.name} className="mt-2" />
                            </div>

                            <div>
                                <InputLabel htmlFor="email" value="Email Address" className="text-gray-700 dark:text-gray-300 font-medium" />
                                <TextInput
                                    id="email"
                                    type="email"
                                    name="email"
                                    value={data.email}
                                    className="mt-2 block w-full rounded-lg border-gray-300 dark:border-gray-600 dark:bg-gray-800 dark:text-gray-300 focus:border-blue-500 dark:focus:border-blue-400 focus:ring-blue-500 dark:focus:ring-blue-400"
                                    autoComplete="username"
                                    onChange={(e) => setData("email", e.target.value)}
                                    required
                                />
                                <InputError message={errors.email} className="mt-2" />
                            </div>
                        </div>

                        <div className="grid grid-cols-1 md:grid-cols-2 gap-6">
                            <div>
                                <InputLabel htmlFor="password" value="Password" className="text-gray-700 dark:text-gray-300 font-medium" />
                                <TextInput
                                    id="password"
                                    type="password"
                                    name="password"
                                    value={data.password}
                                    className="mt-2 block w-full rounded-lg border-gray-300 dark:border-gray-600 dark:bg-gray-800 dark:text-gray-300 focus:border-blue-500 dark:focus:border-blue-400 focus:ring-blue-500 dark:focus:ring-blue-400"
                                    autoComplete="new-password"
                                    onChange={(e) => setData("password", e.target.value)}
                                    required
                                />
                                <InputError message={errors.password} className="mt-2" />
                            </div>

                            <div>
                                <InputLabel htmlFor="password_confirmation" value="Confirm Password" className="text-gray-700 dark:text-gray-300 font-medium" />
                                <TextInput
                                    id="password_confirmation"
                                    type="password"
                                    name="password_confirmation"
                                    value={data.password_confirmation}
                                    className="mt-2 block w-full rounded-lg border-gray-300 dark:border-gray-600 dark:bg-gray-800 dark:text-gray-300 focus:border-blue-500 dark:focus:border-blue-400 focus:ring-blue-500 dark:focus:ring-blue-400"
                                    autoComplete="new-password"
                                    onChange={(e) => setData("password_confirmation", e.target.value)}
                                    required
                                />
                                <InputError message={errors.password_confirmation} className="mt-2" />
                            </div>
                        </div>

                        <div>
                            <InputLabel htmlFor="experience_years" value="Years of Experience" className="text-gray-700 dark:text-gray-300 font-medium" />
                            <TextInput
                                id="experience_years"
                                type="number"
                                name="experience_years"
                                min="0"
                                value={data.experience_years}
                                className="mt-2 block w-full rounded-lg border-gray-300 dark:border-gray-600 dark:bg-gray-800 dark:text-gray-300 focus:border-blue-500 dark:focus:border-blue-400 focus:ring-blue-500 dark:focus:ring-blue-400"
                                onChange={(e) => setData("experience_years", e.target.value)}
                                required
                            />
                            <InputError message={errors.experience_years} className="mt-2" />
                        </div>

                        <div className="grid grid-cols-1 md:grid-cols-2 gap-6">
                            <div>
                                <InputLabel htmlFor="country_id" value="Country" className="text-gray-700 dark:text-gray-300 font-medium" />
                                <select
                                    id="country_id"
                                    name="country_id"
                                    value={data.country_id}
                                    className="mt-2 block w-full rounded-lg border-gray-300 dark:border-gray-600 dark:bg-gray-800 dark:text-gray-300 focus:border-blue-500 dark:focus:border-blue-400 focus:ring-blue-500 dark:focus:ring-blue-400"
                                    onChange={handleCountryChange}
                                    required
                                >
                                    <option value="">Select Country</option>
                                    {countries.map((country) => (
                                        <option key={country.id} value={country.id}>
                                            {country.name}
                                        </option>
                                    ))}
                                </select>
                                <InputError message={errors.country_id} className="mt-2" />
                            </div>

                            <div>
                                <InputLabel htmlFor="phone" value="Phone Number" className="text-gray-700 dark:text-gray-300 font-medium" />
                                <TextInput
                                    id="phone"
                                    name="phone"
                                    value={data.phone}
                                    className="mt-2 block w-full rounded-lg border-gray-300 dark:border-gray-600 dark:bg-gray-800 dark:text-gray-300 focus:border-blue-500 dark:focus:border-blue-400 focus:ring-blue-500 dark:focus:ring-blue-400"
                                    onChange={(e) => setData("phone", e.target.value)}
                                    required
                                />
                                <InputError message={errors.phone} className="mt-2" />
                            </div>
                        </div>

                        <div>
                            <InputLabel htmlFor="education_level_id" value="Education Level" className="text-gray-700 dark:text-gray-300 font-medium" />
                            <select
                                id="education_level_id"
                                name="education_level_id"
                                value={data.education_level_id}
                                className="mt-2 block w-full rounded-lg border-gray-300 dark:border-gray-600 dark:bg-gray-800 dark:text-gray-300 focus:border-blue-500 dark:focus:border-blue-400 focus:ring-blue-500 dark:focus:ring-blue-400"
                                onChange={(e) => setData("education_level_id", e.target.value)}
                                required
                            >
                                <option value="">Select Education Level</option>
                                {educationLevels.map((level) => (
                                    <option key={level.id} value={level.id}>
                                        {level.name}
                                    </option>
                                ))}
                            </select>
                            <InputError message={errors.education_level_id} className="mt-2" />
                        </div>
                    </div>

                    {/* Expertise Section */}
                    <div className="space-y-6">
                        <div className="border-b border-gray-200 dark:border-gray-700 pb-4">
                            <h2 className="text-xl font-semibold text-gray-900 dark:text-white mb-2">
                                Areas of Expertise
                            </h2>
                            <p className="text-sm text-gray-600 dark:text-gray-400">
                                Select your specializations and skills
                            </p>
                        </div>

                        {/* Subjects */}
                        <div>
                            <InputLabel htmlFor="subject_select" value={`Subjects of Expertise (Select up to ${maxSubjects})`} className="text-gray-700 dark:text-gray-300 font-medium" />
                            <select
                                id="subject_select"
                                name="subject_select"
                                className="mt-2 block w-full rounded-lg border-gray-300 dark:border-gray-600 dark:bg-gray-800 dark:text-gray-300 focus:border-blue-500 dark:focus:border-blue-400 focus:ring-blue-500 dark:focus:ring-blue-400"
                                onChange={handleSubjectChange}
                            >
                                <option value="">Select a Subject</option>
                                {subjects.map((subject) => (
                                    <option key={subject.id} value={subject.id}>
                                        {subject.name}
                                    </option>
                                ))}
                            </select>
                            
                            {/* Selected subjects tags */}
                            {selectedSubjects.length > 0 && (
                                <div className="mt-3 flex flex-wrap gap-2">
                                    {selectedSubjects.map((subject) => (
                                        <div
                                            key={subject.id}
                                            className="bg-blue-100 dark:bg-blue-900 text-blue-800 dark:text-blue-200 px-3 py-1 rounded-full flex items-center text-sm font-medium"
                                        >
                                            <span>{subject.name}</span>
                                            <button
                                                type="button"
                                                className="ml-2 text-blue-600 dark:text-blue-400 hover:text-blue-800 dark:hover:text-blue-200"
                                                onClick={() => removeSubject(subject)}
                                            >
                                                ×
                                            </button>
                                        </div>
                                    ))}
                                </div>
                            )}
                            
                            <div className="mt-2 text-sm text-gray-500 dark:text-gray-400">
                                {selectedSubjects.length > 0
                                    ? `${selectedSubjects.length} selected, ${maxSubjects - selectedSubjects.length} more allowed`
                                    : "No subjects selected yet"}
                            </div>
                            {subjectError && (
                                <div className="mt-2 text-sm text-red-600 dark:text-red-400">
                                    {subjectError}
                                </div>
                            )}
                            <InputError message={errors.subjects} className="mt-2" />
                        </div>

                        {/* Languages */}
                        <div>
                            <InputLabel htmlFor="language_select" value={`Languages (Select up to ${maxLanguages})`} className="text-gray-700 dark:text-gray-300 font-medium" />
                            <select
                                id="language_select"
                                name="language_select"
                                className="mt-2 block w-full rounded-lg border-gray-300 dark:border-gray-600 dark:bg-gray-800 dark:text-gray-300 focus:border-blue-500 dark:focus:border-blue-400 focus:ring-blue-500 dark:focus:ring-blue-400"
                                onChange={handleLanguageChange}
                            >
                                <option value="">Select a Language</option>
                                {languages.map((language) => (
                                    <option key={language.id} value={language.id}>
                                        {language.name}
                                    </option>
                                ))}
                            </select>
                            
                            {/* Selected languages tags */}
                            {selectedLanguages.length > 0 && (
                                <div className="mt-3 flex flex-wrap gap-2">
                                    {selectedLanguages.map((language) => (
                                        <div
                                            key={language.id}
                                            className="bg-cyan-100 dark:bg-cyan-900 text-cyan-800 dark:text-cyan-200 px-3 py-1 rounded-full flex items-center text-sm font-medium"
                                        >
                                            <span>{language.name}</span>
                                            <button
                                                type="button"
                                                className="ml-2 text-cyan-600 dark:text-cyan-400 hover:text-cyan-800 dark:hover:text-cyan-200"
                                                onClick={() => removeLanguage(language)}
                                            >
                                                ×
                                            </button>
                                        </div>
                                    ))}
                                </div>
                            )}
                            
                            <div className="mt-2 text-sm text-gray-500 dark:text-gray-400">
                                {selectedLanguages.length > 0
                                    ? `${selectedLanguages.length} selected, ${maxLanguages - selectedLanguages.length} more allowed`
                                    : "No languages selected yet"}
                            </div>
                            {languageError && (
                                <div className="mt-2 text-sm text-red-600 dark:text-red-400">
                                    {languageError}
                                </div>
                            )}
                            <InputError message={errors.languages} className="mt-2" />
                        </div>

                        {/* Citation Styles */}
                        <div>
                            <InputLabel htmlFor="citation_style_select" value={`Citation Styles (Select up to ${maxCitationStyles})`} className="text-gray-700 dark:text-gray-300 font-medium" />
                            <select
                                id="citation_style_select"
                                name="citation_style_select"
                                className="mt-2 block w-full rounded-lg border-gray-300 dark:border-gray-600 dark:bg-gray-800 dark:text-gray-300 focus:border-blue-500 dark:focus:border-blue-400 focus:ring-blue-500 dark:focus:ring-blue-400"
                                onChange={handleCitationStyleChange}
                            >
                                <option value="">Select a Citation Style</option>
                                {citationStyles.map((style) => (
                                    <option key={style.id} value={style.id}>
                                        {style.name}
                                    </option>
                                ))}
                            </select>
                            
                            {/* Selected citation styles tags */}
                            {selectedCitationStyles.length > 0 && (
                                <div className="mt-3 flex flex-wrap gap-2">
                                    {selectedCitationStyles.map((style) => (
                                        <div
                                            key={style.id}
                                            className="bg-indigo-100 dark:bg-indigo-900 text-indigo-800 dark:text-indigo-200 px-3 py-1 rounded-full flex items-center text-sm font-medium"
                                        >
                                            <span>{style.name}</span>
                                            <button
                                                type="button"
                                                className="ml-2 text-indigo-600 dark:text-indigo-400 hover:text-indigo-800 dark:hover:text-indigo-200"
                                                onClick={() => removeCitationStyle(style)}
                                            >
                                                ×
                                            </button>
                                        </div>
                                    ))}
                                </div>
                            )}
                            
                            <div className="mt-2 text-sm text-gray-500 dark:text-gray-400">
                                {selectedCitationStyles.length > 0
                                    ? `${selectedCitationStyles.length} selected, ${maxCitationStyles - selectedCitationStyles.length} more allowed`
                                    : "No citation styles selected yet"}
                            </div>
                            {citationStyleError && (
                                <div className="mt-2 text-sm text-red-600 dark:text-red-400">
                                    {citationStyleError}
                                </div>
                            )}
                            <InputError message={errors.citation_styles} className="mt-2" />

                            {/* Array-based errors */}
                            {Object.keys(errors)
                                .filter((key) => key.startsWith("citation_styles."))
                                .map((key) => (
                                    <div
                                        key={key}
                                        className="mt-2 text-sm text-red-600 dark:text-red-400"
                                    >
                                        {errors[key]}
                                    </div>
                                ))}
                        </div>
                    </div>

                    {/* Bio Section */}
                    <div className="space-y-6">
                        <div className="border-b border-gray-200 dark:border-gray-700 pb-4">
                            <h2 className="text-xl font-semibold text-gray-900 dark:text-white mb-2">
                                Professional Bio
                            </h2>
                            <p className="text-sm text-gray-600 dark:text-gray-400">
                                Tell us about your writing experience and expertise
                            </p>
                        </div>

                        <div>
                            <InputLabel htmlFor="bio" value="Professional Bio" className="text-gray-700 dark:text-gray-300 font-medium" />
                            <textarea
                                id="bio"
                                name="bio"
                                value={data.bio}
                                rows="4"
                                className="mt-2 block w-full rounded-lg border-gray-300 dark:border-gray-600 dark:bg-gray-800 dark:text-gray-300 focus:border-blue-500 dark:focus:border-blue-400 focus:ring-blue-500 dark:focus:ring-blue-400 resize-none"
                                onChange={(e) => setData("bio", e.target.value)}
                                placeholder="Describe your writing experience, areas of expertise, and what makes you a great writer..."
                                required
                            ></textarea>
                            <InputError message={errors.bio} className="mt-2" />
                        </div>
                    </div>

                    {/* Submit Section */}
                    <div className="flex flex-col sm:flex-row items-center justify-between pt-6 border-t border-gray-200 dark:border-gray-700">
                        <Link
                            href={route("login")}
                            className="text-sm text-gray-600 dark:text-gray-400 hover:text-blue-600 dark:hover:text-blue-400 transition-colors duration-200"
                        >
                            Already registered? Sign in here
                        </Link>
                        <PrimaryButton 
                            className="mt-4 sm:mt-0 px-8 py-3 bg-gradient-to-r from-blue-500 to-cyan-500 hover:from-blue-600 hover:to-cyan-600 focus:ring-2 focus:ring-blue-500 focus:ring-offset-2 transition-all duration-200 transform hover:scale-105" 
                            disabled={processing}
                        >
                            {processing ? (
                                <div className="flex items-center">
                                    <svg className="animate-spin -ml-1 mr-3 h-5 w-5 text-white" xmlns="http://www.w3.org/2000/svg" fill="none" viewBox="0 0 24 24">
                                        <circle className="opacity-25" cx="12" cy="12" r="10" stroke="currentColor" strokeWidth="4"></circle>
                                        <path className="opacity-75" fill="currentColor" d="M4 12a8 8 0 018-8V0C5.373 0 0 5.373 0 12h4zm2 5.291A7.962 7.962 0 014 12H0c0 3.042 1.135 5.824 3 7.938l3-2.647z"></path>
                                    </svg>
                                    Creating Account...
                                </div>
                            ) : (
                                "Create Writer Account"
                            )}
                        </PrimaryButton>
                    </div>
                </form>
            </div>
        </GuestLayout>
    );
}
