import { Head, Link, router } from "@inertiajs/react";
import { useState, useEffect } from "react";
import { motion, AnimatePresence } from "framer-motion";
import {
    Search,
    Filter,
    ArrowRight,
    Calendar,
    Clock,
    User,
    Tag,
    BookOpen,
    ChevronLeft,
    ChevronRight,
    Grid,
    List,
    SortAsc,
    SortDesc,
    // Add more icons for categories
    PenTool,
    FileText,
    GraduationCap,
    Lightbulb,
    Target,
    Award,
    Book,
    Edit,
    Bookmark,
} from "lucide-react";
import MainLayout from "@/Layouts/MainLayout";
import OptimizedImage from "@/Components/OptimizedImage";

// Icon mapping object
const iconMap = {
    "pen-tool": PenTool,
    "file-text": FileText,
    "graduation-cap": GraduationCap,
    lightbulb: Lightbulb,
    target: Target,
    award: Award,
    book: Book,
    "book-open": BookOpen,
    edit: Edit,
    bookmark: Bookmark,
    search: Search,
    user: User,
    tag: Tag,
    // Add more mappings as needed
};

export default function BlogCategory({ category, posts, categories, filters }) {
    // Ensure filters is always an object, not an array
    const safeFilters = (Array.isArray(filters) ? {} : filters) || {};

    // State management
    const [searchQuery, setSearchQuery] = useState(safeFilters.search || "");
    const [sortBy, setSortBy] = useState(safeFilters.sort || "newest");
    const [viewMode, setViewMode] = useState("grid");
    const [loading, setLoading] = useState(false);

    // Function to render icon
    const renderIcon = (iconName) => {
        if (!iconName) return null;

        // If it's an emoji, render directly
        if (
            iconName.length <= 2 &&
            /[\u{1F600}-\u{1F64F}]|[\u{1F300}-\u{1F5FF}]|[\u{1F680}-\u{1F6FF}]|[\u{2600}-\u{26FF}]|[\u{2700}-\u{27BF}]/u.test(
                iconName
            )
        ) {
            return <span className="text-2xl">{iconName}</span>;
        }

        // If it's a string, try to map to icon component
        const IconComponent =
            iconMap[iconName] || iconMap[iconName.toLowerCase()] || BookOpen;
        return <IconComponent className="w-6 h-6" />;
    };

    // Handle search with debounce
    useEffect(() => {
        const timer = setTimeout(() => {
            if (searchQuery !== (safeFilters.search || "")) {
                handleFilterChange();
            }
        }, 500);

        return () => clearTimeout(timer);
    }, [searchQuery]);

    // Handle filter changes
    const handleFilterChange = () => {
        setLoading(true);

        const params = {
            search: searchQuery || undefined,
            sort: sortBy !== "newest" ? sortBy : undefined,
        };

        // Remove undefined values
        Object.keys(params).forEach((key) => {
            if (params[key] === undefined) {
                delete params[key];
            }
        });

        router.get(`/blog/category/${category.slug}`, params, {
            preserveState: true,
            preserveScroll: true,
            onFinish: () => setLoading(false),
        });
    };

    // Handle sort change
    const handleSortChange = (sort) => {
        setSortBy(sort);
        setLoading(true);

        const params = {
            search: searchQuery || undefined,
            sort: sort !== "newest" ? sort : undefined,
        };

        Object.keys(params).forEach((key) => {
            if (params[key] === undefined) {
                delete params[key];
            }
        });

        router.get(`/blog/category/${category.slug}`, params, {
            preserveState: true,
            preserveScroll: true,
            onFinish: () => setLoading(false),
        });
    };

    // Format date helper
    const formatDate = (dateString) => {
        return new Date(dateString).toLocaleDateString("en-US", {
            year: "numeric",
            month: "long",
            day: "numeric",
        });
    };

    // Animation variants
    const fadeIn = {
        hidden: { opacity: 0, y: 20 },
        visible: { opacity: 1, y: 0, transition: { duration: 0.6 } },
    };

    const staggerContainer = {
        hidden: { opacity: 0 },
        visible: {
            opacity: 1,
            transition: { staggerChildren: 0.1 },
        },
    };

    const cardHover = {
        hover: {
            y: -5,
            transition: { duration: 0.2 },
        },
    };

    return (
        <MainLayout>
            <Head>
                <title>
                    {category.meta_title ||
                        `${category.name} - Academic Writing Blog`}
                </title>
                <meta
                    name="description"
                    content={category.meta_description || category.description}
                />
                <meta
                    name="keywords"
                    content={`${category.name}, academic writing, essay tips, study guides`}
                />
                <link
                    rel="canonical"
                    href={`${window.location.origin}/blog/category/${category.slug}`}
                />

                {/* Open Graph */}
                <meta
                    property="og:title"
                    content={
                        category.meta_title ||
                        `${category.name} - Academic Writing Blog`
                    }
                />
                <meta
                    property="og:description"
                    content={category.meta_description || category.description}
                />
                <meta property="og:type" content="website" />
                <meta
                    property="og:url"
                    content={`${window.location.origin}/blog/category/${category.slug}`}
                />
                <meta property="og:site_name" content="Academic Scribe" />
                <meta
                    property="og:image"
                    content="https://academicscribe.com/images/blog-category-og.jpg"
                />

                {/* Twitter Card */}
                <meta name="twitter:card" content="summary_large_image" />
                <meta name="twitter:site" content="@AcademicScribe" />
                <meta
                    name="twitter:title"
                    content={
                        category.meta_title ||
                        `${category.name} - Academic Writing Blog`
                    }
                />
                <meta
                    name="twitter:description"
                    content={category.meta_description || category.description}
                />
                <meta
                    name="twitter:image"
                    content="https://academicscribe.com/images/blog-category-twitter.jpg"
                />

                {/* Structured Data */}
                <script
                    type="application/ld+json"
                    dangerouslySetInnerHTML={{
                        __html: JSON.stringify({
                            "@context": "https://schema.org",
                            "@type": "CollectionPage",
                            name: category.name,
                            description: category.description,
                            url: `https://academicscribe.com/blog/category/${category.slug}`,
                            isPartOf: {
                                "@type": "Blog",
                                name: "Academic Scribe Blog",
                                url: "https://academicscribe.com/blog",
                            },
                        }),
                    }}
                />
            </Head>

            <div className="min-h-screen bg-gray-50 dark:bg-gray-900">
                {/* Breadcrumb */}
                <div className="bg-white dark:bg-gray-800 border-b border-gray-200 dark:border-gray-700">
                    <div className="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8 py-4">
                        <nav className="flex" aria-label="Breadcrumb">
                            <ol className="flex items-center space-x-2">
                                <li>
                                    <Link
                                        href="/"
                                        className="text-sm text-gray-500 dark:text-gray-400 hover:text-gray-700 dark:hover:text-gray-300"
                                    >
                                        Home
                                    </Link>
                                </li>
                                <li>
                                    <ChevronRight className="w-4 h-4 text-gray-400 mx-2" />
                                </li>
                                <li>
                                    <Link
                                        href="/blog"
                                        className="text-sm text-gray-500 dark:text-gray-400 hover:text-gray-700 dark:hover:text-gray-300"
                                    >
                                        Blog
                                    </Link>
                                </li>
                                <li>
                                    <ChevronRight className="w-4 h-4 text-gray-400 mx-2" />
                                </li>
                                <li>
                                    <span className="text-sm text-gray-900 dark:text-white font-medium">
                                        {category.name}
                                    </span>
                                </li>
                            </ol>
                        </nav>
                    </div>
                </div>

                {/* Category Header */}
                <div className="bg-white dark:bg-gray-800 border-b border-gray-200 dark:border-gray-700">
                    <div className="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8 py-12">
                        <motion.div
                            initial="hidden"
                            animate="visible"
                            variants={fadeIn}
                            className="text-center"
                        >
                            <div className="flex items-center justify-center mb-4">
                                {category.icon && (
                                    <div className="w-12 h-12 bg-blue-100 dark:bg-blue-900/30 rounded-lg flex items-center justify-center mr-4">
                                        {renderIcon(category.icon)}
                                    </div>
                                )}
                                <div className="text-left">
                                    <h1 className="text-4xl md:text-5xl font-bold text-gray-900 dark:text-white">
                                        {category.name}
                                    </h1>
                                    <p className="text-xl text-gray-600 dark:text-gray-300 mt-2">
                                        {category.description}
                                    </p>
                                </div>
                            </div>
                            <div className="flex items-center justify-center gap-4 text-sm text-gray-500 dark:text-gray-400">
                                <span>
                                    {posts.total}{" "}
                                    {posts.total === 1 ? "post" : "posts"}
                                </span>
                                {category.color && (
                                    <span
                                        className="w-3 h-3 rounded-full"
                                        style={{
                                            backgroundColor: category.color,
                                        }}
                                    />
                                )}
                            </div>
                        </motion.div>
                    </div>
                </div>

                {/* Main Content */}
                <div className="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8 py-12">
                    <div className="grid grid-cols-1 lg:grid-cols-4 gap-8">
                        {/* Sidebar */}
                        <div className="lg:col-span-1">
                            <motion.div
                                initial="hidden"
                                animate="visible"
                                variants={fadeIn}
                                className="sticky top-8 space-y-6"
                            >
                                {/* Search */}
                                <div className="bg-white dark:bg-gray-800 rounded-xl shadow-sm p-6">
                                    <h3 className="font-semibold text-gray-900 dark:text-white mb-4">
                                        Search in {category.name}
                                    </h3>
                                    <div className="relative">
                                        <Search className="absolute left-3 top-1/2 transform -translate-y-1/2 text-gray-400 w-4 h-4" />
                                        <input
                                            type="text"
                                            placeholder="Search articles..."
                                            value={searchQuery}
                                            onChange={(e) =>
                                                setSearchQuery(e.target.value)
                                            }
                                            className="w-full pl-10 pr-4 py-2 border border-gray-300 dark:border-gray-600 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-transparent dark:bg-gray-700 dark:text-white"
                                        />
                                    </div>
                                </div>

                                {/* Other Categories */}
                                <div className="bg-white dark:bg-gray-800 rounded-xl shadow-sm p-6">
                                    <h3 className="font-semibold text-gray-900 dark:text-white mb-4">
                                        Other Categories
                                    </h3>
                                    <div className="space-y-2">
                                        {categories
                                            .filter(
                                                (cat) =>
                                                    cat.slug !== category.slug
                                            )
                                            .map((cat) => (
                                                <Link
                                                    key={cat.id}
                                                    href={`/blog/category/${cat.slug}`}
                                                    className="block px-3 py-2 rounded-lg text-sm text-gray-600 dark:text-gray-400 hover:bg-gray-100 dark:hover:bg-gray-700 transition-colors"
                                                >
                                                    <div className="flex items-center justify-between">
                                                        <span>{cat.name}</span>
                                                        <span className="text-xs bg-gray-200 dark:bg-gray-600 px-2 py-1 rounded-full">
                                                            {cat.published_posts_count ||
                                                                0}
                                                        </span>
                                                    </div>
                                                </Link>
                                            ))}
                                    </div>
                                </div>

                                {/* Sort Options */}
                                <div className="bg-white dark:bg-gray-800 rounded-xl shadow-sm p-6">
                                    <h3 className="font-semibold text-gray-900 dark:text-white mb-4">
                                        Sort By
                                    </h3>
                                    <div className="space-y-2">
                                        <button
                                            onClick={() =>
                                                handleSortChange("newest")
                                            }
                                            className={`w-full text-left px-3 py-2 rounded-lg text-sm transition-colors flex items-center ${
                                                sortBy === "newest"
                                                    ? "bg-blue-100 text-blue-800 dark:bg-blue-900/30 dark:text-blue-400"
                                                    : "text-gray-600 dark:text-gray-400 hover:bg-gray-100 dark:hover:bg-gray-700"
                                            }`}
                                        >
                                            <SortDesc className="w-4 h-4 mr-2" />
                                            Newest First
                                        </button>
                                        <button
                                            onClick={() =>
                                                handleSortChange("oldest")
                                            }
                                            className={`w-full text-left px-3 py-2 rounded-lg text-sm transition-colors flex items-center ${
                                                sortBy === "oldest"
                                                    ? "bg-blue-100 text-blue-800 dark:bg-blue-900/30 dark:text-blue-400"
                                                    : "text-gray-600 dark:text-gray-400 hover:bg-gray-100 dark:hover:bg-gray-700"
                                            }`}
                                        >
                                            <SortAsc className="w-4 h-4 mr-2" />
                                            Oldest First
                                        </button>
                                    </div>
                                </div>
                            </motion.div>
                        </div>

                        {/* Posts Grid */}
                        <div className="lg:col-span-3">
                            <motion.div
                                initial="hidden"
                                animate="visible"
                                variants={staggerContainer}
                            >
                                {/* Results Header */}
                                <motion.div variants={fadeIn} className="mb-8">
                                    <div className="flex items-center justify-between">
                                        <div>
                                            <h2 className="text-2xl font-bold text-gray-900 dark:text-white">
                                                {searchQuery
                                                    ? `Search Results for "${searchQuery}"`
                                                    : `${category.name} Posts`}
                                            </h2>
                                            <p className="text-gray-600 dark:text-gray-400 mt-1">
                                                {posts.total}{" "}
                                                {posts.total === 1
                                                    ? "post"
                                                    : "posts"}{" "}
                                                found
                                            </p>
                                        </div>
                                        <div className="flex items-center gap-2">
                                            {loading && (
                                                <div className="animate-spin rounded-full h-5 w-5 border-b-2 border-blue-600"></div>
                                            )}
                                            <div className="flex items-center bg-white dark:bg-gray-800 rounded-lg border border-gray-200 dark:border-gray-700 p-1">
                                                <button
                                                    onClick={() =>
                                                        setViewMode("grid")
                                                    }
                                                    className={`p-2 rounded-md transition-colors ${
                                                        viewMode === "grid"
                                                            ? "bg-blue-100 text-blue-600 dark:bg-blue-900/30 dark:text-blue-400"
                                                            : "text-gray-400 hover:text-gray-600 dark:hover:text-gray-300"
                                                    }`}
                                                >
                                                    <Grid className="w-4 h-4" />
                                                </button>
                                                <button
                                                    onClick={() =>
                                                        setViewMode("list")
                                                    }
                                                    className={`p-2 rounded-md transition-colors ${
                                                        viewMode === "list"
                                                            ? "bg-blue-100 text-blue-600 dark:bg-blue-900/30 dark:text-blue-400"
                                                            : "text-gray-400 hover:text-gray-600 dark:hover:text-gray-300"
                                                    }`}
                                                >
                                                    <List className="w-4 h-4" />
                                                </button>
                                            </div>
                                        </div>
                                    </div>
                                </motion.div>

                                {/* Posts Grid/List */}
                                {posts.data && posts.data.length > 0 ? (
                                    <div
                                        className={`mb-8 ${
                                            viewMode === "grid"
                                                ? "grid grid-cols-1 md:grid-cols-2 gap-6"
                                                : "space-y-6"
                                        }`}
                                    >
                                        {posts.data.map((post) => (
                                            <motion.article
                                                key={post.id}
                                                variants={fadeIn}
                                                whileHover="hover"
                                                className="group"
                                            >
                                                <motion.div
                                                    variants={cardHover}
                                                    className={`bg-white dark:bg-gray-800 rounded-xl shadow-sm overflow-hidden border border-gray-200 dark:border-gray-700 hover:shadow-md transition-shadow duration-300 ${
                                                        viewMode === "list"
                                                            ? "flex"
                                                            : ""
                                                    }`}
                                                >
                                                    {post.featured_image && (
                                                        <div
                                                            className={`relative ${
                                                                viewMode ===
                                                                "list"
                                                                    ? "w-48 flex-shrink-0"
                                                                    : ""
                                                            }`}
                                                        >
                                                            <OptimizedImage
                                                                src={
                                                                    post.featured_image
                                                                }
                                                                alt={
                                                                    post.featured_image_alt ||
                                                                    post.title
                                                                }
                                                                width={800}
                                                                height={450}
                                                                className={`object-cover ${
                                                                    viewMode ===
                                                                    "list"
                                                                        ? "w-full h-full"
                                                                        : "w-full h-48"
                                                                }`}
                                                            />
                                                            <div className="absolute top-4 left-4">
                                                                <span className="inline-flex items-center px-2 py-1 rounded-full text-xs font-medium bg-white/90 text-gray-700">
                                                                    {post.difficulty ||
                                                                        "beginner"}
                                                                </span>
                                                            </div>
                                                        </div>
                                                    )}
                                                    <div
                                                        className={`p-6 ${
                                                            viewMode ===
                                                            "list"
                                                                ? "flex-1"
                                                                : ""
                                                        }`}
                                                    >
                                                        <div className="flex items-center gap-2 mb-3">
                                                            <span className="text-xs text-gray-500 dark:text-gray-400">
                                                                <Clock className="w-3 h-3 mr-1 inline" />
                                                                {post.read_time ||
                                                                    5}{" "}
                                                                min read
                                                            </span>
                                                            <span className="text-xs text-gray-400">
                                                                •
                                                            </span>
                                                            <span className="text-xs text-gray-500 dark:text-gray-400">
                                                                <Calendar className="w-3 h-3 mr-1 inline" />
                                                                {formatDate(
                                                                    post.published_at
                                                                )}
                                                            </span>
                                                        </div>

                                                        <Link href={`/blog/${post.slug}`}>
                                                            <h3
                                                                className={`font-semibold text-gray-900 dark:text-white mb-2 hover:text-blue-600 dark:hover:text-blue-400 transition-colors ${
                                                                    viewMode ===
                                                                    "list"
                                                                        ? "text-xl line-clamp-2"
                                                                        : "text-lg line-clamp-2"
                                                                }`}
                                                            >
                                                                {post.title}
                                                            </h3>
                                                        </Link>

                                                        <p
                                                            className={`text-gray-600 dark:text-gray-400 mb-4 ${
                                                                viewMode ===
                                                                "list"
                                                                    ? "text-base line-clamp-2"
                                                                    : "text-sm line-clamp-3"
                                                            }`}
                                                        >
                                                            {post.excerpt}
                                                        </p>

                                                        {post.tags &&
                                                            post.tags
                                                                .length >
                                                                0 && (
                                                                <div className="flex flex-wrap gap-1 mb-4">
                                                                    {post.tags
                                                                        .slice(
                                                                            0,
                                                                            3
                                                                        )
                                                                        .map(
                                                                            (
                                                                                tag
                                                                            ) => (
                                                                                <span
                                                                                    key={
                                                                                        tag.id
                                                                                    }
                                                                                    className="inline-flex items-center px-2 py-1 rounded-md text-xs font-medium bg-blue-100 text-blue-800 dark:bg-blue-900/30 dark:text-blue-400"
                                                                                >
                                                                                    <Tag className="w-3 h-3 mr-1" />
                                                                                    {
                                                                                        tag.name
                                                                                    }
                                                                                </span>
                                                                            )
                                                                        )}
                                                                    {post
                                                                        .tags
                                                                        .length >
                                                                        3 && (
                                                                        <span className="inline-flex items-center px-2 py-1 rounded-md text-xs font-medium bg-gray-100 text-gray-700 dark:bg-gray-700 dark:text-gray-300">
                                                                            +
                                                                            {post
                                                                                .tags
                                                                                .length -
                                                                                3}{" "}
                                                                            more
                                                                        </span>
                                                                    )}
                                                                </div>
                                                            )}

                                                        <div className="flex items-center justify-between">
                                                            <div className="flex items-center text-xs text-gray-500 dark:text-gray-400">
                                                                <User className="w-3 h-3 mr-1" />
                                                                {post.author
                                                                    ?.name ||
                                                                    "Academic Scribe"}
                                                            </div>
                                                            <ArrowRight className="w-4 h-4 text-blue-600 dark:text-blue-400 group-hover:translate-x-1 transition-transform" />
                                                        </div>
                                                    </div>
                                                </motion.div>
                                            </motion.article>
                                        ))}
                                    </div>
                                ) : (
                                    <motion.div
                                        variants={fadeIn}
                                        className="text-center py-12"
                                    >
                                        <BookOpen className="w-16 h-16 text-gray-400 mx-auto mb-4" />
                                        <h3 className="text-xl font-semibold text-gray-900 dark:text-white mb-2">
                                            No posts found
                                        </h3>
                                        <p className="text-gray-600 dark:text-gray-400 mb-6">
                                            {searchQuery
                                                ? "Try adjusting your search terms or browse other categories"
                                                : `No posts available in ${category.name} category at the moment`}
                                        </p>
                                        <div className="flex justify-center gap-4">
                                            {searchQuery && (
                                                <button
                                                    onClick={() =>
                                                        setSearchQuery("")
                                                    }
                                                    className="px-4 py-2 bg-blue-600 text-white rounded-lg hover:bg-blue-700 transition-colors"
                                                >
                                                    Clear Search
                                                </button>
                                            )}
                                            <Link
                                                href="/blog"
                                                className="px-4 py-2 bg-gray-600 text-white rounded-lg hover:bg-gray-700 transition-colors"
                                            >
                                                Browse All Posts
                                            </Link>
                                        </div>
                                    </motion.div>
                                )}

                                {/* Pagination */}
                                {posts.data && posts.data.length > 0 && (
                                    <motion.div
                                        variants={fadeIn}
                                        className="flex justify-center"
                                    >
                                        <div className="flex items-center space-x-2">
                                            {posts.prev_page_url && (
                                                <Link
                                                    href={posts.prev_page_url}
                                                    preserveState
                                                    className="flex items-center px-4 py-2 text-sm font-medium text-gray-500 bg-white dark:bg-gray-800 border border-gray-300 dark:border-gray-600 rounded-lg hover:bg-gray-50 dark:hover:bg-gray-700 transition-colors"
                                                >
                                                    <ChevronLeft className="w-4 h-4 mr-1" />
                                                    Previous
                                                </Link>
                                            )}

                                            <div className="flex items-center space-x-1">
                                                {Array.from(
                                                    { length: posts.last_page },
                                                    (_, i) => i + 1
                                                ).map((page) => (
                                                    <Link
                                                        key={page}
                                                        href={`/blog/category/${category.slug}?page=${page}`}
                                                        preserveState
                                                        className={`px-3 py-2 text-sm font-medium rounded-lg transition-colors ${
                                                            page ===
                                                            posts.current_page
                                                                ? "bg-blue-600 text-white"
                                                                : "text-gray-700 dark:text-gray-300 bg-white dark:bg-gray-800 border border-gray-300 dark:border-gray-600 hover:bg-gray-50 dark:hover:bg-gray-700"
                                                        }`}
                                                    >
                                                        {page}
                                                    </Link>
                                                ))}
                                            </div>

                                            {posts.next_page_url && (
                                                <Link
                                                    href={posts.next_page_url}
                                                    preserveState
                                                    className="flex items-center px-4 py-2 text-sm font-medium text-gray-500 bg-white dark:bg-gray-800 border border-gray-300 dark:border-gray-600 rounded-lg hover:bg-gray-50 dark:hover:bg-gray-700 transition-colors"
                                                >
                                                    Next
                                                    <ChevronRight className="w-4 h-4 ml-1" />
                                                </Link>
                                            )}
                                        </div>
                                    </motion.div>
                                )}
                            </motion.div>
                        </div>
                    </div>
                </div>
            </div>
        </MainLayout>
    );
}
