import { Head, Link, router } from "@inertiajs/react";
import { useState, useEffect, useMemo, useCallback } from "react";
import { motion, AnimatePresence } from "framer-motion";
import {
    Search,
    Filter,
    ArrowRight,
    Calendar,
    Clock,
    User,
    Tag,
    TrendingUp,
    BookOpen,
    ChevronLeft,
    ChevronRight,
} from "lucide-react";
import MainLayout from "@/Layouts/MainLayout";
import OptimizedImage from "@/Components/OptimizedImage";

export default function BlogIndex({
    posts,
    categories,
    featured,
    meta,
    filters = {},
}) {
    // State management
    const [searchQuery, setSearchQuery] = useState(filters.search || "");
    const [selectedCategory, setSelectedCategory] = useState(
        filters.category || "all"
    );
    const [sortBy, setSortBy] = useState(filters.sort || "newest");
    const [isFilterOpen, setIsFilterOpen] = useState(false);
    const [loading, setLoading] = useState(false);

    // Unified filter handler
    const applyFilters = useCallback(
        (params = {}) => {
            setLoading(true);

            // Get current values or use provided params
            const filterParams = {
                search:
                    params.search !== undefined ? params.search : searchQuery,
                category:
                    params.category !== undefined
                        ? params.category
                        : selectedCategory,
                sort: params.sort !== undefined ? params.sort : sortBy,
            };

            // Clean up the params - remove empty values
            const cleanParams = {};
            Object.keys(filterParams).forEach((key) => {
                const value = filterParams[key];
                if (
                    value &&
                    value !== "all" &&
                    value !== "newest" &&
                    value.toString().trim() !== ""
                ) {
                    cleanParams[key] = value;
                }
            });

            router.get("/blog", cleanParams, {
                preserveState: true,
                preserveScroll: true,
                onFinish: () => setLoading(false),
                onError: (errors) => {
                    console.error("Filter error:", errors);
                    setLoading(false);
                },
            });
        },
        [searchQuery, selectedCategory, sortBy]
    );

    // Handle search with debounce
    useEffect(() => {
        const timer = setTimeout(() => {
            applyFilters({ search: searchQuery });
        }, 500);

        return () => clearTimeout(timer);
    }, [searchQuery]); // Only depend on searchQuery

    // Handle category filter
    const handleCategoryChange = useCallback(
        (category) => {
            setSelectedCategory(category);
            applyFilters({ category });
        },
        [applyFilters]
    );

    // Handle sort change
    const handleSortChange = useCallback(
        (sort) => {
            setSortBy(sort);
            applyFilters({ sort });
        },
        [applyFilters]
    );

    // Clear all filters
    const clearFilters = useCallback(() => {
        setSearchQuery("");
        setSelectedCategory("all");
        setSortBy("newest");

        router.get(
            "/blog",
            {},
            {
                preserveState: true,
                preserveScroll: true,
                onFinish: () => setLoading(false),
            }
        );
    }, []);

    // Format date helper
    const formatDate = (dateString) => {
        return new Date(dateString).toLocaleDateString("en-US", {
            year: "numeric",
            month: "long",
            day: "numeric",
        });
    };

    // Animation variants
    const fadeIn = {
        hidden: { opacity: 0, y: 20 },
        visible: { opacity: 1, y: 0, transition: { duration: 0.6 } },
    };

    const staggerContainer = {
        hidden: { opacity: 0 },
        visible: {
            opacity: 1,
            transition: { staggerChildren: 0.1 },
        },
    };

    const cardHover = {
        hover: {
            y: -5,
            transition: { duration: 0.2 },
        },
    };

    return (
        <MainLayout>
            <Head>
                <title>{meta.title}</title>
                <meta name="description" content={meta.description} />
                <meta
                    name="keywords"
                    content="academic writing blog, essay tips, writing guides, student resources"
                />
                <link rel="canonical" href={`${window.location.origin}/blog`} />

                {/* Open Graph */}
                <meta property="og:title" content={meta.title} />
                <meta property="og:description" content={meta.description} />
                <meta property="og:type" content="website" />
                <meta
                    property="og:url"
                    content={`${window.location.origin}/blog`}
                />
                <meta property="og:site_name" content="Academic Scribe" />
                <meta
                    property="og:image"
                    content="https://academicscribe.com/images/blog-og-image.jpg"
                />

                {/* Twitter Card */}
                <meta name="twitter:card" content="summary_large_image" />
                <meta name="twitter:site" content="@AcademicScribe" />
                <meta name="twitter:title" content={meta.title} />
                <meta name="twitter:description" content={meta.description} />
                <meta
                    name="twitter:image"
                    content="https://academicscribe.com/images/blog-twitter-image.jpg"
                />

                {/* Structured Data */}
                <script
                    type="application/ld+json"
                    dangerouslySetInnerHTML={{
                        __html: JSON.stringify({
                            "@context": "https://schema.org",
                            "@type": "Blog",
                            name: "Academic Scribe Blog",
                            description:
                                "Expert academic writing tips, guides, and resources for students",
                            url: "https://academicscribe.com/blog",
                            publisher: {
                                "@type": "Organization",
                                name: "Academic Scribe",
                                url: "https://academicscribe.com",
                            },
                        }),
                    }}
                />
            </Head>

            <div className="min-h-screen bg-gray-50 dark:bg-gray-900">
                {/* Hero Section */}
                <div className="bg-white dark:bg-gray-800 border-b border-gray-200 dark:border-gray-700">
                    <div className="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8 py-12">
                        <motion.div
                            initial="hidden"
                            animate="visible"
                            variants={fadeIn}
                            className="text-center"
                        >
                            <h1 className="text-4xl md:text-5xl font-bold text-gray-900 dark:text-white mb-4">
                                Academic Writing Blog
                            </h1>
                            <p className="text-xl text-gray-600 dark:text-gray-300 max-w-3xl mx-auto">
                                Discover expert tips, guides, and strategies to
                                excel in your academic writing journey
                            </p>
                        </motion.div>
                    </div>
                </div>

                {/* Featured Posts */}
                {featured && featured.length > 0 && (
                    <div className="bg-white dark:bg-gray-800 border-b border-gray-200 dark:border-gray-700">
                        <div className="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8 py-12">
                            <motion.div
                                initial="hidden"
                                animate="visible"
                                variants={staggerContainer}
                            >
                                <motion.div variants={fadeIn} className="mb-8">
                                    <h2 className="text-2xl font-bold text-gray-900 dark:text-white mb-2">
                                        Featured Posts
                                    </h2>
                                    <div className="w-20 h-1 bg-blue-600 rounded-full"></div>
                                </motion.div>

                                <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-3 gap-6">
                                    {featured.map((post) => (
                                        <motion.article
                                            key={post.id}
                                            variants={fadeIn}
                                            whileHover="hover"
                                            className="group"
                                        >
                                            <motion.div
                                                variants={cardHover}
                                                className="bg-white dark:bg-gray-800 rounded-xl shadow-sm overflow-hidden border border-gray-200 dark:border-gray-700 hover:shadow-md transition-shadow duration-300"
                                            >
                                                {post.featured_image && (
                                                    <div className="relative">
                                                        <OptimizedImage
                                                            src={
                                                                post.featured_image
                                                            }
                                                            alt={
                                                                post.featured_image_alt ||
                                                                post.title
                                                            }
                                                            width={800}
                                                            height={450}
                                                            className="w-full h-48 object-cover"
                                                        />
                                                        <div className="absolute top-4 left-4">
                                                            <span className="inline-flex items-center px-2 py-1 rounded-full text-xs font-medium bg-blue-600 text-white">
                                                                <TrendingUp className="w-3 h-3 mr-1" />
                                                                Featured
                                                            </span>
                                                        </div>
                                                    </div>
                                                )}
                                                <div className="p-6">
                                                    <div className="flex items-center gap-2 mb-3">
                                                        <span className="inline-flex items-center px-2 py-1 rounded-md text-xs font-medium bg-gray-100 text-gray-700 dark:bg-gray-700 dark:text-gray-300">
                                                            {
                                                                post
                                                                    .category
                                                                    ?.name
                                                            }
                                                        </span>
                                                        <span className="text-xs text-gray-500 dark:text-gray-400">
                                                            {post.read_time ||
                                                                5}{" "}
                                                            min read
                                                        </span>
                                                    </div>
                                                    <Link href={`/blog/${post.slug}`}>
                                                        <h3 className="text-lg font-semibold text-gray-900 dark:text-white mb-2 hover:text-blue-600 dark:hover:text-blue-400 transition-colors line-clamp-2">
                                                            {post.title}
                                                        </h3>
                                                    </Link>
                                                    <p className="text-gray-600 dark:text-gray-400 text-sm mb-4 line-clamp-3">
                                                        {post.excerpt}
                                                    </p>
                                                    <div className="flex items-center justify-between">
                                                        <div className="flex items-center text-xs text-gray-500 dark:text-gray-400">
                                                            <Calendar className="w-3 h-3 mr-1" />
                                                            {formatDate(
                                                                post.published_at
                                                            )}
                                                        </div>
                                                        <ArrowRight className="w-4 h-4 text-blue-600 dark:text-blue-400 group-hover:translate-x-1 transition-transform" />
                                                    </div>
                                                </div>
                                            </motion.div>
                                        </motion.article>
                                    ))}
                                </div>
                            </motion.div>
                        </div>
                    </div>
                )}

                {/* Main Content */}
                <div className="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8 py-12">
                    <div className="grid grid-cols-1 lg:grid-cols-4 gap-8">
                        {/* Sidebar */}
                        <div className="lg:col-span-1">
                            <motion.div
                                initial="hidden"
                                animate="visible"
                                variants={fadeIn}
                                className="sticky top-8 space-y-6"
                            >
                                {/* Search */}
                                <div className="bg-white dark:bg-gray-800 rounded-xl shadow-sm p-6">
                                    <h3 className="font-semibold text-gray-900 dark:text-white mb-4">
                                        Search Posts
                                    </h3>
                                    <div className="relative">
                                        <Search className="absolute left-3 top-1/2 transform -translate-y-1/2 text-gray-400 w-4 h-4" />
                                        <input
                                            type="search"
                                            placeholder="Search articles..."
                                            value={searchQuery}
                                            onChange={(e) =>
                                                setSearchQuery(e.target.value)
                                            }
                                            className="w-full pl-10 pr-4 py-2 border border-gray-300 dark:border-gray-600 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-transparent dark:bg-gray-700 dark:text-white"
                                        />
                                    </div>
                                </div>

                                {/* Categories */}
                                <div className="bg-white dark:bg-gray-800 rounded-xl shadow-sm p-6">
                                    <h3 className="font-semibold text-gray-900 dark:text-white mb-4">
                                        Categories
                                    </h3>
                                    <div className="space-y-2">
                                        <button
                                            onClick={() =>
                                                handleCategoryChange("all")
                                            }
                                            disabled={loading}
                                            className={`w-full text-left px-3 py-2 rounded-lg text-sm transition-colors disabled:opacity-50 ${
                                                selectedCategory === "all"
                                                    ? "bg-blue-100 text-blue-800 dark:bg-blue-900/30 dark:text-blue-400"
                                                    : "text-gray-600 dark:text-gray-400 hover:bg-gray-100 dark:hover:bg-gray-700"
                                            }`}
                                        >
                                            All Categories
                                        </button>
                                        {categories &&
                                            categories.map((category) => (
                                                <button
                                                    key={category.id}
                                                    onClick={() =>
                                                        handleCategoryChange(
                                                            category.slug
                                                        )
                                                    }
                                                    disabled={loading}
                                                    className={`w-full text-left px-3 py-2 rounded-lg text-sm transition-colors flex items-center justify-between disabled:opacity-50 ${
                                                        selectedCategory ===
                                                        category.slug
                                                            ? "bg-blue-100 text-blue-800 dark:bg-blue-900/30 dark:text-blue-400"
                                                            : "text-gray-600 dark:text-gray-400 hover:bg-gray-100 dark:hover:bg-gray-700"
                                                    }`}
                                                >
                                                    <span>{category.name}</span>
                                                    <span className="text-xs bg-gray-200 dark:bg-gray-600 px-2 py-1 rounded-full">
                                                        {category.published_posts_count ||
                                                            0}
                                                    </span>
                                                </button>
                                            ))}
                                    </div>
                                </div>

                                {/* Sort Options */}
                                <div className="bg-white dark:bg-gray-800 rounded-xl shadow-sm p-6">
                                    <h3 className="font-semibold text-gray-900 dark:text-white mb-4">
                                        Sort By
                                    </h3>
                                    <div className="space-y-2">
                                        <button
                                            onClick={() =>
                                                handleSortChange("newest")
                                            }
                                            disabled={loading}
                                            className={`w-full text-left px-3 py-2 rounded-lg text-sm transition-colors disabled:opacity-50 ${
                                                sortBy === "newest"
                                                    ? "bg-blue-100 text-blue-800 dark:bg-blue-900/30 dark:text-blue-400"
                                                    : "text-gray-600 dark:text-gray-400 hover:bg-gray-100 dark:hover:bg-gray-700"
                                            }`}
                                        >
                                            Newest First
                                        </button>
                                        <button
                                            onClick={() =>
                                                handleSortChange("oldest")
                                            }
                                            disabled={loading}
                                            className={`w-full text-left px-3 py-2 rounded-lg text-sm transition-colors disabled:opacity-50 ${
                                                sortBy === "oldest"
                                                    ? "bg-blue-100 text-blue-800 dark:bg-blue-900/30 dark:text-blue-400"
                                                    : "text-gray-600 dark:text-gray-400 hover:bg-gray-100 dark:hover:bg-gray-700"
                                            }`}
                                        >
                                            Oldest First
                                        </button>
                                        <button
                                            onClick={() =>
                                                handleSortChange("popular")
                                            }
                                            disabled={loading}
                                            className={`w-full text-left px-3 py-2 rounded-lg text-sm transition-colors disabled:opacity-50 ${
                                                sortBy === "popular"
                                                    ? "bg-blue-100 text-blue-800 dark:bg-blue-900/30 dark:text-blue-400"
                                                    : "text-gray-600 dark:text-gray-400 hover:bg-gray-100 dark:hover:bg-gray-700"
                                            }`}
                                        >
                                            Most Popular
                                        </button>
                                    </div>
                                </div>

                                {/* Clear Filters */}
                                {(searchQuery ||
                                    selectedCategory !== "all" ||
                                    sortBy !== "newest") && (
                                    <div className="bg-white dark:bg-gray-800 rounded-xl shadow-sm p-6">
                                        <button
                                            onClick={clearFilters}
                                            disabled={loading}
                                            className="w-full px-4 py-2 bg-red-100 text-red-800 dark:bg-red-900/30 dark:text-red-400 rounded-lg hover:bg-red-200 dark:hover:bg-red-900/50 transition-colors text-sm font-medium disabled:opacity-50"
                                        >
                                            Clear All Filters
                                        </button>
                                    </div>
                                )}
                            </motion.div>
                        </div>

                        {/* Posts Grid */}
                        <div className="lg:col-span-3">
                            <motion.div
                                initial="hidden"
                                animate="visible"
                                variants={staggerContainer}
                            >
                                {/* Results Header */}
                                <motion.div variants={fadeIn} className="mb-8">
                                    <div className="flex items-center justify-between">
                                        <div>
                                            <h2 className="text-2xl font-bold text-gray-900 dark:text-white">
                                                {searchQuery
                                                    ? `Search Results for "${searchQuery}"`
                                                    : selectedCategory !== "all"
                                                    ? `Posts in ${
                                                          categories?.find(
                                                              (c) =>
                                                                  c.slug ===
                                                                  selectedCategory
                                                          )?.name ||
                                                          selectedCategory
                                                      }`
                                                    : "Latest Posts"}
                                            </h2>
                                            <p className="text-gray-600 dark:text-gray-400 mt-1">
                                                {posts?.total || 0}{" "}
                                                {(posts?.total || 0) === 1
                                                    ? "post"
                                                    : "posts"}{" "}
                                                found
                                            </p>
                                        </div>
                                        <div className="flex items-center gap-2">
                                            {loading && (
                                                <div className="animate-spin rounded-full h-5 w-5 border-b-2 border-blue-600"></div>
                                            )}
                                            <Filter className="w-5 h-5 text-gray-400" />
                                        </div>
                                    </div>
                                </motion.div>

                                {/* Posts Grid */}
                                {posts?.data && posts.data.length > 0 ? (
                                    <div className="grid grid-cols-1 md:grid-cols-2 gap-6 mb-8">
                                        {posts.data.map((post) => (
                                            <motion.article
                                                key={post.id}
                                                variants={fadeIn}
                                                whileHover="hover"
                                                className="group"
                                            >
                                                <motion.div
                                                    variants={cardHover}
                                                    className="bg-white dark:bg-gray-800 rounded-xl shadow-sm overflow-hidden border border-gray-200 dark:border-gray-700 hover:shadow-md transition-shadow duration-300"
                                                >
                                                    {post.featured_image && (
                                                        <div className="relative">
                                                                <OptimizedImage
                                                                    src={
                                                                        post.featured_image
                                                                    }
                                                                    alt={
                                                                        post.featured_image_alt ||
                                                                        post.title
                                                                    }
                                                                    width={800}
                                                                    height={450}
                                                                    className="w-full h-48 object-cover"
                                                                />
                                                            <div className="absolute top-4 left-4">
                                                                <span className="inline-flex items-center px-2 py-1 rounded-full text-xs font-medium bg-white/90 text-gray-700">
                                                                    {
                                                                        post
                                                                            .category
                                                                            ?.name
                                                                    }
                                                                </span>
                                                            </div>
                                                        </div>
                                                    )}
                                                    <div className="p-6">
                                                        <div className="flex items-center gap-2 mb-3">
                                                            <span className="inline-flex items-center px-2 py-1 rounded-md text-xs font-medium bg-gray-100 text-gray-700 dark:bg-gray-700 dark:text-gray-300">
                                                                <BookOpen className="w-3 h-3 mr-1" />
                                                                {post.difficulty ||
                                                                    "beginner"}
                                                            </span>
                                                            <span className="text-xs text-gray-500 dark:text-gray-400">
                                                                <Clock className="w-3 h-3 mr-1 inline" />
                                                                {post.read_time ||
                                                                    5}{" "}
                                                                min read
                                                            </span>
                                                        </div>

                                                        <Link href={`/blog/${post.slug}`}>
                                                            <h3 className="text-lg font-semibold text-gray-900 dark:text-white mb-2 hover:text-blue-600 dark:hover:text-blue-400 transition-colors line-clamp-2">
                                                                {post.title}
                                                            </h3>
                                                        </Link>

                                                        <p className="text-gray-600 dark:text-gray-400 text-sm mb-4 line-clamp-3">
                                                            {post.excerpt}
                                                        </p>

                                                        {post.tags &&
                                                            post.tags
                                                                .length >
                                                                0 && (
                                                                <div className="flex flex-wrap gap-1 mb-4">
                                                                    {post.tags
                                                                        .slice(
                                                                            0,
                                                                            3
                                                                        )
                                                                        .map(
                                                                            (
                                                                                tag
                                                                            ) => (
                                                                                <span
                                                                                    key={
                                                                                        tag.id
                                                                                    }
                                                                                    className="inline-flex items-center px-2 py-1 rounded-md text-xs font-medium bg-blue-100 text-blue-800 dark:bg-blue-900/30 dark:text-blue-400"
                                                                                >
                                                                                    <Tag className="w-3 h-3 mr-1" />
                                                                                    {
                                                                                        tag.name
                                                                                    }
                                                                                </span>
                                                                            )
                                                                        )}
                                                                    {post
                                                                        .tags
                                                                        .length >
                                                                        3 && (
                                                                        <span className="inline-flex items-center px-2 py-1 rounded-md text-xs font-medium bg-gray-100 text-gray-700 dark:bg-gray-700 dark:text-gray-300">
                                                                            +
                                                                            {post
                                                                                .tags
                                                                                .length -
                                                                                3}{" "}
                                                                            more
                                                                        </span>
                                                                    )}
                                                                </div>
                                                            )}

                                                        <div className="flex items-center justify-between">
                                                            <div className="flex items-center text-xs text-gray-500 dark:text-gray-400">
                                                                <User className="w-3 h-3 mr-1" />
                                                                {post.author
                                                                    ?.name ||
                                                                    "Academic Scribe"}
                                                                <span className="mx-2">
                                                                    •
                                                                </span>
                                                                <Calendar className="w-3 h-3 mr-1" />
                                                                {formatDate(
                                                                    post.published_at
                                                                )}
                                                            </div>
                                                            <ArrowRight className="w-4 h-4 text-blue-600 dark:text-blue-400 group-hover:translate-x-1 transition-transform" />
                                                        </div>
                                                    </div>
                                                </motion.div>
                                            </motion.article>
                                        ))}
                                    </div>
                                ) : (
                                    <motion.div
                                        variants={fadeIn}
                                        className="text-center py-12"
                                    >
                                        <BookOpen className="w-16 h-16 text-gray-400 mx-auto mb-4" />
                                        <h3 className="text-xl font-semibold text-gray-900 dark:text-white mb-2">
                                            No posts found
                                        </h3>
                                        <p className="text-gray-600 dark:text-gray-400 mb-6">
                                            {searchQuery
                                                ? "Try adjusting your search terms or browse by category"
                                                : "No posts available at the moment"}
                                        </p>
                                        {(searchQuery ||
                                            selectedCategory !== "all" ||
                                            sortBy !== "newest") && (
                                            <button
                                                onClick={clearFilters}
                                                className="px-4 py-2 bg-blue-600 text-white rounded-lg hover:bg-blue-700 transition-colors"
                                            >
                                                Clear Filters
                                            </button>
                                        )}
                                    </motion.div>
                                )}

                                {/* Pagination */}
                                {posts?.data &&
                                    posts.data.length > 0 &&
                                    posts.last_page > 1 && (
                                        <motion.div
                                            variants={fadeIn}
                                            className="flex justify-center"
                                        >
                                            <div className="flex items-center space-x-2">
                                                {posts.prev_page_url && (
                                                    <Link
                                                        href={
                                                            posts.prev_page_url
                                                        }
                                                        preserveState
                                                        className="flex items-center px-4 py-2 text-sm font-medium text-gray-500 bg-white dark:bg-gray-800 border border-gray-300 dark:border-gray-600 rounded-lg hover:bg-gray-50 dark:hover:bg-gray-700 transition-colors"
                                                    >
                                                        <ChevronLeft className="w-4 h-4 mr-1" />
                                                        Previous
                                                    </Link>
                                                )}

                                                <div className="flex items-center space-x-1">
                                                    {(() => {
                                                        const currentPage =
                                                            posts.current_page;
                                                        const lastPage =
                                                            posts.last_page;
                                                        const pages = [];

                                                        // Calculate page range to display
                                                        let startPage =
                                                            Math.max(
                                                                1,
                                                                currentPage - 2
                                                            );
                                                        let endPage = Math.min(
                                                            lastPage,
                                                            currentPage + 2
                                                        );

                                                        // Adjust if we're near the beginning or end
                                                        if (
                                                            endPage -
                                                                startPage <
                                                            4
                                                        ) {
                                                            if (
                                                                startPage === 1
                                                            ) {
                                                                endPage =
                                                                    Math.min(
                                                                        lastPage,
                                                                        startPage +
                                                                            4
                                                                    );
                                                            } else if (
                                                                endPage ===
                                                                lastPage
                                                            ) {
                                                                startPage =
                                                                    Math.max(
                                                                        1,
                                                                        endPage -
                                                                            4
                                                                    );
                                                            }
                                                        }

                                                        // Add first page if not in range
                                                        if (startPage > 1) {
                                                            pages.push(1);
                                                            if (startPage > 2) {
                                                                pages.push(
                                                                    "..."
                                                                );
                                                            }
                                                        }

                                                        // Add pages in range
                                                        for (
                                                            let i = startPage;
                                                            i <= endPage;
                                                            i++
                                                        ) {
                                                            pages.push(i);
                                                        }

                                                        // Add last page if not in range
                                                        if (
                                                            endPage < lastPage
                                                        ) {
                                                            if (
                                                                endPage <
                                                                lastPage - 1
                                                            ) {
                                                                pages.push(
                                                                    "..."
                                                                );
                                                            }
                                                            pages.push(
                                                                lastPage
                                                            );
                                                        }

                                                        return pages.map(
                                                            (page, index) => {
                                                                if (
                                                                    page ===
                                                                    "..."
                                                                ) {
                                                                    return (
                                                                        <span
                                                                            key={`ellipsis-${index}`}
                                                                            className="px-3 py-2 text-sm text-gray-500"
                                                                        >
                                                                            ...
                                                                        </span>
                                                                    );
                                                                }

                                                                const isActive =
                                                                    page ===
                                                                    currentPage;
                                                                const searchParams =
                                                                    new URLSearchParams();

                                                                // Preserve current filters
                                                                if (searchQuery)
                                                                    searchParams.set(
                                                                        "search",
                                                                        searchQuery
                                                                    );
                                                                if (
                                                                    selectedCategory &&
                                                                    selectedCategory !==
                                                                        "all"
                                                                )
                                                                    searchParams.set(
                                                                        "category",
                                                                        selectedCategory
                                                                    );
                                                                if (
                                                                    sortBy &&
                                                                    sortBy !==
                                                                        "newest"
                                                                )
                                                                    searchParams.set(
                                                                        "sort",
                                                                        sortBy
                                                                    );
                                                                if (page > 1)
                                                                    searchParams.set(
                                                                        "page",
                                                                        page.toString()
                                                                    );

                                                                const queryString =
                                                                    searchParams.toString();
                                                                const href = `/blog${
                                                                    queryString
                                                                        ? `?${queryString}`
                                                                        : ""
                                                                }`;

                                                                return (
                                                                    <Link
                                                                        key={
                                                                            page
                                                                        }
                                                                        href={
                                                                            href
                                                                        }
                                                                        preserveState
                                                                        className={`px-3 py-2 text-sm font-medium rounded-lg transition-colors ${
                                                                            isActive
                                                                                ? "bg-blue-600 text-white"
                                                                                : "text-gray-700 dark:text-gray-300 bg-white dark:bg-gray-800 border border-gray-300 dark:border-gray-600 hover:bg-gray-50 dark:hover:bg-gray-700"
                                                                        }`}
                                                                    >
                                                                        {page}
                                                                    </Link>
                                                                );
                                                            }
                                                        );
                                                    })()}
                                                </div>

                                                {posts.next_page_url && (
                                                    <Link
                                                        href={
                                                            posts.next_page_url
                                                        }
                                                        preserveState
                                                        className="flex items-center px-4 py-2 text-sm font-medium text-gray-500 bg-white dark:bg-gray-800 border border-gray-300 dark:border-gray-600 rounded-lg hover:bg-gray-50 dark:hover:bg-gray-700 transition-colors"
                                                    >
                                                        Next
                                                        <ChevronRight className="w-4 h-4 ml-1" />
                                                    </Link>
                                                )}
                                            </div>
                                        </motion.div>
                                    )}
                            </motion.div>
                        </div>
                    </div>
                </div>
            </div>
        </MainLayout>
    );
}
