import { useState } from "react";
import { Head, Link, router } from "@inertiajs/react";
import { Search, FileText, Calendar, User, DollarSign, CheckCircle, AlertCircle, Download, Eye } from "lucide-react";
import ClientLayout from "@/Layouts/ClientLayout";
import { Button } from "@/Components/ui/Button";
import { Badge } from "@/Components/ui/Badge";
import Pagination from "@/Components/Pagination";

export default function CompletedProjects({ auth, completedProjects, filters }) {
    const [searchQuery, setSearchQuery] = useState(filters.search || "");

    const handleSearch = (e) => {
        e.preventDefault();
        router.get(route('projects.completed'), { search: searchQuery }, {
            preserveState: true,
            preserveScroll: true,
        });
    };

    const clearSearch = () => {
        setSearchQuery("");
        router.get(route('projects.completed'), {}, {
            preserveState: true,
            preserveScroll: true,
        });
    };

    const getStatusBadge = (status) => {
        const statusConfig = {
            'approved': { variant: 'success', text: 'Approved' },
            'cancelled': { variant: 'destructive', text: 'Cancelled' }
        };

        const config = statusConfig[status] || { variant: 'default', text: status };
        return <Badge variant={config.variant}>{config.text}</Badge>;
    };

    const formatDate = (dateString) => {
        if (!dateString) return 'N/A';
        return new Date(dateString).toLocaleDateString('en-US', {
            year: 'numeric',
            month: 'short',
            day: 'numeric'
        });
    };

    const formatCurrency = (amount, currency = 'USD') => {
        return new Intl.NumberFormat('en-US', {
            style: 'currency',
            currency: currency
        }).format(amount || 0);
    };

    return (
        <ClientLayout>
            <Head title="Completed Projects" />

            <div className="py-12">
                <div className="max-w-7xl mx-auto sm:px-6 lg:px-8">
                    {/* Header Section */}
                    <div className="bg-white dark:bg-gray-800 overflow-hidden shadow-sm sm:rounded-lg mb-6">
                        <div className="p-6">
                            <div className="flex flex-col sm:flex-row sm:items-center sm:justify-between gap-4">
                                <div>
                                    <h1 className="text-2xl font-bold text-gray-900 dark:text-white">
                                        Completed Projects
                                    </h1>
                                    <p className="text-gray-600 dark:text-gray-400 mt-1">
                                        View all your completed academic work and projects
                                    </p>
                                </div>
                                <div className="flex items-center gap-3">
                                    <Link href="/orders">
                                        <Button variant="outline">
                                            <FileText className="w-4 h-4 mr-2" />
                                            All Orders
                                        </Button>
                                    </Link>
                                </div>
                            </div>
                        </div>
                    </div>

                    {/* Search and Filters */}
                    <div className="bg-white dark:bg-gray-800 overflow-hidden shadow-sm sm:rounded-lg mb-6">
                        <div className="p-6">
                            <form onSubmit={handleSearch} className="flex flex-col sm:flex-row gap-4">
                                <div className="flex-1 relative">
                                    <Search className="absolute left-3 top-1/2 transform -translate-y-1/2 text-gray-400 w-4 h-4" />
                                    <input
                                        type="text"
                                        placeholder="Search by order number, title, subject, or paper type..."
                                        className="w-full pl-10 pr-4 py-2 border border-gray-300 dark:border-gray-600 rounded-md focus:ring-2 focus:ring-blue-500 focus:border-transparent dark:bg-gray-700 dark:text-white"
                                        value={searchQuery}
                                        onChange={(e) => setSearchQuery(e.target.value)}
                                    />
                                </div>
                                <div className="flex gap-2">
                                    <Button type="submit" variant="default">
                                        <Search className="w-4 h-4 mr-2" />
                                        Search
                                    </Button>
                                    {filters.search && (
                                        <Button type="button" variant="outline" onClick={clearSearch}>
                                            Clear
                                        </Button>
                                    )}
                                </div>
                            </form>
                        </div>
                    </div>

                    {/* Projects List */}
                    <div className="bg-white dark:bg-gray-800 overflow-hidden shadow-sm sm:rounded-lg">
                        <div className="p-6">
                            {completedProjects.data.length === 0 ? (
                                <div className="text-center py-12">
                                    <FileText className="mx-auto h-12 w-12 text-gray-400" />
                                    <h3 className="mt-2 text-sm font-medium text-gray-900 dark:text-white">
                                        No completed projects found
                                    </h3>
                                    <p className="mt-1 text-sm text-gray-500 dark:text-gray-400">
                                        {filters.search 
                                            ? "Try adjusting your search terms or clear the search to see all projects."
                                            : "You don't have any completed projects yet. Your completed work will appear here once orders are finished."
                                        }
                                    </p>
                                    {filters.search && (
                                        <div className="mt-6">
                                            <Button variant="outline" onClick={clearSearch}>
                                                Clear Search
                                            </Button>
                                        </div>
                                    )}
                                </div>
                            ) : (
                                <div className="space-y-4">
                                    {completedProjects.data.map((project) => (
                                        <div
                                            key={project.id}
                                            className="border border-gray-200 dark:border-gray-700 rounded-lg p-6 hover:shadow-md transition-shadow duration-200"
                                        >
                                            <div className="flex flex-col lg:flex-row lg:items-center lg:justify-between gap-4">
                                                <div className="flex-1">
                                                    <div className="flex items-start justify-between mb-3">
                                                        <div className="flex-1">
                                                            <h3 className="text-lg font-semibold text-gray-900 dark:text-white mb-1">
                                                                {project.title}
                                                            </h3>
                                                            <div className="flex flex-wrap items-center gap-2 text-sm text-gray-600 dark:text-gray-400">
                                                                <span className="flex items-center gap-1">
                                                                    <FileText className="w-4 h-4" />
                                                                    {project.type_of_paper}
                                                                </span>
                                                                <span>•</span>
                                                                <span>{project.subject}</span>
                                                                <span>•</span>
                                                                <span>{project.academic_level}</span>
                                                                <span>•</span>
                                                                <span>{project.pages} pages</span>
                                                            </div>
                                                        </div>
                                                        <div className="flex items-center gap-2">
                                                            {getStatusBadge(project.order_status)}
                                                            {project.is_refunded && (
                                                                <Badge color="warning">Refunded</Badge>
                                                            )}
                                                            {project.is_partially_refunded && (
                                                                <Badge color="warning">Partially Refunded</Badge>
                                                            )}
                                                        </div>
                                                    </div>

                                                    <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-4 gap-4 text-sm">
                                                        <div className="flex items-center gap-2">
                                                            <Calendar className="w-4 h-4 text-gray-400" />
                                                            <span className="text-gray-600 dark:text-gray-400">Completed:</span>
                                                            <span className="font-medium text-gray-900 dark:text-white">
                                                                {formatDate(project.completion_date)}
                                                            </span>
                                                        </div>
                                                        
                                                        <div className="flex items-center gap-2">
                                                            <User className="w-4 h-4 text-gray-400" />
                                                            <span className="text-gray-600 dark:text-gray-400">Writer:</span>
                                                            <span className="font-medium text-gray-900 dark:text-white">
                                                                {project.writer_name || 'Unassigned'}
                                                            </span>
                                                        </div>
                                                        
                                                        <div className="flex items-center gap-2">
                                                            <DollarSign className="w-4 h-4 text-gray-400" />
                                                            <span className="text-gray-600 dark:text-gray-400">Total Paid:</span>
                                                            <span className="font-medium text-gray-900 dark:text-white">
                                                                {formatCurrency(project.total_paid, project.currency)}
                                                            </span>
                                                        </div>
                                                        
                                                        {project.refund_amount > 0 && (
                                                            <div className="flex items-center gap-2">
                                                                <AlertCircle className="w-4 h-4 text-red-400" />
                                                                <span className="text-gray-600 dark:text-gray-400">Refunded:</span>
                                                                <span className="font-medium text-red-600 dark:text-red-400">
                                                                    {formatCurrency(project.refund_amount, project.currency)}
                                                                </span>
                                                            </div>
                                                        )}
                                                    </div>

                                                    {project.instructions && (
                                                        <div className="mt-4">
                                                            <p className="text-sm text-gray-600 dark:text-gray-400">
                                                                <span className="font-medium">Instructions:</span> {project.instructions}
                                                            </p>
                                                        </div>
                                                    )}
                                                </div>

                                                <div className="flex flex-col sm:flex-row gap-2">
                                                    <Link href={`/orders/${project.id}`}>
                                                        <Button variant="outline" size="sm">
                                                            <Eye className="w-4 h-4 mr-2" />
                                                            View Details
                                                        </Button>
                                                    </Link>
                                                    {project.order_status === 'approved' && (
                                                        <Button variant="default" size="sm">
                                                            <Download className="w-4 h-4 mr-2" />
                                                            Download
                                                        </Button>
                                                    )}
                                                </div>
                                            </div>
                                        </div>
                                    ))}
                                </div>
                            )}

                            {/* Pagination */}
                            {completedProjects.data.length > 0 && (
                                <div className="mt-6">
                                    <Pagination links={completedProjects.links} />
                                </div>
                            )}
                        </div>
                    </div>
                </div>
            </div>
        </ClientLayout>
    );
} 