import { Head, Link, useForm } from "@inertiajs/react";
import React from "react";
import { FileText, Plus, Clock, CheckCircle, AlertCircle, ArrowRight, Eye } from "lucide-react";

import ClientLayout from "@/Layouts/ClientLayout";
import { Badge } from "@/Components/ui/Badge";
import { Button } from "@/Components/ui/Button";
import { Card, CardContent, CardHeader, CardTitle } from "@/Components/ui/Card";

export default function ClientRevisionIndex({ order, revisions, canRequestRevision, revisionTypes }) {
    const { get } = useForm();

    const getStatusBadge = (status) => {
        const statusConfig = {
            requested: { color: "bg-blue-100 text-blue-800", icon: Clock, label: "Requested" },
            in_progress: { color: "bg-yellow-100 text-yellow-800", icon: Clock, label: "In Progress" },
            completed: { color: "bg-green-100 text-green-800", icon: CheckCircle, label: "Completed" },
            rejected: { color: "bg-red-100 text-red-800", icon: AlertCircle, label: "Rejected" },
            escalated: { color: "bg-orange-100 text-orange-800", icon: AlertCircle, label: "Escalated" },
        };

        const config = statusConfig[status] || statusConfig.requested;
        const Icon = config.icon;

        return (
            <Badge className={`${config.color} dark:${config.color.replace('100', '900')} dark:${config.color.replace('800', '200')}`}>
                <Icon size={12} className="mr-1" />
                {config.label}
            </Badge>
        );
    };

    const getRevisionTypeLabel = (type) => {
        return revisionTypes[type] || type;
    };

    const formatDate = (dateString) => {
        return new Date(dateString).toLocaleDateString('en-US', {
            year: 'numeric',
            month: 'short',
            day: 'numeric',
            hour: '2-digit',
            minute: '2-digit'
        });
    };

    return (
        <ClientLayout>
            <Head title={`Revisions - Order #${order.order_number}`} />

            <div className="max-w-7xl mx-auto py-6 px-4 sm:px-6 lg:px-8">
                {/* Header */}
                <div className="mb-8">
                    <div className="flex items-center justify-between">
                        <div>
                            <h1 className="text-3xl font-bold text-gray-900 dark:text-white">
                                Order Revisions
                            </h1>
                            <p className="mt-2 text-gray-600 dark:text-gray-400">
                                Order #{order.order_number} - {order.title || 'Untitled Order'}
                            </p>
                        </div>
                        <div className="flex space-x-3">
                            <Link href={route('orders.show', order.id)}>
                                <Button variant="outline">
                                    <Eye size={16} className="mr-2" />
                                    View Order
                                </Button>
                            </Link>
                            {canRequestRevision && (
                                <Link href={route('client.revisions.create', order.id)}>
                                    <Button>
                                        <Plus size={16} className="mr-2" />
                                        Request Revision
                                    </Button>
                                </Link>
                            )}
                        </div>
                    </div>
                </div>

                {/* Order Summary */}
                <Card className="mb-6">
                    <CardHeader>
                        <CardTitle>Order Summary</CardTitle>
                    </CardHeader>
                    <CardContent>
                        <div className="grid grid-cols-1 md:grid-cols-3 gap-4">
                            <div>
                                <span className="text-sm font-medium text-gray-500">Status:</span>
                                <p className="text-sm text-gray-900 dark:text-white">{order.order_status.replace('_', ' ').toUpperCase()}</p>
                            </div>
                            <div>
                                <span className="text-sm font-medium text-gray-500">Writer:</span>
                                <p className="text-sm text-gray-900 dark:text-white">
                                    {order.writer ? order.writer.name : 'Unassigned'}
                                </p>
                            </div>
                            <div>
                                <span className="text-sm font-medium text-gray-500">Total Revisions:</span>
                                <p className="text-sm text-gray-900 dark:text-white">{revisions.length}</p>
                            </div>
                        </div>
                    </CardContent>
                </Card>

                {/* Revisions List */}
                <Card>
                    <CardHeader>
                        <CardTitle className="flex items-center">
                            <FileText className="h-5 w-5 mr-2" />
                            Revision History
                        </CardTitle>
                    </CardHeader>
                    <CardContent>
                        {revisions.length === 0 ? (
                            <div className="text-center py-8">
                                <FileText className="h-12 w-12 text-gray-400 mx-auto mb-4" />
                                <p className="text-gray-500 dark:text-gray-400 mb-4">
                                    No revisions have been requested for this order yet.
                                </p>
                                {canRequestRevision && (
                                    <Link href={route('client.revisions.create', order.id)}>
                                        <Button>
                                            <Plus size={16} className="mr-2" />
                                            Request First Revision
                                        </Button>
                                    </Link>
                                )}
                            </div>
                        ) : (
                            <div className="space-y-4">
                                {revisions.map((revision) => (
                                    <div
                                        key={revision.id}
                                        className="border border-gray-200 dark:border-gray-700 rounded-lg p-4 hover:bg-gray-50 dark:hover:bg-gray-800 transition-colors"
                                    >
                                        <div className="flex items-start justify-between">
                                            <div className="flex-1">
                                                <div className="flex items-center space-x-3 mb-3">
                                                    <h3 className="font-medium text-gray-900 dark:text-white">
                                                        Revision #{revision.revision_number}
                                                    </h3>
                                                    {getStatusBadge(revision.status)}
                                                    {revision.is_client_requested && (
                                                        <Badge className="bg-purple-100 text-purple-800 dark:bg-purple-900 dark:text-purple-200">
                                                            Client Requested
                                                        </Badge>
                                                    )}
                                                </div>
                                                
                                                <div className="grid grid-cols-1 md:grid-cols-2 gap-4 mb-3">
                                                    <div>
                                                        <span className="text-sm font-medium text-gray-500">Type:</span>
                                                        <p className="text-sm text-gray-900 dark:text-white">
                                                            {getRevisionTypeLabel(revision.revision_type)}
                                                        </p>
                                                    </div>
                                                    <div>
                                                        <span className="text-sm font-medium text-gray-500">Requested:</span>
                                                        <p className="text-sm text-gray-900 dark:text-white">
                                                            {formatDate(revision.requested_at)}
                                                        </p>
                                                    </div>
                                                    {revision.due_date && (
                                                        <div>
                                                            <span className="text-sm font-medium text-gray-500">Due Date:</span>
                                                            <p className="text-sm text-gray-900 dark:text-white">
                                                                {formatDate(revision.due_date)}
                                                            </p>
                                                        </div>
                                                    )}
                                                    {revision.assigned_to && (
                                                        <div>
                                                            <span className="text-sm font-medium text-gray-500">Assigned to:</span>
                                                            <p className="text-sm text-gray-900 dark:text-white">
                                                                {revision.assigned_to.name}
                                                            </p>
                                                        </div>
                                                    )}
                                                </div>

                                                {revision.revision_reason && (
                                                    <div className="mb-3">
                                                        <span className="text-sm font-medium text-gray-500">Reason:</span>
                                                        <p className="text-sm text-gray-900 dark:text-white mt-1">
                                                            {revision.revision_reason}
                                                        </p>
                                                    </div>
                                                )}

                                                {revision.specific_instructions && (
                                                    <div className="mb-3">
                                                        <span className="text-sm font-medium text-gray-500">Instructions:</span>
                                                        <p className="text-sm text-gray-900 dark:text-white mt-1">
                                                            {revision.specific_instructions}
                                                        </p>
                                                    </div>
                                                )}

                                                {revision.quality_rating && (
                                                    <div className="mb-3">
                                                        <span className="text-sm font-medium text-gray-500">Quality Rating:</span>
                                                        <div className="flex items-center mt-1">
                                                            <div className="flex text-yellow-400">
                                                                {[...Array(5)].map((_, i) => (
                                                                    <svg
                                                                        key={i}
                                                                        className={`h-4 w-4 ${i < revision.quality_rating ? 'text-yellow-400' : 'text-gray-300'}`}
                                                                        fill="currentColor"
                                                                        viewBox="0 0 20 20"
                                                                    >
                                                                        <path d="M9.049 2.927c.3-.921 1.603-.921 1.902 0l1.07 3.292a1 1 0 00.95.69h3.462c.969 0 1.371 1.24.588 1.81l-2.8 2.034a1 1 0 00-.364 1.118l1.07 3.292c.3.921-.755 1.688-1.54 1.118l-2.8-2.034a1 1 0 00-1.175 0l-2.8 2.034c-.784.57-1.838-.197-1.539-1.118l1.07-3.292a1 1 0 00-.364-1.118L2.98 8.72c-.783-.57-.38-1.81.588-1.81h3.461a1 1 0 00.951-.69l1.07-3.292z" />
                                                                    </svg>
                                                                ))}
                                                            </div>
                                                            <span className="ml-2 text-sm text-gray-600 dark:text-gray-400">
                                                                {revision.quality_rating}/5
                                                            </span>
                                                        </div>
                                                    </div>
                                                )}
                                            </div>
                                            
                                            <div className="flex flex-col space-y-2 ml-4">
                                                <Link
                                                    href={route('client.revisions.show', [order.id, revision.id])}
                                                    className="inline-flex items-center text-blue-600 hover:text-blue-800 dark:text-blue-400 dark:hover:text-blue-300 text-sm font-medium"
                                                >
                                                    <Eye size={16} className="mr-1" />
                                                    View Details
                                                </Link>
                                            </div>
                                        </div>
                                    </div>
                                ))}
                            </div>
                        )}
                    </CardContent>
                </Card>
            </div>
        </ClientLayout>
    );
} 