import { Head, Link } from "@inertiajs/react";
import React from "react";
import { FileText, ArrowLeft, Clock, CheckCircle, AlertCircle, User, Calendar, MessageSquare } from "lucide-react";

import ClientLayout from "@/Layouts/ClientLayout";
import { Badge } from "@/Components/ui/Badge";
import { Button } from "@/Components/ui/Button";
import { Card, CardContent, CardHeader, CardTitle } from "@/Components/ui/Card";

export default function ClientRevisionShow({ order, revision, canRequestRevision }) {
    const getStatusBadge = (status) => {
        const statusConfig = {
            requested: { color: "bg-blue-100 text-blue-800", icon: Clock, label: "Requested" },
            in_progress: { color: "bg-yellow-100 text-yellow-800", icon: Clock, label: "In Progress" },
            completed: { color: "bg-green-100 text-green-800", icon: CheckCircle, label: "Completed" },
            rejected: { color: "bg-red-100 text-red-800", icon: AlertCircle, label: "Rejected" },
            escalated: { color: "bg-orange-100 text-orange-800", icon: AlertCircle, label: "Escalated" },
        };

        const config = statusConfig[status] || statusConfig.requested;
        const Icon = config.icon;

        return (
            <Badge className={`${config.color} dark:${config.color.replace('100', '900')} dark:${config.color.replace('800', '200')}`}>
                <Icon size={12} className="mr-1" />
                {config.label}
            </Badge>
        );
    };

    const getRevisionTypeLabel = (type) => {
        const typeLabels = {
            quality: "Quality Issues",
            requirements: "Requirements Not Met",
            formatting: "Formatting Issues",
            plagiarism: "Plagiarism Concerns",
            deadline: "Deadline Issues",
            communication: "Communication Issues",
            other: "Other Issues",
        };

        return typeLabels[type] || type;
    };

    const formatDate = (dateString) => {
        return new Date(dateString).toLocaleDateString('en-US', {
            year: 'numeric',
            month: 'long',
            day: 'numeric',
            hour: '2-digit',
            minute: '2-digit'
        });
    };

    const formatTimeAgo = (dateString) => {
        const now = new Date();
        const date = new Date(dateString);
        const diffInHours = Math.floor((now - date) / (1000 * 60 * 60));
        
        if (diffInHours < 1) return 'Less than an hour ago';
        if (diffInHours < 24) return `${diffInHours} hour${diffInHours > 1 ? 's' : ''} ago`;
        
        const diffInDays = Math.floor(diffInHours / 24);
        return `${diffInDays} day${diffInDays > 1 ? 's' : ''} ago`;
    };

    const isOverdue = revision.due_date && new Date() > new Date(revision.due_date) && revision.status !== 'completed';

    return (
        <ClientLayout>
            <Head title={`Revision #${revision.revision_number} - Order #${order.order_number}`} />

            <div className="max-w-6xl mx-auto py-6 px-4 sm:px-6 lg:px-8">
                {/* Header */}
                <div className="mb-8">
                    <div className="flex items-center space-x-4 mb-4">
                        <Link href={route('client.revisions.index', order.id)}>
                            <Button variant="ghost" size="sm">
                                <ArrowLeft size={16} className="mr-2" />
                                Back to Revisions
                            </Button>
                        </Link>
                    </div>
                    <div className="flex items-center justify-between">
                        <div>
                            <h1 className="text-3xl font-bold text-gray-900 dark:text-white">
                                Revision #{revision.revision_number}
                            </h1>
                            <p className="mt-2 text-gray-600 dark:text-gray-400">
                                Order #{order.order_number} - {order.title || 'Untitled Order'}
                            </p>
                        </div>
                        <div className="flex items-center space-x-3">
                            {getStatusBadge(revision.status)}
                            {revision.is_client_requested && (
                                <Badge className="bg-purple-100 text-purple-800 dark:bg-purple-900 dark:text-purple-200">
                                    Client Requested
                                </Badge>
                            )}
                            {canRequestRevision && revision.status === 'completed' && (
                                <Link href={route('client.revisions.create', order.id)}>
                                    <Button>
                                        Request Another Revision
                                    </Button>
                                </Link>
                            )}
                        </div>
                    </div>
                </div>

                <div className="grid grid-cols-1 lg:grid-cols-3 gap-6">
                    {/* Main Content */}
                    <div className="lg:col-span-2 space-y-6">
                        {/* Revision Details */}
                        <Card>
                            <CardHeader>
                                <CardTitle className="flex items-center">
                                    <FileText className="h-5 w-5 mr-2" />
                                    Revision Details
                                </CardTitle>
                            </CardHeader>
                            <CardContent className="space-y-4">
                                <div className="grid grid-cols-1 md:grid-cols-2 gap-4">
                                    <div>
                                        <span className="text-sm font-medium text-gray-500">Type:</span>
                                        <p className="text-sm text-gray-900 dark:text-white">
                                            {getRevisionTypeLabel(revision.revision_type)}
                                        </p>
                                    </div>
                                    <div>
                                        <span className="text-sm font-medium text-gray-500">Status:</span>
                                        <p className="text-sm text-gray-900 dark:text-white">
                                            {revision.status.replace('_', ' ').toUpperCase()}
                                        </p>
                                    </div>
                                    <div>
                                        <span className="text-sm font-medium text-gray-500">Requested:</span>
                                        <p className="text-sm text-gray-900 dark:text-white">
                                            {formatDate(revision.requested_at)}
                                        </p>
                                    </div>
                                    <div>
                                        <span className="text-sm font-medium text-gray-500">Due Date:</span>
                                        <p className="text-sm text-gray-900 dark:text-white">
                                            {revision.due_date ? formatDate(revision.due_date) : 'Not set'}
                                        </p>
                                    </div>
                                </div>

                                {isOverdue && (
                                    <div className="bg-red-50 dark:bg-red-900/20 border border-red-200 dark:border-red-800 rounded-lg p-3">
                                        <div className="flex items-center">
                                            <AlertCircle className="h-4 w-4 text-red-500 mr-2" />
                                            <span className="text-sm text-red-700 dark:text-red-300">
                                                This revision is overdue. Please contact support if you have concerns.
                                            </span>
                                        </div>
                                    </div>
                                )}

                                {revision.revision_reason && (
                                    <div>
                                        <span className="text-sm font-medium text-gray-500">Revision Reason:</span>
                                        <p className="text-sm text-gray-900 dark:text-white mt-1 p-3 bg-gray-50 dark:bg-gray-800 rounded-lg">
                                            {revision.revision_reason}
                                        </p>
                                    </div>
                                )}

                                {revision.specific_instructions && (
                                    <div>
                                        <span className="text-sm font-medium text-gray-500">Specific Instructions:</span>
                                        <p className="text-sm text-gray-900 dark:text-white mt-1 p-3 bg-gray-50 dark:bg-gray-800 rounded-lg">
                                            {revision.specific_instructions}
                                        </p>
                                    </div>
                                )}

                                {revision.quality_rating && (
                                    <div>
                                        <span className="text-sm font-medium text-gray-500">Quality Rating:</span>
                                        <div className="flex items-center mt-1">
                                            <div className="flex text-yellow-400">
                                                {[...Array(5)].map((_, i) => (
                                                    <svg
                                                        key={i}
                                                        className={`h-5 w-5 ${i < revision.quality_rating ? 'text-yellow-400' : 'text-gray-300'}`}
                                                        fill="currentColor"
                                                        viewBox="0 0 20 20"
                                                    >
                                                        <path d="M9.049 2.927c.3-.921 1.603-.921 1.902 0l1.07 3.292a1 1 0 00.95.69h3.462c.969 0 1.371 1.24.588 1.81l-2.8 2.034a1 1 0 00-.364 1.118l1.07 3.292c.3.921-.755 1.688-1.54 1.118l-2.8-2.034a1 1 0 00-1.175 0l-2.8 2.034c-.784.57-1.838-.197-1.539-1.118l1.07-3.292a1 1 0 00-.364-1.118L2.98 8.72c-.783-.57-.38-1.81.588-1.81h3.461a1 1 0 00.951-.69l1.07-3.292z" />
                                                    </svg>
                                                ))}
                                            </div>
                                            <span className="ml-2 text-sm text-gray-600 dark:text-gray-400">
                                                {revision.quality_rating}/5
                                            </span>
                                        </div>
                                    </div>
                                )}

                                {revision.quality_notes && (
                                    <div>
                                        <span className="text-sm font-medium text-gray-500">Quality Notes:</span>
                                        <p className="text-sm text-gray-900 dark:text-white mt-1 p-3 bg-gray-50 dark:bg-gray-800 rounded-lg">
                                            {revision.quality_notes}
                                        </p>
                                    </div>
                                )}
                            </CardContent>
                        </Card>

                        {/* Timeline */}
                        <Card>
                            <CardHeader>
                                <CardTitle className="flex items-center">
                                    <Calendar className="h-5 w-5 mr-2" />
                                    Revision Timeline
                                </CardTitle>
                            </CardHeader>
                            <CardContent>
                                <div className="space-y-4">
                                    <div className="flex items-start space-x-3">
                                        <div className="w-3 h-3 bg-blue-500 rounded-full mt-2 flex-shrink-0"></div>
                                        <div className="flex-1">
                                            <p className="text-sm font-medium text-gray-900 dark:text-white">
                                                Revision Requested
                                            </p>
                                            <p className="text-sm text-gray-600 dark:text-gray-400">
                                                {formatDate(revision.requested_at)} ({formatTimeAgo(revision.requested_at)})
                                            </p>
                                        </div>
                                    </div>

                                    {revision.started_at && (
                                        <div className="flex items-start space-x-3">
                                            <div className="w-3 h-3 bg-yellow-500 rounded-full mt-2 flex-shrink-0"></div>
                                            <div className="flex-1">
                                                <p className="text-sm font-medium text-gray-900 dark:text-white">
                                                    Work Started
                                                </p>
                                                <p className="text-sm text-gray-600 dark:text-gray-400">
                                                    {formatDate(revision.started_at)} ({formatTimeAgo(revision.started_at)})
                                                </p>
                                            </div>
                                        </div>
                                    )}

                                    {revision.completed_at && (
                                        <div className="flex items-start space-x-3">
                                            <div className="w-3 h-3 bg-green-500 rounded-full mt-2 flex-shrink-0"></div>
                                            <div className="flex-1">
                                                <p className="text-sm font-medium text-gray-900 dark:text-white">
                                                    Revision Completed
                                                </p>
                                                <p className="text-sm text-gray-600 dark:text-gray-400">
                                                    {formatDate(revision.completed_at)} ({formatTimeAgo(revision.completed_at)})
                                                </p>
                                            </div>
                                        </div>
                                    )}

                                    {revision.rejected_at && (
                                        <div className="flex items-start space-x-3">
                                            <div className="w-3 h-3 bg-red-500 rounded-full mt-2 flex-shrink-0"></div>
                                            <div className="flex-1">
                                                <p className="text-sm font-medium text-gray-900 dark:text-white">
                                                    Revision Rejected
                                                </p>
                                                <p className="text-sm text-gray-600 dark:text-gray-400">
                                                    {formatDate(revision.rejected_at)} ({formatTimeAgo(revision.rejected_at)})
                                                </p>
                                            </div>
                                        </div>
                                    )}
                                </div>
                            </CardContent>
                        </Card>
                    </div>

                    {/* Sidebar */}
                    <div className="space-y-6">
                        {/* Order Summary */}
                        <Card>
                            <CardHeader>
                                <CardTitle>Order Summary</CardTitle>
                            </CardHeader>
                            <CardContent className="space-y-3">
                                <div>
                                    <span className="text-sm font-medium text-gray-500">Order Number:</span>
                                    <p className="text-sm text-gray-900 dark:text-white">{order.order_number}</p>
                                </div>
                                <div>
                                    <span className="text-sm font-medium text-gray-500">Status:</span>
                                    <p className="text-sm text-gray-900 dark:text-white">{order.order_status.replace('_', ' ').toUpperCase()}</p>
                                </div>
                                <div>
                                    <span className="text-sm font-medium text-gray-500">Writer:</span>
                                    <p className="text-sm text-gray-900 dark:text-white">
                                        {order.writer ? order.writer.name : 'Unassigned'}
                                    </p>
                                </div>
                                <div>
                                    <span className="text-sm font-medium text-gray-500">Total Revisions:</span>
                                    <p className="text-sm text-gray-900 dark:text-white">{order.revisions?.length || 0}</p>
                                </div>
                            </CardContent>
                        </Card>

                        {/* Assignment Info */}
                        {revision.assigned_to && (
                            <Card>
                                <CardHeader>
                                    <CardTitle className="flex items-center">
                                        <User className="h-5 w-5 mr-2" />
                                        Assigned Writer
                                    </CardTitle>
                                </CardHeader>
                                <CardContent>
                                    <div className="flex items-center space-x-3">
                                        <div className="w-10 h-10 bg-blue-100 dark:bg-blue-900 rounded-full flex items-center justify-center">
                                            <span className="text-blue-600 dark:text-blue-400 font-medium">
                                                {revision.assigned_to.name.charAt(0).toUpperCase()}
                                            </span>
                                        </div>
                                        <div>
                                            <p className="text-sm font-medium text-gray-900 dark:text-white">
                                                {revision.assigned_to.name}
                                            </p>
                                            <p className="text-xs text-gray-500 dark:text-gray-400">
                                                Assigned to revision
                                            </p>
                                        </div>
                                    </div>
                                </CardContent>
                            </Card>
                        )}

                        {/* Actions */}
                        <Card>
                            <CardHeader>
                                <CardTitle>Actions</CardTitle>
                            </CardHeader>
                            <CardContent className="space-y-3">
                                <Link href={route('orders.show', order.id)} className="w-full">
                                    <Button variant="outline" className="w-full">
                                        <FileText size={16} className="mr-2" />
                                        View Order
                                    </Button>
                                </Link>
                                <Link href={route('client.revisions.index', order.id)} className="w-full">
                                    <Button variant="outline" className="w-full">
                                        <FileText size={16} className="mr-2" />
                                        All Revisions
                                    </Button>
                                </Link>
                                {revision.conversation && (
                                    <Link href={`/conversations/${revision.conversation.id}`} className="w-full">
                                        <Button variant="outline" className="w-full">
                                            <MessageSquare size={16} className="mr-2" />
                                            View Messages
                                        </Button>
                                    </Link>
                                )}
                            </CardContent>
                        </Card>
                    </div>
                </div>
            </div>
        </ClientLayout>
    );
} 