import React, { useState, useEffect, useCallback } from "react";
import { Head } from "@inertiajs/react";
import ClientLayout from "@/Layouts/ClientLayout";
import WriterLayout from "@/Layouts/WriterLayout";
import AdminLayout from "@/Layouts/AdminLayout";
import {
    Bell,
    Filter,
    Check,
    Search,
    X,
    Calendar,
    ChevronLeft,
    ChevronRight,
    RefreshCw,
    Trash2,
    Loader2,
    CheckSquare,
    Square,
    LayoutGrid,
    List,
    Settings,
    Info,
    MessageSquare,
    DollarSign,
    Clock,
    AlertTriangle,
    ArrowUpDown,
    Eye,
    EyeOff,
} from "lucide-react";
import axios from "axios";
import NotificationItem from "@/Components/Notifications/NotificationItem";
import { debounce } from "lodash";

export default function Notifications({
    auth,
    notifications: initialNotifications,
    filters: initialFilters = {},
    unreadCount: initialUnreadCount,
}) {
    const [notifications, setNotifications] = useState(
        initialNotifications.data || []
    );
    const [selectedItems, setSelectedItems] = useState([]);
    const [isSelectAll, setIsSelectAll] = useState(false);
    const [filters, setFilters] = useState({
        page: initialFilters.page || 1,
        per_page: initialFilters.per_page || 15,
        read: initialFilters.read || null,
        type: initialFilters.type || null,
        category: initialFilters.category || null,
        search: initialFilters.search || "",
        from_date: initialFilters.from_date || null,
        to_date: initialFilters.to_date || null,
        sort_field: initialFilters.sort_field || "created_at",
        sort_direction: initialFilters.sort_direction || "desc",
    });
    const [pagination, setPagination] = useState({
        current_page: initialNotifications.current_page || 1,
        last_page: initialNotifications.last_page || 1,
        per_page: initialNotifications.per_page || 15,
        total: initialNotifications.total || 0,
    });
    const [unreadCount, setUnreadCount] = useState(initialUnreadCount || 0);
    const [isLoading, setIsLoading] = useState(false);
    const [isBatchProcessing, setIsBatchProcessing] = useState(false);
    const [isRefreshing, setIsRefreshing] = useState(false);
    const [error, setError] = useState(null);
    const [isShowingFilters, setIsShowingFilters] = useState(false);
    const [viewMode, setViewMode] = useState("grid"); // 'grid' or 'list'
    const [dateRange, setDateRange] = useState({
        fromDate: initialFilters.from_date || "",
        toDate: initialFilters.to_date || "",
    });

    // Define notification categories
    const notificationCategories = [
        { value: null, label: "All Categories" },
        { value: "order", label: "Orders", icon: Info },
        { value: "payment", label: "Payments", icon: DollarSign },
        { value: "message", label: "Messages", icon: MessageSquare },
        { value: "system", label: "System", icon: Settings },
        { value: "alert", label: "Alerts", icon: AlertTriangle },
    ];

    // Define notification read status options
    const readStatusOptions = [
        { value: null, label: "All" },
        { value: false, label: "Unread", icon: EyeOff },
        { value: true, label: "Read", icon: Eye },
    ];

    // Helper function to safely get route URLs
    const getRouteUrl = (routeName, params = {}) => {
        try {
            // Check if route exists in Ziggy routes
            if (route().has(routeName)) {
                return route(routeName, params);
            }

            // Fallback to hardcoded URLs if route doesn't exist
            const routeMap = {
                "notifications.list": "/notifications/api/list",
                "notifications.mark-all-read": "/notifications/api/read-all",
                "notifications.mark-read": `/notifications/api/${params.id}/read`,
                "notifications.mark-multiple-read": "/notifications/api/read",
                "notifications.delete": `/notifications/api/${params.id}`,
            };

            return routeMap[routeName] || "#";
        } catch (error) {
            console.error(`Error getting route URL for ${routeName}:`, error);
            return "#";
        }
    };

    // Fetch notifications with current filters - using useCallback to avoid recreating on every render
    const fetchNotifications = useCallback(
        async (newFilters = null) => {
            setIsLoading(true);
            setError(null);

            const queryParams = newFilters || filters;

            try {
                const response = await axios.get(
                    getRouteUrl("notifications.list"),
                    {
                        params: queryParams,
                    }
                );

                if (response.data.success) {
                    setNotifications(response.data.data.data || []);
                    setPagination({
                        current_page: response.data.data.current_page || 1,
                        last_page: response.data.data.last_page || 1,
                        per_page: response.data.data.per_page || 15,
                        total: response.data.data.total || 0,
                    });
                    fetchUnreadCount(); // Update unread count
                    setSelectedItems([]); // Clear selection when data changes
                    setIsSelectAll(false);
                } else {
                    setError(
                        response.data.message || "Failed to load notifications"
                    );
                }
            } catch (error) {
                console.error("Error fetching notifications:", error);
                setError(
                    error.response?.data?.message ||
                        "Failed to load notifications. Please try again."
                );
            } finally {
                setIsLoading(false);
            }
        },
        [filters]
    );

    // Fetch unread notification count
    const fetchUnreadCount = async () => {
        try {
            const response = await axios.get(
                getRouteUrl("notifications.count")
            );
            if (response.data.success) {
                setUnreadCount(response.data.count);
            }
        } catch (error) {
            console.error("Error fetching unread count:", error);
        }
    };

    // Apply filters
    const applyFilters = (newFilters = {}) => {
        const updatedFilters = { ...filters, ...newFilters, page: 1 };
        setFilters(updatedFilters);
        fetchNotifications(updatedFilters);
    };

    // Reset all filters
    const resetFilters = () => {
        const defaultFilters = {
            page: 1,
            per_page: 15,
            read: null,
            type: null,
            category: null,
            search: "",
            from_date: null,
            to_date: null,
            sort_field: "created_at",
            sort_direction: "desc",
        };
        setFilters(defaultFilters);
        setDateRange({ fromDate: "", toDate: "" });
        fetchNotifications(defaultFilters);
    };

    // Handle page change
    const changePage = (page) => {
        if (page < 1 || page > pagination.last_page) return;
        const updatedFilters = { ...filters, page };
        setFilters(updatedFilters);
        fetchNotifications(updatedFilters);
    };

    // Handle per page change
    const changePerPage = (perPage) => {
        const updatedFilters = { ...filters, per_page: perPage, page: 1 };
        setFilters(updatedFilters);
        fetchNotifications(updatedFilters);
    };

    // Handle sort change
    const handleSort = (field) => {
        const direction =
            filters.sort_field === field && filters.sort_direction === "asc"
                ? "desc"
                : "asc";

        const updatedFilters = {
            ...filters,
            sort_field: field,
            sort_direction: direction,
            page: 1,
        };

        setFilters(updatedFilters);
        fetchNotifications(updatedFilters);
    };

    // Create a debounced search function
    const debouncedSearch = useCallback(
        debounce((searchTerm) => {
            applyFilters({ search: searchTerm });
        }, 500),
        []
    );

    // Handle search input change
    const handleSearchChange = (e) => {
        const searchTerm = e.target.value;
        setFilters((prev) => ({ ...prev, search: searchTerm }));
        debouncedSearch(searchTerm);
    };

    // Handle date range changes
    const handleDateChange = (field, value) => {
        setDateRange((prev) => ({ ...prev, [field]: value }));
    };

    // Apply date range filter
    const applyDateRange = () => {
        applyFilters({
            from_date: dateRange.fromDate || null,
            to_date: dateRange.toDate || null,
        });
    };

    // Mark multiple notifications as read
    const markMultipleAsRead = async () => {
        if (selectedItems.length === 0) return;

        setIsBatchProcessing(true);

        try {
            const response = await axios.post(
                getRouteUrl("notifications.mark-multiple-read"),
                { ids: selectedItems }
            );

            if (response.data.success) {
                // Update UI for marked items
                const updatedNotifications = notifications.map(
                    (notification) => {
                        if (selectedItems.includes(notification.id)) {
                            return {
                                ...notification,
                                read_at: new Date().toISOString(),
                            };
                        }
                        return notification;
                    }
                );

                setNotifications(updatedNotifications);
                setSelectedItems([]);
                setIsSelectAll(false);
                fetchUnreadCount();

                // Show success message
                if (window.toast?.success) {
                    window.toast.success(
                        `${selectedItems.length} notifications marked as read`
                    );
                }
            } else {
                throw new Error(
                    response.data.message ||
                        "Failed to mark notifications as read"
                );
            }
        } catch (error) {
            console.error("Error marking notifications as read:", error);
            if (window.toast?.error) {
                window.toast.error(
                    error.response?.data?.message ||
                        "Failed to mark notifications as read. Please try again."
                );
            }
        } finally {
            setIsBatchProcessing(false);
        }
    };

    // Mark all notifications as read
    const markAllAsRead = async () => {
        if (unreadCount === 0) return;

        setIsBatchProcessing(true);

        try {
            const response = await axios.post(
                getRouteUrl("notifications.mark-all-read")
            );

            if (response.data.success) {
                // Refresh the data to get updated notification status
                fetchNotifications();

                // Show success message
                if (window.toast?.success) {
                    window.toast.success("All notifications marked as read");
                }
            } else {
                throw new Error(
                    response.data.message ||
                        "Failed to mark all notifications as read"
                );
            }
        } catch (error) {
            console.error("Error marking all notifications as read:", error);
            if (window.toast?.error) {
                window.toast.error(
                    error.response?.data?.message ||
                        "Failed to mark all notifications as read. Please try again."
                );
            }
        } finally {
            setIsBatchProcessing(false);
        }
    };

    // Delete a notification
    const deleteNotification = async (id) => {
        if (!id) return;

        if (!confirm("Are you sure you want to delete this notification?")) {
            return;
        }

        try {
            const response = await axios.delete(
                getRouteUrl("notifications.delete", { id })
            );

            if (response.data.success) {
                // Remove from UI
                setNotifications((prev) =>
                    prev.filter((item) => item.id !== id)
                );

                // Update selected items if needed
                setSelectedItems((prev) =>
                    prev.filter((itemId) => itemId !== id)
                );

                // Show success message
                if (window.toast?.success) {
                    window.toast.success("Notification deleted successfully");
                }

                // Update counts if needed
                fetchUnreadCount();
            } else {
                throw new Error(
                    response.data.message || "Failed to delete notification"
                );
            }
        } catch (error) {
            console.error("Error deleting notification:", error);
            if (window.toast?.error) {
                window.toast.error(
                    error.response?.data?.message ||
                        "Failed to delete notification. Please try again."
                );
            }
        }
    };

    // Handle notification update (when marked as read)
    const handleNotificationUpdate = (updatedNotification) => {
        if (!updatedNotification?.id) return;

        // Update in the list
        setNotifications((prev) =>
            prev.map((notification) =>
                notification.id === updatedNotification.id
                    ? updatedNotification
                    : notification
            )
        );

        // Update unread count if changed from unread to read
        if (!updatedNotification.read_at) {
            fetchUnreadCount();
        }
    };

    // Handle bulk selection
    const toggleSelectAll = () => {
        if (isSelectAll) {
            setSelectedItems([]);
        } else {
            const allIds = notifications.map((notification) => notification.id);
            setSelectedItems(allIds);
        }
        setIsSelectAll(!isSelectAll);
    };

    // Toggle selection of a single item
    const toggleSelectItem = (id) => {
        setSelectedItems((prev) => {
            if (prev.includes(id)) {
                return prev.filter((itemId) => itemId !== id);
            } else {
                return [...prev, id];
            }
        });
    };

    // Check if an item is selected
    const isSelected = (id) => {
        return selectedItems.includes(id);
    };

    // Handle refresh
    const handleRefresh = () => {
        setIsRefreshing(true);
        fetchNotifications().finally(() => {
            setIsRefreshing(false);
        });
    };

    // Toggle filters display
    const toggleFilters = () => {
        setIsShowingFilters(!isShowingFilters);
    };

    // Handle reads status changes
    const handleReadStatusChange = (value) => {
        let readValue = null;
        if (value === "true") readValue = true;
        if (value === "false") readValue = false;

        applyFilters({ read: readValue });
    };

    // Choose the appropriate layout based on user type
    // Choose the appropriate layout based on user type
    const getLayout = () => {
        if (!auth?.user?.user_type) return ClientLayout;

        switch (auth.user.user_type) {
            case "admin":
                return AdminLayout;
            case "writer":
                return WriterLayout;
            default:
                return ClientLayout;
        }
    };

    const Layout = getLayout();

    // Render notification grid/list based on view mode
    const renderNotificationList = () => {
        if (notifications.length === 0) {
            return (
                <div className="col-span-full flex flex-col items-center justify-center py-12 px-4">
                    <Bell className="h-16 w-16 text-gray-300 dark:text-gray-600 mb-4" />
                    <h3 className="text-lg font-medium text-gray-900 dark:text-white mb-2">
                        No notifications found
                    </h3>
                    <p className="text-gray-500 dark:text-gray-400 text-center max-w-md">
                        {filters.search ||
                        filters.category ||
                        filters.read !== null
                            ? "Try adjusting your filters or search criteria"
                            : "You don't have any notifications yet"}
                    </p>
                </div>
            );
        }

        if (viewMode === "list") {
            // List view for desktop
            return (
                <div className="overflow-hidden rounded-lg border border-gray-200 dark:border-gray-700">
                    <div className="divide-y divide-gray-200 dark:divide-gray-700">
                        {notifications.map((notification) => (
                            <div
                                key={notification.id}
                                className="flex items-center"
                            >
                                <div className="p-2 flex-shrink-0">
                                    <label className="cursor-pointer">
                                        <input
                                            type="checkbox"
                                            className="hidden"
                                            checked={isSelected(
                                                notification.id
                                            )}
                                            onChange={() =>
                                                toggleSelectItem(
                                                    notification.id
                                                )
                                            }
                                        />
                                        <div className="text-gray-400 dark:text-gray-600 hover:text-primary dark:hover:text-primary-foreground transition-colors">
                                            {isSelected(notification.id) ? (
                                                <CheckSquare className="h-5 w-5 text-primary" />
                                            ) : (
                                                <Square className="h-5 w-5" />
                                            )}
                                        </div>
                                    </label>
                                </div>
                                <div className="flex-grow">
                                    <NotificationItem
                                        notification={notification}
                                        onUpdate={handleNotificationUpdate}
                                        getRouteUrl={getRouteUrl}
                                    />
                                </div>
                                <div className="p-4 flex-shrink-0">
                                    <button
                                        onClick={(e) => {
                                            e.preventDefault();
                                            e.stopPropagation();
                                            deleteNotification(notification.id);
                                        }}
                                        className="text-gray-400 hover:text-red-500 dark:text-gray-600 dark:hover:text-red-400 transition-colors"
                                        title="Delete notification"
                                    >
                                        <Trash2 className="h-4 w-4" />
                                    </button>
                                </div>
                            </div>
                        ))}
                    </div>
                </div>
            );
        } else {
            // Grid view (card-based) - better for mobile
            return (
                <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-3 gap-4">
                    {notifications.map((notification) => (
                        <div
                            key={notification.id}
                            className="relative rounded-lg border border-gray-200 dark:border-gray-700 overflow-hidden bg-white dark:bg-gray-800"
                        >
                            <div className="absolute top-2 left-2 z-10">
                                <label className="cursor-pointer">
                                    <input
                                        type="checkbox"
                                        className="hidden"
                                        checked={isSelected(notification.id)}
                                        onChange={() =>
                                            toggleSelectItem(notification.id)
                                        }
                                    />
                                    <div className="w-5 h-5 flex items-center justify-center rounded bg-white dark:bg-gray-700 border border-gray-300 dark:border-gray-600 shadow-sm">
                                        {isSelected(notification.id) && (
                                            <Check className="h-3 w-3 text-primary" />
                                        )}
                                    </div>
                                </label>
                            </div>
                            <div className="absolute top-2 right-2 z-10">
                                <button
                                    onClick={(e) => {
                                        e.preventDefault();
                                        e.stopPropagation();
                                        deleteNotification(notification.id);
                                    }}
                                    className="p-1 rounded-full bg-white dark:bg-gray-700 text-gray-500 hover:text-red-500 dark:text-gray-400 dark:hover:text-red-400 transition-colors shadow-sm"
                                    title="Delete notification"
                                >
                                    <Trash2 className="h-3.5 w-3.5" />
                                </button>
                            </div>
                            <NotificationItem
                                notification={notification}
                                onUpdate={handleNotificationUpdate}
                                getRouteUrl={getRouteUrl}
                            />
                        </div>
                    ))}
                </div>
            );
        }
    };

    return (
        <Layout>
            <Head title="Notifications" />

            <div className="bg-white dark:bg-gray-800 rounded-lg shadow overflow-hidden">
                <div className="p-6 border-b border-gray-200 dark:border-gray-700 bg-gradient-to-r from-blue-100 to-cyan-100 dark:bg-none dark:bg-gray-900 border ">
                    <div className="flex flex-col md:flex-row md:items-center justify-between gap-4">
                        <div className="flex items-center gap-3">
                            <Bell className="h-6 w-6 text-blue-600 dark:text-blue-400" />
                            <h1 className="text-2xl font-bold text-gray-900 dark:text-white">
                                Notifications
                            </h1>

                            {unreadCount > 0 && (
                                <span className="bg-red-500 text-white text-xs font-bold px-2 py-1 rounded-full">
                                    {unreadCount} unread
                                </span>
                            )}
                        </div>

                        <div className="flex flex-wrap items-center gap-3">
                            <button
                                onClick={toggleFilters}
                                className={`flex items-center gap-2 px-3 py-2 rounded-md text-sm font-medium transition-colors ${
                                    isShowingFilters
                                        ? "bg-blue-100 dark:bg-blue-800 text-blue-700 dark:text-blue-200"
                                        : "bg-white dark:bg-gray-700 text-gray-700 dark:text-gray-200 border border-gray-300 dark:border-gray-600"
                                }`}
                            >
                                <Filter className="h-4 w-4" />
                                <span>Filters</span>
                            </button>

                            <button
                                onClick={handleRefresh}
                                disabled={isRefreshing || isLoading}
                                className="flex items-center gap-2 px-3 py-2 bg-white dark:bg-gray-700 border border-gray-300 dark:border-gray-600 rounded-md text-sm font-medium text-gray-700 dark:text-gray-200 hover:bg-gray-50 dark:hover:bg-gray-600 transition-colors"
                            >
                                {isRefreshing ? (
                                    <Loader2 className="h-4 w-4 animate-spin" />
                                ) : (
                                    <RefreshCw className="h-4 w-4" />
                                )}
                                <span className="hidden sm:inline">
                                    Refresh
                                </span>
                            </button>

                            <div className="flex items-center border border-gray-300 dark:border-gray-600 rounded-md overflow-hidden">
                                <button
                                    onClick={() => setViewMode("grid")}
                                    className={`p-2 ${
                                        viewMode === "grid"
                                            ? "bg-blue-100 dark:bg-blue-800 text-blue-700 dark:text-blue-200"
                                            : "bg-white dark:bg-gray-700 text-gray-700 dark:text-gray-200"
                                    }`}
                                    title="Grid view"
                                >
                                    <LayoutGrid className="h-4 w-4" />
                                </button>
                                <button
                                    onClick={() => setViewMode("list")}
                                    className={`p-2 ${
                                        viewMode === "list"
                                            ? "bg-blue-100 dark:bg-blue-800 text-blue-700 dark:text-blue-200"
                                            : "bg-white dark:bg-gray-700 text-gray-700 dark:text-gray-200"
                                    }`}
                                    title="List view"
                                >
                                    <List className="h-4 w-4" />
                                </button>
                            </div>

                            {unreadCount > 0 && (
                                <button
                                    onClick={markAllAsRead}
                                    disabled={isBatchProcessing || isLoading}
                                    className="flex items-center gap-2 px-4 py-2 bg-white dark:bg-gray-700 border border-gray-300 dark:border-gray-600 rounded-md text-sm font-medium text-gray-700 dark:text-gray-200 hover:bg-gray-50 dark:hover:bg-gray-600 transition-colors"
                                >
                                    {isBatchProcessing ? (
                                        <Loader2 className="h-4 w-4 animate-spin" />
                                    ) : (
                                        <Check className="h-4 w-4" />
                                    )}
                                    <span>Mark all as read</span>
                                </button>
                            )}
                        </div>
                    </div>
                </div>

                {/* Filters panel */}
                {isShowingFilters && (
                    <div className="p-4 border-b border-gray-200 dark:border-gray-700 bg-gray-50 dark:bg-gray-850 animate-in slide-in-from-top duration-300">
                        <div className="grid grid-cols-1 md:grid-cols-3 gap-4">
                            <div>
                                <label className="block text-sm font-medium text-gray-700 dark:text-gray-300 mb-1">
                                    Status
                                </label>
                                <select
                                    value={
                                        filters.read === null
                                            ? ""
                                            : filters.read.toString()
                                    }
                                    onChange={(e) =>
                                        handleReadStatusChange(e.target.value)
                                    }
                                    className="w-full rounded-md border border-gray-300 dark:border-gray-600 bg-white dark:bg-gray-700 py-2 px-3 text-sm text-gray-900 dark:text-gray-100"
                                >
                                    {readStatusOptions.map((option) => (
                                        <option
                                            key={option.label}
                                            value={
                                                option.value === null
                                                    ? ""
                                                    : option.value.toString()
                                            }
                                        >
                                            {option.label}
                                        </option>
                                    ))}
                                </select>
                            </div>
                            <div>
                                <label className="block text-sm font-medium text-gray-700 dark:text-gray-300 mb-1">
                                    Category
                                </label>
                                <select
                                    value={filters.category || ""}
                                    onChange={(e) => {
                                        const value = e.target.value;
                                        applyFilters({
                                            category:
                                                value === "" ? null : value,
                                        });
                                    }}
                                    className="w-full rounded-md border border-gray-300 dark:border-gray-600 bg-white dark:bg-gray-700 py-2 px-3 text-sm text-gray-900 dark:text-gray-100"
                                >
                                    {notificationCategories.map((category) => (
                                        <option
                                            key={category.label}
                                            value={category.value || ""}
                                        >
                                            {category.label}
                                        </option>
                                    ))}
                                </select>
                            </div>
                            <div>
                                <label className="block text-sm font-medium text-gray-700 dark:text-gray-300 mb-1">
                                    Date Range
                                </label>
                                <div className="flex items-center gap-2">
                                    <input
                                        type="date"
                                        value={dateRange.fromDate}
                                        onChange={(e) =>
                                            handleDateChange(
                                                "fromDate",
                                                e.target.value
                                            )
                                        }
                                        className="flex-1 rounded-md border border-gray-300 dark:border-gray-600 bg-white dark:bg-gray-700 py-2 px-3 text-sm text-gray-900 dark:text-gray-100"
                                        placeholder="From"
                                    />
                                    <span className="text-gray-500 dark:text-gray-400">
                                        to
                                    </span>
                                    <input
                                        type="date"
                                        value={dateRange.toDate}
                                        onChange={(e) =>
                                            handleDateChange(
                                                "toDate",
                                                e.target.value
                                            )
                                        }
                                        className="flex-1 rounded-md border border-gray-300 dark:border-gray-600 bg-white dark:bg-gray-700 py-2 px-3 text-sm text-gray-900 dark:text-gray-100"
                                        placeholder="To"
                                    />
                                    <button
                                        onClick={applyDateRange}
                                        className="p-2 bg-blue-600 hover:bg-blue-700 text-white rounded-md"
                                        title="Apply date range"
                                    >
                                        <Check className="h-4 w-4" />
                                    </button>
                                </div>
                            </div>
                        </div>
                        <div className="flex justify-between mt-4">
                            <div className="text-sm text-gray-500 dark:text-gray-400">
                                {pagination.total} notifications found
                            </div>
                            <button
                                onClick={resetFilters}
                                className="text-sm text-blue-600 dark:text-blue-400 hover:text-blue-800 dark:hover:text-blue-300"
                            >
                                Reset filters
                            </button>
                        </div>
                    </div>
                )}

                {/* Search bar */}
                <div className="p-4 border-b border-gray-200 dark:border-gray-700 bg-white dark:bg-gray-800">
                    <div className="relative max-w-md mx-auto md:mx-0">
                        <Search className="absolute left-3 top-1/2 transform -translate-y-1/2 text-gray-400 dark:text-gray-500 h-4 w-4" />
                        <input
                            type="text"
                            placeholder="Search notifications..."
                            className="w-full pl-10 pr-4 py-2 border border-gray-300 dark:border-gray-600 rounded-md bg-white dark:bg-gray-700 text-gray-900 dark:text-gray-100"
                            value={filters.search || ""}
                            onChange={handleSearchChange}
                        />
                        {filters.search && (
                            <button
                                onClick={() => applyFilters({ search: "" })}
                                className="absolute right-3 top-1/2 transform -translate-y-1/2 text-gray-400 hover:text-gray-600 dark:text-gray-500 dark:hover:text-gray-300"
                            >
                                <X className="h-4 w-4" />
                            </button>
                        )}
                    </div>
                </div>

                {/* Batch actions */}
                {selectedItems.length > 0 && (
                    <div className="p-2 border-b border-gray-200 dark:border-gray-700 bg-blue-50 dark:bg-blue-900/20 flex items-center justify-between">
                        <div className="flex items-center gap-2 text-sm text-gray-600 dark:text-gray-300">
                            <CheckSquare className="h-4 w-4 text-blue-600 dark:text-blue-400" />
                            <span>{selectedItems.length} selected</span>
                        </div>
                        <div className="flex items-center gap-2">
                            <button
                                onClick={markMultipleAsRead}
                                disabled={isBatchProcessing}
                                className="flex items-center gap-1 px-3 py-1 text-xs font-medium rounded-md bg-blue-600 text-white hover:bg-blue-700 transition-colors"
                            >
                                {isBatchProcessing ? (
                                    <Loader2 className="h-3 w-3 animate-spin" />
                                ) : (
                                    <Check className="h-3 w-3" />
                                )}
                                <span>Mark as read</span>
                            </button>
                            <button
                                onClick={() => setSelectedItems([])}
                                className="flex items-center gap-1 px-3 py-1 text-xs font-medium rounded-md bg-gray-200 dark:bg-gray-700 text-gray-700 dark:text-gray-300 hover:bg-gray-300 dark:hover:bg-gray-600 transition-colors"
                            >
                                <X className="h-3 w-3" />
                                <span>Clear selection</span>
                            </button>
                        </div>
                    </div>
                )}

                {/* Notifications list/grid */}
                <div className="p-4">
                    {/* Select all header for list view */}
                    {viewMode === "list" && notifications.length > 0 && (
                        <div className="flex items-center mb-2 px-2">
                            <label className="cursor-pointer flex items-center gap-2">
                                <input
                                    type="checkbox"
                                    className="hidden"
                                    checked={isSelectAll}
                                    onChange={toggleSelectAll}
                                />
                                <div className="text-gray-400 dark:text-gray-600 hover:text-primary dark:hover:text-primary-foreground transition-colors">
                                    {isSelectAll ? (
                                        <CheckSquare className="h-5 w-5 text-primary" />
                                    ) : (
                                        <Square className="h-5 w-5" />
                                    )}
                                </div>
                                <span className="text-sm text-gray-500 dark:text-gray-400">
                                    Select all
                                </span>
                            </label>

                            {/* Sort controls */}
                            <div className="ml-auto flex items-center gap-4">
                                <button
                                    onClick={() => handleSort("created_at")}
                                    className="flex items-center gap-1 text-sm text-gray-500 dark:text-gray-400 hover:text-gray-700 dark:hover:text-gray-200"
                                >
                                    <span>Date</span>
                                    <ArrowUpDown className="h-3 w-3" />
                                </button>
                                <button
                                    onClick={() => handleSort("read_at")}
                                    className="flex items-center gap-1 text-sm text-gray-500 dark:text-gray-400 hover:text-gray-700 dark:hover:text-gray-200"
                                >
                                    <span>Status</span>
                                    <ArrowUpDown className="h-3 w-3" />
                                </button>
                            </div>
                        </div>
                    )}

                    {/* Loading state */}
                    {isLoading ? (
                        <div className="flex justify-center items-center py-12">
                            <div className="animate-spin rounded-full h-12 w-12 border-b-2 border-blue-600 dark:border-blue-400"></div>
                        </div>
                    ) : error ? (
                        <div className="py-12 text-center">
                            <AlertTriangle className="h-12 w-12 text-red-500 dark:text-red-400 mx-auto mb-4" />
                            <h3 className="text-lg font-medium text-red-600 dark:text-red-400 mb-2">
                                {error}
                            </h3>
                            <button
                                onClick={handleRefresh}
                                className="mt-4 px-4 py-2 bg-gray-100 dark:bg-gray-700 hover:bg-gray-200 dark:hover:bg-gray-600 rounded-md text-sm text-gray-700 dark:text-gray-300 flex items-center gap-2 mx-auto"
                            >
                                <RefreshCw className="h-4 w-4" />
                                <span>Try again</span>
                            </button>
                        </div>
                    ) : (
                        renderNotificationList()
                    )}
                </div>

                {/* Pagination */}
                {pagination.last_page > 1 && (
                    <div className="p-4 border-t border-gray-200 dark:border-gray-700 bg-gray-50 dark:bg-gray-850 flex flex-col sm:flex-row items-center justify-between gap-4">
                        <div className="text-sm text-gray-500 dark:text-gray-400">
                            Showing{" "}
                            {(pagination.current_page - 1) *
                                pagination.per_page +
                                1}{" "}
                            to{" "}
                            {Math.min(
                                pagination.current_page * pagination.per_page,
                                pagination.total
                            )}{" "}
                            of {pagination.total} notifications
                        </div>

                        <div className="flex items-center gap-2">
                            <button
                                onClick={() =>
                                    changePage(pagination.current_page - 1)
                                }
                                disabled={pagination.current_page === 1}
                                className={`p-2 rounded-md ${
                                    pagination.current_page === 1
                                        ? "bg-gray-200 dark:bg-gray-700 text-gray-400 dark:text-gray-500 cursor-not-allowed"
                                        : "bg-white dark:bg-gray-700 text-gray-700 dark:text-gray-200 hover:bg-gray-100 dark:hover:bg-gray-600"
                                }`}
                                aria-label="Previous page"
                            >
                                <ChevronLeft className="h-5 w-5" />
                            </button>

                            <div className="flex items-center gap-1">
                                {/* Page numbers - show limited range for many pages */}
                                {Array.from({
                                    length: Math.min(5, pagination.last_page),
                                }).map((_, i) => {
                                    // Calculate which page numbers to show
                                    let pageNum;
                                    if (pagination.last_page <= 5) {
                                        pageNum = i + 1;
                                    } else if (pagination.current_page <= 3) {
                                        pageNum = i + 1;
                                    } else if (
                                        pagination.current_page >=
                                        pagination.last_page - 2
                                    ) {
                                        pageNum = pagination.last_page - 4 + i;
                                    } else {
                                        pageNum =
                                            pagination.current_page - 2 + i;
                                    }

                                    // Only render if pageNum is valid
                                    if (
                                        pageNum > 0 &&
                                        pageNum <= pagination.last_page
                                    ) {
                                        return (
                                            <button
                                                key={pageNum}
                                                onClick={() =>
                                                    changePage(pageNum)
                                                }
                                                className={`min-w-[2.5rem] h-10 rounded-md ${
                                                    pageNum ===
                                                    pagination.current_page
                                                        ? "bg-blue-600 text-white"
                                                        : "bg-white dark:bg-gray-700 text-gray-700 dark:text-gray-200 hover:bg-gray-100 dark:hover:bg-gray-600"
                                                }`}
                                            >
                                                {pageNum}
                                            </button>
                                        );
                                    }
                                    return null;
                                })}
                            </div>

                            <button
                                onClick={() =>
                                    changePage(pagination.current_page + 1)
                                }
                                disabled={
                                    pagination.current_page ===
                                    pagination.last_page
                                }
                                className={`p-2 rounded-md ${
                                    pagination.current_page ===
                                    pagination.last_page
                                        ? "bg-gray-200 dark:bg-gray-700 text-gray-400 dark:text-gray-500 cursor-not-allowed"
                                        : "bg-white dark:bg-gray-700 text-gray-700 dark:text-gray-200 hover:bg-gray-100 dark:hover:bg-gray-600"
                                }`}
                                aria-label="Next page"
                            >
                                <ChevronRight className="h-5 w-5" />
                            </button>
                        </div>

                        {/* Per page selector */}
                        <div className="flex items-center gap-2 sm:hidden md:flex">
                            <span className="text-sm text-gray-500 dark:text-gray-400">
                                Show
                            </span>
                            <select
                                value={filters.per_page}
                                onChange={(e) =>
                                    changePerPage(parseInt(e.target.value))
                                }
                                className="rounded-md border border-gray-300 dark:border-gray-600 bg-white dark:bg-gray-700 py-1 px-2 text-sm text-gray-900 dark:text-gray-100"
                            >
                                <option value="10">10</option>
                                <option value="15">15</option>
                                <option value="25">25</option>
                                <option value="50">50</option>
                            </select>
                            <span className="text-sm text-gray-500 dark:text-gray-400">
                                per page
                            </span>
                        </div>
                    </div>
                )}
            </div>
        </Layout>
    );
}
