import { Head, Link, useForm, router } from "@inertiajs/react";
import React from "react";
import { Eye, Search, X, Plus, RefreshCw, Gavel } from "lucide-react";

import ExpiresIn from "@/Components/ExpiresIn";
import ClientLayout from "@/Layouts/ClientLayout";
import CleanPagination from "@/Components/CleanPagination";

export default function MyOrders({ orders }) {
    const urlParams = new URLSearchParams(window.location.search);
    const searchParam = urlParams.get("search") || "";

    const { data, setData, get } = useForm({
        search: searchParam,
    });
    const handleSearch = (e) => {
        e.preventDefault();
        get(route("orders.index"), {
            preserveScroll: true,
            onSuccess: () => {
                setData("search", "");
            },
        });
    };

    const clearSearch = () => {
        // Use router directly instead of the form's get method
        router.visit(route("orders.index"), {
            method: "get",
            data: { search: "" },
            preserveScroll: true,
            onSuccess: () => {
                // Update local form state after successful navigation
                setData("search", "");
            },
        });
    };

    // Function to format currency with the correct currency symbol
    const formatCurrency = (amount, currencyCode = "USD") => {
        // Get the appropriate currency symbol
        const currencySymbol = getCurrencySymbol(currencyCode);

        // Format the number with 2 decimal places
        const formattedAmount = parseFloat(amount).toFixed(2);

        // Return the formatted currency string
        return `${currencySymbol}${formattedAmount}`;
    };

    // Helper function to get currency symbol from currency code
    const getCurrencySymbol = (currencyCode) => {
        switch (currencyCode?.toUpperCase()) {
            case "EUR":
                return "€";
            case "GBP":
                return "£";
            case "CAD":
                return "C$";
            case "AUD":
                return "A$";
            default:
                return "$";
        }
    };

    // Function to get refund status badge
    const getRefundStatusBadge = (order) => {
        if (order.is_refunded) {
            return (
                <span className="px-3 py-1 ml-2 rounded-full text-xs font-medium bg-red-100 text-red-800 dark:bg-red-900 dark:text-red-200 flex items-center">
                    <RefreshCw size={12} className="mr-1" />
                    Fully Refunded
                </span>
            );
        } else if (order.is_partially_refunded) {
            return (
                <span className="px-3 py-1 ml-2 rounded-full text-xs font-medium bg-orange-100 text-orange-800 dark:bg-orange-900 dark:text-orange-200 flex items-center">
                    <RefreshCw size={12} className="mr-1" />
                    Partially Refunded
                </span>
            );
        }
        return null;
    };

    // Function to get bidding status badge
    const getBiddingStatusBadge = (order) => {
        if (order.order_status === "bidding") {
            return (
                <span className="px-3 py-1 ml-2 rounded-full text-xs font-medium bg-blue-100 text-blue-800 dark:bg-blue-900 dark:text-blue-200 flex items-center">
                    <Gavel size={12} className="mr-1" />
                    Bidding Active
                </span>
            );
        }
        return null;
    };

    // Function to get bids count or assignment status
    const getBidsInfo = (order) => {
        if (order.is_assigned) {
            return (
                <span className="px-3 py-1 ml-2 rounded-full text-xs font-medium bg-green-100 text-green-800 dark:bg-green-900 dark:text-green-200 flex items-center">
                    ✓ Already Assigned
                </span>
            );
        } else if (order.order_status === "bidding" && order.bids_count > 0) {
            return (
                <span className="px-3 py-1 ml-2 rounded-full text-xs font-medium bg-purple-100 text-purple-800 dark:bg-purple-900 dark:text-purple-200 flex items-center">
                    {order.bids_count} {order.bids_count === 1 ? 'Bid' : 'Bids'}
                </span>
            );
        } else if (order.order_status === "bidding") {
            return (
                <span className="px-3 py-1 ml-2 rounded-full text-xs font-medium bg-gray-100 text-gray-800 dark:bg-gray-900 dark:text-gray-200 flex items-center">
                    No Bids Yet
                </span>
            );
        }
        return null;
    };

    return (
        <ClientLayout
            header={
                <div className="text-xl font-semibold leading-tight text-gray-800 dark:text-gray-100">
                    My Orders
                </div>
            }
        >
            <Head title="My Orders" />

            <div className="py-12">
                <div className="mx-auto max-w-7xl sm:px-2 lg:px-">
                    <div className="overflow-hidden dark:bg-gray-800 shadow-sm sm:rounded-lg">
                        <div className="p-6 text-gray-900 dark:text-gray-200">
                            <div className="mb-6">
                                <div className="flex flex-col md:flex-row md:items-center md:justify-between">
                                    <div className="flex flex-col md:flex-row md:items-center">
                                        <form
                                            onSubmit={handleSearch}
                                            className="mb-2 md:mb-0"
                                        >
                                            <div className="relative">
                                                <Search className="absolute left-3 top-1/2 transform -translate-y-1/2 text-gray-400 h-5 w-5" />
                                                <input
                                                    type="search"
                                                    placeholder="Search orders..."
                                                    className="w-full md:w-64 pl-10 pr-4 py-2 border border-gray-300 dark:border-gray-600 rounded-lg focus:ring-2 focus:ring-indigo-500 dark:focus:ring-indigo-400 focus:border-transparent bg-white dark:bg-gray-700 text-gray-900 dark:text-gray-100 placeholder-gray-500 dark:placeholder-gray-400"
                                                    value={data.search}
                                                    onChange={(e) =>
                                                        setData(
                                                            "search",
                                                            e.target.value
                                                        )
                                                    }
                                                />
                                                <button
                                                    type="submit"
                                                    className="absolute left-3 top-2.5 text-gray-500"
                                                ></button>
                                            </div>
                                        </form>
                                        {/* Search Tag - Desktop (right of search box) */}
                                        <div className="hidden md:flex md:flex-row items-center">
                                            {searchParam && (
                                                <div className="ml-4 bg-indigo-100 text-indigo-800 py-1 px-3 rounded-md flex items-center text-sm">
                                                    <span className="mr-2">
                                                        Search: {searchParam}
                                                    </span>
                                                    <button
                                                        onClick={clearSearch}
                                                        type="button"
                                                        className="text-indigo-600 hover:text-indigo-800"
                                                    >
                                                        <X size={16} />
                                                    </button>
                                                </div>
                                            )}
                                        </div>
                                    </div>

                                    {/* Add New Button - Both Desktop and Mobile */}
                                    <Link
                                        href={route("order.create")}
                                        className="flex items-center justify-center mt-4 md:mt-0 bg-gradient-to-r from-blue-600 to-cyan-500 hover:from-blue-700 hover:to-cyan-600 dark:bg-indigo-600 dark:hover:bg-indigo-700 text-white px-4 py-2 rounded-md transition-colors duration-200"
                                    >
                                        <Plus className="w-5 h-5 mr-2" />
                                        <span>Place Order</span>
                                    </Link>
                                </div>

                                {/* Search Tag - Mobile (below search box) */}
                                {searchParam && (
                                    <div className="md:hidden mt-2 flex">
                                        <div className="bg-indigo-100 text-indigo-800 py-1 px-3 rounded-md flex items-center text-sm">
                                            <span className="mr-2">
                                                Search: {searchParam}
                                            </span>
                                            <button
                                                onClick={clearSearch}
                                                type="button"
                                                className="text-indigo-600 hover:text-indigo-800"
                                            >
                                                <X size={16} />
                                            </button>
                                        </div>
                                    </div>
                                )}
                            </div>
                            {orders.data.length > 0 ? (
                                <>
                                    <div className="hidden lg:flex bg-gradient-to-r from-blue-600 to-cyan-500 dark:bg-none text-white dark:bg-gray-700 rounded-t-lg p-4 font-bold dark:text-gray-300">
                                        <div className="flex-1">
                                            Order Number
                                        </div>
                                        <div className="flex-1">Posted On</div>
                                        <div className="flex-1">Time Left</div>
                                        <div className="flex-1">
                                            Order Status
                                        </div>
                                        <div className="flex-1">
                                            Payment Status
                                        </div>
                                        <div className="flex-1">Bids/Status</div>
                                        <div className="flex-1">Amount</div>
                                        <div className="flex-1 text-center">
                                            Action
                                        </div>
                                    </div>

                                    <div className="space-y-4 lg:space-y-0">
                                        {orders.data.map((order) => (
                                            <div
                                                key={order.id}
                                                className="bg-gradient-to-r from-blue-100 to-cyan-100 dark:bg-none dark:bg-gray-900 shadow rounded-lg border border-gray-300 dark:border-gray-600 overflow-hidden"
                                            >
                                                                                                <div className="hidden lg:flex items-center p-4 hover:bg-cyan-100 dark:hover:bg-gray-800">
                                                    <div className="flex-1 font-medium text-indigo-600 dark:text-indigo-400 hover:text-indigo-500 dark:hover:text-indigo-300">
                                                        <Link
                                                            href={`/orders/${order.id}`}
                                                        >
                                                            {order.order_number}
                                                        </Link>
                                                    </div>
                                                    <div className="flex-1">
                                                        {new Date(
                                                            order.dateposted
                                                        ).toLocaleString(
                                                            "en-US",
                                                            {
                                                                year: "numeric",
                                                                month: "short",
                                                                day: "numeric",
                                                                hour: "2-digit",
                                                                minute: "2-digit",
                                                            }
                                                        )}
                                                    </div>

                                                    <div className="flex-1">
                                                        <ExpiresIn
                                                            deadline={
                                                                order.deadline
                                                            }
                                                        />
                                                    </div>
                                                    <div className="flex-1">
                                                        <div className="flex items-center">
                                                            <span
                                                                className={`px-3 py-2 rounded-full text-sm font-medium ${
                                                                    order.order_status ===
                                                                    "completed"
                                                                        ? "bg-green-500 text-green-50"
                                                                        : order.order_status ===
                                                                          "pending"
                                                                        ? "bg-yellow-500 text-yellow-100"
                                                                        : order.order_status ===
                                                                          "bidding"
                                                                        ? "bg-blue-500 text-blue-50"
                                                                        : "bg-red-700 text-red-50"
                                                                }`}
                                                            >
                                                                {
                                                                    order.order_status
                                                                }
                                                            </span>
                                                            {getRefundStatusBadge(
                                                                order
                                                            )}
                                                            {getBiddingStatusBadge(
                                                                order
                                                            )}
                                                        </div>
                                                    </div>
                                                    <div className="flex-1">
                                                        <span
                                                            className={`px-3 py-2 rounded-full text-sm font-medium ${
                                                                order.payment_status ===
                                                                "paid"
                                                                    ? "bg-green-500 text-green-50"
                                                                    : order.payment_status ===
                                                                      "pending"
                                                                    ? "bg-yellow-500 text-yellow-100"
                                                                    : "bg-red-700 text-red-100"
                                                            }`}
                                                        >
                                                            {
                                                                order.payment_status
                                                            }
                                                        </span>
                                                    </div>
                                                    <div className="flex-1">
                                                        {getBidsInfo(order)}
                                                    </div>
                                                    <div className="flex-1">
                                                        <div className="flex flex-col">
                                                            <span className="font-medium">
                                                                {formatCurrency(
                                                                    order.net_amount,
                                                                    order.currency
                                                                )}
                                                            </span>
                                                            {order.refund_info &&
                                                                order
                                                                    .refund_info
                                                                    .has_refunds && (
                                                                    <span className="text-xs text-red-600 dark:text-red-400">
                                                                        Refunded:{" "}
                                                                        {formatCurrency(
                                                                            order
                                                                                .refund_info
                                                                                .refunded_amount,
                                                                            order.currency
                                                                        )}
                                                                    </span>
                                                                )}
                                                        </div>
                                                    </div>

                                                    <div className="flex-1">
                                                        <Link
                                                            href={`/orders/${order.id}`}
                                                        >
                                                            <button className="w-full bg-gradient-to-r from-blue-900 to-cyan-500 dark:bg-indigo-500 text-white py-2 rounded-md hover:bg-blue-700 dark:hover:bg-indigo-400 transition-colors duration-200">
                                                                <Eye className="w-6 h-6 inline-block mr-2" />
                                                            </button>
                                                        </Link>
                                                    </div>
                                                </div>

                                                <div className="lg:hidden p-4 bg-gradient-to-r from-blue-100 to-cyan-100 dark:bg-none dark:bg-gray-900 shadow rounded-lg border border-gray-300 dark:border-gray-600">
                                                    {/* Header Section */}
                                                    <div className="flex justify-between items-start mb-4">
                                                        <div className="flex-1">
                                                            <h2 className="text-lg font-semibold text-indigo-600 dark:text-indigo-400 mb-1">
                                                                <Link
                                                                    href={`/orders/${order.id}`}
                                                                    className="hover:text-indigo-500 dark:hover:text-indigo-300 transition-colors"
                                                                >
                                                                    Order #{order.order_number}
                                                                </Link>
                                                            </h2>
                                                            <p className="text-sm text-gray-600 dark:text-gray-400">
                                                                {new Date(order.dateposted).toLocaleString("en-US", {
                                                                    year: "numeric",
                                                                    month: "short",
                                                                    day: "numeric",
                                                                    hour: "2-digit",
                                                                    minute: "2-digit",
                                                                })}
                                                            </p>
                                                        </div>
                                                        <div className="flex flex-col items-end space-y-1">
                                                            <span
                                                                className={`px-3 py-1 rounded-full text-xs font-medium ${
                                                                    order.order_status === "completed"
                                                                        ? "bg-green-500 text-green-50"
                                                                        : order.order_status === "pending"
                                                                        ? "bg-yellow-500 text-yellow-100"
                                                                        : order.order_status === "bidding"
                                                                        ? "bg-blue-500 text-blue-50"
                                                                        : "bg-red-700 text-red-50"
                                                                }`}
                                                            >
                                                                {order.order_status}
                                                            </span>
                                                            <span
                                                                className={`px-3 py-1 rounded-full text-xs font-medium ${
                                                                    order.payment_status === "paid"
                                                                        ? "bg-green-500 text-green-50"
                                                                        : order.payment_status === "pending"
                                                                        ? "bg-yellow-500 text-yellow-100"
                                                                        : "bg-red-700 text-red-100"
                                                                }`}
                                                            >
                                                                {order.payment_status}
                                                            </span>
                                                        </div>
                                                    </div>

                                                    {/* Status Badges Row */}
                                                    <div className="flex flex-wrap gap-2 mb-4">
                                                        {getRefundStatusBadge(order)}
                                                        {getBiddingStatusBadge(order)}
                                                        {getBidsInfo(order)}
                                                    </div>

                                                    {/* Info Grid */}
                                                    <div className="grid grid-cols-2 gap-4 mb-4">
                                                        <div className="bg-white dark:bg-gray-800 p-3 rounded-lg border border-gray-200 dark:border-gray-700">
                                                            <p className="text-xs text-gray-500 dark:text-gray-400 font-medium uppercase tracking-wide mb-1">
                                                                Time Left
                                                            </p>
                                                            <div className="text-sm font-medium">
                                                                <ExpiresIn deadline={order.deadline} />
                                                            </div>
                                                        </div>
                                                        <div className="bg-white dark:bg-gray-800 p-3 rounded-lg border border-gray-200 dark:border-gray-700">
                                                            <p className="text-xs text-gray-500 dark:text-gray-400 font-medium uppercase tracking-wide mb-1">
                                                                Amount
                                                            </p>
                                                            <div className="text-sm font-medium text-green-600 dark:text-green-400">
                                                                {formatCurrency(order.net_amount, order.currency)}
                                                            </div>
                                                            {order.refund_info && order.refund_info.has_refunds && (
                                                                <div className="text-xs text-red-600 dark:text-red-400 mt-1">
                                                                    Refunded: {formatCurrency(order.refund_info.refunded_amount, order.currency)}
                                                                </div>
                                                            )}
                                                        </div>
                                                    </div>

                                                    {/* Action Button */}
                                                    <div className="mt-4">
                                                        <Link
                                                            href={`/orders/${order.id}`}
                                                            className="inline-flex w-full items-center justify-center px-4 py-3 bg-gradient-to-r from-blue-600 to-cyan-500 border border-transparent rounded-lg font-semibold text-sm text-white hover:from-blue-700 hover:to-cyan-600 active:bg-blue-900 focus:outline-none focus:ring-2 focus:ring-blue-500 focus:ring-offset-2 disabled:opacity-25 transition-all duration-200 shadow-sm"
                                                        >
                                                            <Eye className="w-4 h-4 mr-2" />
                                                            View Order Details
                                                        </Link>
                                                    </div>
                                                </div>
                                            </div>
                                        ))}
                                    </div>
                                </>
                            ) : (
                                <div className="text-center p-12">
                                    <div className="inline-flex items-center justify-center w-16 h-16 rounded-full bg-indigo-100 text-indigo-600 mb-4">
                                        <svg
                                            className="w-8 h-8"
                                            fill="none"
                                            viewBox="0 0 24 24"
                                            stroke="currentColor"
                                        >
                                            <path
                                                strokeLinecap="round"
                                                strokeLinejoin="round"
                                                strokeWidth={2}
                                                d="M12 9v2m0 4h.01m-6.938 4h13.856c1.54 0 2.502-1.667 1.732-3L13.732 4c-.77-1.333-2.694-1.333-3.464 0L3.34 16c-.77 1.333.192 3 1.732 3z"
                                            />
                                        </svg>
                                    </div>
                                    <h3 className="text-lg font-medium text-gray-900 dark:text-gray-100 mb-2">
                                        No orders found
                                    </h3>
                                    <p className="text-gray-500 dark:text-gray-400 mb-6">
                                        {searchParam
                                            ? `No orders matching "${searchParam}" were found.`
                                            : "You haven't placed any orders yet."}
                                    </p>
                                    <Link
                                        href={route("order.create")}
                                        className="inline-flex items-center px-4 py-2 bg-gradient-to-r from-blue-600 to-indigo-500 border border-transparent rounded-md font-semibold text-xs text-white uppercase tracking-widest hover:from-blue-700 hover:to-indigo-600 active:bg-indigo-900 focus:outline-none focus:border-indigo-900 focus:ring ring-indigo-300 disabled:opacity-25 transition ease-in-out duration-150"
                                    >
                                        <Plus className="w-4 h-4 mr-2" />
                                        Place Your First Order
                                    </Link>
                                </div>
                            )}

                            <div className="mt-4">
                                <CleanPagination
                                    links={orders.links}
                                    total={orders.total}
                                    perPage={orders.per_page}
                                    currentPage={orders.current_page}
                                />
                            </div>
                        </div>
                    </div>
                </div>
            </div>
        </ClientLayout>
    );
}
