import React, { useState, useEffect } from "react";
import { Head, Link } from "@inertiajs/react";
import ClientLayout from "@/Layouts/ClientLayout";
import PayPalButton from "@/Components/PayPalButton";
import {
    FileText,
    Clock,
    Notebook,
    Timer,
    MoveLeft,
    Vote,
    ShoppingCart,
    User,
    Upload,
    ArrowDown,
    CheckCircle,
    X,
    AlertCircle,
    GraduationCap,
    Briefcase,
    Award,
    CheckCircle2,
    CreditCard,
    Calendar,
    DollarSign,
    Users,
    PenTool,
    XCircle,
    Repeat,
    Info,
    Download,
} from "lucide-react";
import ClearDeadlineCountdown from "@/Components/ClearDeadlineCountdown";
import FileUpload from "@/Components/FileUpload";
import FileManagement from "@/Components/FileManagement";
import { format, formatDistance } from "date-fns";
import axios from "axios";

export default function Show({
    order,
    bids = [],
    isAdmin = false,
    refundInfo = null,
}) {
    const [activeTab, setActiveTab] = useState("details"); // State to manage active tab

    // Add these state variables
    const [showBidActionModal, setShowBidActionModal] = useState(false);
    const [selectedBid, setSelectedBid] = useState(null);

    const [flashMessage, setFlashMessage] = useState("");
    const [flashType, setFlashType] = useState(""); // "success" or "error"

    const [showWriterDetailsModal, setShowWriterDetailsModal] = useState(false);
    const [writerDetails, setWriterDetails] = useState(null);
    const [isLoadingWriterDetails, setIsLoadingWriterDetails] = useState(false);
    const [writerDetailsError, setWriterDetailsError] = useState("");

    // Function to fetch writer details
    const fetchWriterDetails = async (writerId) => {
        setIsLoadingWriterDetails(true);
        setWriterDetailsError("");

        try {
            const response = await axios.get(`/api/writer-details/${writerId}`);
            setWriterDetails(response.data);
        } catch (error) {
            setWriterDetailsError(
                error.response?.data?.message ||
                    "Failed to fetch writer details. Please try again."
            );
        } finally {
            setIsLoadingWriterDetails(false);
        }
    };

    // Functions to handle modal
    const openWriterDetailsModal = (writerId) => {
        fetchWriterDetails(writerId);
        setShowWriterDetailsModal(true);
    };

    const closeWriterDetailsModal = () => {
        setShowWriterDetailsModal(false);
        setWriterDetails(null);
        setWriterDetailsError("");
    };

    // Add after the state declarations
    const openBidActionModal = (bid) => {
        setSelectedBid(bid);
        setShowBidActionModal(true);
    };

    const closeBidActionModal = () => {
        setShowBidActionModal(false);
        setSelectedBid(null);
    };

    const acceptBid = async () => {
        try {
            const response = await axios.post(
                `/orders/${order.id}/bids/${selectedBid.id}/accept`
            );
            setFlashMessage(
                response.data.message || "Bid accepted successfully!"
            );
            setFlashType("success");
            closeBidActionModal();

            // Redirect to refresh the page after a short delay
            setTimeout(() => {
                window.location.href = `/orders/${order.id}`;
            }, 1500);
        } catch (error) {
            setFlashMessage(
                error.response?.data?.message || "Failed to accept bid"
            );
            setFlashType("error");
        }
    };

    // Add this effect to display flash messages
    useEffect(() => {
        if (flashMessage) {
            const timer = setTimeout(() => {
                setFlashMessage("");
                setFlashType("");
            }, 3000);
            return () => clearTimeout(timer);
        }
    }, [flashMessage]);

    const capitalize = (string) => {
        return string.charAt(0).toUpperCase() + string.slice(1);
    };

    // Navigate to payment options page
    const goToPaymentOptions = () => {
        window.location.href = route("orders.payment", order.id);
    };

    // Get currency symbol
    const getCurrencySymbol = (currencyCode = order.currency || "USD") => {
        const symbols = {
            USD: "$",
            EUR: "€",
            GBP: "£",
            CAD: "C$",
            AUD: "A$",
        };
        return symbols[currencyCode] || "$";
    };

    // Format currency helper with order currency
    const formatCurrency = (amount, useCurrency = true) => {
        const currency = useCurrency ? order.currency || "USD" : "USD";
        const symbol = getCurrencySymbol(currency);

        return `${symbol}${new Intl.NumberFormat("en-US", {
            minimumFractionDigits: 2,
            maximumFractionDigits: 2,
        }).format(amount)}`;
    };

    // Format USD equivalent if currency is not USD
    const getUSDEquivalent = (amount) => {
        if (
            !order.currency ||
            order.currency === "USD" ||
            !order.currency_rate
        ) {
            return null;
        }

        // Convert back to USD
        const usdAmount = amount / order.currency_rate;
        return `($${usdAmount.toFixed(2)} USD)`;
    };

    // Format date helper
    const formatDate = (dateString, formatStr = "PPP p") => {
        if (!dateString) return "N/A";
        try {
            return format(new Date(dateString), formatStr);
        } catch (e) {
            return dateString;
        }
    };

    // Get status badge class
    const getStatusBadgeClass = (status) => {
        switch (status?.toLowerCase()) {
            case "bidding":
                return "bg-blue-100 text-blue-800 dark:bg-blue-900/50 dark:text-blue-300";
            case "pending":
            case "unpaid":
                return "bg-yellow-100 text-yellow-800 dark:bg-yellow-900/50 dark:text-yellow-300";
            case "writing":
            case "in progress":
                return "bg-purple-100 text-purple-800 dark:bg-purple-900/50 dark:text-purple-300";
            case "completed":
            case "paid":
                return "bg-green-100 text-green-800 dark:bg-green-900/50 dark:text-green-300";
            case "cancelled":
            case "failed":
                return "bg-red-100 text-red-800 dark:bg-red-900/50 dark:text-red-300";
            case "refunded":
                return "bg-orange-100 text-orange-800 dark:bg-orange-900/50 dark:text-orange-300";
            case "partially_refunded":
            case "partially refunded":
                return "bg-amber-100 text-amber-800 dark:bg-amber-900/50 dark:text-amber-300";
            default:
                return "bg-gray-100 text-gray-800 dark:bg-gray-700 dark:text-gray-300";
        }
    };

    // Get status icon
    const getStatusIcon = (status) => {
        switch (status?.toLowerCase()) {
            case "bidding":
                return <Users size={20} />;
            case "pending":
            case "unpaid":
                return <Clock size={20} />;
            case "writing":
            case "in progress":
                return <PenTool size={20} />;
            case "completed":
            case "paid":
                return <CheckCircle size={20} />;
            case "cancelled":
            case "failed":
                return <XCircle size={20} />;
            case "refunded":
                return <Repeat size={20} />;
            case "partially_refunded":
            case "partially refunded":
                return <Repeat size={20} />;
            default:
                return <Info size={20} />;
        }
    };

    return (
        <ClientLayout>
            <Head title={`Order #${order.order_number}`} />
            <div className="w-full mx-auto px-4 sm:px-4 lg:px-4 py-2 space-y-6 dark:bg-gray-900 overflow-x-hidden">
                <div className="flex justify-end items-center mb-6">
                    <a
                        href="/orders"
                        className="flex items-center px-3 py-1.5 sm:px-4 sm:py-2 text-indigo-600 dark:text-gray-100 bg-indigo-100 dark:bg-indigo-900 hover:bg-indigo-100 dark:hover:bg-indigo-800 rounded-full shadow-sm border border-indigo-200 dark:border-indigo-700 transition-colors duration-200 text-sm"
                    >
                        <MoveLeft className="mr-1 h-3 w-3 sm:h-4 sm:w-4" />
                        <span className="font-medium">All Orders</span>
                    </a>
                </div>

                {/* Tabs */}
                <div className="border-b border-gray-200 dark:border-gray-700">
                    <nav
                        className="flex w-full border-b border-gray-200 dark:border-gray-700 px-2 sm:px-6 lg:px-4 rounded-lg"
                        aria-label="Tabs"
                    >
                        <button
                            onClick={() => setActiveTab("details")}
                            className={`
                                relative w-1/2 px-4 py-3 font-medium text-sm transition-colors duration-200
                                ${
                                    activeTab === "details"
                                        ? "bg-blue-600 dark:bg-gray-800 text-white dark:text-indigo-400 border-t border-l border-r border-gray-200 dark:border-gray-700 rounded-lg"
                                        : "text-gray-600 dark:text-gray-300 border border-gray-200 dark:border-gray-700 hover:text-indigo-500 dark:hover:text-indigo-300"
                                }
                            `}
                        >
                            <span className="flex items-center justify-center">
                                <svg
                                    className="w-4 h-4 mr-2"
                                    fill="none"
                                    stroke="currentColor"
                                    viewBox="0 0 24 24"
                                    xmlns="http://www.w3.org/2000/svg"
                                >
                                    <path
                                        strokeLinecap="round"
                                        strokeLinejoin="round"
                                        strokeWidth={2}
                                        d="M9 5H7a2 2 0 00-2 2v12a2 2 0 002 2h10a2 2 0 002-2V7a2 2 0 00-2-2h-2M9 5a2 2 0 002 2h2a2 2 0 002-2M9 5a2 2 0 012-2h2a2 2 0 012 2"
                                    />
                                </svg>
                                Order Details
                            </span>
                            {activeTab === "details" && (
                                <span className="absolute bottom-0 left-0 w-full h-0.5 bg-indigo-500"></span>
                            )}
                        </button>

                        <button
                            onClick={() => setActiveTab("bids")}
                            className={`
                                relative w-1/2 px-4 py-3 font-medium text-sm transition-colors duration-200
                                ${
                                    activeTab === "bids"
                                        ? "bg-blue-600 dark:bg-gray-800 text-white dark:text-indigo-400 border-t border-l border-r border-gray-200 dark:border-gray-700 rounded-lg"
                                        : "text-gray-600 dark:text-gray-300 border border-gray-200 dark:border-gray-700 hover:text-indigo-500 dark:hover:text-indigo-300"
                                }
                            `}
                        >
                            <span className="flex items-center justify-center">
                                <svg
                                    className="w-4 h-4 mr-2"
                                    fill="none"
                                    stroke="currentColor"
                                    viewBox="0 0 24 24"
                                    xmlns="http://www.w3.org/2000/svg"
                                >
                                    <path
                                        strokeLinecap="round"
                                        strokeLinejoin="round"
                                        strokeWidth={2}
                                        d="M12 8c-1.657 0-3 .895-3 2s1.343 2 3 2 3 .895 3 2-1.343 2-3 2m0-8c1.11 0 2.08.402 2.599 1M12 8V7m0 1v8m0 0v1m0-1c-1.11 0-2.08-.402-2.599-1M21 12a9 9 0 11-18 0 9 9 0 0118 0z"
                                    />
                                </svg>
                                Bids
                            </span>
                            {activeTab === "bids" && (
                                <span className="absolute bottom-0 left-0 w-full h-0.5 bg-indigo-500"></span>
                            )}
                        </button>
                    </nav>

                    <div className="p-4">
                        {activeTab === "details" && (
                            <div className="text-gray-700 dark:text-gray-300">
                                <div>
                                    <div className="flex flex-wrap gap-6 items-start">
                                        {/* Header Section */}
                                        <div className="flex-1 space-y-6">
                                            <div className="bg-gradient-to-r from-orange-300 to-cyan-300 dark:bg-none dark:bg-gray-800 rounded-lg shadow-sm p-4 sm:p-6">
                                                <div className="flex flex-col sm:flex-row justify-between items-start sm:items-center mb-4">
                                                    <div>
                                                        <h1 className="text-lg sm:text-xl font-bold text-gray-900 dark:text-white">
                                                            {order.title}
                                                        </h1>
                                                        <div className="flex items-center mt-2">
                                                            <span className="text-sm text-gray-600 dark:text-gray-400">
                                                                Order #
                                                                {
                                                                    order.order_number
                                                                }
                                                            </span>
                                                            <span className="mx-2 text-gray-300 dark:text-gray-600">
                                                                |
                                                            </span>
                                                            <span className="text-sm text-gray-600 dark:text-gray-400">
                                                                Posted:{" "}
                                                                {formatDate(
                                                                    order.dateposted
                                                                )}
                                                            </span>
                                                        </div>
                                                    </div>
                                                    <div className="flex mt-3 sm:mt-0">
                                                        <span
                                                            className={`px-3 py-1 text-xs rounded-full flex items-center ${getStatusBadgeClass(
                                                                order.order_status
                                                            )}`}
                                                        >
                                                            {getStatusIcon(
                                                                order.order_status
                                                            )}
                                                            <span className="ml-1">
                                                                {capitalize(
                                                                    order.order_status ||
                                                                        "Unknown"
                                                                )}
                                                            </span>
                                                        </span>
                                                        <span
                                                            className={`ml-2 px-3 py-1 text-xs rounded-full flex items-center ${getStatusBadgeClass(
                                                                order.payment_status
                                                            )}`}
                                                        >
                                                            {order.payment_status.toLowerCase() ===
                                                            "paid" ? (
                                                                <CheckCircle2
                                                                    size={16}
                                                                />
                                                            ) : (
                                                                <CreditCard
                                                                    size={16}
                                                                />
                                                            )}
                                                            <span className="ml-1">
                                                                {capitalize(
                                                                    order.payment_status ||
                                                                        "Unknown"
                                                                )}
                                                            </span>
                                                        </span>
                                                    </div>
                                                </div>

                                                {/* Countdown Timer */}
                                                <div className="bg-blue-50 dark:bg-blue-900/30 border border-blue-100 dark:border-blue-800 rounded-lg p-4 mb-6">
                                                    <div className="flex items-center justify-between">
                                                        <div className="flex items-center">
                                                            <Clock className="h-5 w-5 text-blue-500 dark:text-blue-400 mr-2" />
                                                            <h3 className="text-sm font-medium text-blue-800 dark:text-blue-300">
                                                                Deadline
                                                            </h3>
                                                        </div>
                                                        <div className="text-right">
                                                            <span className="text-sm font-medium text-gray-700 dark:text-gray-300">
                                                                {formatDate(
                                                                    order.deadline
                                                                )}
                                                            </span>
                                                            <div className="text-xs text-gray-500 dark:text-gray-400">
                                                                Timezone:{" "}
                                                                {order.client_timezone ||
                                                                    "UTC"}
                                                            </div>
                                                        </div>
                                                    </div>
                                                    <div className="mt-2">
                                                        <ClearDeadlineCountdown
                                                            deadlineDate={
                                                                order.deadline
                                                            }
                                                        />
                                                    </div>
                                                </div>

                                                {/* Order Summary Grid */}
                                                <div className="grid grid-cols-1 md:grid-cols-2 gap-4 mb-6">
                                                    <div className="bg-white dark:bg-gray-700 rounded-lg shadow-sm p-4 border border-gray-200 dark:border-gray-600">
                                                        <h3 className="text-sm font-medium text-gray-700 dark:text-gray-300 mb-3 flex items-center">
                                                            <FileText className="h-4 w-4 mr-2 text-indigo-500 dark:text-indigo-400" />
                                                            Assignment Details
                                                        </h3>
                                                        <div className="grid grid-cols-2 gap-3 text-sm">
                                                            <div>
                                                                <p className="text-gray-500 dark:text-gray-400">
                                                                    Subject
                                                                </p>
                                                                <p className="font-medium text-gray-900 dark:text-white">
                                                                    {order.subject ||
                                                                        "Not specified"}
                                                                </p>
                                                            </div>
                                                            <div>
                                                                <p className="text-gray-500 dark:text-gray-400">
                                                                    Paper Type
                                                                </p>
                                                                <p className="font-medium text-gray-900 dark:text-white">
                                                                    {order.type_of_paper ||
                                                                        "Not specified"}
                                                                </p>
                                                            </div>
                                                            <div>
                                                                <p className="text-gray-500 dark:text-gray-400">
                                                                    Academic
                                                                    Level
                                                                </p>
                                                                <p className="font-medium text-gray-900 dark:text-white">
                                                                    {order.academic_level ||
                                                                        "Not specified"}
                                                                </p>
                                                            </div>
                                                            <div>
                                                                <p className="text-gray-500 dark:text-gray-400">
                                                                    Sources
                                                                </p>
                                                                <p className="font-medium text-gray-900 dark:text-white">
                                                                    {
                                                                        order.sources
                                                                    }{" "}
                                                                    {order.sources ===
                                                                    1
                                                                        ? "source"
                                                                        : "sources"}
                                                                </p>
                                                            </div>
                                                            <div>
                                                                <p className="text-gray-500 dark:text-gray-400">
                                                                    Pages
                                                                </p>
                                                                <p className="font-medium text-gray-900 dark:text-white">
                                                                    {
                                                                        order.pages
                                                                    }{" "}
                                                                    {order.pages ===
                                                                    1
                                                                        ? "page"
                                                                        : "pages"}{" "}
                                                                    (
                                                                    {
                                                                        order.spacing
                                                                    }
                                                                    )
                                                                </p>
                                                            </div>
                                                            <div>
                                                                <p className="text-gray-500 dark:text-gray-400">
                                                                    PowerPoint
                                                                    Slides
                                                                </p>
                                                                <p className="font-medium text-gray-900 dark:text-white">
                                                                    {
                                                                        order.powerpoint_slides
                                                                    }{" "}
                                                                    {order.powerpoint_slides ===
                                                                    1
                                                                        ? "slide"
                                                                        : "slides"}
                                                                </p>
                                                            </div>
                                                        </div>
                                                    </div>

                                                    <div className="bg-white dark:bg-gray-700 rounded-lg shadow-sm p-4 border border-gray-200 dark:border-gray-600">
                                                        <h3 className="text-sm font-medium text-gray-700 dark:text-gray-300 mb-3 flex items-center">
                                                            <DollarSign className="h-4 w-4 mr-2 text-indigo-500 dark:text-indigo-400" />
                                                            Payment Details
                                                        </h3>
                                                        <div className="space-y-2 text-sm">
                                                            <div className="flex justify-between">
                                                                <span className="text-gray-500 dark:text-gray-400">
                                                                    Total Amount
                                                                </span>
                                                                <span className="font-medium text-gray-900 dark:text-white">
                                                                    {formatCurrency(
                                                                        order.order_amount
                                                                    )}
                                                                </span>
                                                            </div>
                                                            {order.discount >
                                                                0 && (
                                                                <div className="flex justify-between">
                                                                    <span className="text-gray-500 dark:text-gray-400">
                                                                        Discount
                                                                    </span>
                                                                    <span className="font-medium text-green-600 dark:text-green-400">
                                                                        -
                                                                        {formatCurrency(
                                                                            order.discount
                                                                        )}
                                                                    </span>
                                                                </div>
                                                            )}
                                                            <div className="flex justify-between pt-2 border-t border-gray-200 dark:border-gray-600">
                                                                <span className="font-medium text-gray-800 dark:text-gray-200">
                                                                    Net Amount
                                                                </span>
                                                                <span className="font-bold text-gray-900 dark:text-white">
                                                                    {formatCurrency(
                                                                        order.net_amount
                                                                    )}
                                                                    {getUSDEquivalent(
                                                                        order.net_amount
                                                                    ) && (
                                                                        <span className="block text-xs text-gray-500 dark:text-gray-400 text-right">
                                                                            {getUSDEquivalent(
                                                                                order.net_amount
                                                                            )}
                                                                        </span>
                                                                    )}
                                                                </span>
                                                            </div>
                                                            {refundInfo &&
                                                                refundInfo.has_refunds && (
                                                                    <div className="flex justify-between pt-2 border-t border-gray-200 dark:border-gray-600">
                                                                        <span className="font-medium text-gray-800 dark:text-gray-200">
                                                                            Refunded
                                                                            Amount
                                                                        </span>
                                                                        <span className="font-bold text-orange-600 dark:text-orange-400">
                                                                            {formatCurrency(
                                                                                refundInfo.refunded_amount
                                                                            )}
                                                                        </span>
                                                                    </div>
                                                                )}
                                                            {order.payment_status.toLowerCase() ===
                                                                "unpaid" && (
                                                                <div className="mt-4">
                                                                    <button
                                                                        onClick={
                                                                            goToPaymentOptions
                                                                        }
                                                                        className="w-full bg-blue-600 hover:bg-blue-700 dark:bg-blue-700 dark:hover:bg-blue-600 text-white p-2 rounded-md font-medium flex items-center justify-center"
                                                                    >
                                                                        <ShoppingCart className="w-4 h-4 mr-2" />
                                                                        Pay Now
                                                                    </button>
                                                                </div>
                                                            )}
                                                        </div>
                                                    </div>
                                                </div>

                                                {/* Instructions */}
                                                <div className="bg-white dark:bg-gray-700 rounded-lg shadow-sm p-4 border border-gray-200 dark:border-gray-600 mb-6">
                                                    <h3 className="text-sm font-medium text-gray-700 dark:text-gray-300 mb-3 flex items-center">
                                                        <Notebook className="h-4 w-4 mr-2 text-indigo-500 dark:text-indigo-400" />
                                                        Instructions
                                                    </h3>
                                                    <div className="text-sm text-gray-700 dark:text-gray-300 whitespace-pre-wrap">
                                                        {order.instructions ||
                                                            "No instructions provided."}
                                                    </div>
                                                </div>

                                                {/* Files */}
                                                <div className="bg-white dark:bg-gray-700 rounded-lg shadow-sm p-4 border border-gray-200 dark:border-gray-600">
                                                    <h3 className="text-sm font-medium text-gray-700 dark:text-gray-300 mb-3 flex items-center">
                                                        <Upload className="h-4 w-4 mr-2 text-indigo-500 dark:text-indigo-400" />
                                                        Files
                                                    </h3>
                                                    {order.orderFiles &&
                                                    order.orderFiles.length >
                                                        0 ? (
                                                        <FileManagement
                                                            files={
                                                                order.orderFiles
                                                            }
                                                            canDelete={false}
                                                            isAdmin={isAdmin}
                                                        />
                                                    ) : (
                                                        <p className="text-sm text-gray-500 dark:text-gray-400">
                                                            No files uploaded.
                                                        </p>
                                                    )}

                                                    {/* File Upload Component */}
                                                    <div className="mt-4">
                                                        <FileUpload
                                                            orderId={order.id}
                                                            fileStatus="instructions"
                                                        />
                                                    </div>
                                                </div>
                                            </div>
                                        </div>
                                    </div>
                                </div>
                            </div>
                        )}

                        {activeTab === "bids" && (
                            <div className="bg-white dark:bg-gray-800 rounded-lg shadow-sm p-4 border border-gray-200 dark:border-gray-700">
                                <h3 className="text-lg font-semibold text-gray-900 dark:text-white mb-4 flex items-center">
                                    <Vote className="h-5 w-5 mr-2 text-indigo-500 dark:text-indigo-400" />
                                    Writer Bids
                                </h3>

                                {order.order_status.toLowerCase() ===
                                "bidding" ? (
                                    <>
                                        {bids && bids.length > 0 ? (
                                            <div className="space-y-4">
                                                {bids.map((bid) => (
                                                    <div
                                                        key={bid.id}
                                                        className="bg-gray-50 dark:bg-gray-700 rounded-lg shadow-sm p-4 border border-gray-200 dark:border-gray-600"
                                                    >
                                                        <div className="flex flex-col sm:flex-row justify-between">
                                                            <div className="flex items-start mb-2 sm:mb-0">
                                                                <div className="flex-shrink-0">
                                                                    <div className="w-10 h-10 rounded-full bg-indigo-100 dark:bg-indigo-900 flex items-center justify-center">
                                                                        <User className="h-5 w-5 text-indigo-600 dark:text-indigo-400" />
                                                                    </div>
                                                                </div>
                                                                <div className="ml-3">
                                                                    <h4 className="text-sm font-medium text-gray-900 dark:text-white">
                                                                        {bid.writer
                                                                            ? bid
                                                                                  .writer
                                                                                  .name
                                                                            : "Unknown Writer"}
                                                                    </h4>
                                                                    <div className="mt-1 text-xs text-gray-500 dark:text-gray-400">
                                                                        <span>
                                                                            Bid
                                                                            amount:{" "}
                                                                            {formatCurrency(
                                                                                bid.amount
                                                                            )}
                                                                            {getUSDEquivalent(
                                                                                bid.amount
                                                                            ) && (
                                                                                <span className="ml-1 text-xs text-gray-500 dark:text-gray-400">
                                                                                    {getUSDEquivalent(
                                                                                        bid.amount
                                                                                    )}
                                                                                </span>
                                                                            )}
                                                                        </span>
                                                                        <span className="mx-1">
                                                                            •
                                                                        </span>
                                                                        <span>
                                                                            Bid
                                                                            placed:{" "}
                                                                            {formatDate(
                                                                                bid.created_at
                                                                            )}
                                                                        </span>
                                                                    </div>
                                                                    <div className="mt-2 text-sm text-gray-700 dark:text-gray-300">
                                                                        {bid.message ||
                                                                            "No message from writer."}
                                                                    </div>
                                                                </div>
                                                            </div>
                                                            <div className="flex flex-col sm:items-end mt-2 sm:mt-0 space-y-2">
                                                                <button
                                                                    onClick={() =>
                                                                        openWriterDetailsModal(
                                                                            bid
                                                                                .writer
                                                                                .id
                                                                        )
                                                                    }
                                                                    className="inline-flex items-center px-3 py-1.5 border border-indigo-300 dark:border-indigo-700 text-xs font-medium rounded-full shadow-sm text-indigo-700 dark:text-indigo-300 bg-white dark:bg-gray-800 hover:bg-indigo-50 dark:hover:bg-indigo-900/30 focus:outline-none focus:ring-2 focus:ring-offset-2 focus:ring-indigo-500 dark:focus:ring-offset-gray-900"
                                                                >
                                                                    <GraduationCap className="h-3.5 w-3.5 mr-1" />
                                                                    View Profile
                                                                </button>
                                                                {order.payment_status.toLowerCase() ===
                                                                    "paid" && (
                                                                    <button
                                                                        onClick={() =>
                                                                            openBidActionModal(
                                                                                bid
                                                                            )
                                                                        }
                                                                        className="inline-flex items-center px-3 py-1.5 border border-green-300 dark:border-green-700 text-xs font-medium rounded-full shadow-sm text-green-700 dark:text-green-300 bg-white dark:bg-gray-800 hover:bg-green-50 dark:hover:bg-green-900/30 focus:outline-none focus:ring-2 focus:ring-offset-2 focus:ring-green-500 dark:focus:ring-offset-gray-900"
                                                                    >
                                                                        <CheckCircle className="h-3.5 w-3.5 mr-1" />
                                                                        Accept
                                                                        Bid
                                                                    </button>
                                                                )}
                                                            </div>
                                                        </div>
                                                    </div>
                                                ))}
                                            </div>
                                        ) : (
                                            <div className="text-center py-10">
                                                <svg
                                                    className="mx-auto h-12 w-12 text-gray-400"
                                                    fill="none"
                                                    stroke="currentColor"
                                                    viewBox="0 0 24 24"
                                                    xmlns="http://www.w3.org/2000/svg"
                                                >
                                                    <path
                                                        strokeLinecap="round"
                                                        strokeLinejoin="round"
                                                        strokeWidth={1.5}
                                                        d="M9 12h6m-6 4h6m2 5H7a2 2 0 01-2-2V5a2 2 0 012-2h5.586a1 1 0 01.707.293l5.414 5.414a1 1 0 01.293.707V19a2 2 0 01-2 2z"
                                                    />
                                                </svg>
                                                <h3 className="mt-2 text-sm font-medium text-gray-900 dark:text-white">
                                                    No bids yet
                                                </h3>
                                                <p className="mt-1 text-sm text-gray-500 dark:text-gray-400">
                                                    Writers will review your
                                                    order and place bids soon.
                                                </p>
                                            </div>
                                        )}
                                        {order.payment_status.toLowerCase() ===
                                            "unpaid" && (
                                            <div className="mt-6 bg-yellow-50 dark:bg-yellow-900/30 border border-yellow-200 dark:border-yellow-800 rounded-lg p-4">
                                                <div className="flex">
                                                    <div className="flex-shrink-0">
                                                        <AlertCircle
                                                            className="h-5 w-5 text-yellow-400"
                                                            aria-hidden="true"
                                                        />
                                                    </div>
                                                    <div className="ml-3">
                                                        <h3 className="text-sm font-medium text-yellow-800 dark:text-yellow-300">
                                                            Payment Required
                                                        </h3>
                                                        <div className="mt-2 text-sm text-yellow-700 dark:text-yellow-400">
                                                            <p>
                                                                You need to pay
                                                                for this order
                                                                before you can
                                                                accept a
                                                                writer's bid.
                                                            </p>
                                                        </div>
                                                        <div className="mt-4">
                                                            <button
                                                                onClick={
                                                                    goToPaymentOptions
                                                                }
                                                                className="inline-flex items-center px-4 py-2 border border-transparent text-sm font-medium rounded-md shadow-sm text-white bg-yellow-600 hover:bg-yellow-700 focus:outline-none focus:ring-2 focus:ring-offset-2 focus:ring-yellow-500 dark:focus:ring-offset-gray-900"
                                                            >
                                                                <CreditCard className="h-4 w-4 mr-2" />
                                                                Pay Now (
                                                                {formatCurrency(
                                                                    order.net_amount
                                                                )}
                                                                )
                                                            </button>
                                                        </div>
                                                    </div>
                                                </div>
                                            </div>
                                        )}
                                    </>
                                ) : (
                                    <div className="rounded-md bg-blue-50 dark:bg-blue-900/30 p-4 border border-blue-200 dark:border-blue-800">
                                        <div className="flex">
                                            <div className="flex-shrink-0">
                                                <Info
                                                    className="h-5 w-5 text-blue-400"
                                                    aria-hidden="true"
                                                />
                                            </div>
                                            <div className="ml-3 flex-1 md:flex md:justify-between">
                                                <p className="text-sm text-blue-700 dark:text-blue-300">
                                                    This order is no longer in
                                                    the bidding phase.
                                                    {order.writer && (
                                                        <span>
                                                            {" "}
                                                            It's currently
                                                            assigned to{" "}
                                                            <span className="font-medium">
                                                                {
                                                                    order.writer
                                                                        .name
                                                                }
                                                            </span>
                                                            .
                                                        </span>
                                                    )}
                                                </p>
                                            </div>
                                        </div>
                                    </div>
                                )}
                            </div>
                        )}
                    </div>
                </div>
            </div>

            {/* Writer Details Modal */}
            {showWriterDetailsModal && (
                <div className="fixed inset-0 overflow-y-auto z-50">
                    <div className="flex items-center justify-center min-h-screen pt-4 px-4 pb-20 text-center sm:block sm:p-0">
                        <div
                            className="fixed inset-0 transition-opacity"
                            aria-hidden="true"
                        >
                            <div className="absolute inset-0 bg-gray-500 dark:bg-gray-900 opacity-75"></div>
                        </div>

                        <span
                            className="hidden sm:inline-block sm:align-middle sm:h-screen"
                            aria-hidden="true"
                        >
                            &#8203;
                        </span>

                        <div className="inline-block align-bottom bg-white dark:bg-gray-800 rounded-lg text-left overflow-hidden shadow-xl transform transition-all sm:my-8 sm:align-middle sm:max-w-lg sm:w-full">
                            <div className="absolute top-0 right-0 pt-4 pr-4">
                                <button
                                    type="button"
                                    onClick={closeWriterDetailsModal}
                                    className="bg-white dark:bg-gray-800 rounded-md text-gray-400 hover:text-gray-500 dark:hover:text-gray-300 focus:outline-none focus:ring-2 focus:ring-offset-2 focus:ring-indigo-500 dark:focus:ring-offset-gray-900"
                                >
                                    <span className="sr-only">Close</span>
                                    <X className="h-6 w-6" aria-hidden="true" />
                                </button>
                            </div>

                            <div className="p-6">
                                <h3 className="text-lg font-medium leading-6 text-gray-900 dark:text-white mb-4">
                                    Writer Profile
                                </h3>

                                {isLoadingWriterDetails ? (
                                    <div className="flex justify-center items-center py-8">
                                        <div className="animate-spin rounded-full h-10 w-10 border-t-2 border-b-2 border-indigo-500"></div>
                                    </div>
                                ) : writerDetailsError ? (
                                    <div className="rounded-md bg-red-50 dark:bg-red-900/30 p-4 border border-red-200 dark:border-red-800">
                                        <div className="flex">
                                            <div className="flex-shrink-0">
                                                <AlertCircle
                                                    className="h-5 w-5 text-red-400"
                                                    aria-hidden="true"
                                                />
                                            </div>
                                            <div className="ml-3">
                                                <h3 className="text-sm font-medium text-red-800 dark:text-red-300">
                                                    Error
                                                </h3>
                                                <div className="mt-2 text-sm text-red-700 dark:text-red-400">
                                                    <p>{writerDetailsError}</p>
                                                </div>
                                            </div>
                                        </div>
                                    </div>
                                ) : writerDetails ? (
                                    <div className="space-y-4">
                                        <div className="flex items-center">
                                            <div className="flex-shrink-0">
                                                <div className="h-12 w-12 rounded-full bg-indigo-100 dark:bg-indigo-900 flex items-center justify-center">
                                                    <User className="h-8 w-8 text-indigo-600 dark:text-indigo-400" />
                                                </div>
                                            </div>
                                            <div className="ml-4">
                                                <h4 className="text-lg font-bold text-gray-900 dark:text-white">
                                                    {writerDetails.writer.name}
                                                </h4>
                                                <p className="text-sm text-gray-500 dark:text-gray-400">
                                                    {writerDetails.writer
                                                        .writerProfile
                                                        ?.educationLevel
                                                        ?.name ||
                                                        "Education not specified"}
                                                </p>
                                            </div>
                                        </div>

                                        <div className="grid grid-cols-2 gap-4 mt-4">
                                            <div className="bg-gray-50 dark:bg-gray-700 rounded-lg p-3">
                                                <div className="text-sm font-medium text-gray-500 dark:text-gray-400">
                                                    Active Orders
                                                </div>
                                                <div className="mt-1 text-2xl font-semibold text-gray-900 dark:text-white">
                                                    {writerDetails.activeOrdersCount ||
                                                        0}
                                                </div>
                                            </div>
                                            <div className="bg-gray-50 dark:bg-gray-700 rounded-lg p-3">
                                                <div className="text-sm font-medium text-gray-500 dark:text-gray-400">
                                                    Completed Orders
                                                </div>
                                                <div className="mt-1 text-2xl font-semibold text-gray-900 dark:text-white">
                                                    {writerDetails.approvedOrdersCount ||
                                                        0}
                                                </div>
                                            </div>
                                        </div>

                                        <div className="bg-gray-50 dark:bg-gray-700 rounded-lg p-4 mt-4">
                                            <h5 className="text-sm font-medium text-gray-700 dark:text-gray-300 mb-2">
                                                Specialization
                                            </h5>
                                            <p className="text-sm text-gray-600 dark:text-gray-400">
                                                {writerDetails.writer
                                                    .writerProfile
                                                    ?.specialization?.name ||
                                                    "Not specified"}
                                            </p>
                                        </div>

                                        {writerDetails.previousOrders &&
                                            writerDetails.previousOrders
                                                .length > 0 && (
                                                <div className="bg-gray-50 dark:bg-gray-700 rounded-lg p-4 mt-4">
                                                    <h5 className="text-sm font-medium text-gray-700 dark:text-gray-300 mb-2">
                                                        Previous Orders with You
                                                    </h5>
                                                    <ul className="space-y-2 mt-2">
                                                        {writerDetails.previousOrders.map(
                                                            (order) => (
                                                                <li
                                                                    key={
                                                                        order.id
                                                                    }
                                                                    className="text-sm"
                                                                >
                                                                    <Link
                                                                        href={route(
                                                                            "orders.show",
                                                                            order.id
                                                                        )}
                                                                        className="text-indigo-600 dark:text-indigo-400 hover:text-indigo-800 dark:hover:text-indigo-200"
                                                                    >
                                                                        Order #
                                                                        {
                                                                            order.order_number
                                                                        }
                                                                        :{" "}
                                                                        {
                                                                            order.title
                                                                        }
                                                                    </Link>
                                                                </li>
                                                            )
                                                        )}
                                                    </ul>
                                                </div>
                                            )}

                                        <div className="mt-6">
                                            <p className="text-sm text-gray-500 dark:text-gray-400">
                                                Consider this writer's
                                                credentials and experience
                                                before accepting their bid.
                                            </p>
                                        </div>
                                    </div>
                                ) : (
                                    <div className="py-4 text-center text-gray-500 dark:text-gray-400">
                                        Writer information not available.
                                    </div>
                                )}
                            </div>
                        </div>
                    </div>
                </div>
            )}

            {/* Bid Action Modal */}
            {showBidActionModal && selectedBid && (
                <div className="fixed inset-0 overflow-y-auto z-50">
                    <div className="flex items-center justify-center min-h-screen pt-4 px-4 pb-20 text-center sm:block sm:p-0">
                        <div
                            className="fixed inset-0 transition-opacity"
                            aria-hidden="true"
                        >
                            <div className="absolute inset-0 bg-gray-500 dark:bg-gray-900 opacity-75"></div>
                        </div>

                        <span
                            className="hidden sm:inline-block sm:align-middle sm:h-screen"
                            aria-hidden="true"
                        >
                            &#8203;
                        </span>

                        <div className="inline-block align-bottom bg-white dark:bg-gray-800 rounded-lg text-left overflow-hidden shadow-xl transform transition-all sm:my-8 sm:align-middle sm:max-w-lg sm:w-full">
                            <div className="bg-white dark:bg-gray-800 px-4 pt-5 pb-4 sm:p-6 sm:pb-4">
                                <div className="sm:flex sm:items-start">
                                    <div className="mx-auto flex-shrink-0 flex items-center justify-center h-12 w-12 rounded-full bg-green-100 dark:bg-green-900 sm:mx-0 sm:h-10 sm:w-10">
                                        <CheckCircle
                                            className="h-6 w-6 text-green-600 dark:text-green-400"
                                            aria-hidden="true"
                                        />
                                    </div>
                                    <div className="mt-3 text-center sm:mt-0 sm:ml-4 sm:text-left">
                                        <h3 className="text-lg leading-6 font-medium text-gray-900 dark:text-white">
                                            Accept Writer's Bid
                                        </h3>
                                        <div className="mt-2">
                                            <p className="text-sm text-gray-500 dark:text-gray-400">
                                                You are about to assign your
                                                order to{" "}
                                                {selectedBid.writer?.name ||
                                                    "this writer"}{" "}
                                                for{" "}
                                                {formatCurrency(
                                                    selectedBid.amount
                                                )}
                                                .
                                                {getUSDEquivalent(
                                                    selectedBid.amount
                                                ) && (
                                                    <span className="ml-1 text-xs text-gray-500 dark:text-gray-400">
                                                        {getUSDEquivalent(
                                                            selectedBid.amount
                                                        )}
                                                    </span>
                                                )}
                                            </p>
                                            <p className="mt-2 text-sm text-gray-500 dark:text-gray-400">
                                                Once assigned, the writer will
                                                start working on your order
                                                immediately. This action cannot
                                                be undone.
                                            </p>
                                        </div>
                                    </div>
                                </div>
                            </div>
                            <div className="bg-gray-50 dark:bg-gray-700 px-4 py-3 sm:px-6 sm:flex sm:flex-row-reverse">
                                <button
                                    type="button"
                                    className="w-full inline-flex justify-center rounded-md border border-transparent shadow-sm px-4 py-2 bg-green-600 text-base font-medium text-white hover:bg-green-700 focus:outline-none focus:ring-2 focus:ring-offset-2 focus:ring-green-500 dark:focus:ring-offset-gray-900 sm:ml-3 sm:w-auto sm:text-sm"
                                    onClick={acceptBid}
                                >
                                    Confirm Assignment
                                </button>
                                <button
                                    type="button"
                                    className="mt-3 w-full inline-flex justify-center rounded-md border border-gray-300 dark:border-gray-600 shadow-sm px-4 py-2 bg-white dark:bg-gray-800 text-base font-medium text-gray-700 dark:text-gray-300 hover:bg-gray-50 dark:hover:bg-gray-700 focus:outline-none focus:ring-2 focus:ring-offset-2 focus:ring-indigo-500 dark:focus:ring-offset-gray-900 sm:mt-0 sm:ml-3 sm:w-auto sm:text-sm"
                                    onClick={closeBidActionModal}
                                >
                                    Cancel
                                </button>
                            </div>
                        </div>
                    </div>
                </div>
            )}

            {/* Flash Message */}
            {flashMessage && (
                <div
                    className={`fixed bottom-5 right-5 z-50 p-4 rounded-md shadow-lg ${
                        flashType === "success"
                            ? "bg-green-50 dark:bg-green-900/30 border border-green-200 dark:border-green-800"
                            : "bg-red-50 dark:bg-red-900/30 border border-red-200 dark:border-red-800"
                    }`}
                >
                    <div className="flex">
                        <div className="flex-shrink-0">
                            {flashType === "success" ? (
                                <CheckCircle
                                    className="h-5 w-5 text-green-400"
                                    aria-hidden="true"
                                />
                            ) : (
                                <AlertCircle
                                    className="h-5 w-5 text-red-400"
                                    aria-hidden="true"
                                />
                            )}
                        </div>
                        <div className="ml-3">
                            <p
                                className={`text-sm font-medium ${
                                    flashType === "success"
                                        ? "text-green-800 dark:text-green-300"
                                        : "text-red-800 dark:text-red-300"
                                }`}
                            >
                                {flashMessage}
                            </p>
                        </div>
                        <div className="ml-auto pl-3">
                            <div className="-mx-1.5 -my-1.5">
                                <button
                                    type="button"
                                    onClick={() => {
                                        setFlashMessage("");
                                        setFlashType("");
                                    }}
                                    className={`inline-flex rounded-md p-1.5 ${
                                        flashType === "success"
                                            ? "text-green-500 hover:bg-green-100 dark:text-green-400 dark:hover:bg-green-900"
                                            : "text-red-500 hover:bg-red-100 dark:text-red-400 dark:hover:bg-red-900"
                                    } focus:outline-none focus:ring-2 focus:ring-offset-2 ${
                                        flashType === "success"
                                            ? "focus:ring-green-500 dark:focus:ring-offset-gray-900"
                                            : "focus:ring-red-500 dark:focus:ring-offset-gray-900"
                                    }`}
                                >
                                    <span className="sr-only">Dismiss</span>
                                    <X className="h-5 w-5" aria-hidden="true" />
                                </button>
                            </div>
                        </div>
                    </div>
                </div>
            )}
        </ClientLayout>
    );
}
