import React, { useState, useEffect } from "react";
import { Head, Link } from "@inertiajs/react";
import ClientLayout from "@/Layouts/ClientLayout";
import PayPalButton from "@/Components/PayPalButton";
import {
    FileText,
    Clock,
    Notebook,
    Timer,
    MoveLeft,
    Vote,
    ShoppingCart,
    User,
    Upload,
    ArrowDown,
    CheckCircle,
    X,
    AlertCircle,
    GraduationCap,
    Briefcase,
    Award,
    CheckCircle2,
    CreditCard,
    Calendar,
    DollarSign,
    Users,
    PenTool,
    XCircle,
    Repeat,
    Info,
    Heading,
    PlusCircle,
} from "lucide-react";
import ClearDeadlineCountdown from "@/Components/ClearDeadlineCountdown";
import FileUpload from "@/Components/FileUpload";
import FileManagement from "@/Components/FileManagement";
import { format, formatDistance } from "date-fns";

export default function Show({
    order,
    bids = [],
    isAdmin = false,
    refundInfo = null,
}) {
    const [activeTab, setActiveTab] = useState("details"); // State to manage active tab

    // Add these state variables
    const [showBidActionModal, setShowBidActionModal] = useState(false);
    const [selectedBid, setSelectedBid] = useState(null);

    const [flashMessage, setFlashMessage] = useState("");
    const [flashType, setFlashType] = useState(""); // "success" or "error"

    const [showWriterDetailsModal, setShowWriterDetailsModal] = useState(false);
    const [writerDetails, setWriterDetails] = useState(null);
    const [isLoadingWriterDetails, setIsLoadingWriterDetails] = useState(false);
    const [writerDetailsError, setWriterDetailsError] = useState("");

    // Function to fetch writer details
    const fetchWriterDetails = async (writerId) => {
        setIsLoadingWriterDetails(true);
        setWriterDetailsError("");

        try {
            const response = await axios.get(`/api/writer-details/${writerId}`);
            setWriterDetails(response.data);
        } catch (error) {
            setWriterDetailsError(
                error.response?.data?.message ||
                    "Failed to fetch writer details. Please try again."
            );
        } finally {
            setIsLoadingWriterDetails(false);
        }
    };

    // Functions to handle modal
    const openWriterDetailsModal = (writerId) => {
        fetchWriterDetails(writerId);
        setShowWriterDetailsModal(true);
    };

    const closeWriterDetailsModal = () => {
        setShowWriterDetailsModal(false);
        setWriterDetails(null);
        setWriterDetailsError("");
    };

    // Add after the state declarations
    const openBidActionModal = (bid) => {
        setSelectedBid(bid);
        setShowBidActionModal(true);
    };

    const closeBidActionModal = () => {
        setShowBidActionModal(false);
        setSelectedBid(null);
    };

    const acceptBid = async () => {
        try {
            const response = await axios.post(
                `/orders/${order.id}/bids/${selectedBid.id}/accept`
            );
            setFlashMessage(
                response.data.message || "Bid accepted successfully!"
            );
            setFlashType("success");
            closeBidActionModal();

            // Redirect to refresh the page after a short delay
            setTimeout(() => {
                window.location.href = `/orders/${order.id}`;
            }, 1500);
        } catch (error) {
            setFlashMessage(
                error.response?.data?.message || "Failed to accept bid"
            );
            setFlashType("error");
        }
    };

    // Add this effect to display flash messages
    useEffect(() => {
        if (flashMessage) {
            const timer = setTimeout(() => {
                setFlashMessage("");
                setFlashType("");
            }, 3000);
            return () => clearTimeout(timer);
        }
    }, [flashMessage]);

    const capitalize = (string) => {
        return string.charAt(0).toUpperCase() + string.slice(1);
    };

    // Navigate to payment options page
    const goToPaymentOptions = () => {
        window.location.href = route("orders.payment", order.id);
    };

    const parsedBids = Array.isArray(bids)
        ? bids.map((bid) => {
              // If the bid is a string, try to parse it
              if (typeof bid === "string") {
                  try {
                      return JSON.parse(bid);
                  } catch (e) {
                      console.error("Failed to parse bid:", bid);
                      return {};
                  }
              }
              return bid;
          })
        : [];

    const formatCurrency = (amount, customCurrency = null) => {
        // Handle invalid amounts
        if (
            amount === undefined ||
            amount === null ||
            isNaN(parseFloat(amount))
        ) {
            amount = 0;
        }

        // Use the order's currency if no custom currency is provided
        const currencyCode = customCurrency || order.currency || "USD";

        // Format the currency
        return new Intl.NumberFormat("en-US", {
            style: "currency",
            currency: currencyCode,
        }).format(parseFloat(amount));
    };

    // Add this function to your component, near other helper functions like formatCurrency
    const getUSDEquivalent = (amount) => {
        // If the order is already in USD, no conversion needed
        if (order.currency === "USD") {
            return null;
        }

        // Convert the amount to USD using the currency rate
        // The currency_rate is how many USD = 1 unit of the order currency
        // So we divide by the rate to get the USD equivalent
        const usdAmount = parseFloat(amount) / parseFloat(order.currency_rate);

        // Format the USD amount with the $ symbol and 2 decimal places
        return `($${usdAmount.toFixed(2)} USD)`;
    };

    // Format date helper
    const formatDate = (dateString, formatStr = "PPP p") => {
        if (!dateString) return "N/A";
        try {
            return format(new Date(dateString), formatStr);
        } catch (e) {
            return dateString;
        }
    };

    // Get status badge class
    const getStatusBadgeClass = (status) => {
        switch (status?.toLowerCase()) {
            case "bidding":
                return "bg-blue-100 text-blue-800 dark:bg-blue-900/50 dark:text-blue-300";
            case "pending":
            case "unpaid":
                return "bg-yellow-100 text-yellow-800 dark:bg-yellow-900/50 dark:text-yellow-300";
            case "writing":
            case "in progress":
                return "bg-purple-100 text-purple-800 dark:bg-purple-900/50 dark:text-purple-300";
            case "completed":
            case "paid":
                return "bg-green-100 text-green-800 dark:bg-green-900/50 dark:text-green-300";
            case "cancelled":
            case "failed":
                return "bg-red-100 text-red-800 dark:bg-red-900/50 dark:text-red-300";
            case "refunded":
                return "bg-orange-100 text-orange-800 dark:bg-orange-900/50 dark:text-orange-300";
            case "partially_refunded":
            case "partially refunded":
                return "bg-amber-100 text-amber-800 dark:bg-amber-900/50 dark:text-amber-300";
            default:
                return "bg-gray-100 text-gray-800 dark:bg-gray-700 dark:text-gray-300";
        }
    };

    // Get status icon
    const getStatusIcon = (status) => {
        switch (status?.toLowerCase()) {
            case "bidding":
                return <Users size={20} />;
            case "pending":
            case "unpaid":
                return <Clock size={20} />;
            case "writing":
            case "in progress":
                return <PenTool size={20} />;
            case "completed":
            case "paid":
                return <CheckCircle size={20} />;
            case "cancelled":
            case "failed":
                return <XCircle size={20} />;
            case "refunded":
                return <Repeat size={20} />;
            case "partially_refunded":
            case "partially refunded":
                return <Repeat size={20} />;
            default:
                return <Info size={20} />;
        }
    };

    return (
        <ClientLayout>
            <Head title={`Order #${order.order_number}`} />
            <div className="w-full mx-auto px-4 sm:px-4 lg:px-4 py-2 space-y-6 dark:bg-gray-900 overflow-x-hidden">
                <div className="flex flex-col sm:flex-row sm:justify-end sm:items-center gap-3 mb-6">
                    <Link
                        href="/place-order"
                        className="inline-flex items-center px-4 py-2 bg-gradient-to-r from-indigo-600 to-blue-600 hover:from-indigo-700 hover:to-blue-700 text-white font-medium rounded-lg transition-all duration-200 shadow hover:shadow-lg gap-2"
                    >
                        <PlusCircle size={14} />
                        <span className="font-medium">Place New Order</span>
                    </Link>
                    <a
                        href="/orders"
                        className="flex items-center px-4 py-2 sm:px-4 sm:py-2 text-indigo-600 dark:text-gray-100 bg-indigo-100 dark:bg-indigo-900 hover:bg-indigo-100 dark:hover:bg-indigo-800 rounded-lg shadow-sm border border-indigo-200 dark:border-indigo-700 transition-colors duration-200 text-sm"
                    >
                        <MoveLeft className="mr-1 h-3 w-3 sm:h-4 sm:w-4" />
                        <span className="font-medium">All Orders</span>
                    </a>
                </div>

                {/* Tabs */}
                <div className="border-b border-gray-200 dark:border-gray-700">
                    <nav
                        className="flex w-full border-b border-gray-200 dark:border-gray-700 px-2 sm:px-6 lg:px-4 rounded-lg"
                        aria-label="Tabs"
                    >
                        <button
                            onClick={() => setActiveTab("details")}
                            className={`
      relative w-1/2 px-4 py-3 font-medium text-sm transition-colors duration-200
      ${
          activeTab === "details"
              ? "bg-blue-600 dark:bg-gray-800 text-white dark:text-indigo-400 border-t border-l border-r border-gray-200 dark:border-gray-700 rounded-lg"
              : "text-gray-600 dark:text-gray-300 border border-gray-200 dark:border-gray-700 hover:text-indigo-500 dark:hover:text-indigo-300"
      }
    `}
                        >
                            <span className="flex items-center justify-center">
                                <svg
                                    className="w-4 h-4 mr-2"
                                    fill="none"
                                    stroke="currentColor"
                                    viewBox="0 0 24 24"
                                    xmlns="http://www.w3.org/2000/svg"
                                >
                                    <path
                                        strokeLinecap="round"
                                        strokeLinejoin="round"
                                        strokeWidth={2}
                                        d="M9 5H7a2 2 0 00-2 2v12a2 2 0 002 2h10a2 2 0 002-2V7a2 2 0 00-2-2h-2M9 5a2 2 0 002 2h2a2 2 0 002-2M9 5a2 2 0 012-2h2a2 2 0 012 2"
                                    />
                                </svg>
                                Order Details
                            </span>
                            {activeTab === "details" && (
                                <span className="absolute bottom-0 left-0 w-full h-0.5 bg-indigo-500"></span>
                            )}
                        </button>

                        <button
                            onClick={() => setActiveTab("bids")}
                            className={`
      relative w-1/2 px-4 py-3 font-medium text-sm transition-colors duration-200
      ${
          activeTab === "bids"
              ? "bg-blue-600 dark:bg-gray-800 text-white dark:text-indigo-400 border-t border-l border-r border-gray-200 dark:border-gray-700 rounded-lg"
              : "text-gray-600 dark:text-gray-300 border border-gray-200 dark:border-gray-700 hover:text-indigo-500 dark:hover:text-indigo-300"
      }
    `}
                        >
                            <span className="flex items-center justify-center">
                                <svg
                                    className="w-4 h-4 mr-2"
                                    fill="none"
                                    stroke="currentColor"
                                    viewBox="0 0 24 24"
                                    xmlns="http://www.w3.org/2000/svg"
                                >
                                    <path
                                        strokeLinecap="round"
                                        strokeLinejoin="round"
                                        strokeWidth={2}
                                        d="M12 8c-1.657 0-3 .895-3 2s1.343 2 3 2 3 .895 3 2-1.343 2-3 2m0-8c1.11 0 2.08.402 2.599 1M12 8V7m0 1v8m0 0v1m0-1c-1.11 0-2.08-.402-2.599-1M21 12a9 9 0 11-18 0 9 9 0 0118 0z"
                                    />
                                </svg>
                                Bids
                            </span>
                            {activeTab === "bids" && (
                                <span className="absolute bottom-0 left-0 w-full h-0.5 bg-indigo-500"></span>
                            )}
                        </button>
                    </nav>

                    <div className="p-4">
                        {activeTab === "details" && (
                            <div className="text-gray-700 dark:text-gray-300">
                                <div>
                                    <div className="flex flex-wrap gap-6 items-start">
                                        {/* Header Section */}
                                        <div className="flex-1 space-y-6">
                                            <div className="bg-gradient-to-r from-blue-100 to-cyan-100 dark:bg-none dark:bg-gray-800 rounded-lg shadow-sm p-4 sm:p-4 border border-gray-300 dark:border-gray-600">
                                                <div className="flex flex-col sm:flex-row justify-between items-start sm:items-center mb-4">
                                                    <div>
                                                        {/* <h1 className="text-lg sm:text-xl font-bold text-gray-900 dark:text-white">
                                                            {order.title}
                                                        </h1> */}
                                                        <div className="flex items-center mt-2">
                                                            <span className="text-lg text-gray-600 dark:text-gray-400">
                                                                Order #
                                                                {
                                                                    order.order_number
                                                                }
                                                            </span>
                                                            <span className="mx-2 text-gray-300 dark:text-gray-600">
                                                                |
                                                            </span>
                                                            <span className="text-sm text-gray-600 dark:text-gray-400">
                                                                Posted:{" "}
                                                                {formatDate(
                                                                    order.dateposted
                                                                )}
                                                            </span>
                                                        </div>
                                                    </div>
                                                    <div className="flex mt-3 sm:mt-0">
                                                        <span
                                                            className={`px-3 py-1 text-xs rounded-full flex items-center ${getStatusBadgeClass(
                                                                order.order_status
                                                            )}`}
                                                        >
                                                            {getStatusIcon(
                                                                order.order_status
                                                            )}
                                                            <span className="ml-1">
                                                                {capitalize(
                                                                    order.order_status ||
                                                                        "Unknown"
                                                                )}
                                                            </span>
                                                        </span>
                                                        <span
                                                            className={`ml-2 px-3 py-1 text-xs rounded-full flex items-center ${getStatusBadgeClass(
                                                                order.payment_status
                                                            )}`}
                                                        >
                                                            {order.payment_status.toLowerCase() ===
                                                            "paid" ? (
                                                                <CheckCircle2
                                                                    size={16}
                                                                />
                                                            ) : (
                                                                <CreditCard
                                                                    size={16}
                                                                />
                                                            )}
                                                            <span className="ml-1">
                                                                {capitalize(
                                                                    order.payment_status ||
                                                                        "Unknown"
                                                                )}
                                                            </span>
                                                        </span>
                                                    </div>
                                                </div>
                                            </div>

                                            <div className="bg-gradient-to-r from-blue-100 to-cyan-100 dark:bg-none border border-gray-300 dark:border-gray-600  text-black  dark:bg-gray-800 rounded-lg shadow-sm p-4 sm:p-6">
                                                <div className="flex flex-col sm:flex-row justify-between items-start sm:items-center space-y-4 sm:space-y-0">
                                                    <div className="flex items-center justify-between">
                                                        <h3 className="text-sm font-medium text-blue-800 dark:text-blue-300 mr-2">
                                                            {/* <Heading className="h-4 w-4 mr-2 text-indigo-500 dark:text-indigo-400" /> */}
                                                            Order Title:
                                                        </h3>

                                                        <p className="text-md sm:text-md font-bold dark:text-white">
                                                            {order.title}
                                                        </p>
                                                    </div>
                                                </div>
                                            </div>

                                            {/* Countdown Timer */}
                                            <div className="bg-gradient-to-r from-blue-100 to-cyan-100 dark:bg-none border border-gray-300 dark:border-gray-600  dark:bg-blue-900/30  rounded-lg p-4 mb-6">
                                                <div className="flex items-center justify-between">
                                                    <div className="flex items-center">
                                                        <Clock className="h-5 w-5 text-blue-500 dark:text-blue-400 mr-2" />
                                                        <h3 className="text-sm font-medium text-blue-800 dark:text-blue-300">
                                                            Deadline
                                                        </h3>
                                                    </div>
                                                    <div className="text-right">
                                                        <span className="text-sm font-medium text-gray-700 dark:text-gray-300">
                                                            {formatDate(
                                                                order.deadline
                                                            )}
                                                        </span>
                                                        <div className="text-xs text-gray-500 dark:text-gray-400">
                                                            Timezone:{" "}
                                                            {order.client_timezone ||
                                                                "UTC"}
                                                        </div>
                                                    </div>
                                                </div>
                                                <div className="mt-2">
                                                    <ClearDeadlineCountdown
                                                        deadline={
                                                            order.deadline
                                                        }
                                                    />
                                                </div>
                                            </div>

                                            {/* Writer Details Section */}
                                            <div className="bg-gradient-to-r from-blue-100 to-cyan-100 dark:bg-none border border-gray-300 dark:border-gray-600 dark:bg-gray-800 rounded-lg shadow-sm p-4 sm:p-6">
                                                <h2 className="text-lg sm:text-xl font-semibold mb-4 flex items-center dark:text-white">
                                                    <Notebook className="w-5 h-5 mr-2" />
                                                    Writer Details
                                                </h2>
                                                {order.writer ? (
                                                    <div className="space-y-3">
                                                        <div className="flex items-center gap-2">
                                                            <User className="w-8 h-8 text-gray-400 dark:text-gray-500" />
                                                            <div>
                                                                <p className="font-medium dark:text-white">
                                                                    Writer Name
                                                                </p>
                                                                <p className="text-sm text-gray-500 dark:text-gray-400">
                                                                    {
                                                                        order
                                                                            .writer
                                                                            .name
                                                                    }
                                                                </p>
                                                            </div>
                                                        </div>
                                                    </div>
                                                ) : (
                                                    <div className="bg-gradient-to-br from-blue-200 to-purple-200 dark:bg-none text-center py-4 sm:py-6 bg-gray-50 dark:bg-gray-700 rounded-lg">
                                                        <User className="w-4 h-4 sm:w-12 sm:h-12 text-black dark:text-gray-500 mx-auto mb-2" />
                                                        <p className="text-black dark:text-gray-400">
                                                            Not assigned to
                                                            writer yet
                                                        </p>
                                                    </div>
                                                )}
                                            </div>

                                            <div className="bg-gradient-to-r from-blue-100 to-cyan-100 dark:bg-none border border-gray-300 dark:border-gray-600  dark:bg-gray-700 rounded-lg shadow-sm p-4 ">
                                                <h3 className="text-sm font-medium text-gray-700 dark:text-gray-300 mb-3 flex items-center">
                                                    <FileText className="h-4 w-4 mr-2 text-indigo-500 dark:text-indigo-400" />
                                                    Assignment Details
                                                </h3>
                                                <div className="grid grid-cols-2 gap-3 text-sm">
                                                    <div>
                                                        <p className="text-gray-500 dark:text-gray-400">
                                                            Subject
                                                        </p>
                                                        <p className="font-medium text-gray-900 dark:text-white">
                                                            {order.subject ||
                                                                "Not specified"}
                                                        </p>
                                                    </div>
                                                    <div>
                                                        <p className="text-gray-500 dark:text-gray-400">
                                                            Paper Type
                                                        </p>
                                                        <p className="font-medium text-gray-900 dark:text-white">
                                                            {order.type_of_paper ||
                                                                "Not specified"}
                                                        </p>
                                                    </div>
                                                    <div>
                                                        <p className="text-gray-500 dark:text-gray-400">
                                                            Academic Level
                                                        </p>
                                                        <p className="font-medium text-gray-900 dark:text-white">
                                                            {order.academic_level ||
                                                                "Not specified"}
                                                        </p>
                                                    </div>
                                                    <div>
                                                        <p className="text-gray-500 dark:text-gray-400">
                                                            Sources
                                                        </p>
                                                        <p className="font-medium text-gray-900 dark:text-white">
                                                            {order.sources}{" "}
                                                            {order.sources === 1
                                                                ? "source"
                                                                : "sources"}
                                                        </p>
                                                    </div>
                                                    <div>
                                                        <p className="text-gray-500 dark:text-gray-400">
                                                            Pages
                                                        </p>
                                                        <p className="font-medium text-gray-900 dark:text-white">
                                                            {order.pages}{" "}
                                                            {order.pages === 1
                                                                ? "page"
                                                                : "pages"}{" "}
                                                            ({order.spacing})
                                                        </p>
                                                    </div>
                                                    <div>
                                                        <p className="text-gray-500 dark:text-gray-400">
                                                            PowerPoint Slides
                                                        </p>
                                                        <p className="font-medium text-gray-900 dark:text-white">
                                                            {
                                                                order.powerpoint_slides
                                                            }{" "}
                                                            {order.powerpoint_slides ===
                                                            1
                                                                ? "slide"
                                                                : "slides"}
                                                        </p>
                                                    </div>
                                                </div>
                                            </div>

                                            {/* Instructions/Description Section */}

                                            <div className="bg-gradient-to-r from-blue-100 to-cyan-100 dark:bg-none border border-gray-300 dark:border-gray-600  dark:bg-gray-800 rounded-lg shadow-sm p-4 sm:p-6">
                                                <div className="flex flex-col space-y-4">
                                                    <h3 className="text-lg sm:text-lg font-bold dark:text-white">
                                                        Instructions/Description
                                                    </h3>
                                                    <p className="text-md dark:text-white">
                                                        {order.instructions}
                                                    </p>
                                                </div>
                                            </div>
                                            {/* File Management Section */}
                                            <div className="bg-gradient-to-r from-blue-100 to-cyan-100 dark:bg-none border border-gray-300 dark:border-gray-600  dark:bg-gray-800 rounded-lg shadow-sm p-4 sm:p-6">
                                                <h2 className="text-lg sm:text-lg font-semibold mb-4 flex items-center dark:text-white">
                                                    <FileText className="w-5 h-5 mr-2" />
                                                    File Management
                                                </h2>
                                                <div className="mb-6">
                                                    <FileUpload
                                                        orderId={order.id}
                                                    />
                                                </div>

                                                <div className="mt-6">
                                                    <h3 className="text-lg font-medium text-gray-700 dark:text-gray-300 mb-3 flex items-center">
                                                        <FileText className="w-5 h-5 mr-2" />
                                                        Uploaded Files
                                                    </h3>
                                                    <FileManagement
                                                        files={
                                                            order.order_files
                                                        }
                                                        isAdmin={isAdmin}
                                                    />
                                                </div>
                                            </div>
                                        </div>

                                        {/* Sidebar */}
                                        <div className="w-full lg:w-1/4 space-y-6">
                                            {/* Payment Status Card */}
                                            <div className="bg-gradient-to-r from-blue-100 to-cyan-100 dark:bg-none border border-gray-300 dark:border-gray-600 dark:bg-gray-800 shadow rounded-lg overflow-hidden">
                                                <div className="bg-gradient-to-r from-indigo-500 to-purple-600 dark:from-gray-700 dark:to-gray-800 px-6 py-4 border-b">
                                                    <h2 className="text-lg font-medium text-white flex items-center">
                                                        <ShoppingCart className="h-5 w-5 mr-2 text-white" />
                                                        Payment Status
                                                    </h2>
                                                </div>
                                                <div className="p-6">
                                                    <div className="mb-4 bg-indigo-50 dark:bg-gray-700 p-3 rounded-lg">
                                                        <p className="text-sm font-medium text-gray-500 dark:text-gray-400">
                                                            Total Amount
                                                        </p>
                                                        <p className="text-lg font-bold text-gray-900 dark:text-gray-100">
                                                            {formatCurrency(
                                                                order.net_amount
                                                            )}
                                                        </p>
                                                    </div>

                                                    {order.payment_status ===
                                                    "unpaid" ? (
                                                        <button
                                                            onClick={
                                                                goToPaymentOptions
                                                            }
                                                            className="w-full px-2 py-2 bg-blue-600 hover:bg-blue-700 text-white rounded-md transition-colors flex items-center justify-center"
                                                        >
                                                            <CreditCard className="w-4 h-4 mr-2" />
                                                            Proceed To Payment
                                                        </button>
                                                    ) : (
                                                        <div className="space-y-3">
                                                            <div className="flex items-center text-green-600 dark:text-green-400 font-medium p-2 bg-green-50 dark:bg-green-900/20 rounded-md">
                                                                <CheckCircle className="w-5 h-5 mr-2" />
                                                                Payment
                                                                completed
                                                            </div>

                                                            <a
                                                                href={route(
                                                                    "orders.payment-history",
                                                                    order.id
                                                                )}
                                                                className=" mt-2 text-sm text-blue-600 dark:text-blue-400 hover:underline flex items-center bg-green-50 dark:bg-green-900/20 rounded-md p-2"
                                                            >
                                                                <FileText className="w-5 h-5 mr-2" />
                                                                View Payment
                                                                History
                                                            </a>
                                                        </div>
                                                    )}
                                                </div>
                                            </div>

                                            {/* Order Details Card */}
                                            <div className="bg-gradient-to-r from-blue-100 to-cyan-100 dark:bg-none dark:bg-gray-800 shadow rounded-lg overflow-hidden">
                                                <div className="bg-gradient-to-r from-indigo-500 to-purple-600 dark:from-gray-700 dark:to-gray-800 px-6 py-4 border-b border-gray-200 dark:border-gray-700">
                                                    <h2 className="text-lg font-medium text-gray-100 dark:text-gray-100 flex items-center">
                                                        <Calendar className="h-5 w-5 mr-2 text-gray-100" />
                                                        Order Details
                                                    </h2>
                                                </div>
                                                <div className="p-6">
                                                    <dl className="space-y-4">
                                                        <div>
                                                            <dt className="text-sm font-medium text-gray-500 dark:text-gray-400">
                                                                Order Amount
                                                            </dt>
                                                            <dd className="mt-1 text-sm font-bold text-gray-900 dark:text-gray-100">
                                                                {formatCurrency(
                                                                    order.order_amount
                                                                )}
                                                            </dd>
                                                        </div>

                                                        {order.discount > 0 && (
                                                            <div>
                                                                <dt className="text-sm font-medium text-gray-500 dark:text-gray-400">
                                                                    Discount
                                                                </dt>
                                                                <dd className="mt-1 text-sm font-medium text-green-600 dark:text-green-400">
                                                                    -
                                                                    {formatCurrency(
                                                                        order.discount
                                                                    )}
                                                                </dd>
                                                            </div>
                                                        )}

                                                        <div className="pt-2 border-t border-gray-200 dark:border-gray-700">
                                                            <dt className="text-sm font-medium text-gray-500 dark:text-gray-400">
                                                                Total Paid
                                                            </dt>
                                                            <dd className="mt-1 text-lg font-bold text-gray-900 dark:text-gray-100">
                                                                {formatCurrency(
                                                                    order.net_amount
                                                                )}
                                                            </dd>
                                                        </div>

                                                        <div className="pt-4 border-t border-gray-200 dark:border-gray-700">
                                                            <dt className="text-sm font-medium text-gray-500 dark:text-gray-400">
                                                                Order Date
                                                            </dt>
                                                            <dd className="mt-1 text-sm text-gray-900 dark:text-gray-100">
                                                                {formatDate(
                                                                    order.dateposted
                                                                )}
                                                            </dd>
                                                        </div>

                                                        <div>
                                                            <dt className="text-sm font-medium text-gray-500 dark:text-gray-400">
                                                                Deadline
                                                            </dt>
                                                            <dd className="mt-1 text-sm text-gray-900 dark:text-gray-100">
                                                                {formatDate(
                                                                    order.deadline
                                                                )}
                                                            </dd>
                                                            {order.deadline && (
                                                                <dd className="mt-1 text-xs text-red-600 dark:text-red-400">
                                                                    {new Date(
                                                                        order.deadline
                                                                    ) >
                                                                    new Date()
                                                                        ? `(${formatDistance(
                                                                              new Date(
                                                                                  order.deadline
                                                                              ),
                                                                              new Date(),
                                                                              {
                                                                                  addSuffix: true,
                                                                              }
                                                                          )})`
                                                                        : "Deadline passed"}
                                                                </dd>
                                                            )}
                                                        </div>
                                                    </dl>
                                                </div>
                                            </div>

                                            {/* Order Timeline Card */}
                                            <div className="bg-gradient-to-r from-blue-100 to-cyan-100 dark:bg-none dark:bg-gray-800 shadow rounded-lg overflow-hidden">
                                                <div className="bg-gradient-to-r from-indigo-500 to-purple-600 dark:from-gray-700 dark:to-gray-800 py-4 border-b px-6  border-gray-200 dark:border-gray-700">
                                                    <h2 className="text-lg font-medium text-gray-100 dark:text-gray-100 flex items-center">
                                                        <Clock className="h-5 w-5 mr-2 text-gray-100" />
                                                        Order Timeline
                                                    </h2>
                                                </div>
                                                <div className="p-6">
                                                    <ol className="relative border-l border-gray-200 dark:border-gray-700 ml-3">
                                                        <li className="mb-6 ml-6">
                                                            <span className="absolute flex items-center justify-center w-6 h-6 bg-blue-100 rounded-full -left-3 ring-8 ring-white dark:ring-gray-800 dark:bg-blue-900">
                                                                <FileText className="w-3 h-3 text-blue-800 dark:text-blue-300" />
                                                            </span>
                                                            <h3 className="mb-1 text-sm font-semibold text-gray-900 dark:text-white">
                                                                Order Created
                                                            </h3>
                                                            <time className="block text-xs font-normal leading-none text-gray-500 dark:text-gray-400">
                                                                {formatDate(
                                                                    order.dateposted
                                                                )}
                                                            </time>
                                                            <p className="text-sm font-normal text-gray-500 dark:text-gray-400">
                                                                Order #
                                                                {
                                                                    order.order_number
                                                                }{" "}
                                                                was placed
                                                            </p>
                                                        </li>

                                                        {order.payment_status ===
                                                            "paid" && (
                                                            <li className="mb-6 ml-6">
                                                                <span className="absolute flex items-center justify-center w-6 h-6 bg-green-100 rounded-full -left-3 ring-8 ring-white dark:ring-gray-800 dark:bg-green-900">
                                                                    <DollarSign className="w-3 h-3 text-green-800 dark:text-green-300" />
                                                                </span>
                                                                <h3 className="mb-1 text-sm font-semibold text-gray-900 dark:text-white">
                                                                    Payment
                                                                    Completed
                                                                </h3>
                                                                <time className="block text-xs font-normal leading-none text-gray-500 dark:text-gray-400">
                                                                    {formatDate(
                                                                        order.payment_date
                                                                    )}
                                                                </time>
                                                                <p className="text-sm font-normal text-gray-500 dark:text-gray-400">
                                                                    {formatCurrency(
                                                                        order.net_amount
                                                                    )}{" "}
                                                                    payment
                                                                    processed
                                                                </p>
                                                            </li>
                                                        )}

                                                        {order.order_status !==
                                                            "bidding" &&
                                                            order.writer && (
                                                                <li className="mb-6 ml-6">
                                                                    <span className="absolute flex items-center justify-center w-6 h-6 bg-purple-100 rounded-full -left-3 ring-8 ring-white dark:ring-gray-800 dark:bg-purple-900">
                                                                        <User className="w-3 h-3 text-purple-800 dark:text-purple-300" />
                                                                    </span>
                                                                    <h3 className="mb-1 text-sm font-semibold text-gray-900 dark:text-white">
                                                                        Writer
                                                                        Assigned
                                                                    </h3>
                                                                    <p className="text-sm font-normal text-gray-500 dark:text-gray-400">
                                                                        {
                                                                            order
                                                                                .writer
                                                                                .name
                                                                        }{" "}
                                                                        was
                                                                        assigned
                                                                        to your
                                                                        order
                                                                    </p>
                                                                </li>
                                                            )}

                                                        {order.order_status ===
                                                            "writing" && (
                                                            <li className="mb-6 ml-6">
                                                                <span className="absolute flex items-center justify-center w-6 h-6 bg-amber-100 rounded-full -left-3 ring-8 ring-white dark:ring-gray-800 dark:bg-amber-900">
                                                                    <PenTool className="w-3 h-3 text-amber-800 dark:text-amber-300" />
                                                                </span>
                                                                <h3 className="mb-1 text-sm font-semibold text-gray-900 dark:text-white">
                                                                    In Progress
                                                                </h3>
                                                                <p className="text-sm font-normal text-gray-500 dark:text-gray-400">
                                                                    Writer is
                                                                    working on
                                                                    your order
                                                                </p>
                                                            </li>
                                                        )}

                                                        {order.order_status ===
                                                            "completed" && (
                                                            <li className="mb-6 ml-6">
                                                                <span className="absolute flex items-center justify-center w-6 h-6 bg-green-100 rounded-full -left-3 ring-8 ring-white dark:ring-gray-800 dark:bg-green-900">
                                                                    <CheckCircle className="w-3 h-3 text-green-800 dark:text-green-300" />
                                                                </span>
                                                                <h3 className="mb-1 text-sm font-semibold text-gray-900 dark:text-white">
                                                                    Order
                                                                    Completed
                                                                </h3>
                                                                <p className="text-sm font-normal text-gray-500 dark:text-gray-400">
                                                                    Your order
                                                                    has been
                                                                    completed
                                                                    successfully
                                                                </p>
                                                            </li>
                                                        )}

                                                        {refundInfo &&
                                                            (refundInfo.is_fully_refunded ||
                                                                refundInfo.is_partially_refunded) && (
                                                                <li className="mb-6 ml-6">
                                                                    <span className="absolute flex items-center justify-center w-6 h-6 bg-orange-100 rounded-full -left-3 ring-8 ring-white dark:ring-gray-800 dark:bg-orange-900">
                                                                        <Repeat className="w-3 h-3 text-orange-800 dark:text-orange-300" />
                                                                    </span>
                                                                    <h3 className="mb-1 text-sm font-semibold text-gray-900 dark:text-white">
                                                                        {refundInfo.is_fully_refunded
                                                                            ? "Fully Refunded"
                                                                            : "Partially Refunded"}
                                                                    </h3>
                                                                    <time className="block text-xs font-normal leading-none text-gray-500 dark:text-gray-400">
                                                                        {formatDate(
                                                                            refundInfo.refund_date
                                                                        )}
                                                                    </time>
                                                                    <p className="text-sm font-normal text-gray-500 dark:text-gray-400">
                                                                        {refundInfo.is_fully_refunded
                                                                            ? `Full refund of ${formatCurrency(
                                                                                  refundInfo.refunded_amount
                                                                              )}`
                                                                            : `Partial refund of ${formatCurrency(
                                                                                  refundInfo.refunded_amount
                                                                              )}`}
                                                                    </p>
                                                                </li>
                                                            )}

                                                        <li className="ml-6">
                                                            <span className="absolute flex items-center justify-center w-6 h-6 bg-gray-100 rounded-full -left-3 ring-8 ring-white dark:ring-gray-800 dark:bg-gray-700">
                                                                <Clock className="w-3 h-3 text-gray-800 dark:text-gray-300" />
                                                            </span>
                                                            <h3 className="mb-1 text-sm font-semibold text-gray-900 dark:text-white">
                                                                Deadline
                                                            </h3>
                                                            <time className="block text-xs font-normal leading-none text-gray-500 dark:text-gray-400">
                                                                {formatDate(
                                                                    order.deadline
                                                                )}
                                                            </time>
                                                            <p className="text-sm font-normal text-gray-500 dark:text-gray-400">
                                                                {new Date(
                                                                    order.deadline
                                                                ) > new Date()
                                                                    ? `Due ${formatDistance(
                                                                          new Date(
                                                                              order.deadline
                                                                          ),
                                                                          new Date(),
                                                                          {
                                                                              addSuffix: true,
                                                                          }
                                                                      )}`
                                                                    : "Deadline has passed"}
                                                            </p>
                                                        </li>
                                                    </ol>
                                                </div>
                                            </div>
                                        </div>
                                    </div>
                                </div>
                            </div>
                        )}
                        {activeTab === "bids" && (
                            <div className=" bg-gradient-to-r from-blue-100 to-cyan-100 dark:bg-none  dark:bg-gray-800 rounded-lg shadow-sm p-4 border border-gray-400 dark:border-gray-700">
                                <h3 className="text-lg font-semibold text-gray-900 dark:text-white mb-4 flex items-center">
                                    <Vote className="h-5 w-5 mr-2 text-indigo-500 dark:text-indigo-400" />
                                    Writer Bids
                                </h3>

                                {order.order_status.toLowerCase() ===
                                "bidding" ? (
                                    <>
                                        {bids && bids.length > 0 ? (
                                            <div className="space-y-4">
                                                {bids.map((bid) => (
                                                    <div
                                                        key={bid.id}
                                                        className=" bg-gradient-to-r from-blue-100 to-cyan-100 dark:bg-none  dark:bg-gray-700 rounded-lg shadow-sm p-4 border border-gray-300 dark:border-gray-600"
                                                    >
                                                        <div className="flex flex-col sm:flex-row justify-between">
                                                            <div className="flex items-start mb-2 sm:mb-0">
                                                                <div className="flex-shrink-0">
                                                                    <div className="w-10 h-10 rounded-full bg-indigo-100 dark:bg-indigo-900 flex items-center justify-center">
                                                                        <User className="h-5 w-5 text-indigo-600 dark:text-indigo-400" />
                                                                    </div>
                                                                </div>
                                                                <div className="ml-3">
                                                                    <h4 className="text-sm font-medium text-gray-900 dark:text-white">
                                                                        {bid.writer
                                                                            ? bid
                                                                                  .writer
                                                                                  .name
                                                                            : "Unknown Writer"}
                                                                    </h4>
                                                                    <div className="mt-1 text-xs text-gray-500 dark:text-gray-400">
                                                                        <span>
                                                                            Bid
                                                                            amount:{" "}
                                                                            {formatCurrency(
                                                                                bid.bid_amount ||
                                                                                    bid.amount ||
                                                                                    0
                                                                            )}
                                                                            {getUSDEquivalent(
                                                                                bid.bid_amount ||
                                                                                    bid.amount
                                                                            ) && (
                                                                                <span className="ml-1 text-xs text-gray-500 dark:text-gray-400">
                                                                                    {getUSDEquivalent(
                                                                                        bid.bid_amount ||
                                                                                            bid.amount
                                                                                    )}
                                                                                </span>
                                                                            )}
                                                                        </span>
                                                                        <span className="mx-1">
                                                                            •
                                                                        </span>
                                                                        <span>
                                                                            Bid
                                                                            placed:{" "}
                                                                            {formatDate(
                                                                                bid.created_at
                                                                            )}
                                                                        </span>
                                                                    </div>
                                                                    <div className="mt-2 text-sm text-gray-700 dark:text-gray-300">
                                                                        {bid.comments ||
                                                                            "No message from writer."}
                                                                    </div>
                                                                </div>
                                                            </div>
                                                            <div className="flex flex-col sm:items-end mt-2 sm:mt-0 space-y-2">
                                                                <button
                                                                    onClick={() =>
                                                                        openWriterDetailsModal(
                                                                            bid
                                                                                .writer
                                                                                .id
                                                                        )
                                                                    }
                                                                    className="inline-flex items-center px-3 py-1.5 border border-indigo-300 dark:border-indigo-700 text-xs font-medium rounded-full shadow-sm text-indigo-700 dark:text-indigo-300 bg-white dark:bg-gray-800 hover:bg-indigo-50 dark:hover:bg-indigo-900/30 focus:outline-none focus:ring-2 focus:ring-offset-2 focus:ring-indigo-500 dark:focus:ring-offset-gray-900"
                                                                >
                                                                    <GraduationCap className="h-3.5 w-3.5 mr-1" />
                                                                    View Profile
                                                                </button>
                                                                {order.payment_status.toLowerCase() ===
                                                                    "paid" && (
                                                                    <button
                                                                        onClick={() =>
                                                                            openBidActionModal(
                                                                                bid
                                                                            )
                                                                        }
                                                                        className="inline-flex items-center px-3 py-1.5 border border-green-300 dark:border-green-700 text-xs font-medium rounded-full shadow-sm text-green-700 dark:text-green-300 bg-white dark:bg-gray-800 hover:bg-green-50 dark:hover:bg-green-900/30 focus:outline-none focus:ring-2 focus:ring-offset-2 focus:ring-green-500 dark:focus:ring-offset-gray-900"
                                                                    >
                                                                        <CheckCircle className="h-3.5 w-3.5 mr-1" />
                                                                        Accept
                                                                        Bid
                                                                    </button>
                                                                )}
                                                            </div>
                                                        </div>
                                                    </div>
                                                ))}
                                            </div>
                                        ) : (
                                            <div className="text-center py-10">
                                                <svg
                                                    className="mx-auto h-12 w-12 text-gray-400"
                                                    fill="none"
                                                    stroke="currentColor"
                                                    viewBox="0 0 24 24"
                                                    xmlns="http://www.w3.org/2000/svg"
                                                >
                                                    <path
                                                        strokeLinecap="round"
                                                        strokeLinejoin="round"
                                                        strokeWidth={1.5}
                                                        d="M9 12h6m-6 4h6m2 5H7a2 2 0 01-2-2V5a2 2 0 012-2h5.586a1 1 0 01.707.293l5.414 5.414a1 1 0 01.293.707V19a2 2 0 01-2 2z"
                                                    />
                                                </svg>
                                                <h3 className="mt-2 text-sm font-medium text-gray-900 dark:text-white">
                                                    No bids yet
                                                </h3>
                                                <p className="mt-1 text-sm text-gray-500 dark:text-gray-400">
                                                    Writers will review your
                                                    order and place bids soon.
                                                </p>
                                            </div>
                                        )}
                                        {order.payment_status.toLowerCase() ===
                                            "unpaid" && (
                                            <div className="mt-6 bg-yellow-50 dark:bg-yellow-900/30 border border-yellow-200 dark:border-yellow-800 rounded-lg p-4">
                                                <div className="flex">
                                                    <div className="flex-shrink-0">
                                                        <AlertCircle
                                                            className="h-5 w-5 text-yellow-400"
                                                            aria-hidden="true"
                                                        />
                                                    </div>
                                                    <div className="ml-3">
                                                        <h3 className="text-sm font-medium text-yellow-800 dark:text-yellow-300">
                                                            Payment Required
                                                        </h3>
                                                        <div className="mt-2 text-sm text-yellow-700 dark:text-yellow-400">
                                                            <p>
                                                                You need to pay
                                                                for this order
                                                                before you can
                                                                accept a
                                                                writer's bid.
                                                            </p>
                                                        </div>
                                                        <div className="mt-4">
                                                            <button
                                                                onClick={
                                                                    goToPaymentOptions
                                                                }
                                                                className="inline-flex items-center px-4 py-2 border border-transparent text-sm font-medium rounded-md shadow-sm text-white bg-yellow-600 hover:bg-yellow-700 focus:outline-none focus:ring-2 focus:ring-offset-2 focus:ring-yellow-500 dark:focus:ring-offset-gray-900"
                                                            >
                                                                <CreditCard className="h-4 w-4 mr-2" />
                                                                Pay Now (
                                                                {formatCurrency(
                                                                    order.net_amount
                                                                )}
                                                                )
                                                            </button>
                                                        </div>
                                                    </div>
                                                </div>
                                            </div>
                                        )}
                                    </>
                                ) : (
                                    <div className="rounded-md bg-blue-50 dark:bg-blue-900/30 p-4 border border-blue-200 dark:border-blue-800">
                                        <div className="flex">
                                            <div className="flex-shrink-0">
                                                <Info
                                                    className="h-5 w-5 text-blue-400"
                                                    aria-hidden="true"
                                                />
                                            </div>
                                            <div className="ml-3 flex-1 md:flex md:justify-between">
                                                <p className="text-sm text-blue-700 dark:text-blue-300">
                                                    This order is no longer in
                                                    the bidding phase.
                                                    {order.writer && (
                                                        <span>
                                                            {" "}
                                                            It's currently
                                                            assigned to{" "}
                                                            <span className="font-medium">
                                                                {
                                                                    order.writer
                                                                        .name
                                                                }
                                                            </span>
                                                            .
                                                        </span>
                                                    )}
                                                </p>
                                            </div>
                                        </div>
                                    </div>
                                )}
                            </div>
                        )}

                        {/* Add the Writer Details Modal */}
                        {showWriterDetailsModal && (
                            <div className="fixed inset-0 bg-black bg-opacity-50 flex items-center justify-center z-50 p-4">
                                <div className="bg-white dark:bg-gray-800 rounded-lg w-full max-w-2xl mx-auto">
                                    <div className="p-6">
                                        <div className="flex justify-between items-center mb-6">
                                            <h3 className="text-xl font-semibold text-gray-900 dark:text-white">
                                                Writer Details
                                            </h3>
                                            <button
                                                onClick={
                                                    closeWriterDetailsModal
                                                }
                                                className="text-gray-400 hover:text-gray-500 dark:hover:text-gray-300"
                                            >
                                                <X className="w-5 h-5" />
                                            </button>
                                        </div>

                                        {isLoadingWriterDetails ? (
                                            <div className="flex items-center justify-center py-8">
                                                <div className="animate-spin rounded-full h-8 w-8 border-b-2 border-blue-500"></div>
                                            </div>
                                        ) : writerDetailsError ? (
                                            <div className="flex items-center justify-center py-8 text-red-500 dark:text-red-400">
                                                <AlertCircle className="w-5 h-5 mr-2" />
                                                {writerDetailsError}
                                            </div>
                                        ) : writerDetails ? (
                                            <div className="space-y-6">
                                                {/* Writer Profile Section */}
                                                <div className="grid grid-cols-1 md:grid-cols-2 gap-6">
                                                    <div className="space-y-4">
                                                        <div className="flex items-center space-x-3">
                                                            <GraduationCap className="w-5 h-5 text-blue-500" />
                                                            <div>
                                                                <p className="text-sm text-gray-500 dark:text-gray-400">
                                                                    Education
                                                                    Level
                                                                </p>
                                                                <p className="font-medium text-gray-900 dark:text-white">
                                                                    {writerDetails
                                                                        .writer
                                                                        .writerProfile
                                                                        ?.educationLevel
                                                                        ?.name ||
                                                                        "Not specified"}
                                                                </p>
                                                            </div>
                                                        </div>

                                                        <div className="flex items-center space-x-3">
                                                            <Briefcase className="w-5 h-5 text-green-500" />
                                                            <div>
                                                                <p className="text-sm text-gray-500 dark:text-gray-400">
                                                                    Specialization
                                                                </p>
                                                                <p className="font-medium text-gray-900 dark:text-white">
                                                                    {writerDetails
                                                                        .writer
                                                                        .writerProfile
                                                                        ?.specialization
                                                                        ?.name ||
                                                                        "Not specified"}
                                                                </p>
                                                            </div>
                                                        </div>

                                                        <div className="flex items-center space-x-3">
                                                            <Award className="w-5 h-5 text-purple-500" />
                                                            <div>
                                                                <p className="text-sm text-gray-500 dark:text-gray-400">
                                                                    Writer Level
                                                                </p>
                                                                <p className="font-medium text-gray-900 dark:text-white">
                                                                    {writerDetails
                                                                        .writer
                                                                        .writerProfile
                                                                        ?.writerLevel
                                                                        ?.name ||
                                                                        "Not specified"}
                                                                </p>
                                                            </div>
                                                        </div>
                                                    </div>

                                                    <div className="space-y-4">
                                                        <div className="flex items-center space-x-3">
                                                            <Clock className="w-5 h-5 text-orange-500" />
                                                            <div>
                                                                <p className="text-sm text-gray-500 dark:text-gray-400">
                                                                    Active
                                                                    Orders
                                                                </p>
                                                                <p className="font-medium text-gray-900 dark:text-white">
                                                                    {writerDetails.activeOrdersCount ||
                                                                        0}
                                                                </p>
                                                            </div>
                                                        </div>

                                                        <div className="flex items-center space-x-3">
                                                            <CheckCircle2 className="w-5 h-5 text-teal-500" />
                                                            <div>
                                                                <p className="text-sm text-gray-500 dark:text-gray-400">
                                                                    Approved
                                                                    Orders
                                                                </p>
                                                                <p className="font-medium text-gray-900 dark:text-white">
                                                                    {writerDetails.approvedOrdersCount ||
                                                                        0}
                                                                </p>
                                                            </div>
                                                        </div>
                                                    </div>
                                                </div>

                                                {/* Previous Orders Section */}
                                                <div className="mt-6">
                                                    <h4 className="text-lg font-medium text-gray-900 dark:text-white mb-4">
                                                        Previous Orders for You
                                                    </h4>
                                                    {writerDetails
                                                        .previousOrders
                                                        ?.length > 0 ? (
                                                        <div className="bg-gray-50 dark:bg-gray-700 rounded-lg p-4">
                                                            <ul className="space-y-3">
                                                                {writerDetails.previousOrders.map(
                                                                    (order) => (
                                                                        <li
                                                                            key={
                                                                                order.id
                                                                            }
                                                                            className="flex items-center justify-between py-2 border-b border-gray-200 dark:border-gray-600 last:border-0"
                                                                        >
                                                                            <span className="text-sm text-gray-600 dark:text-gray-300">
                                                                                {
                                                                                    order.title
                                                                                }
                                                                            </span>
                                                                            <span className="text-xs px-2 py-1 rounded-full bg-green-100 text-green-800 dark:bg-green-900 dark:text-green-200">
                                                                                Completed
                                                                            </span>
                                                                        </li>
                                                                    )
                                                                )}
                                                            </ul>
                                                        </div>
                                                    ) : (
                                                        <p className="text-gray-500 dark:text-gray-400">
                                                            No previous orders
                                                            with this writer.
                                                        </p>
                                                    )}
                                                </div>
                                            </div>
                                        ) : (
                                            <div className="flex items-center justify-center py-8 text-gray-500 dark:text-gray-400">
                                                No writer details available.
                                            </div>
                                        )}
                                    </div>
                                </div>
                            </div>
                        )}

                        {/* Bid Action Modal */}
                        {showBidActionModal && (
                            <div className="fixed inset-0 bg-black bg-opacity-50 flex items-center justify-center z-50">
                                <div className="bg-white dark:bg-gray-800 rounded-lg p-6 w-full max-w-md mx-4">
                                    <h3 className="text-lg font-semibold mb-4 dark:text-white">
                                        Accept Bid
                                    </h3>
                                    <div>
                                        <p className="mb-4 text-gray-600 dark:text-gray-400">
                                            Are you sure you want to accept this
                                            bid from{" "}
                                            <span className="font-medium">
                                                {selectedBid.writer.name}
                                            </span>{" "}
                                            for{" "}
                                            <span className="font-medium">
                                                ${selectedBid.bid_amount}
                                            </span>
                                            ?
                                        </p>
                                        <p className="mb-4 text-gray-600 dark:text-gray-400">
                                            The writer will complete the order
                                            by{" "}
                                            <span className="font-medium">
                                                {new Date(
                                                    selectedBid.completion_time
                                                ).toLocaleString()}
                                            </span>
                                            .
                                        </p>
                                        <p className="mb-4 text-sm text-yellow-600 dark:text-yellow-400">
                                            Note: This action cannot be undone.
                                            The order status will change to
                                            "assigned" and other bids will be
                                            automatically rejected.
                                        </p>
                                    </div>
                                    <div className="flex justify-end space-x-3">
                                        <button
                                            onClick={closeBidActionModal}
                                            className="px-4 py-2 bg-gray-200 text-gray-800 rounded-md hover:bg-gray-300 dark:bg-gray-700 dark:text-gray-200 dark:hover:bg-gray-600"
                                        >
                                            Cancel
                                        </button>
                                        <button
                                            onClick={acceptBid}
                                            className="px-4 py-2 bg-green-600 hover:bg-green-700 text-white rounded-md transition-colors"
                                        >
                                            Accept Bid
                                        </button>
                                    </div>
                                </div>
                            </div>
                        )}
                    </div>
                </div>
            </div>
        </ClientLayout>
    );
}
