import React, { useMemo } from "react";
import { Head } from "@inertiajs/react";
import MainLayout from "@/Layouts/MainLayout";
import OptimizedImage from "@/Components/OptimizedImage";

// Import all pricing components
import PricingHero from "@/Components/Pricing/PricingHero";
import PricingTiers from "@/Components/Pricing/PricingTiers";
import ServicePricingGrid from "@/Components/Pricing/ServicePricingGrid";
import ServiceImageSection from "@/Components/Pricing/ServiceImageSection";
import PricingReasons from "@/Components/Pricing/PricingReasons";
import PricingFAQs from "@/Components/Pricing/PricingFAQs";
import PricingCTA from "@/Components/Pricing/PricingCTA";
import PricingKeywords from "@/Components/Pricing/PricingKeywords";
import PricingTestimonials from "@/Components/Pricing/PricingTestimonials";
import PricingResources from "@/Components/Pricing/PricingResources";
import PricingSecurity from "@/Components/Pricing/PricingSecurity";
import PricingContact from "@/Components/Pricing/PricingContact";
import PriceCalculatorSection from "@/Components/CMS/PriceCalculatorSection";

export default function Pricing({ page, contentBlocks, seoData, openGraph, structuredData, isCMS }) {
    // Sort content blocks by order field for dynamic rendering
    const sortedBlocks = useMemo(() => {
        if (!isCMS || !contentBlocks) return [];
        return [...contentBlocks].sort((a, b) => (a.order || 0) - (b.order || 0));
    }, [contentBlocks, isCMS]);

    // Render individual block based on type
    const renderBlock = (block) => {
        switch (block.block_type) {
            case 'pricing_hero':
                return <PricingHero key={block.id} content={[block]} />;
            case 'pricing_tiers':
                return <PricingTiers key={block.id} content={[block]} />;
            case 'service_pricing_grid':
                return <ServicePricingGrid key={block.id} content={[block]} />;
            case 'service_image':
                return <ServiceImageSection key={block.id} content={[block]} />;
            case 'pricing_reasons':
                return <PricingReasons key={block.id} content={[block]} />;
            case 'pricing_faqs':
                return <PricingFAQs key={block.id} content={[block]} />;
            case 'pricing_cta':
                return <PricingCTA key={block.id} content={[block]} />;
            case 'pricing_keywords':
                return <PricingKeywords key={block.id} content={[block]} />;
            case 'pricing_testimonials':
                return <PricingTestimonials key={block.id} content={[block]} />;
            case 'pricing_resources':
                return <PricingResources key={block.id} content={[block]} />;
            case 'pricing_security':
                return <PricingSecurity key={block.id} content={[block]} />;
            case 'pricing_contact':
                return <PricingContact key={block.id} content={[block]} />;
            case 'price_calculator':
                return <PriceCalculatorSection key={block.id} content={block.content} />;
            default:
                return null;
        }
    };

    // If CMS data is available, render with components
    if (isCMS) {
        return (
            <MainLayout>
                <Head>
                    <title>{seoData?.meta_title || 'Pricing'}</title>
                    <meta name="description" content={seoData?.meta_description} />
                    <meta name="keywords" content={seoData?.meta_keywords} />
                    
                    {/* Open Graph */}
                    <meta property="og:title" content={seoData?.og_title} />
                    <meta property="og:description" content={seoData?.og_description} />
                    <meta property="og:type" content="website" />
                    <meta property="og:url" content={seoData?.canonical_url} />
                    <meta property="og:image" content={seoData?.og_image} />

                    {/* Twitter Card */}
                    <meta name="twitter:card" content="summary_large_image" />
                    <meta name="twitter:title" content={seoData?.twitter_title} />
                    <meta name="twitter:description" content={seoData?.twitter_description} />
                    <meta name="twitter:image" content={seoData?.twitter_image} />

                    {/* Canonical URL */}
                    <link rel="canonical" href={seoData?.canonical_url} />

                    {/* Structured Data */}
                    {structuredData && (
                        <script type="application/ld+json">
                            {JSON.stringify(structuredData)}
                        </script>
                    )}
                    <script type="application/ld+json">
                        {JSON.stringify({
                            "@context": "https://schema.org",
                            "@type": "BreadcrumbList",
                            itemListElement: [
                                { "@type": "ListItem", position: 1, name: "Home", item: "https://academicscribe.com/" },
                                { "@type": "ListItem", position: 2, name: "Pricing", item: "https://academicscribe.com/pricing-page" },
                            ],
                        })}
                    </script>
                </Head>

                <div className="pricing-page">
                    {sortedBlocks.map((block) => renderBlock(block))}
                </div>
            </MainLayout>
        );
    }

    // Fallback to static data (existing implementation)
    const pricingTiers = [
        {
            level: "High School",
            startingPrice: "8.5",
            description:
                "Affordable essay writing service for high school students with professional quality",
        },
        {
            level: "University",
            startingPrice: "10.5",
            description:
                "College essay writing service prices that fit student budgets",
        },
        {
            level: "Masters",
            startingPrice: "14",
            description:
                "Graduate research paper prices for advanced academic writing",
        },
        {
            level: "PhD",
            startingPrice: "18",
            description:
                "PhD thesis writing service rates for expert dissertation help",
        },
    ];

    const serviceLinks = [
        {
            name: "Essay Writing",
            url: "/services/essay-writing",
            price: "$10-18",
            description:
                "Professional essay writing service for all academic levels",
        },
        {
            name: "Research Papers",
            url: "/services/research-papers",
            price: "$12-20",
            description: "Expert research paper writing with thorough analysis",
        },
        {
            name: "Dissertation Writing",
            url: "/services/dissertation-writing",
            price: "$12-20",
            description: "Comprehensive dissertation writing and consultation",
        },
        {
            name: "Assignments",
            url: "/services/assignments",
            price: "$8-18",
            description: "All types of academic assignments and homework help",
        },
        {
            name: "Case Studies",
            url: "/services/case-studies",
            price: "$12-18",
            description: "Detailed case study analysis and writing",
        },
        {
            name: "Literature Reviews",
            url: "/services/literature-reviews",
            price: "$10-18",
            description: "Comprehensive literature review and analysis",
        },
        {
            name: "Thesis Proposals",
            url: "/services/thesis-proposals",
            price: "$18-25",
            description: "Professional thesis proposal writing and guidance",
        },
    ];

    const reasons = [
        {
            title: "Cheapest Essays Online with Quality Guarantee",
            description:
                "Get the cheapest essays online without compromising quality. Our cheapest essay writing service delivers professional results at unbeatable prices.",
            link: "/services/essay-writing",
            linkText: "View Cheapest Essay Services",
        },
        {
            title: "Best Research Paper Writing Rates",
            description:
                "Compare research paper writing prices and see why we offer the most competitive rates for expert research and analysis.",
            link: "/services/research-papers",
            linkText: "View Research Paper Services",
        },
        {
            title: "Professional Dissertation Writing Rates",
            description:
                "Our dissertation writing cost includes expert research, quality writing, and unlimited revisions for your academic success.",
            link: "/services/dissertation-writing",
            linkText: "View Dissertation Services",
        },
        {
            title: "Assignment Help Pricing",
            description:
                "Affordable assignment help cost for all types of academic work including case studies, literature reviews, and more.",
            link: "/services/assignments",
            linkText: "View Assignment Services",
        },
        {
            title: "Cheapest Custom Essays with PhD Writers",
            description:
                "Order custom essay cheap from PhD-qualified writers. Our cheapest custom essays include original research, plagiarism reports, and unlimited revisions.",
            link: "/services/essay-writing",
            linkText: "Order Cheapest Custom Essays",
        },
        {
            title: "Academic Writing Service Pricing",
            description:
                "Transparent academic writing service cost comparison shows we offer the best value across all service types.",
            link: "/services",
            linkText: "Compare All Services",
        },
    ];

    const faqs = [
        {
            question: "Are your essays really the cheapest online?",
            answer: "Yes! Our cheapest essays online with quality guarantee start at $8.50/page, which is 30-50% cheaper than competitors. We offer the cheapest essay writing service price per page with quality guarantee, plagiarism reports, and unlimited revisions included.",
            link: "/services/essay-writing",
            linkText: "View our cheapest essay services",
        },
        {
            question: "How do you offer cheapest essays without compromising quality?",
            answer: "We maintain quality while offering cheapest essays through efficient processes, bulk writer partnerships, and streamlined operations. Our cheapest custom essays are written by PhD-qualified writers with the same quality standards as premium services.",
            link: "/services/essay-writing",
            linkText: "Learn about our quality guarantee",
        },
        {
            question: "Can I buy cheap essays with same-day delivery?",
            answer: "Yes! Our cheapest essays with same-day delivery start at $12.75/page (50% rush fee). Buy cheap essay online with same day delivery. We offer urgent cheapest essay help for tight deadlines while maintaining quality standards.",
            link: "/services/essay-writing",
            linkText: "Get urgent cheapest essays",
        },
        {
            question: "What's included when I order custom essay cheap?",
            answer: "When you order custom essay cheap online with quality guarantee from us, you get original content, plagiarism report, unlimited revisions, 24/7 support, and money-back guarantee. All cheapest essays online with quality guarantee include these features at no extra cost.",
            link: "/services/essay-writing",
            linkText: "Order cheapest custom essays",
        },
        {
            question: "What are your urgent essay writing service prices?",
            answer: "Rush dissertation writing cost and 24 hour research paper writing prices include a 50-100% urgency fee. Same day essay writing rates start at $24 per page.",
            link: "/services/essay-writing",
            linkText: "Get urgent essay writing help",
        },
        {
            question:
                "How do your academic writing service pricing compare to competitors?",
            answer: "Our academic writing service cost comparison shows we offer 20-30% lower rates than competitors while maintaining superior quality and customer service.",
            link: "/services",
            linkText: "Compare all our services",
        },
        {
            question:
                "Can I purchase essay writing help for specific academic levels?",
            answer: "Yes! We offer college essay writing service prices, university dissertation writing cost, and undergraduate assignment help cost tailored to each academic level.",
            link: "/services/assignments",
            linkText: "View assignment help services",
        },
        {
            question: "What's included when I buy academic papers online?",
            answer: "When you buy academic papers online from us, you get original content, free revisions, plagiarism report, and 24/7 customer support at no extra cost.",
            link: "/services",
            linkText: "See all included services",
        },
    ];

    const structuredDataStatic = {
        "@context": "https://schema.org",
        "@type": "Service",
        name: "Academic Writing Services",
        description:
            "Cheap custom essay writing, affordable research papers, and dissertation help with transparent pricing.",
        provider: {
            "@type": "Organization",
            name: "Academic Scribe",
        },
        offers: pricingTiers.map((tier) => ({
            "@type": "Offer",
            name: `${tier.level} Academic Writing Service`,
            price: tier.startingPrice,
            priceCurrency: "USD",
            description: tier.description,
        })),
    };

    return (
        <MainLayout>
            <Head>
                <title>Cheap Custom Essay Writing Service Prices | From $8.50/Page | Academic Scribe</title>
                <meta
                    name="description"
                    content="Affordable, original essays from $8.50/page. Cheap custom essay writing service with plagiarism reports, free revisions, fast turnaround, and secure payments."
                />

                {/* Open Graph */}
                <meta
                    property="og:title"
                    content="Cheapest Essays Online | Buy Cheap Essays Starting $8.50"
                />
                <meta
                    property="og:description"
                    content="Get the cheapest essays online with quality guarantee. Buy cheap essays starting $8.50/page. Custom essay writing at lowest prices with instant delivery."
                />
                <meta property="og:type" content="website" />
                <meta property="og:url" content="https://academicscribe.com/pricing-page" />
                <meta
                    property="og:image"
                    content="/images/academic-writing-pricing.jpg"
                />

                {/* Twitter Card */}
                <meta name="twitter:card" content="summary_large_image" />
                <meta
                    name="twitter:title"
                    content="Cheap Custom Essay Writing Service Prices | From $8.50/Page | Academic Scribe"
                />
                <meta
                    name="twitter:description"
                    content="Affordable, original essays from $8.50/page. Cheap custom essay writing service with plagiarism reports, free revisions, fast turnaround, and secure payments."
                />
                <meta
                    name="twitter:image"
                    content="/images/academic-writing-pricing.jpg"
                />

                {/* Canonical URL */}
                <link rel="canonical" href="https://academicscribe.com/pricing-page" />

                {/* Structured Data */}
                <script type="application/ld+json">
                    {JSON.stringify(structuredDataStatic)}
                </script>
                <script type="application/ld+json">
                    {JSON.stringify({
                        "@context": "https://schema.org",
                        "@type": "BreadcrumbList",
                        itemListElement: [
                            { "@type": "ListItem", position: 1, name: "Home", item: "https://academicscribe.com/" },
                            { "@type": "ListItem", position: 2, name: "Pricing", item: "https://academicscribe.com/pricing-page" },
                        ],
                    })}
                </script>
                <script type="application/ld+json">
                    {JSON.stringify({
                        "@context": "https://schema.org",
                        "@type": "FAQPage",
                        mainEntity: faqs.map((f) => ({
                            "@type": "Question",
                            name: f.question,
                            acceptedAnswer: { "@type": "Answer", text: f.answer },
                        })),
                    })}
                </script>
            </Head>

            <div className="pricing-page">
                {/* Hero Section */}
                <section className="bg-gradient-to-r from-blue-600 to-purple-600 py-16 text-white">
                    <div className="max-w-6xl mx-auto px-4 sm:px-6 lg:px-8 text-center">
                        <h1 className="text-4xl md:text-5xl font-bold mb-6">
                            Cheap Custom Essay Writing Service Prices (From $8.50/Page)
                        </h1>
                        <p className="text-xl md:text-2xl mb-8 max-w-3xl mx-auto">
                            Get the cheapest essays online with quality guarantee! Buy cheap essays starting $8.50/page. Custom essay writing at lowest prices with instant delivery and money-back guarantee. Order custom essay cheap online with quality guarantee from PhD writers.
                        </p>
                        <div className="flex flex-col sm:flex-row gap-4 justify-center">
                            <a
                                href="/place-order"
                                className="bg-white text-blue-600 hover:bg-gray-100 px-8 py-4 rounded-lg font-bold text-lg transition-colors"
                            >
                                Get Cheapest Essays Now
                            </a>
                            <a
                                href="/services/essay-writing"
                                className="border-2 border-white text-white hover:bg-white hover:text-blue-600 px-8 py-4 rounded-lg font-bold text-lg transition-colors"
                            >
                                View Essay Services
                            </a>
                        </div>
                        <div className="mt-6 text-sm text-blue-100">
                            <span className="mr-2">Related:</span>
                            <a href="/services/essay-writing" className="underline hover:text-white mr-3">cheap custom essay</a>
                            <a href="/services/essay-writing" className="underline hover:text-white mr-3">cheap custom essay writing service</a>
                            <a href="/services/essay-writing" className="underline hover:text-white mr-3">buy essays online cheap</a>
                            <a href="/services/essay-writing" className="underline hover:text-white">order essay online cheap</a>
                        </div>
                    </div>
                </section>

                {/* Pricing Tiers Section */}
                <section id="pricing" className="py-16 bg-gray-50">
                    <div className="max-w-6xl mx-auto px-4 sm:px-6 lg:px-8">
                        <div className="text-center mb-12">
                            <h2 className="text-3xl md:text-4xl font-bold text-gray-900 mb-4">
                                Cheapest Essay Writing Service Prices & Savings
                            </h2>
                            <p className="text-xl text-gray-600">
                                Compare our cheapest essay writing service price per page - Save 30-50% compared to competitors! Buy cheap essays online with quality guarantee. The prices below are the average starting prices for a 2 weeks deadline.
                            </p>
                        </div>

                        <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-4 gap-8">
                            {pricingTiers.map((tier, index) => (
                                <div
                                    key={index}
                                    className="bg-white rounded-lg shadow-lg p-8 text-center hover:shadow-xl transition-shadow"
                                >
                                    <h3 className="text-2xl font-bold text-gray-900 mb-4">
                                        {tier.level}
                                    </h3>
                                    <div className="mb-6">
                                        <span className="text-4xl font-bold text-blue-600">
                                            ${tier.startingPrice}
                                        </span>
                                        <span className="text-gray-600">
                                            /page
                                        </span>
                                    </div>
                                    <p className="text-gray-600 mb-6">
                                        {tier.description}
                                    </p>
                                    <div className="space-y-2">
                                        <a
                                            href="/place-order"
                                            className="w-full bg-blue-600 hover:bg-blue-700 text-white py-3 px-6 rounded-lg font-semibold transition-colors inline-block"
                                        >
                                            Buy Affordable {tier.level} Essays
                                        </a>
                                        <a
                                            href="/services/essay-writing"
                                            className="w-full border-2 border-blue-600 text-blue-600 hover:bg-blue-600 hover:text-white py-2 px-6 rounded-lg font-semibold transition-colors inline-block text-sm"
                                        >
                                            View Essay Services
                                        </a>
                                    </div>
                                </div>
                            ))}
                        </div>

                        {/* Price Comparison Section */}
                        <div className="mt-12 bg-green-50 rounded-lg p-8 max-w-4xl mx-auto">
                            <h3 className="text-2xl font-bold text-gray-900 mb-6 text-center">
                                🏆 Why We're the Cheapest Essay Writing Service
                            </h3>
                            <div className="grid grid-cols-1 md:grid-cols-3 gap-6">
                                <div className="text-center">
                                    <div className="text-3xl font-bold text-red-600 mb-2">$15-20</div>
                                    <div className="text-sm text-gray-600 mb-2">Competitor Prices</div>
                                    <div className="text-xs text-gray-500">Per page</div>
                                </div>
                                <div className="text-center">
                                    <div className="text-3xl font-bold text-green-600 mb-2">$8.50-18</div>
                                    <div className="text-sm text-gray-600 mb-2">Our Cheapest Prices</div>
                                    <div className="text-xs text-gray-500">Per page</div>
                                </div>
                                <div className="text-center">
                                    <div className="text-3xl font-bold text-blue-600 mb-2">Save 30-50%</div>
                                    <div className="text-sm text-gray-600 mb-2">You Save</div>
                                    <div className="text-xs text-gray-500">Every page</div>
                                </div>
                            </div>
                            <div className="text-center mt-6">
                                <a
                                    href="/place-order"
                                    className="bg-green-600 hover:bg-green-700 text-white px-8 py-3 rounded-lg font-bold text-lg transition-colors inline-block"
                                >
                                    Order Cheapest Essays Now - Save 30%+
                                </a>
                            </div>
                        </div>

                        {/* Internal Links Section */}
                        <div className="mt-12 bg-blue-50 rounded-lg p-8 max-w-4xl mx-auto">
                            <h3 className="text-2xl font-bold text-gray-900 mb-6 text-center">
                                Why Choose Our Cheapest Essay Writing Service?
                            </h3>
                            <div className="grid grid-cols-1 md:grid-cols-2 gap-6">
                                <div className="text-center">
                                    <h4 className="text-lg font-semibold text-gray-900 mb-3">
                                        Meet Our Expert Writers
                                    </h4>
                                    <p className="text-gray-600 mb-4">
                                        Our cheapest essays are written by PhD-qualified writers with verified credentials and extensive experience.
                                    </p>
                                    <a
                                        href="/writers"
                                        className="bg-blue-600 hover:bg-blue-700 text-white px-6 py-3 rounded-lg font-semibold transition-colors inline-block"
                                    >
                                        Meet Our PhD-Qualified Writers
                                    </a>
                                </div>
                                <div className="text-center">
                                    <h4 className="text-lg font-semibold text-gray-900 mb-3">
                                        Our Quality Guarantees
                                    </h4>
                                    <p className="text-gray-600 mb-4">
                                        Get cheapest essays with comprehensive guarantees including plagiarism-free content and unlimited revisions.
                                    </p>
                                    <a
                                        href="/guarantees"
                                        className="bg-green-600 hover:bg-green-700 text-white px-6 py-3 rounded-lg font-semibold transition-colors inline-block"
                                    >
                                        View Our Quality Guarantees
                                    </a>
                                </div>
                            </div>
                        </div>

                        <div className="text-center mt-8">
                            <p className="text-gray-600 mb-4">
                                * Cheapest essay prices may vary based on deadline and complexity. All prices include quality guarantee and free revisions.
                            </p>
                            <div className="bg-blue-50 rounded-lg p-6 max-w-4xl mx-auto">
                                <h3 className="text-lg font-semibold text-gray-900 mb-2">
                                    ⚡ URGENT: Cheapest Essays with Same-Day Delivery
                                </h3>
                                <p className="text-gray-700 mb-3">
                                    Need cheapest essays urgently? Get same-day delivery for just 50% extra! Buy cheap essay online with same day delivery. Rush orders available with emergency cheapest essay help starting at $12.75/page.
                                </p>
                                <a
                                    href="/services/essay-writing"
                                    className="text-blue-600 hover:text-blue-800 font-semibold text-sm inline-flex items-center"
                                >
                                    Get Cheapest Essays Urgently
                                    <svg
                                        className="w-4 h-4 ml-1"
                                        fill="none"
                                        stroke="currentColor"
                                        viewBox="0 0 24 24"
                                    >
                                        <path
                                            strokeLinecap="round"
                                            strokeLinejoin="round"
                                            strokeWidth={2}
                                            d="M9 5l7 7-7 7"
                                        />
                                    </svg>
                                </a>
                            </div>
                        </div>
                    </div>
                </section>

                {/* Our Services & Pricing Section */}
                <section className="py-16 bg-white">
                    <div className="max-w-6xl mx-auto px-4 sm:px-6 lg:px-8">
                        <div className="text-center mb-12">
                            <h2 className="text-3xl md:text-4xl font-bold text-gray-900 mb-4">
                                All Academic Writing Services & Pricing
                            </h2>
                        <p className="text-xl text-gray-600">
                            Explore our complete range of cheapest academic writing services with transparent pricing and quality guarantee. Buy cheap essays online with quality guarantee from professional writers.
                        </p>
                        </div>

                        <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-3 gap-8">
                            {serviceLinks.map((service, index) => (
                                <div
                                    key={index}
                                    className="bg-gray-50 rounded-lg shadow-sm p-6 hover:shadow-md transition-shadow border-l-4 border-blue-600"
                                >
                                    <div className="flex justify-between items-start mb-3">
                                        <h3 className="text-xl font-semibold text-gray-900">
                                            {service.name}
                                        </h3>
                                        <span className="text-lg font-bold text-blue-600">
                                            {service.price}
                                        </span>
                                    </div>
                                    <p className="text-gray-600 mb-4">
                                        {service.description}
                                    </p>
                                    <div className="flex space-x-2">
                                        <a
                                            href={service.url}
                                            className="flex-1 bg-blue-600 hover:bg-blue-700 text-white py-2 px-4 rounded-lg font-semibold transition-colors text-center text-sm"
                                        >
                                            Learn More
                                        </a>
                                        <a
                                            href="/place-order"
                                            className="flex-1 border-2 border-blue-600 text-blue-600 hover:bg-blue-600 hover:text-white py-2 px-4 rounded-lg font-semibold transition-colors text-center text-sm"
                                        >
                                            Order Now
                                        </a>
                                    </div>
                                </div>
                            ))}
                        </div>

                        <div className="text-center mt-8">
                            <a
                                href="/services"
                                className="bg-blue-600 hover:bg-blue-700 text-white px-8 py-4 rounded-lg font-bold text-lg transition-colors inline-block"
                            >
                                View All Essay Writing Services
                            </a>
                        </div>
                    </div>
                </section>

                {/* Service Image */}
                <section className="py-16 bg-gray-50">
                    <div className="max-w-6xl mx-auto px-4 sm:px-6 lg:px-8 text-center">
                        <h2 className="text-3xl font-bold text-gray-900 mb-8">
                            Professional Academic Writing Service Cost
                            Comparison
                        </h2>
                        {/* <img
                            src="/images/academic-writing-pricing.jpg"
                            alt="Buy essays online cheap - professional academic writers providing quality essay writing service prices, dissertation writing cost, and research paper writing rates for students"
                            className="w-full max-w-4xl mx-auto rounded-lg shadow-lg"
                        /> */}

                        <OptimizedImage
                            src="/images/academic-writing-pricing.jpg"
                            alt="Buy essays online cheap - professional academic writers providing quality essay writing service prices, dissertation writing cost, and research paper writing rates for students"
                            className="w-full h-[400px] sm:h-[500px] lg:h-[600px] object-cover"
                            priority={true}
                            width={600}
                            height={600}
                            sizes="(max-width: 768px) 100vw, (max-width: 1024px) 50vw, 600px"
                        />
                        <p className="text-gray-600 mt-4 max-w-2xl mx-auto">
                            When you purchase essay writing help from us, you
                            get expert writers, original content, and the best
                            academic writing service pricing in the industry.
                        </p>
                        <div className="mt-6">
                            <a
                                href="/services"
                                className="text-blue-600 hover:text-blue-800 font-semibold inline-flex items-center"
                            >
                                Explore All Cheapest Essay Services
                                <svg
                                    className="w-4 h-4 ml-1"
                                    fill="none"
                                    stroke="currentColor"
                                    viewBox="0 0 24 24"
                                >
                                    <path
                                        strokeLinecap="round"
                                        strokeLinejoin="round"
                                        strokeWidth={2}
                                        d="M9 5l7 7-7 7"
                                    />
                                </svg>
                            </a>
                        </div>
                    </div>
                </section>

                {/* Reasons to Choose Us */}
                <section className="py-16 bg-white">
                    <div className="max-w-6xl mx-auto px-4 sm:px-6 lg:px-8">
                        <div className="text-center mb-12">
                            <h2 className="text-3xl md:text-4xl font-bold text-gray-900 mb-4">
                                Why Our Prices Beat the Competition
                            </h2>
                            <p className="text-xl text-gray-600">
                                Cheapest essay writing service prices with guaranteed quality and professional writers. Save 30-50% compared to competitors.
                            </p>
                        </div>

                        <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-3 gap-8">
                            {reasons.map((reason, index) => (
                                <div
                                    key={index}
                                    className="bg-gray-50 rounded-lg shadow-sm p-6 hover:shadow-md transition-shadow"
                                >
                                    <h3 className="text-xl font-semibold text-gray-900 mb-3">
                                        {reason.title}
                                    </h3>
                                    <p className="text-gray-600 mb-4">
                                        {reason.description}
                                    </p>
                                    <a
                                        href={reason.link}
                                        className="text-blue-600 hover:text-blue-800 font-semibold text-sm inline-flex items-center"
                                    >
                                        {reason.linkText}
                                        <svg
                                            className="w-4 h-4 ml-1"
                                            fill="none"
                                            stroke="currentColor"
                                            viewBox="0 0 24 24"
                                        >
                                            <path
                                                strokeLinecap="round"
                                                strokeLinejoin="round"
                                                strokeWidth={2}
                                                d="M9 5l7 7-7 7"
                                            />
                                        </svg>
                                    </a>
                                </div>
                            ))}
                        </div>
                    </div>
                </section>

                {/* FAQ Section */}
                <section className="py-16 bg-gray-50">
                    <div className="max-w-4xl mx-auto px-4 sm:px-6 lg:px-8">
                        <div className="text-center mb-12">
                            <h2 className="text-3xl md:text-4xl font-bold text-gray-900 mb-4">
                                Academic Writing Service Pricing FAQ
                            </h2>
                            <p className="text-xl text-gray-600">
                                Common questions about essay writing service
                                cost per page and dissertation writing service
                                rates
                            </p>
                        </div>

                        <div className="space-y-6">
                            {faqs.map((faq, index) => (
                                <div
                                    key={index}
                                    className="bg-white rounded-lg shadow-sm"
                                >
                                    <details className="group">
                                        <summary className="flex justify-between items-center p-6 cursor-pointer">
                                            <h3 className="text-lg font-semibold text-gray-900 pr-4">
                                                {faq.question}
                                            </h3>
                                            <svg
                                                className="w-5 h-5 text-gray-500 transition-transform group-open:rotate-180"
                                                fill="none"
                                                stroke="currentColor"
                                                viewBox="0 0 24 24"
                                            >
                                                <path
                                                    strokeLinecap="round"
                                                    strokeLinejoin="round"
                                                    strokeWidth={2}
                                                    d="M19 9l-7 7-7-7"
                                                />
                                            </svg>
                                        </summary>
                                        <div className="px-6 pb-6">
                                            <p className="text-gray-600 leading-relaxed mb-3">
                                                {faq.answer}
                                            </p>
                                            {faq.link && (
                                                <a
                                                    href={faq.link}
                                                    className="text-blue-600 hover:text-blue-800 font-semibold text-sm inline-flex items-center"
                                                >
                                                    {faq.linkText}
                                                    <svg
                                                        className="w-4 h-4 ml-1"
                                                        fill="none"
                                                        stroke="currentColor"
                                                        viewBox="0 0 24 24"
                                                    >
                                                        <path
                                                            strokeLinecap="round"
                                                            strokeLinejoin="round"
                                                            strokeWidth={2}
                                                            d="M9 5l7 7-7 7"
                                                        />
                                                    </svg>
                                                </a>
                                            )}
                                        </div>
                                    </details>
                                </div>
                            ))}
                        </div>
                    </div>
                </section>

                {/* Final CTA Section */}
                <section className="py-16 bg-gradient-to-r from-blue-600 to-purple-600">
                    <div className="max-w-4xl mx-auto px-4 sm:px-6 lg:px-8 text-center">
                        <h2 className="text-3xl md:text-4xl font-bold text-white mb-4">
                            Ready to Buy Affordable Essays Online?
                        </h2>
                        <p className="text-xl text-blue-100 mb-8 max-w-2xl mx-auto">
                            Get cheapest essays online from expert writers with quality guarantee. Order cheap essays with our reliable pricing and save 30-50% compared to competitors. Buy cheap essays online with quality guarantee from PhD writers.
                        </p>

                        {/* Limited Time Offer */}
                        <div className="bg-red-50 border-2 border-red-200 rounded-lg p-6 max-w-4xl mx-auto mb-8">
                            <h3 className="text-xl font-bold text-red-800 mb-2 text-center">
                                🚨 LIMITED TIME: Extra 15% OFF Essay Orders
                            </h3>
                            <p className="text-red-700 text-center mb-4">
                                Limited writer availability today! This offer expires in 24 hours.
                            </p>
                            <div className="text-center">
                                <a
                                    href="/place-order"
                                    className="bg-red-600 hover:bg-red-700 text-white px-8 py-3 rounded-lg font-bold text-lg transition-colors inline-block"
                                >
                                    Claim 15% OFF Now
                                </a>
                            </div>
                        </div>

                        <div className="flex flex-col sm:flex-row gap-4 justify-center mb-8">
                            <a
                                href="/place-order"
                                className="bg-white text-blue-600 hover:bg-gray-100 px-8 py-4 rounded-lg font-bold text-lg transition-colors"
                            >
                                Buy Essays Now
                            </a>
                            <a
                                href="/services"
                                className="border-2 border-white text-white hover:bg-white hover:text-blue-600 px-8 py-4 rounded-lg font-bold text-lg transition-colors"
                            >
                                View All Services
                            </a>
                        </div>

                        {/* Trust Indicators */}
                        <div className="flex flex-wrap justify-center items-center gap-8 text-blue-100">
                            <div className="flex items-center">
                                <svg
                                    className="w-5 h-5 mr-2"
                                    fill="currentColor"
                                    viewBox="0 0 20 20"
                                >
                                    <path
                                        fillRule="evenodd"
                                        d="M16.707 5.293a1 1 0 010 1.414l-8 8a1 1 0 01-1.414 0l-4-4a1 1 0 011.414-1.414L8 12.586l7.293-7.293a1 1 0 011.414 0z"
                                        clipRule="evenodd"
                                    />
                                </svg>
                                <span className="text-sm">
                                    Money-Back Guarantee
                                </span>
                            </div>
                            <div className="flex items-center">
                                <svg
                                    className="w-5 h-5 mr-2"
                                    fill="currentColor"
                                    viewBox="0 0 20 20"
                                >
                                    <path
                                        fillRule="evenodd"
                                        d="M16.707 5.293a1 1 0 010 1.414l-8 8a1 1 0 01-1.414 0l-4-4a1 1 0 011.414-1.414L8 12.586l7.293-7.293a1 1 0 011.414 0z"
                                        clipRule="evenodd"
                                    />
                                </svg>
                                <span className="text-sm">Free Revisions</span>
                            </div>
                            <div className="flex items-center">
                                <svg
                                    className="w-5 h-5 mr-2"
                                    fill="currentColor"
                                    viewBox="0 0 20 20"
                                >
                                    <path
                                        fillRule="evenodd"
                                        d="M16.707 5.293a1 1 0 010 1.414l-8 8a1 1 0 01-1.414 0l-4-4a1 1 0 011.414-1.414L8 12.586l7.293-7.293a1 1 0 011.414 0z"
                                        clipRule="evenodd"
                                    />
                                </svg>
                                <span className="text-sm">24/7 Support</span>
                            </div>
                            <div className="flex items-center">
                                <svg
                                    className="w-5 h-5 mr-2"
                                    fill="currentColor"
                                    viewBox="0 0 20 20"
                                >
                                    <path
                                        fillRule="evenodd"
                                        d="M16.707 5.293a1 1 0 010 1.414l-8 8a1 1 0 01-1.414 0l-4-4a1 1 0 011.414-1.414L8 12.586l7.293-7.293a1 1 0 011.414 0z"
                                        clipRule="evenodd"
                                    />
                                </svg>
                                <span className="text-sm">Plagiarism-Free</span>
                            </div>
                        </div>
                    </div>
                </section>

                {/* Additional Keywords Section */}
                <section className="py-16 bg-white">
                    <div className="max-w-6xl mx-auto px-4 sm:px-6 lg:px-8">
                        <div className="text-center mb-12">
                            <h2 className="text-3xl md:text-4xl font-bold text-gray-900 mb-4">
                                Complete Academic Writing Service Cost Guide
                            </h2>
                            <p className="text-xl text-gray-600 mb-8">
                                Comprehensive pricing for all academic writing
                                services
                            </p>
                        </div>

                        <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-3 gap-8">
                            <div className="bg-gray-50 rounded-lg p-6">
                                <h3 className="text-xl font-semibold text-gray-900 mb-3">
                                    <a
                                        href="/services/assignments"
                                        className="hover:text-blue-600 transition-colors"
                                    >
                                        Assignment Help Pricing
                                    </a>
                                </h3>
                                <ul className="text-gray-600 space-y-2">
                                    <li>
                                        •{" "}
                                        <a
                                            href="/services/case-studies"
                                            className="hover:text-blue-600 transition-colors"
                                        >
                                            Case study writing cost: $15-22/page
                                        </a>
                                    </li>
                                    <li>
                                        •{" "}
                                        <a
                                            href="/services/literature-reviews"
                                            className="hover:text-blue-600 transition-colors"
                                        >
                                            Literature review writing rates:
                                            $16-24/page
                                        </a>
                                    </li>
                                    <li>
                                        •{" "}
                                        <a
                                            href="/services/thesis-proposals"
                                            className="hover:text-blue-600 transition-colors"
                                        >
                                            Thesis proposal writing prices:
                                            $18-25/page
                                        </a>
                                    </li>
                                    <li>
                                        •{" "}
                                        <a
                                            href="/services/assignments"
                                            className="hover:text-blue-600 transition-colors"
                                        >
                                            Undergraduate assignment help cost:
                                            $12-18/page
                                        </a>
                                    </li>
                                </ul>
                            </div>

                            <div className="bg-gray-50 rounded-lg p-6">
                                <h3 className="text-xl font-semibold text-gray-900 mb-3">
                                    Quality vs Price Guarantee
                                </h3>
                                <ul className="text-gray-600 space-y-2">
                                    <li>
                                        •{" "}
                                        <a
                                            href="/services/essay-writing"
                                            className="hover:text-blue-600 transition-colors"
                                        >
                                            Quality essay writing affordable
                                            prices
                                        </a>
                                    </li>
                                    <li>
                                        •{" "}
                                        <a
                                            href="/services/research-papers"
                                            className="hover:text-blue-600 transition-colors"
                                        >
                                            Expert research paper writing cost
                                        </a>
                                    </li>
                                    <li>
                                        •
                                        <a
                                            href="/services"
                                            className="hover:text-blue-600 transition-colors"
                                        >
                                            Reliable essay service pricing
                                        </a>
                                    </li>
                                    <li>
                                        •{" "}
                                        <a
                                            href="/services/dissertation-writing"
                                            className="hover:text-blue-600 transition-colors"
                                        >
                                            Professional dissertation writing
                                            rates
                                        </a>
                                    </li>
                                </ul>
                            </div>

                            <div className="bg-gray-50 rounded-lg p-6">
                                <h3 className="text-xl font-semibold text-gray-900 mb-3">
                                    Academic Level Pricing
                                </h3>
                                <ul className="text-gray-600 space-y-2">
                                    <li>
                                        •{" "}
                                        <a
                                            href="/services/essay-writing"
                                            className="hover:text-blue-600 transition-colors"
                                        >
                                            College essay writing service prices
                                        </a>
                                    </li>
                                    <li>
                                        •{" "}
                                        <a
                                            href="/services/dissertation-writing"
                                            className="hover:text-blue-600 transition-colors"
                                        >
                                            University dissertation writing cost
                                        </a>
                                    </li>
                                    <li>
                                        •{" "}
                                        <a
                                            href="/services/research-papers"
                                            className="hover:text-blue-600 transition-colors"
                                        >
                                            Graduate research paper prices
                                        </a>
                                    </li>
                                    <li>
                                        •{" "}
                                        <a
                                            href="/services/dissertation-writing"
                                            className="hover:text-blue-600 transition-colors"
                                        >
                                            PhD thesis writing service rates
                                        </a>
                                    </li>
                                </ul>
                            </div>
                        </div>

                        <div className="mt-12 text-center">
                            <div className="bg-blue-50 rounded-lg p-8 max-w-4xl mx-auto">
                                <h3 className="text-2xl font-bold text-gray-900 mb-4">
                                    Why We're the Best Academic Writing Service
                                    Pricing Choice
                                </h3>
                                <div className="grid grid-cols-1 md:grid-cols-2 gap-6 text-left">
                                    <div>
                                        <h4 className="font-semibold text-gray-900 mb-2">
                                            Affordable vs Expensive Essay
                                            Services
                                        </h4>
                                        <p className="text-gray-600 mb-2">
                                            We offer the perfect balance of
                                            quality and affordability, beating
                                            expensive competitors by 30%.
                                        </p>
                                        <a
                                            href="/services/essay-writing"
                                            className="text-blue-600 hover:text-blue-800 font-semibold text-sm"
                                        >
                                            Compare Cheapest Essay Prices →
                                        </a>
                                    </div>
                                    <div>
                                        <h4 className="font-semibold text-gray-900 mb-2">
                                            Compare Essay Writing Prices
                                        </h4>
                                        <p className="text-gray-600 mb-2">
                                            Our transparent pricing beats the
                                            competition with no hidden fees or
                                            surprise charges.
                                        </p>
                                        <a
                                            href="/services"
                                            className="text-blue-600 hover:text-blue-800 font-semibold text-sm"
                                        >
                                            View All Cheapest Services →
                                        </a>
                                    </div>
                                </div>
                            </div>
                        </div>
                    </div>
                </section>

                {/* Customer Success Stories */}
                <section className="py-16 bg-gray-50">
                    <div className="max-w-6xl mx-auto px-4 sm:px-6 lg:px-8">
                        <div className="text-center mb-12">
                            <h2 className="text-3xl md:text-4xl font-bold text-gray-900 mb-4">
                                What Our Customers Say About Our Pricing
                            </h2>
                            <p className="text-xl text-gray-600">
                                Real feedback from students who chose our
                                services
                            </p>
                        </div>

                        <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-3 gap-8">
                            <div className="bg-white rounded-lg p-6 shadow-sm">
                                <div className="flex items-center mb-4">
                                    <div className="flex text-yellow-400">
                                        {[...Array(5)].map((_, i) => (
                                            <svg
                                                key={i}
                                                className="w-5 h-5"
                                                fill="currentColor"
                                                viewBox="0 0 20 20"
                                            >
                                                <path d="M9.049 2.927c.3-.921 1.603-.921 1.902 0l1.07 3.292a1 1 0 00.95.69h3.462c.969 0 1.371 1.24.588 1.81l-2.8 2.034a1 1 0 00-.364 1.118l1.07 3.292c.3.921-.755 1.688-1.54 1.118l-2.8-2.034a1 1 0 00-1.175 0l-2.8 2.034c-.784.57-1.838-.197-1.539-1.118l1.07-3.292a1 1 0 00-.364-1.118L2.98 8.72c-.783-.57-.38-1.81.588-1.81h3.461a1 1 0 00.951-.69l1.07-3.292z" />
                                            </svg>
                                        ))}
                                    </div>
                                    <span className="ml-2 text-sm text-gray-600">
                                        Essay Writing
                                    </span>
                                </div>
                                <blockquote className="text-gray-700 mb-4 italic">
                                    "Best essay writing service prices I found!
                                    Quality work at affordable rates."
                                </blockquote>
                                <div className="flex items-center justify-between">
                                    <div>
                                        <div className="font-semibold text-gray-900">
                                            Sarah M.
                                        </div>
                                        <div className="text-sm text-gray-500">
                                            University Student
                                        </div>
                                    </div>
                                    <a
                                        href="/services/essay-writing"
                                        className="text-blue-600 hover:text-blue-800 text-sm font-semibold"
                                    >
                                        Get Cheapest Essay Help →
                                    </a>
                                </div>
                            </div>

                            <div className="bg-white rounded-lg p-6 shadow-sm">
                                <div className="flex items-center mb-4">
                                    <div className="flex text-yellow-400">
                                        {[...Array(5)].map((_, i) => (
                                            <svg
                                                key={i}
                                                className="w-5 h-5"
                                                fill="currentColor"
                                                viewBox="0 0 20 20"
                                            >
                                                <path d="M9.049 2.927c.3-.921 1.603-.921 1.902 0l1.07 3.292a1 1 0 00.95.69h3.462c.969 0 1.371 1.24.588 1.81l-2.8 2.034a1 1 0 00-.364 1.118l1.07 3.292c.3.921-.755 1.688-1.54 1.118l-2.8-2.034a1 1 0 00-1.175 0l-2.8 2.034c-.784.57-1.838-.197-1.539-1.118l1.07-3.292a1 1 0 00-.364-1.118L2.98 8.72c-.783-.57-.38-1.81.588-1.81h3.461a1 1 0 00.951-.69l1.07-3.292z" />
                                            </svg>
                                        ))}
                                    </div>
                                    <span className="ml-2 text-sm text-gray-600">
                                        Research Paper
                                    </span>
                                </div>
                                <blockquote className="text-gray-700 mb-4 italic">
                                    "Research paper writing rates were very
                                    reasonable. Excellent value for money!"
                                </blockquote>
                                <div className="flex items-center justify-between">
                                    <div>
                                        <div className="font-semibold text-gray-900">
                                            Mike R.
                                        </div>
                                        <div className="text-sm text-gray-500">
                                            Graduate Student
                                        </div>
                                    </div>
                                    <a
                                        href="/services/research-papers"
                                        className="text-blue-600 hover:text-blue-800 text-sm font-semibold"
                                    >
                                        Get Research Help →
                                    </a>
                                </div>
                            </div>

                            <div className="bg-white rounded-lg p-6 shadow-sm">
                                <div className="flex items-center mb-4">
                                    <div className="flex text-yellow-400">
                                        {[...Array(5)].map((_, i) => (
                                            <svg
                                                key={i}
                                                className="w-5 h-5"
                                                fill="currentColor"
                                                viewBox="0 0 20 20"
                                            >
                                                <path d="M9.049 2.927c.3-.921 1.603-.921 1.902 0l1.07 3.292a1 1 0 00.95.69h3.462c.969 0 1.371 1.24.588 1.81l-2.8 2.034a1 1 0 00-.364 1.118l1.07 3.292c.3.921-.755 1.688-1.54 1.118l-2.8-2.034a1 1 0 00-1.175 0l-2.8 2.034c-.784.57-1.838-.197-1.539-1.118l1.07-3.292a1 1 0 00-.364-1.118L2.98 8.72c-.783-.57-.38-1.81.588-1.81h3.461a1 1 0 00.951-.69l1.07-3.292z" />
                                            </svg>
                                        ))}
                                    </div>
                                    <span className="ml-2 text-sm text-gray-600">
                                        Dissertation
                                    </span>
                                </div>
                                <blockquote className="text-gray-700 mb-4 italic">
                                    "Dissertation writing cost was fair and the
                                    quality exceeded my expectations."
                                </blockquote>
                                <div className="flex items-center justify-between">
                                    <div>
                                        <div className="font-semibold text-gray-900">
                                            Lisa K.
                                        </div>
                                        <div className="text-sm text-gray-500">
                                            PhD Candidate
                                        </div>
                                    </div>
                                    <a
                                        href="/services/dissertation-writing"
                                        className="text-blue-600 hover:text-blue-800 text-sm font-semibold"
                                    >
                                        Get Dissertation Help →
                                    </a>
                                </div>
                            </div>
                        </div>

                        <div className="text-center mt-8">
                            <a
                                href="/reviews"
                                className="border-2 border-blue-600 text-blue-600 hover:bg-blue-600 hover:text-white px-6 py-3 rounded-lg font-semibold transition-colors"
                            >
                                Read More Reviews
                            </a>
                        </div>
                    </div>
                </section>

                {/* Blog Links & Resources Section */}
                <section className="py-16 bg-white">
                    <div className="max-w-6xl mx-auto px-4 sm:px-6 lg:px-8">
                        <div className="text-center mb-12">
                            <h2 className="text-3xl md:text-4xl font-bold text-gray-900 mb-4">
                                Cheapest Essay Writing Resources & Guides
                            </h2>
                            <p className="text-xl text-gray-600">
                                Learn more about essay writing and find the cheapest essay services with our comprehensive guides
                            </p>
                        </div>

                        <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-3 gap-8">
                            {/* Blog Links */}
                            <div className="bg-gray-50 rounded-lg p-6">
                                <h3 className="text-xl font-semibold text-gray-900 mb-4">
                                    📚 Essay Writing Guides
                                </h3>
                                <ul className="space-y-3">
                                   
                                    <li>
                                        <a
                                            href="/blog/buy-essays-online-ultimate-guide-2025"
                                            className="text-blue-600 hover:text-blue-800 font-medium hover:underline"
                                        >
                                            How to Choose Essay Writing Service
                                        </a>
                                    </li>
                                    <li>
                                        <a
                                            href="/blog/8-essay-writing-tips-for-students-2025"
                                            className="text-blue-600 hover:text-blue-800 font-medium hover:underline"
                                        >
                                            8 Essay Writing Tips for Students (2025)
                                        </a>
                                    </li>
                                    <li>
                                        <a
                                            href="/blog/essay-topic-ideas"
                                            className="text-blue-600 hover:text-blue-800 font-medium hover:underline"
                                        >
                                            Essay Topic Ideas
                                        </a>
                                    </li>
                                    <li>
                                        <a
                                            href="/blog/essay-structure-guide"
                                            className="text-blue-600 hover:text-blue-800 font-medium hover:underline"
                                        >
                                            Essay Structure Guide
                                        </a>
                                    </li>
                                </ul>
                            </div>

                            {/* Trust & Authority Links */}
                            <div className="bg-gray-50 rounded-lg p-6">
                                <h3 className="text-xl font-semibold text-gray-900 mb-4">
                                    🛡️ Trust & Authority
                                </h3>
                                <ul className="space-y-3">
                                    <li>
                                        <a
                                            href="https://www.bbb.org"
                                            target="_blank"
                                            rel="nofollow"
                                            className="text-blue-600 hover:text-blue-800 font-medium hover:underline"
                                        >
                                            Better Business Bureau
                                        </a>
                                    </li>
                                    <li>
                                        <a
                                            href="https://www.trustpilot.com"
                                            target="_blank"
                                            rel="nofollow"
                                            className="text-blue-600 hover:text-blue-800 font-medium hover:underline"
                                        >
                                            Trustpilot Reviews
                                        </a>
                                    </li>
                                    <li>
                                        <a
                                            href="https://www.turnitin.com"
                                            target="_blank"
                                            rel="nofollow"
                                            className="text-blue-600 hover:text-blue-800 font-medium hover:underline"
                                        >
                                            Turnitin Plagiarism Check
                                        </a>
                                    </li>
                                </ul>
                            </div>

                            {/* Educational Resources */}
                            <div className="bg-gray-50 rounded-lg p-6">
                                <h3 className="text-xl font-semibold text-gray-900 mb-4">
                                    🎓 Educational Resources
                                </h3>
                                <ul className="space-y-3">
                                    <li>
                                        <a
                                            href="https://owl.purdue.edu"
                                            target="_blank"
                                            rel="nofollow"
                                            className="text-blue-600 hover:text-blue-800 font-medium hover:underline"
                                        >
                                            Purdue OWL Writing Guide
                                        </a>
                                    </li>
                                    <li>
                                        <a
                                            href="https://www.apastyle.org"
                                            target="_blank"
                                            rel="nofollow"
                                            className="text-blue-600 hover:text-blue-800 font-medium hover:underline"
                                        >
                                            APA Style Guide
                                        </a>
                                    </li>
                                    <li>
                                        <a
                                            href="https://www.mla.org"
                                            target="_blank"
                                            rel="nofollow"
                                            className="text-blue-600 hover:text-blue-800 font-medium hover:underline"
                                        >
                                            MLA Style Guide
                                        </a>
                                    </li>
                                </ul>
                            </div>
                        </div>
                    </div>
                </section>

                {/* Trust and Security Section */}
                <section className="py-16 bg-white">
                    <div className="max-w-6xl mx-auto px-4 sm:px-6 lg:px-8">
                        <div className="text-center mb-12">
                            <h2 className="text-3xl md:text-4xl font-bold text-gray-900 mb-4">
                                Your Security & Privacy Matter
                            </h2>
                            <p className="text-xl text-gray-600">
                                We protect your personal information and ensure
                                secure transactions
                            </p>
                        </div>

                        <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-4 gap-8">
                            <div className="text-center">
                                <div className="bg-green-100 rounded-full w-16 h-16 flex items-center justify-center mx-auto mb-4">
                                    <svg
                                        className="w-8 h-8 text-green-600"
                                        fill="currentColor"
                                        viewBox="0 0 20 20"
                                    >
                                        <path
                                            fillRule="evenodd"
                                            d="M2.166 4.999A11.954 11.954 0 0010 1.944 11.954 11.954 0 0017.834 5c.11.65.166 1.32.166 2.001 0 5.225-3.34 9.67-8 11.317C5.34 16.67 2 12.225 2 7c0-.682.057-1.35.166-2.001zm11.541 3.708a1 1 0 00-1.414-1.414L9 10.586 7.707 9.293a1 1 0 00-1.414 1.414l2 2a1 1 0 001.414 0l4-4z"
                                            clipRule="evenodd"
                                        />
                                    </svg>
                                </div>
                                <h3 className="text-lg font-semibold text-gray-900 mb-2">
                                    SSL Encryption
                                </h3>
                                <p className="text-gray-600 text-sm">
                                    All data transmitted is encrypted with
                                    256-bit SSL security
                                </p>
                            </div>

                            <div className="text-center">
                                <div className="bg-blue-100 rounded-full w-16 h-16 flex items-center justify-center mx-auto mb-4">
                                    <svg
                                        className="w-8 h-8 text-blue-600"
                                        fill="currentColor"
                                        viewBox="0 0 20 20"
                                    >
                                        <path
                                            fillRule="evenodd"
                                            d="M5 9V7a5 5 0 0110 0v2a2 2 0 012 2v5a2 2 0 01-2 2H5a2 2 0 01-2-2v-5a2 2 0 012-2zm8-2v2H7V7a3 3 0 016 0z"
                                            clipRule="evenodd"
                                        />
                                    </svg>
                                </div>
                                <h3 className="text-lg font-semibold text-gray-900 mb-2">
                                    Privacy Protected
                                </h3>
                                <p className="text-gray-600 text-sm">
                                    Your personal information is never shared
                                    with third parties
                                </p>
                            </div>

                            <div className="text-center">
                                <div className="bg-purple-100 rounded-full w-16 h-16 flex items-center justify-center mx-auto mb-4">
                                    <svg
                                        className="w-8 h-8 text-purple-600"
                                        fill="currentColor"
                                        viewBox="0 0 20 20"
                                    >
                                        <path
                                            fillRule="evenodd"
                                            d="M4 4a2 2 0 00-2 2v4a2 2 0 002 2V6h10a2 2 0 00-2-2H4zm2 6a2 2 0 012-2h8a2 2 0 012 2v4a2 2 0 01-2 2H8a2 2 0 01-2-2v-4zm6 4a2 2 0 100-4 2 2 0 000 4z"
                                            clipRule="evenodd"
                                        />
                                    </svg>
                                </div>
                                <h3 className="text-lg font-semibold text-gray-900 mb-2">
                                    Secure Payments
                                </h3>
                                <p className="text-gray-600 text-sm">
                                    Multiple payment options with bank-level
                                    security
                                </p>
                            </div>

                            <div className="text-center">
                                <div className="bg-yellow-100 rounded-full w-16 h-16 flex items-center justify-center mx-auto mb-4">
                                    <svg
                                        className="w-8 h-8 text-yellow-600"
                                        fill="currentColor"
                                        viewBox="0 0 20 20"
                                    >
                                        <path
                                            fillRule="evenodd"
                                            d="M6.267 3.455a3.066 3.066 0 001.745-.723 3.066 3.066 0 013.976 0 3.066 3.066 0 001.745.723 3.066 3.066 0 012.812 2.812c.051.643.304 1.254.723 1.745a3.066 3.066 0 010 3.976 3.066 3.066 0 00-.723 1.745 3.066 3.066 0 01-2.812 2.812 3.066 3.066 0 00-1.745.723 3.066 3.066 0 01-3.976 0 3.066 3.066 0 00-1.745-.723 3.066 3.066 0 01-2.812-2.812 3.066 3.066 0 00-.723-1.745 3.066 3.066 0 010-3.976 3.066 3.066 0 00.723-1.745 3.066 3.066 0 012.812-2.812zm7.44 5.252a1 1 0 00-1.414-1.414L9 10.586 7.707 9.293a1 1 0 00-1.414 1.414l2 2a1 1 0 001.414 0l4-4z"
                                            clipRule="evenodd"
                                        />
                                    </svg>
                                </div>
                                <h3 className="text-lg font-semibold text-gray-900 mb-2">
                                    Verified Service
                                </h3>
                                <p className="text-gray-600 text-sm">
                                    Trusted by thousands with verified customer
                                    reviews
                                </p>
                            </div>
                        </div>

                        {/* Payment Methods */}
                        <div className="mt-12 text-center">
                            <h3 className="text-lg font-semibold text-gray-900 mb-6">
                                Accepted Payment Methods
                            </h3>
                            <div className="flex flex-wrap justify-center items-center gap-6 opacity-60">
                                <div className="bg-white rounded-lg p-3 shadow-sm">
                                    <span className="text-blue-600 font-bold text-lg">
                                        VISA
                                    </span>
                                </div>
                                <div className="bg-white rounded-lg p-3 shadow-sm">
                                    <span className="text-red-600 font-bold text-lg">
                                        MasterCard
                                    </span>
                                </div>
                                <div className="bg-white rounded-lg p-3 shadow-sm">
                                    <span className="text-blue-800 font-bold text-lg">
                                        PayPal
                                    </span>
                                </div>
                           
                               
                            </div>
                        </div>
                    </div>
                </section>

                {/* Contact Support Section */}
                <section className="py-16 bg-gray-50">
                    <div className="max-w-4xl mx-auto px-4 sm:px-6 lg:px-8 text-center">
                        <h2 className="text-3xl md:text-4xl font-bold text-gray-900 mb-4">
                            Still Have Questions About Pricing?
                        </h2>
                        <p className="text-xl text-gray-600 mb-8">
                            Our friendly support team is here to help you choose
                            the right service and pricing option
                        </p>

                        <div className="grid grid-cols-1 md:grid-cols-3 gap-8 mb-8">
                            <div className="text-center">
                                <div className="bg-blue-100 rounded-full w-16 h-16 flex items-center justify-center mx-auto mb-4">
                                    <svg
                                        className="w-8 h-8 text-blue-600"
                                        fill="currentColor"
                                        viewBox="0 0 20 20"
                                    >
                                        <path d="M2 3a1 1 0 011-1h2.153a1 1 0 01.986.836l.74 4.435a1 1 0 01-.54 1.06l-1.548.773a11.037 11.037 0 006.105 6.105l.774-1.548a1 1 0 011.059-.54l4.435.74a1 1 0 01.836.986V17a1 1 0 01-1 1h-2C7.82 18 2 12.18 2 5V3z" />
                                    </svg>
                                </div>
                                <h3 className="text-lg font-semibold text-gray-900 mb-2">
                                    Call Us
                                </h3>
                                <p className="text-gray-600 mb-2">
                                    Speak with our pricing experts
                                </p>
                                <a
                                    href="tel:+1-800-123-4567"
                                    className="text-blue-600 hover:text-blue-800 font-semibold"
                                >
                                    +1 (800) 123-4567
                                </a>
                            </div>

                            <div className="text-center">
                                <div className="bg-green-100 rounded-full w-16 h-16 flex items-center justify-center mx-auto mb-4">
                                    <svg
                                        className="w-8 h-8 text-green-600"
                                        fill="currentColor"
                                        viewBox="0 0 20 20"
                                    >
                                        <path d="M2.003 5.884L10 9.882l7.997-3.998A2 2 0 0016 4H4a2 2 0 00-1.997 1.884z" />
                                        <path d="M18 8.118l-8 4-8-4V14a2 2 0 002 2h12a2 2 0 002-2V8.118z" />
                                    </svg>
                                </div>
                                <h3 className="text-lg font-semibold text-gray-900 mb-2">
                                    Email Us
                                </h3>
                                <p className="text-gray-600 mb-2">
                                    Get detailed pricing information
                                </p>
                                <a
                                    href="mailto:info@academicscribe.com"
                                    className="text-blue-600 hover:text-blue-800 font-semibold"
                                >
                                    info@academicscribe.com
                                </a>
                            </div>

                            <div className="text-center">
                                <div className="bg-purple-100 rounded-full w-16 h-16 flex items-center justify-center mx-auto mb-4">
                                    <svg
                                        className="w-8 h-8 text-purple-600"
                                        fill="currentColor"
                                        viewBox="0 0 20 20"
                                    >
                                        <path
                                            fillRule="evenodd"
                                            d="M18 10c0 3.866-3.582 7-8 7a8.841 8.841 0 01-4.083-.98L2 17l1.338-3.123C2.493 12.767 2 11.434 2 10c0-3.866 3.582-7 8-7s8 3.134 8 7zM7 9H5v2h2V9zm8 0h-2v2h2V9zM9 9h2v2H9V9z"
                                            clipRule="evenodd"
                                        />
                                    </svg>
                                </div>
                                <h3 className="text-lg font-semibold text-gray-900 mb-2">
                                    Live Chat
                                </h3>
                                <p className="text-gray-600 mb-2">
                                    Chat with us right now
                                </p>
                                <button className="text-blue-600 hover:text-blue-800 font-semibold">
                                    Start Chat
                                </button>
                            </div>
                        </div>

                        <div className="bg-blue-50 rounded-lg p-6">
                            <h3 className="text-lg font-semibold text-gray-900 mb-2">
                                💡 Pro Tip: Custom Quote Available
                            </h3>
                            <p className="text-gray-600 mb-3">
                                Need a custom solution or have a large project?
                                Contact us for a personalized quote that fits
                                your specific requirements and budget.
                            </p>
                            <a
                                href="/contact"
                                className="text-blue-600 hover:text-blue-800 font-semibold inline-flex items-center"
                            >
                                Get Custom Quote
                                <svg
                                    className="w-4 h-4 ml-1"
                                    fill="none"
                                    stroke="currentColor"
                                    viewBox="0 0 24 24"
                                >
                                    <path
                                        strokeLinecap="round"
                                        strokeLinejoin="round"
                                        strokeWidth={2}
                                        d="M9 5l7 7-7 7"
                                    />
                                </svg>
                            </a>
                        </div>
                    </div>
                </section>

                {/* Mobile Bottom Navigation */}
                <div className="fixed bottom-0 left-0 right-0 bg-white border-t border-gray-200 p-4 lg:hidden z-50">
                    <div className="flex justify-between items-center">
                        <div className="text-center">
                            <div className="text-sm text-gray-600">
                                Essays starting
                            </div>
                            <div className="text-lg font-bold text-blue-600">
                                $8.50/page
                            </div>
                        </div>
                        <div className="flex space-x-2">
                            <a
                                href="/services"
                                className="border-blue-600 text-blue-600 hover:bg-blue-600 hover:text-white px-4 py-2 rounded-lg font-semibold transition-colors border-2 text-sm"
                            >
                                Services
                            </a>
                            <a
                                href="/place-order"
                                className="bg-blue-600 hover:bg-blue-700 text-white px-6 py-2 rounded-lg font-semibold transition-colors text-sm"
                            >
                                Order Now
                            </a>
                        </div>
                    </div>
                </div>

                {/* Scroll to Top Button */}
                <button
                    onClick={() =>
                        window.scrollTo({ top: 0, behavior: "smooth" })
                    }
                    className="fixed bottom-8 right-8 z-40 bg-blue-600 hover:bg-blue-700 text-white p-3 rounded-full shadow-lg transition-all duration-300 transform hover:scale-110 lg:bottom-24"
                    aria-label="Scroll to top"
                >
                    <svg
                        className="w-6 h-6"
                        fill="none"
                        stroke="currentColor"
                        viewBox="0 0 24 24"
                    >
                        <path
                            strokeLinecap="round"
                            strokeLinejoin="round"
                            strokeWidth={2}
                            d="M5 10l7-7m0 0l7 7m-7-7v18"
                        />
                    </svg>
                </button>
            </div>
        </MainLayout>
    );
}
