import React from "react";
import { Head } from "@inertiajs/react";
import MainLayout from "@/Layouts/MainLayout";
import ReviewHero from "@/Components/Reviews/ReviewHero";
import RatingOverview from "@/Components/Reviews/RatingOverview";
import FeaturedTestimonials from "@/Components/Reviews/FeaturedTestimonials";
import ReviewsGrid from "@/Components/Reviews/ReviewsGrid";
import ReviewsByService from "@/Components/Reviews/ReviewsByService";
import ReviewTrustIndicators from "@/Components/Reviews/ReviewTrustIndicators";
import ReviewServiceLinks from "@/Components/Reviews/ReviewServiceLinks";
import ServiceFAQs from "@/Components/Services/ServiceFAQs";
import ServiceCTA from "@/Components/Services/ServiceCTA";

export default function Reviews(props) {
    const { seoData, contentBlocks, structuredData, isCMS } = props;

    // Extract content from CMS blocks
    const getContentBlock = (blockKey) => {
        if (!isCMS || !contentBlocks) return null;
        const block = contentBlocks.find((b) => b.block_key === blockKey);
        return block ? block.content : null;
    };

    // Get all content blocks from CMS
    const hero = getContentBlock('reviews_hero');
    const ratingOverview = getContentBlock('rating_overview');
    const featured = getContentBlock('featured_testimonials');
    const serviceFiltersData = getContentBlock('service_filters');
    const customerReviewsData = getContentBlock('customer_reviews');
    const reviewsByService = getContentBlock('reviews_by_service');
    const trustIndicators = getContentBlock('review_trust');
    const faqs = getContentBlock('reviews_faqs');
    const cta = getContentBlock('reviews_cta');
    const serviceLinks = getContentBlock('review_service_links');

    // FALLBACK DATA (exact copy from original for SEO if CMS not available)
    const fallbackHero = {
        title: 'Customer Reviews & Student Testimonials',
        description: 'Read authentic reviews from 2,800+ satisfied students who trusted us with their academic success. See why we maintain a 4.9/5 star rating across all our academic writing services.',
        stats: [
            { value: '2,847', label: 'Total Reviews' },
            { value: '4.9', label: 'Average Rating' },
            { value: '98%', label: 'Satisfaction Rate' },
            { value: '100%', label: 'Verified Reviews' },
        ]
    };

    const fallbackRatingOverview = {
        title: 'Verified Customer Reviews & Ratings',
        description: 'Our customer reviews speak for themselves. With over 2,800 verified student testimonials and a 4.9/5 star rating, we\'re proud to be the most trusted academic writing service.',
        stats: {
            totalReviews: 2847,
            averageRating: 4.9,
            fiveStars: 2654,
            fourStars: 156,
            threeStars: 28,
            twoStars: 6,
            oneStars: 3,
        },
        features: [
            { title: '100% Verified Reviews', description: 'All reviews are from real customers who completed orders' },
            { title: 'Moderated for Quality', description: 'We ensure all feedback is constructive and helpful' },
            { title: 'Regular Updates', description: 'Fresh reviews added weekly from recent orders' },
            { title: '24/7 Customer Support', description: 'Round-the-clock assistance' },
        ]
    };

    const fallbackServiceFilters = {
        filters: [
            { value: "all", label: "All Services", count: 2847 },
            { value: "essay-writing", label: "Essay Writing Reviews", count: 1245 },
            { value: "research-papers", label: "Research Paper Reviews", count: 687 },
            { value: "dissertation-writing", label: "Dissertation Reviews", count: 423 },
            { value: "assignments", label: "Assignment Help Reviews", count: 298 },
            { value: "case-studies", label: "Case Study Reviews", count: 134 },
            { value: "literature-reviews", label: "Literature Review Reviews", count: 60 },
        ]
    };

    const fallbackCustomerReviews = {
        title: 'All Customer Reviews by Service',
        description: 'Browse verified reviews for specific academic writing services',
        reviews: [
            {
                id: 1,
                name: "Sarah Johnson",
                service: "Essay Writing",
                serviceSlug: "essay-writing",
                rating: 5,
                title: "Outstanding Essay Writing Service - Highly Recommended!",
                review: "Exceptional essay writing service! The quality exceeded my expectations. My writer delivered a perfectly structured argumentative essay with impeccable grammar and compelling arguments. The plagiarism report showed 100% originality. Customer support was responsive throughout the process. Definitely using this service again for future essays.",
                date: "2024-01-15",
                verified: true,
                helpful: 47,
                course: "English Literature",
                university: "Harvard University",
                grade: "A+",
                orderType: "Argumentative Essay",
            },
            {
                id: 2,
                name: "Michael Chen",
                service: "Research Papers",
                serviceSlug: "research-papers",
                rating: 5,
                title: "Excellent Research Paper Quality - Professional Writers",
                review: "Amazing research paper service! The writer demonstrated deep understanding of my topic and provided comprehensive analysis with credible sources. The methodology was sound and the findings were well-presented. Received my psychology research paper on time with free revisions. The quality was PhD-level work. Highly satisfied with the results.",
                date: "2024-01-12",
                verified: true,
                helpful: 52,
                course: "Psychology",
                university: "Stanford University",
                grade: "A",
                orderType: "Research Paper",
            },
            {
                id: 3,
                name: "Emma Rodriguez",
                service: "Dissertation Writing",
                serviceSlug: "dissertation-writing",
                rating: 5,
                title: "Professional Dissertation Help - Exceeded Expectations",
                review: "Outstanding dissertation writing service! My PhD dissertation was handled with utmost professionalism. The writer had expertise in my field and provided excellent literature review, methodology, and analysis chapters. The work was original, well-researched, and properly formatted. Customer service was excellent throughout the lengthy process. Worth every penny!",
                date: "2024-01-10",
                verified: true,
                helpful: 89,
                course: "Business Administration",
                university: "MIT",
                grade: "A+",
                orderType: "PhD Dissertation",
            },
            {
                id: 4,
                name: "David Thompson",
                service: "Assignment Help",
                serviceSlug: "assignments",
                rating: 5,
                title: "Reliable Assignment Help - Always On Time",
                review: "Fantastic assignment help service! I've used them for multiple assignments and they never disappoint. The writers understand requirements perfectly and deliver high-quality work on time. My recent marketing assignment received excellent feedback from my professor. The customer support team is available 24/7 and very helpful. Highly recommend for urgent assignments.",
                date: "2024-01-08",
                verified: true,
                helpful: 34,
                course: "Marketing",
                university: "UCLA",
                grade: "A-",
                orderType: "Marketing Assignment",
            },
            {
                id: 5,
                name: "Lisa Wang",
                service: "Case Studies",
                serviceSlug: "case-studies",
                rating: 5,
                title: "Impressive Case Study Analysis - Expert Writers",
                review: "Excellent case study writing service! The writer provided thorough analysis with practical recommendations. The case study was well-structured with clear problem identification, analysis, and solutions. Used real-world examples and industry insights. My business professor was impressed with the quality. Great value for money and professional service throughout.",
                date: "2024-01-05",
                verified: true,
                helpful: 28,
                course: "Business Strategy",
                university: "Wharton",
                grade: "A",
                orderType: "Business Case Study",
            },
            {
                id: 6,
                name: "James Wilson",
                service: "Literature Reviews",
                serviceSlug: "literature-reviews",
                rating: 5,
                title: "Comprehensive Literature Review - Scholarly Quality",
                review: "Outstanding literature review service! The writer conducted comprehensive research and provided critical analysis of current literature in my field. The review was well-organized with proper citations and identified research gaps effectively. The quality was publication-ready. Excellent communication throughout the process. Highly recommend for graduate students.",
                date: "2024-01-03",
                verified: true,
                helpful: 41,
                course: "Education",
                university: "Columbia University",
                grade: "A+",
                orderType: "Systematic Literature Review",
            },
            {
                id: 7,
                name: "Rachel Green",
                service: "Essay Writing",
                serviceSlug: "essay-writing",
                rating: 5,
                title: "Perfect Essay Writing - Exceeded My Expectations",
                review: "Amazing essay writing experience! The writer crafted a compelling narrative essay with excellent flow and engaging content. The introduction was captivating and the conclusion was powerful. Received detailed feedback and free revisions. The final essay was polished and professional. My English professor gave excellent marks. Will definitely order again!",
                date: "2024-01-01",
                verified: true,
                helpful: 36,
                course: "Creative Writing",
                university: "NYU",
                grade: "A+",
                orderType: "Narrative Essay",
            },
            {
                id: 8,
                name: "Alex Kumar",
                service: "Research Papers",
                serviceSlug: "research-papers",
                rating: 5,
                title: "Top-Quality Research Paper - Highly Professional",
                review: "Exceptional research paper service! The writer demonstrated expertise in computer science and delivered a technically sound paper with innovative solutions. The literature review was comprehensive and the methodology was rigorous. All requirements were met perfectly. The paper contributed valuable insights to my field. Excellent customer service and timely delivery.",
                date: "2023-12-28",
                verified: true,
                helpful: 43,
                course: "Computer Science",
                university: "Carnegie Mellon",
                grade: "A",
                orderType: "Technical Research Paper",
            },
            {
                id: 9,
                name: "Sophie Martinez",
                service: "Dissertation Writing",
                serviceSlug: "dissertation-writing",
                rating: 5,
                title: "Excellent Dissertation Support - Professional Service",
                review: "Outstanding dissertation writing service! The writer helped me through every chapter with expert guidance. The research was thorough, analysis was insightful, and writing was scholarly. The dissertation committee was impressed with the quality. Received continuous support and revisions until perfect. This service made my PhD journey much smoother. Highly recommended!",
                date: "2023-12-25",
                verified: true,
                helpful: 67,
                course: "Sociology",
                university: "University of Chicago",
                grade: "A+",
                orderType: "PhD Dissertation",
            },
            {
                id: 10,
                name: "Ryan O'Connor",
                service: "Assignment Help",
                serviceSlug: "assignments",
                rating: 5,
                title: "Reliable Assignment Writing - Always Delivers",
                review: "Fantastic assignment help service! I've been using them for my MBA coursework and they consistently deliver high-quality work. The writers understand business concepts well and provide practical insights. My recent strategic management assignment received top marks. Fast turnaround, excellent quality, and great customer support. My go-to service for assignments.",
                date: "2023-12-22",
                verified: true,
                helpful: 29,
                course: "Strategic Management",
                university: "Kellogg",
                grade: "A",
                orderType: "MBA Assignment",
            },
            {
                id: 11,
                name: "Maria Gonzalez",
                service: "Case Studies",
                serviceSlug: "case-studies",
                rating: 5,
                title: "Professional Case Study Writing - Excellent Analysis",
                review: "Excellent case study writing service! The writer provided comprehensive analysis with actionable recommendations. The case study was well-researched with relevant data and industry insights. My healthcare management professor praised the practical approach and professional presentation. Great attention to detail and timely delivery. Highly satisfied with the quality and service.",
                date: "2023-12-20",
                verified: true,
                helpful: 31,
                course: "Healthcare Management",
                university: "Johns Hopkins",
                grade: "A-",
                orderType: "Healthcare Case Study",
            },
            {
                id: 12,
                name: "Kevin Park",
                service: "Literature Reviews",
                serviceSlug: "literature-reviews",
                rating: 5,
                title: "Comprehensive Literature Review - Scholarly Excellence",
                review: "Outstanding literature review service! The writer conducted extensive research and provided critical synthesis of current literature. The review identified key themes, research gaps, and future directions effectively. The quality was suitable for publication. Excellent organization and proper academic formatting. My advisor was impressed with the scholarly quality. Highly recommend!",
                date: "2023-12-18",
                verified: true,
                helpful: 38,
                course: "Environmental Science",
                university: "UC Berkeley",
                grade: "A+",
                orderType: "Systematic Review",
            },
        ]
    };

    const fallbackFeaturedTestimonials = {
        title: 'Featured Student Testimonials',
        description: 'Hear directly from students who achieved academic success with our help',
        testimonials: [
            {
                name: "Jennifer Adams",
                service: "Essay Writing",
                serviceSlug: "essay-writing",
                quote: "The essay writing service transformed my academic performance. Professional writers, original content, and excellent customer support. My grades improved significantly after using their services.",
                rating: 5,
                university: "Yale University",
                course: "Political Science",
            },
            {
                name: "Thomas Lee",
                service: "Research Papers",
                serviceSlug: "research-papers",
                quote: "Exceptional research paper quality! The writer had PhD-level expertise and delivered comprehensive analysis with credible sources. The methodology was rigorous and findings were well-presented.",
                rating: 5,
                university: "Princeton University",
                course: "Economics",
            },
            {
                name: "Amanda Foster",
                service: "Dissertation Writing",
                serviceSlug: "dissertation-writing",
                quote: "My dissertation was handled with utmost professionalism. The writer provided expert guidance through every chapter. The final work exceeded my committee's expectations. Highly recommended!",
                rating: 5,
                university: "Oxford University",
                course: "Literature",
            },
        ]
    };

    const fallbackCta = {
        title: 'Join 2,800+ Satisfied Students Today',
        description: 'Experience the quality that 2,800+ students have praised. Order your academic paper today and see why we maintain a 4.9/5 star rating across all services.',
        primaryButtonText: 'Order Your Paper Now',
        primaryButtonLink: '/place-order',
        secondaryButtonText: 'View Pricing',
        secondaryButtonLink: '/pricing',
        trustBadges: [
            '4.9/5 Star Rating',
            '2,800+ Happy Students',
            '100% Verified Reviews',
            'Money-Back Guarantee',
        ]
    };

    return (
        <MainLayout>
            <Head>
                <title>{seoData?.title || 'Customer Reviews & Testimonials | 2800+ Verified Student Reviews | Academic Scribe'}</title>
                <meta name="description" content={seoData?.description || 'Read 2800+ verified customer reviews and student testimonials for our essay writing, research paper, dissertation, and assignment help services. 4.9/5 star rating from satisfied students worldwide.'} />
                <meta name="keywords" content={seoData?.keywords || 'customer reviews, student testimonials, essay writing reviews, research paper reviews, dissertation reviews, assignment help reviews, academic writing testimonials, verified customer feedback, 5-star reviews, satisfied students, quality reviews, service ratings, customer satisfaction, genuine reviews, authentic testimonials'} />
                <link rel="canonical" href={seoData?.canonicalUrl || 'https://academicscribe.com/reviews'} />
                <meta property="og:title" content={seoData?.ogTitle || '2800+ Verified Customer Reviews | Academic Writing Service Reviews'} />
                <meta property="og:description" content={seoData?.ogDescription || 'Read authentic student testimonials and customer reviews. 4.9/5 star rating from satisfied customers. Verified reviews for essay writing, research papers, dissertations, and more.'} />
                <meta property="og:type" content="website" />
                <meta property="og:url" content={seoData?.ogUrl || 'https://academicscribe.com/reviews'} />
                <meta property="og:image" content={seoData?.ogImage || '/images/customer-reviews-og.jpg'} />
                <meta name="twitter:card" content="summary_large_image" />
                <meta name="twitter:title" content={seoData?.twitterTitle || '2800+ Verified Customer Reviews | Student Testimonials'} />
                <meta name="twitter:description" content={seoData?.twitterDescription || 'Read authentic student testimonials and customer reviews. 4.9/5 star rating from satisfied customers worldwide.'} />
                <meta name="twitter:image" content={seoData?.twitterImage || '/images/customer-reviews-twitter.jpg'} />
                
                {/* Structured Data */}
                {structuredData && (
                    <script type="application/ld+json">
                        {JSON.stringify(structuredData)}
                    </script>
                )}
            </Head>

            <div className="reviews-page">
                {/* 1. Hero Section with Stats */}
                <ReviewHero 
                    hero={hero || fallbackHero}
                    stats={hero?.stats || fallbackHero.stats}
                />

                {/* 2. Rating Overview */}
                <RatingOverview 
                    title={ratingOverview?.title || fallbackRatingOverview.title}
                    description={ratingOverview?.description || fallbackRatingOverview.description}
                    stats={ratingOverview?.stats || fallbackRatingOverview.stats}
                    features={ratingOverview?.features || fallbackRatingOverview.features}
                />

                {/* 3. Featured Testimonials */}
                <FeaturedTestimonials 
                    title={featured?.title || fallbackFeaturedTestimonials.title}
                    description={featured?.description || fallbackFeaturedTestimonials.description}
                    testimonials={featured?.testimonials || fallbackFeaturedTestimonials.testimonials}
                />

                {/* 4. Filters and All Reviews (Interactive Section) */}
                <ReviewsGrid 
                    title={customerReviewsData?.title || fallbackCustomerReviews.title}
                    description={customerReviewsData?.description || fallbackCustomerReviews.description}
                    reviews={customerReviewsData?.reviews || fallbackCustomerReviews.reviews}
                    serviceFilters={serviceFiltersData?.filters || fallbackServiceFilters.filters}
                />

                {/* 5. Reviews by Service */}
                <ReviewsByService 
                    title={reviewsByService?.title}
                    description={reviewsByService?.description}
                    services={reviewsByService?.services || []}
                />

                {/* 6. Trust Indicators */}
                <ReviewTrustIndicators 
                    title={trustIndicators?.title}
                    description={trustIndicators?.description}
                    indicators={trustIndicators?.indicators || []}
                />

                {/* 7. Final CTA */}
                <section className="py-16 bg-gradient-to-r from-green-600 to-blue-600 text-white">
                    <div className="max-w-6xl mx-auto px-4 sm:px-6 lg:px-8 text-center">
                        <h2 className="text-3xl md:text-4xl font-bold mb-4">
                            {cta?.title || fallbackCta.title}
                        </h2>
                        <p className="text-xl text-green-100 mb-8 max-w-2xl mx-auto">
                            {cta?.description || fallbackCta.description}
                        </p>

                        <div className="flex flex-col sm:flex-row gap-4 justify-center mb-8">
                            <a
                                href={cta?.primaryButtonLink || fallbackCta.primaryButtonLink}
                                className="bg-white text-green-600 hover:bg-gray-100 px-8 py-4 rounded-lg font-bold text-lg transition-colors"
                            >
                                {cta?.primaryButtonText || fallbackCta.primaryButtonText}
                            </a>
                            <a
                                href={cta?.secondaryButtonLink || fallbackCta.secondaryButtonLink}
                                className="border-2 border-white text-white hover:bg-white hover:text-green-600 px-8 py-4 rounded-lg font-bold text-lg transition-colors"
                            >
                                {cta?.secondaryButtonText || fallbackCta.secondaryButtonText}
                            </a>
                        </div>

                        {/* Trust Indicators */}
                        <div className="flex flex-wrap justify-center items-center gap-8 text-green-100">
                            {(cta?.trustBadges || fallbackCta.trustBadges).map((badge, index) => (
                                <div key={index} className="flex items-center">
                                    <svg
                                        className="w-5 h-5 mr-2"
                                        fill="currentColor"
                                        viewBox="0 0 20 20"
                                    >
                                        <path
                                            fillRule="evenodd"
                                            d="M16.707 5.293a1 1 0 010 1.414l-8 8a1 1 0 01-1.414 0l-4-4a1 1 0 011.414-1.414L8 12.586l7.293-7.293a1 1 0 011.414 0z"
                                            clipRule="evenodd"
                                        />
                                    </svg>
                                    <span className="text-sm">{badge}</span>
                                </div>
                            ))}
                        </div>
                    </div>
                </section>

                {/* 8. FAQs Section */}
                <ServiceFAQs 
                    title={faqs?.title}
                    description={faqs?.description}
                    faqs={faqs?.faqs || []}
                />

                {/* 9. Service Links Footer */}
                <ReviewServiceLinks 
                    title={serviceLinks?.title}
                    description={serviceLinks?.description}
                    services={serviceLinks?.services || []}
                />
            </div>
        </MainLayout>
    );
}
