import { useEffect, useMemo, useState } from "react";
import { Head, usePage } from "@inertiajs/react";
import MainLayout from "@/Layouts/MainLayout";
import ServiceHero from "@/Components/Services/ServiceHero";
import ServiceOverview from "@/Components/Services/ServiceOverview";
import ServiceProcess from "@/Components/Services/ServiceProcess";
import ServiceBenefits from "@/Components/Services/ServiceBenefits";
import ServiceTestimonials from "@/Components/Services/ServiceTestimonials";
import ServiceFAQs from "@/Components/Services/ServiceFAQs";
import ServiceCTA from "@/Components/Services/ServiceCTA";
import ScrollableContentSection from "@/Components/Services/ScrollableContentSection";
import { preloadServiceImage } from "@/utils/preloadImages";
import { buildServicePageState, normalizeRichText } from "@/Components/Services/servicePageUtils";
import PriceCalculatorSection from "@/Components/CMS/PriceCalculatorSection";
import { PriceCalculatorPopup, PriceCalculatorTrigger } from "@/Components/PriceCalculator";

const renderHtmlContent = (value, className = "") => {
    if (!value) return null;

    return (
        <div
            className={`prose prose-sm mx-auto text-gray-600 dark:text-slate-300 content-area ${className}`.trim()}
            dangerouslySetInnerHTML={{ __html: value }}
        />
    );
};

const safeStringify = (input) => {
    try {
        return JSON.stringify(input, (key, value) => {
            if (typeof value === "symbol") {
                return value.toString();
            }
            return value;
        });
    } catch (error) {
        console.error("Failed to stringify structured data payload", error);
        return "{}";
    }
};

const buildMeta = (props) => {
    const fallbackUrl = `/services/${props.page?.slug ?? ""}`;
    const seo = props.seoData ?? {};
    const structuredData = seo.structured_data ?? {};

    return {
        title: seo.meta_title ?? props.page?.title ?? "Academic Service",
        description: seo.meta_description ?? "",
        robots: seo.robots ?? "index, follow",
        language: seo.language ?? "en-US",
        canonicalUrl: seo.canonicalUrl ?? fallbackUrl,
        keywords: seo.meta_keywords ?? "",
        openGraph: {
            // Prioritize database values from openGraph prop (from SEOService), then fallback to seo fields
            title: props.openGraph?.title ?? seo.og_title ?? seo.meta_title ?? props.page?.title ?? "",
            description: props.openGraph?.description ?? seo.og_description ?? seo.meta_description ?? "",
            type: props.openGraph?.type ?? seo.og_type ?? "website",
            url: props.openGraph?.url ?? seo.og_url ?? seo.canonicalUrl ?? fallbackUrl,
            siteName: props.openGraph?.site_name ?? seo.og_site_name ?? "Academic Scribe",
            image: props.openGraph?.image ?? seo.og_image ?? null,
            imageAlt: props.openGraph?.image_alt ?? seo.og_image_alt ?? null,
            width: props.openGraph?.image_width ?? seo.og_image_width ?? null,
            height: props.openGraph?.image_height ?? seo.og_image_height ?? null,
            locale: props.openGraph?.locale ?? seo.og_locale ?? "en_US",
        },
        twitter: {
            // Prioritize database values from twitterCard prop (from SEOService), then fallback to seo fields
            card: props.twitterCard?.card ?? seo.twitter_card ?? "summary_large_image",
            site: props.twitterCard?.site ?? seo.twitter_site ?? null,
            creator: props.twitterCard?.creator ?? seo.twitter_creator ?? null,
            title: props.twitterCard?.title ?? seo.twitter_title ?? seo.og_title ?? seo.meta_title ?? null,
            description: props.twitterCard?.description ?? seo.twitter_description ?? seo.og_description ?? seo.meta_description ?? null,
            image: props.twitterCard?.image ?? seo.twitter_image ?? seo.og_image ?? null,
        },
        structuredData,
    };
};

const Breadcrumbs = ({ pageTitle }) => (
    <nav className="bg-gray-50 py-3" aria-label="Breadcrumb">
        <div className="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8">
            <ol className="flex items-center space-x-2 text-sm">
                <li>
                    <a
                        href="/"
                        className="text-gray-500 hover:text-blue-600 transition-colors"
                    >
                        Home
                    </a>
                </li>
                <li className="text-gray-400">/</li>
                <li>
                    <a
                        href="/services"
                        className="text-gray-500 hover:text-blue-600 transition-colors"
                    >
                        Services
                    </a>
                </li>
                <li className="text-gray-400">/</li>
                <li
                    className="text-gray-900 font-medium dark:text-white"
                    aria-current="page"
                >
                    {pageTitle || "Service"}
                </li>
            </ol>
        </div>
    </nav>
);

const LinkGridSection = ({ block, background = "bg-gray-50" }) => {
    if (!block || !block.links || block.links.length === 0) {
        return null;
    }

    return (
        <section className={`${background} py-12 dark:bg-slate-900/70`}>
            <div className="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8">
                <div className="text-center mb-10">
                    <h2 className="text-2xl font-bold text-gray-900 dark:text-white mb-4">
                        {block.title || "Explore More"}
                    </h2>
                    {renderHtmlContent(block.description)}
                </div>
                <div className="grid md:grid-cols-4 gap-6">
                    {block.links.map((link, index) => (
                        <div
                            key={`${link.title}-${index}`}
                            className="bg-white dark:bg-slate-900/60 border border-slate-100 dark:border-slate-700 rounded-xl shadow-sm hover:shadow-lg transition-shadow p-6 flex flex-col h-full"
                        >
                            <h3 className="text-lg font-semibold text-gray-900 dark:text-white mb-3">
                                <a
                                    href={link.url}
                                    className="hover:text-blue-600 dark:hover:text-blue-400"
                                >
                                    {link.title}
                                </a>
                            </h3>
                            {renderHtmlContent(link.description, "text-sm text-gray-600 dark:text-slate-300 flex-1 mb-4")}
                            <a
                                href={link.url}
                                className="text-blue-600 dark:text-blue-400 hover:underline font-medium text-sm"
                            >
                                Learn More →
                            </a>
                        </div>
                    ))}
                </div>
            </div>
        </section>
    );
};

const ResourceLinksSection = ({ block }) => {
    if (!block || !block.links || block.links.length === 0) {
        return null;
    }

    return (
        <section className="py-8 bg-gray-900 dark:bg-gray-950 text-white">
            <div className="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8">
                <div className="text-center">
                    <h3 className="text-lg font-semibold mb-4">
                        {block.title || "Resources"}
                    </h3>
                    {block.description && (
                        <div className="text-sm text-gray-400 mb-4 content-area">
                            <div
                                dangerouslySetInnerHTML={{ __html: block.description }}
                            />
                        </div>
                    )}
                    <div className="flex flex-wrap items-center justify-center gap-4 text-sm">
                        {block.links.map((link, index) => (
                            <span key={`${link.title}-${index}`} className="flex items-center gap-4">
                                <a
                                    href={link.url}
                                    className="hover:text-blue-400 transition-colors"
                                >
                                    {link.title}
                                </a>
                                {index < block.links.length - 1 && (
                                    <span className="text-gray-600">|</span>
                                )}
                            </span>
                        ))}
                    </div>
                </div>
            </div>
        </section>
    );
};

export default function DynamicService(props) {
    const { isCMS, contentBlocks, page } = props;
    const { auth } = usePage().props;

    if (!isCMS || !contentBlocks) {
        return (
            <MainLayout>
                <div className="min-h-screen flex items-center justify-center">
                    <div className="text-center">
                        <h1 className="text-2xl font-bold text-gray-900 mb-4">
                            Page Not Available
                        </h1>
                        <p className="text-gray-600">
                            This service page is being prepared. Please try again later.
                        </p>
                    </div>
                </div>
            </MainLayout>
        );
    }

    const meta = useMemo(() => buildMeta(props), [props]);
    
    // Sort blocks by order field to render dynamically
    const sortedBlocks = useMemo(() => {
        return [...(contentBlocks || [])].sort((a, b) => (a.order ?? 0) - (b.order ?? 0));
    }, [contentBlocks]);

    const standardized = useMemo(
        () => buildServicePageState(contentBlocks),
        [contentBlocks]
    );

    useEffect(() => {
        if (page?.slug) {
            preloadServiceImage(page.slug);
        }
    }, [page?.slug]);

    const scrollableDescription =
        standardized.scrollable.description ??
        normalizeRichText(
            "Scroll through extended guarantees, pricing frameworks, writer standards, and quality policies tailored for this service."
        );

    // Component mapping for dynamic rendering
    const renderBlock = (block) => {
        const blockType = block.type || block.block_type;
        
        switch (blockType) {
            case 'service_hero':
            case 'hero':
                return standardized.hero && <ServiceHero key={block.id} {...standardized.hero} />;
                
            case 'service_overview':
            case 'overview':
                return standardized.overview && (
                    <div key={block.id} className="service-content">
                        <ServiceOverview {...standardized.overview} />
                    </div>
                );
                
            case 'service_process':
            case 'process':
                return standardized.process && (
                    <div key={block.id} className="service-content">
                        <ServiceProcess {...standardized.process} />
                    </div>
                );
                
            case 'service_benefits':
            case 'benefits':
                return standardized.benefits && (
                    <div key={block.id} className="service-content">
                        <ServiceBenefits {...standardized.benefits} />
                    </div>
                );
                
            case 'service_testimonials':
            case 'testimonials':
                return standardized.testimonials && (
                    <div key={block.id} className="service-content">
                        <ServiceTestimonials {...standardized.testimonials} />
                    </div>
                );
                
            case 'service_faqs':
            case 'faqs':
                return standardized.faqs && (
                    <div key={block.id} className="service-content">
                        <ServiceFAQs {...standardized.faqs} />
                    </div>
                );
                
            case 'service_cta':
            case 'cta':
                return standardized.cta && <ServiceCTA key={block.id} {...standardized.cta} />;
                
            case 'price_calculator':
                return standardized.priceCalculator && (
                    <PriceCalculatorSection key={block.id} content={standardized.priceCalculator} />
                );
                
            case 'service_links':
            case 'internal_links':
                // Determine which link section based on key or category
                const linkCategory = block.content?.linkCategory || block.key || '';
                if (linkCategory.includes('related')) {
                    return <LinkGridSection
                        key={block.id}
                        block={standardized.links.relatedServices}
                        background="bg-gray-50 dark:bg-slate-900/70 transition-colors"
                    />;
                } else if (linkCategory.includes('support')) {
                    return <LinkGridSection
                        key={block.id}
                        block={standardized.links.supportPages}
                        background="bg-blue-50 dark:bg-slate-900/60 transition-colors"
                    />;
                } else if (linkCategory.includes('resource')) {
                    return <ResourceLinksSection key={block.id} block={standardized.links.resourceLinks} />;
                }
                return null;
                
            case 'service_scrollable':
                // Scrollable sections are grouped, only render if this is the first one
                const scrollableIndex = sortedBlocks.findIndex(b => 
                    (b.type || b.block_type) === 'service_scrollable'
                );
                if (block.id === sortedBlocks[scrollableIndex]?.id && standardized.scrollable.sections.length > 0) {
                    return (
                        <ScrollableContentSection
                            key={block.id}
                            eyebrow={standardized.scrollable.eyebrow}
                            title={standardized.scrollable.title}
                            description={scrollableDescription}
                            sections={standardized.scrollable.sections}
                            height={standardized.scrollable.preferredHeight}
                        />
                    );
                }
                return null;
                
            default:
                return null;
        }
    };

    return (
        <MainLayout>
            <Head>
                <title>{String(meta.title || "")}</title>
                {meta.description && typeof meta.description === 'string' && (
                    <meta name="description" content={meta.description} />
                )}
                {meta.keywords && typeof meta.keywords === 'string' && (
                    <meta name="keywords" content={meta.keywords} />
                )}
                {meta.robots && typeof meta.robots === 'string' && (
                    <meta name="robots" content={meta.robots} />
                )}
                {meta.language && typeof meta.language === 'string' && (
                    <meta name="language" content={meta.language} />
                )}
                {meta.canonicalUrl && typeof meta.canonicalUrl === 'string' && (
                    <link rel="canonical" href={meta.canonicalUrl} />
                )}

                {/* Open Graph */}
                {meta.openGraph?.title && typeof meta.openGraph.title === 'string' && (
                    <meta property="og:title" content={meta.openGraph.title} />
                )}
                {meta.openGraph?.description && typeof meta.openGraph.description === 'string' && (
                    <meta property="og:description" content={meta.openGraph.description} />
                )}
                {meta.openGraph?.type && typeof meta.openGraph.type === 'string' && (
                    <meta property="og:type" content={meta.openGraph.type} />
                )}
                {meta.openGraph?.url && typeof meta.openGraph.url === 'string' && (
                    <meta property="og:url" content={meta.openGraph.url} />
                )}
                {meta.openGraph?.siteName && typeof meta.openGraph.siteName === 'string' && (
                    <meta property="og:site_name" content={meta.openGraph.siteName} />
                )}
                {meta.openGraph?.image && typeof meta.openGraph.image === 'string' && (
                    <meta property="og:image" content={meta.openGraph.image} />
                )}
                {meta.openGraph?.width && typeof meta.openGraph.width === 'string' && (
                    <meta property="og:image:width" content={meta.openGraph.width} />
                )}
                {meta.openGraph?.height && typeof meta.openGraph.height === 'string' && (
                    <meta property="og:image:height" content={meta.openGraph.height} />
                )}
                {meta.openGraph?.imageAlt && typeof meta.openGraph.imageAlt === 'string' && (
                    <meta property="og:image:alt" content={meta.openGraph.imageAlt} />
                )}
                {meta.openGraph?.locale && typeof meta.openGraph.locale === 'string' && (
                    <meta property="og:locale" content={meta.openGraph.locale} />
                )}

                {/* Twitter */}
                {meta.twitter?.card && typeof meta.twitter.card === 'string' && (
                    <meta name="twitter:card" content={meta.twitter.card} />
                )}
                {meta.twitter?.site && typeof meta.twitter.site === 'string' && (
                    <meta name="twitter:site" content={meta.twitter.site} />
                )}
                {meta.twitter?.creator && typeof meta.twitter.creator === 'string' && (
                    <meta name="twitter:creator" content={meta.twitter.creator} />
                )}
                {meta.twitter?.title && typeof meta.twitter.title === 'string' && (
                    <meta name="twitter:title" content={meta.twitter.title} />
                )}
                {meta.twitter?.description && typeof meta.twitter.description === 'string' && (
                    <meta name="twitter:description" content={meta.twitter.description} />
                )}
                {meta.twitter?.image && typeof meta.twitter.image === 'string' && (
                    <meta name="twitter:image" content={meta.twitter.image} />
                )}

                {/* Structured data */}
                {meta.structuredData?.service && (
                    <script type="application/ld+json">
                        {safeStringify(meta.structuredData.service)}
                    </script>
                )}
                {meta.structuredData?.organization && (
                    <script type="application/ld+json">
                        {safeStringify(meta.structuredData.organization)}
                    </script>
                )}
                {meta.structuredData?.breadcrumb && (
                    <script type="application/ld+json">
                        {safeStringify(meta.structuredData.breadcrumb)}
                    </script>
                )}
                {meta.structuredData?.faq && (
                    <script type="application/ld+json">
                        {safeStringify(meta.structuredData.faq)}
                    </script>
                )}
                {meta.structuredData?.website && (
                    <script type="application/ld+json">
                        {safeStringify(meta.structuredData.website)}
                    </script>
                )}
                {meta.structuredData?.webpage && (
                    <script type="application/ld+json">
                        {safeStringify(meta.structuredData.webpage)}
                    </script>
                )}
            </Head>

            <Breadcrumbs pageTitle={page?.title} />

            {/* Render blocks dynamically in the order specified in CMS */}
            {sortedBlocks.map((block) => renderBlock(block))}

        </MainLayout>
    );
}

