import { useState } from "react";
import { useForm } from "@inertiajs/react";
import PrimaryButton from "@/Components/PrimaryButton";
import { Bell, Mail, MessageSquare } from "lucide-react";

export default function NotificationSettings({ notificationPreferences, onSuccess }) {
    const { data, setData, post, processing, recentlySuccessful } = useForm({
        // Order notifications
        order_status_updates: notificationPreferences?.order_status_updates ?? true,
        order_status_updates_email: notificationPreferences?.order_status_updates_email ?? true,
        order_status_updates_in_app: notificationPreferences?.order_status_updates_in_app ?? true,
        
        writer_assignment: notificationPreferences?.writer_assignment ?? true,
        writer_assignment_email: notificationPreferences?.writer_assignment_email ?? true,
        writer_assignment_in_app: notificationPreferences?.writer_assignment_in_app ?? true,
        
        new_bids: notificationPreferences?.new_bids ?? true,
        new_bids_email: notificationPreferences?.new_bids_email ?? true,
        new_bids_in_app: notificationPreferences?.new_bids_in_app ?? true,
        
        deadline_reminders: notificationPreferences?.deadline_reminders ?? true,
        deadline_reminders_email: notificationPreferences?.deadline_reminders_email ?? true,
        deadline_reminders_in_app: notificationPreferences?.deadline_reminders_in_app ?? true,
        
        order_completion: notificationPreferences?.order_completion ?? true,
        order_completion_email: notificationPreferences?.order_completion_email ?? true,
        order_completion_in_app: notificationPreferences?.order_completion_in_app ?? true,
        
        revision_requests: notificationPreferences?.revision_requests ?? true,
        revision_requests_email: notificationPreferences?.revision_requests_email ?? true,
        revision_requests_in_app: notificationPreferences?.revision_requests_in_app ?? true,
        
        // Payment notifications
        payment_confirmations: notificationPreferences?.payment_confirmations ?? true,
        payment_confirmations_email: notificationPreferences?.payment_confirmations_email ?? true,
        payment_confirmations_in_app: notificationPreferences?.payment_confirmations_in_app ?? true,
        
        payment_failures: notificationPreferences?.payment_failures ?? true,
        payment_failures_email: notificationPreferences?.payment_failures_email ?? true,
        payment_failures_in_app: notificationPreferences?.payment_failures_in_app ?? true,
        
        refunds: notificationPreferences?.refunds ?? true,
        refunds_email: notificationPreferences?.refunds_email ?? true,
        refunds_in_app: notificationPreferences?.refunds_in_app ?? true,
        
        invoice_ready: notificationPreferences?.invoice_ready ?? true,
        invoice_ready_email: notificationPreferences?.invoice_ready_email ?? true,
        invoice_ready_in_app: notificationPreferences?.invoice_ready_in_app ?? true,
        
        // Communication notifications
        new_messages: notificationPreferences?.new_messages ?? true,
        new_messages_email: notificationPreferences?.new_messages_email ?? true,
        new_messages_in_app: notificationPreferences?.new_messages_in_app ?? true,
        
        writer_questions: notificationPreferences?.writer_questions ?? true,
        writer_questions_email: notificationPreferences?.writer_questions_email ?? true,
        writer_questions_in_app: notificationPreferences?.writer_questions_in_app ?? true,
        
        // System notifications
        system_announcements: notificationPreferences?.system_announcements ?? true,
        system_announcements_email: notificationPreferences?.system_announcements_email ?? true,
        system_announcements_in_app: notificationPreferences?.system_announcements_in_app ?? true,
        
        promotional_emails: notificationPreferences?.promotional_emails ?? true,
        
        coupon_alerts: notificationPreferences?.coupon_alerts ?? true,
        coupon_alerts_email: notificationPreferences?.coupon_alerts_email ?? true,
        coupon_alerts_in_app: notificationPreferences?.coupon_alerts_in_app ?? true,
    });

    const handleSubmit = (e) => {
        e.preventDefault();
        post(route("settings.notifications.update"), {
            preserveScroll: true,
            onSuccess: () => {
                onSuccess("Notification preferences updated successfully.");
            },
        });
    };

    const NotificationGroup = ({ title, icon: Icon, notifications }) => (
        <div className="mb-6">
            <div className="flex items-center gap-2 mb-4">
                {Icon && <Icon className="h-5 w-5 text-gray-500 dark:text-gray-400" />}
                <h3 className="text-lg font-semibold text-gray-900 dark:text-white">
                    {title}
                </h3>
            </div>
            <div className="space-y-3 pl-7">
                {notifications.map((notif) => (
                    <div
                        key={notif.key}
                        className="flex items-center justify-between p-3 bg-gray-50 dark:bg-gray-700/50 rounded-lg"
                    >
                        <div>
                            <label className="text-sm font-medium text-gray-900 dark:text-white">
                                {notif.label}
                            </label>
                            {notif.description && (
                                <p className="text-xs text-gray-500 dark:text-gray-400 mt-1">
                                    {notif.description}
                                </p>
                            )}
                        </div>
                        <div className="flex items-center gap-4">
                            <label className="flex items-center gap-2">
                                <input
                                    type="checkbox"
                                    checked={data[`${notif.key}_email`] ?? false}
                                    onChange={(e) =>
                                        setData(`${notif.key}_email`, e.target.checked)
                                    }
                                    disabled={!data[notif.key]}
                                    className="rounded border-gray-300 text-indigo-600 focus:ring-indigo-500"
                                />
                                <span className="text-xs text-gray-600 dark:text-gray-400 flex items-center gap-1">
                                    <Mail className="h-3 w-3" />
                                    Email
                                </span>
                            </label>
                            <label className="flex items-center gap-2">
                                <input
                                    type="checkbox"
                                    checked={data[`${notif.key}_in_app`] ?? false}
                                    onChange={(e) =>
                                        setData(`${notif.key}_in_app`, e.target.checked)
                                    }
                                    disabled={!data[notif.key]}
                                    className="rounded border-gray-300 text-indigo-600 focus:ring-indigo-500"
                                />
                                <span className="text-xs text-gray-600 dark:text-gray-400 flex items-center gap-1">
                                    <Bell className="h-3 w-3" />
                                    In-App
                                </span>
                            </label>
                            <label className="flex items-center gap-2">
                                <input
                                    type="checkbox"
                                    checked={data[notif.key] ?? false}
                                    onChange={(e) => {
                                        setData(notif.key, e.target.checked);
                                        if (!e.target.checked) {
                                            setData(`${notif.key}_email`, false);
                                            setData(`${notif.key}_in_app`, false);
                                        }
                                    }}
                                    className="rounded border-gray-300 text-indigo-600 focus:ring-indigo-500"
                                />
                                <span className="text-xs font-medium text-gray-700 dark:text-gray-300">
                                    Enable
                                </span>
                            </label>
                        </div>
                    </div>
                ))}
            </div>
        </div>
    );

    return (
        <form onSubmit={handleSubmit}>
            <div className="space-y-6">
                <NotificationGroup
                    title="Order Notifications"
                    icon={Bell}
                    notifications={[
                        {
                            key: "order_status_updates",
                            label: "Order Status Updates",
                            description: "Get notified when your order status changes",
                        },
                        {
                            key: "writer_assignment",
                            label: "Writer Assignment",
                            description: "Notifications when a writer is assigned to your order",
                        },
                        {
                            key: "new_bids",
                            label: "New Bids",
                            description: "Alert when writers submit bids on your orders",
                        },
                        {
                            key: "deadline_reminders",
                            label: "Deadline Reminders",
                            description: "Get reminded before order deadlines",
                        },
                        {
                            key: "order_completion",
                            label: "Order Completion",
                            description: "Notification when your order is completed",
                        },
                        {
                            key: "revision_requests",
                            label: "Revision Requests",
                            description: "Alerts for revision requests from writers",
                        },
                    ]}
                />

                <NotificationGroup
                    title="Payment Notifications"
                    icon={Bell}
                    notifications={[
                        {
                            key: "payment_confirmations",
                            label: "Payment Confirmations",
                            description: "Confirmations when payments are successful",
                        },
                        {
                            key: "payment_failures",
                            label: "Payment Failures",
                            description: "Alerts when payment attempts fail",
                        },
                        {
                            key: "refunds",
                            label: "Refunds",
                            description: "Notifications about refunds processed",
                        },
                        {
                            key: "invoice_ready",
                            label: "Invoice Ready",
                            description: "Alert when invoices are available",
                        },
                    ]}
                />

                <NotificationGroup
                    title="Communication Notifications"
                    icon={MessageSquare}
                    notifications={[
                        {
                            key: "new_messages",
                            label: "New Messages",
                            description: "Notifications for new messages from writers",
                        },
                        {
                            key: "writer_questions",
                            label: "Writer Questions",
                            description: "Alerts when writers have questions about your order",
                        },
                    ]}
                />

                <div className="mb-6">
                    <h3 className="text-lg font-semibold text-gray-900 dark:text-white mb-4">
                        System Notifications
                    </h3>
                    <div className="space-y-3 pl-7">
                        <div className="flex items-center justify-between p-3 bg-gray-50 dark:bg-gray-700/50 rounded-lg">
                            <div>
                                <label className="text-sm font-medium text-gray-900 dark:text-white">
                                    System Announcements
                                </label>
                                <p className="text-xs text-gray-500 dark:text-gray-400 mt-1">
                                    Important system updates and announcements
                                </p>
                            </div>
                            <div className="flex items-center gap-4">
                                <label className="flex items-center gap-2">
                                    <input
                                        type="checkbox"
                                        checked={data.system_announcements_email ?? false}
                                        onChange={(e) =>
                                            setData("system_announcements_email", e.target.checked)
                                        }
                                        disabled={!data.system_announcements}
                                        className="rounded border-gray-300 text-indigo-600 focus:ring-indigo-500"
                                    />
                                    <span className="text-xs text-gray-600 dark:text-gray-400 flex items-center gap-1">
                                        <Mail className="h-3 w-3" />
                                        Email
                                    </span>
                                </label>
                                <label className="flex items-center gap-2">
                                    <input
                                        type="checkbox"
                                        checked={data.system_announcements_in_app ?? false}
                                        onChange={(e) =>
                                            setData("system_announcements_in_app", e.target.checked)
                                        }
                                        disabled={!data.system_announcements}
                                        className="rounded border-gray-300 text-indigo-600 focus:ring-indigo-500"
                                    />
                                    <span className="text-xs text-gray-600 dark:text-gray-400 flex items-center gap-1">
                                        <Bell className="h-3 w-3" />
                                        In-App
                                    </span>
                                </label>
                                <label className="flex items-center gap-2">
                                    <input
                                        type="checkbox"
                                        checked={data.system_announcements ?? false}
                                        onChange={(e) => {
                                            setData("system_announcements", e.target.checked);
                                            if (!e.target.checked) {
                                                setData("system_announcements_email", false);
                                                setData("system_announcements_in_app", false);
                                            }
                                        }}
                                        className="rounded border-gray-300 text-indigo-600 focus:ring-indigo-500"
                                    />
                                    <span className="text-xs font-medium text-gray-700 dark:text-gray-300">
                                        Enable
                                    </span>
                                </label>
                            </div>
                        </div>

                        <div className="flex items-center justify-between p-3 bg-gray-50 dark:bg-gray-700/50 rounded-lg">
                            <div>
                                <label className="text-sm font-medium text-gray-900 dark:text-white">
                                    Promotional Emails
                                </label>
                                <p className="text-xs text-gray-500 dark:text-gray-400 mt-1">
                                    Receive promotional emails and special offers
                                </p>
                            </div>
                            <label className="flex items-center gap-2">
                                <input
                                    type="checkbox"
                                    checked={data.promotional_emails ?? false}
                                    onChange={(e) => setData("promotional_emails", e.target.checked)}
                                    className="rounded border-gray-300 text-indigo-600 focus:ring-indigo-500"
                                />
                            </label>
                        </div>

                        <div className="flex items-center justify-between p-3 bg-gray-50 dark:bg-gray-700/50 rounded-lg">
                            <div>
                                <label className="text-sm font-medium text-gray-900 dark:text-white">
                                    Coupon Alerts
                                </label>
                                <p className="text-xs text-gray-500 dark:text-gray-400 mt-1">
                                    Notifications about new coupons and discounts
                                </p>
                            </div>
                            <div className="flex items-center gap-4">
                                <label className="flex items-center gap-2">
                                    <input
                                        type="checkbox"
                                        checked={data.coupon_alerts_email ?? false}
                                        onChange={(e) =>
                                            setData("coupon_alerts_email", e.target.checked)
                                        }
                                        disabled={!data.coupon_alerts}
                                        className="rounded border-gray-300 text-indigo-600 focus:ring-indigo-500"
                                    />
                                    <span className="text-xs text-gray-600 dark:text-gray-400 flex items-center gap-1">
                                        <Mail className="h-3 w-3" />
                                        Email
                                    </span>
                                </label>
                                <label className="flex items-center gap-2">
                                    <input
                                        type="checkbox"
                                        checked={data.coupon_alerts_in_app ?? false}
                                        onChange={(e) =>
                                            setData("coupon_alerts_in_app", e.target.checked)
                                        }
                                        disabled={!data.coupon_alerts}
                                        className="rounded border-gray-300 text-indigo-600 focus:ring-indigo-500"
                                    />
                                    <span className="text-xs text-gray-600 dark:text-gray-400 flex items-center gap-1">
                                        <Bell className="h-3 w-3" />
                                        In-App
                                    </span>
                                </label>
                                <label className="flex items-center gap-2">
                                    <input
                                        type="checkbox"
                                        checked={data.coupon_alerts ?? false}
                                        onChange={(e) => {
                                            setData("coupon_alerts", e.target.checked);
                                            if (!e.target.checked) {
                                                setData("coupon_alerts_email", false);
                                                setData("coupon_alerts_in_app", false);
                                            }
                                        }}
                                        className="rounded border-gray-300 text-indigo-600 focus:ring-indigo-500"
                                    />
                                    <span className="text-xs font-medium text-gray-700 dark:text-gray-300">
                                        Enable
                                    </span>
                                </label>
                            </div>
                        </div>
                    </div>
                </div>

                <div className="flex justify-end">
                    <PrimaryButton disabled={processing}>
                        {processing ? "Saving..." : "Save Preferences"}
                    </PrimaryButton>
                </div>
            </div>
        </form>
    );
}







