import ProfileCompletionCard from "@/Components/Writer/ProfileCompletionCard";
import WriterLayout from "@/Layouts/WriterLayout";
import { Head, Link } from "@inertiajs/react";
import UrgentOrdersWidget from "@/Components/UrgentOrdersWidget";
import {
    GraduationCap,
    ClipboardCheck,
    Clock,
    FileText,
    CheckCircle,
    XCircle,
    ArrowRight,
    Briefcase,
    BarChart,
    AlertTriangle,
    Calendar,
    FileQuestion,
} from "lucide-react";

export default function Dashboard({
    testStatus,
    questionsStatus,
    essayStatus,
    daysUntilRetry,
    retryDate,
    recentBids,
    activeOrders,
    availableOrdersCount,
    writerProfile,
    hasTestInProgress,
    profileCompletionPercentage,
    missingRequiredFields,
    isFullyQualified,
    urgentOrders,
    urgentOrdersCount,
}) {
    // Helper function to render the appropriate test status section
    const renderTestStatusSection = () => {
        // If there's a test in progress, show that first
        if (hasTestInProgress) {
            return (
                <div className="bg-yellow-50 dark:bg-yellow-900/20 border-l-4 border-yellow-400 dark:border-yellow-500 p-4 mb-6">
                    <div className="flex">
                        <div className="flex-shrink-0">
                            <ClipboardCheck className="h-5 w-5 text-yellow-400 dark:text-yellow-300" />
                        </div>
                        <div className="ml-3">
                            <h3 className="text-sm font-medium text-yellow-800 dark:text-yellow-200">
                                Test In Progress
                            </h3>
                            <div className="mt-2 text-sm text-yellow-700 dark:text-yellow-300">
                                <p>
                                    You have a multiple-choice test in progress.
                                    Please complete it to continue the
                                    qualification process.
                                </p>
                            </div>
                            <div className="mt-4">
                                <Link
                                    href={route("writer.test.take")}
                                    className="inline-flex items-center px-3 py-2 border border-transparent text-sm leading-4 font-medium rounded-md text-yellow-700 bg-yellow-100 dark:text-yellow-100 dark:bg-yellow-800/60 hover:bg-yellow-200 dark:hover:bg-yellow-800 focus:outline-none focus:ring-2 focus:ring-offset-2 focus:ring-yellow-500 dark:focus:ring-offset-gray-800 transition-colors"
                                >
                                    Continue Test
                                    <ArrowRight className="ml-2 -mr-0.5 h-4 w-4" />
                                </Link>
                            </div>
                        </div>
                    </div>
                </div>
            );
        }

        // If questions test is not passed, show questions test status
        if (questionsStatus !== "passed") {
            switch (questionsStatus) {
                case "not_started":
                    return (
                        <div className="bg-blue-50 dark:bg-blue-900/20 border-l-4 border-blue-400 dark:border-blue-500 p-4 mb-6">
                            <div className="flex">
                                <div className="flex-shrink-0">
                                    <GraduationCap className="h-5 w-5 text-blue-400 dark:text-blue-300" />
                                </div>
                                <div className="ml-3">
                                    <h3 className="text-sm font-medium text-blue-800 dark:text-blue-200">
                                        Writer Qualification Required
                                    </h3>
                                    <div className="mt-2 text-sm text-blue-700 dark:text-blue-300">
                                        <p>
                                            Before you can start bidding on
                                            orders, you need to complete our
                                            writer qualification process. This
                                            includes a multiple-choice test and
                                            an essay submission.
                                        </p>
                                    </div>
                                    <div className="mt-4">
                                        <Link
                                            href={route(
                                                "writer.test.instructions"
                                            )}
                                            className="inline-flex items-center px-3 py-2 border border-transparent text-sm leading-4 font-medium rounded-md text-blue-700 bg-blue-100 dark:text-blue-100 dark:bg-blue-800/60 hover:bg-blue-200 dark:hover:bg-blue-800 focus:outline-none focus:ring-2 focus:ring-offset-2 focus:ring-blue-500 dark:focus:ring-offset-gray-800 transition-colors"
                                        >
                                            Start Qualification Process
                                            <ArrowRight className="ml-2 -mr-0.5 h-4 w-4" />
                                        </Link>
                                    </div>
                                </div>
                            </div>
                        </div>
                    );

                case "questions_pending":
                    return (
                        <div className="bg-yellow-50 dark:bg-yellow-900/20 border-l-4 border-yellow-400 dark:border-yellow-500 p-4 mb-6">
                            <div className="flex">
                                <div className="flex-shrink-0">
                                    <ClipboardCheck className="h-5 w-5 text-yellow-400 dark:text-yellow-300" />
                                </div>
                                <div className="ml-3">
                                    <h3 className="text-sm font-medium text-yellow-800 dark:text-yellow-200">
                                        Test In Progress
                                    </h3>
                                    <div className="mt-2 text-sm text-yellow-700 dark:text-yellow-300">
                                        <p>
                                            You have a multiple-choice test in
                                            progress. Please complete it to
                                            continue the qualification process.
                                        </p>
                                    </div>
                                    <div className="mt-4">
                                        <Link
                                            href={route("writer.test.take")}
                                            className="inline-flex items-center px-3 py-2 border border-transparent text-sm leading-4 font-medium rounded-md text-yellow-700 bg-yellow-100 dark:text-yellow-100 dark:bg-yellow-800/60 hover:bg-yellow-200 dark:hover:bg-yellow-800 focus:outline-none focus:ring-2 focus:ring-offset-2 focus:ring-yellow-500 dark:focus:ring-offset-gray-800 transition-colors"
                                        >
                                            Continue Test
                                            <ArrowRight className="ml-2 -mr-0.5 h-4 w-4" />
                                        </Link>
                                    </div>
                                </div>
                            </div>
                        </div>
                    );

                case "questions_failed":
                    return (
                        <div className="bg-red-50 dark:bg-red-900/20 border-l-4 border-red-400 dark:border-red-500 p-4 mb-6">
                            <div className="flex">
                                <div className="flex-shrink-0">
                                    <XCircle className="h-5 w-5 text-red-400 dark:text-red-300" />
                                </div>
                                <div className="ml-3">
                                    <h3 className="text-sm font-medium text-red-800 dark:text-red-200">
                                        Multiple-Choice Test Not Passed
                                    </h3>
                                    <div className="mt-2 text-sm text-red-700 dark:text-red-300">
                                        <p>
                                            Unfortunately, you did not pass the
                                            multiple-choice qualification test.
                                            {daysUntilRetry > 0 ? (
                                                <span>
                                                    {" "}
                                                    You can retry on{" "}
                                                    <strong>
                                                        {retryDate}
                                                    </strong>{" "}
                                                    ({daysUntilRetry} days from
                                                    now).
                                                </span>
                                            ) : (
                                                <span>
                                                    {" "}
                                                    You can now retry the test.
                                                </span>
                                            )}
                                        </p>
                                    </div>
                                    <div className="mt-4">
                                        <Link
                                            href={route("writer.test.results")}
                                            className="inline-flex items-center px-3 py-2 border border-transparent text-sm leading-4 font-medium rounded-md text-red-700 bg-red-100 dark:text-red-100 dark:bg-red-800/60 hover:bg-red-200 dark:hover:bg-red-800 focus:outline-none focus:ring-2 focus:ring-offset-2 focus:ring-red-500 dark:focus:ring-offset-gray-800 transition-colors mr-3"
                                        >
                                            View Test Results
                                        </Link>

                                        {daysUntilRetry <= 0 && (
                                            <Link
                                                href={route(
                                                    "writer.test.instructions"
                                                )}
                                                className="inline-flex items-center px-3 py-2 border border-transparent text-sm leading-4 font-medium rounded-md text-blue-700 bg-blue-100 dark:text-blue-100 dark:bg-blue-800/60 hover:bg-blue-200 dark:hover:bg-blue-800 focus:outline-none focus:ring-2 focus:ring-offset-2 focus:ring-blue-500 dark:focus:ring-offset-gray-800 transition-colors"
                                            >
                                                Retry Test
                                                <ArrowRight className="ml-2 -mr-0.5 h-4 w-4" />
                                            </Link>
                                        )}
                                    </div>
                                </div>
                            </div>
                        </div>
                    );
            }
        } else {
            // Questions test is passed, show essay test status
            switch (essayStatus) {
                case "not_started":
                    return (
                        <div className="bg-green-50 dark:bg-green-900/20 border-l-4 border-green-400 dark:border-green-500 p-4 mb-6">
                            <div className="flex">
                                <div className="flex-shrink-0">
                                    <CheckCircle className="h-5 w-5 text-green-400 dark:text-green-300" />
                                </div>
                                <div className="ml-3">
                                    <h3 className="text-sm font-medium text-green-800 dark:text-green-200">
                                        Multiple-Choice Test Passed!
                                    </h3>
                                    <div className="mt-2 text-sm text-green-700 dark:text-green-300">
                                        <p>
                                            Congratulations! You passed the
                                            multiple-choice test with a score of{" "}
                                            {writerProfile?.questions_score}%.
                                            The next step is to complete the
                                            essay test.
                                        </p>
                                    </div>
                                    <div className="mt-4">
                                        <Link
                                            href={route("writer.test.results")}
                                            className="inline-flex items-center px-3 py-2 border border-transparent text-sm leading-4 font-medium rounded-md text-green-700 bg-green-100 dark:text-green-100 dark:bg-green-800/60 hover:bg-green-200 dark:hover:bg-green-800 focus:outline-none focus:ring-2 focus:ring-offset-2 focus:ring-green-500 dark:focus:ring-offset-gray-800 transition-colors mr-3"
                                        >
                                            View Test Results
                                        </Link>

                                        <Link
                                            href={route(
                                                "writer.essay.instructions"
                                            )}
                                            className="inline-flex items-center px-3 py-2 border border-transparent text-sm leading-4 font-medium rounded-md text-blue-700 bg-blue-100 dark:text-blue-100 dark:bg-blue-800/60 hover:bg-blue-200 dark:hover:bg-blue-800 focus:outline-none focus:ring-2 focus:ring-offset-2 focus:ring-blue-500 dark:focus:ring-offset-gray-800 transition-colors"
                                        >
                                            Start Essay Test
                                            <ArrowRight className="ml-2 -mr-0.5 h-4 w-4" />
                                        </Link>
                                    </div>
                                </div>
                            </div>
                        </div>
                    );

                case "essay_pending":
                    return (
                        <div className="bg-yellow-50 dark:bg-yellow-900/20 border-l-4 border-yellow-400 dark:border-yellow-500 p-4 mb-6">
                            <div className="flex">
                                <div className="flex-shrink-0">
                                    <ClipboardCheck className="h-5 w-5 text-yellow-400 dark:text-yellow-300" />
                                </div>

                                <div className="ml-3">
                                    <h3 className="text-sm font-medium text-yellow-800 dark:text-yellow-200">
                                        Essay Test In Progress
                                    </h3>
                                    <div className="mt-2 text-sm text-yellow-700 dark:text-yellow-300">
                                        <p>
                                            You have an essay test in progress.
                                            Please complete it to finish the
                                            qualification process.
                                        </p>
                                    </div>
                                    <div className="mt-4">
                                        <Link
                                            href={route("writer.essay.write")}
                                            className="inline-flex items-center px-3 py-2 border border-transparent text-sm leading-4 font-medium rounded-md text-yellow-700 bg-yellow-100 dark:text-yellow-100 dark:bg-yellow-800/60 hover:bg-yellow-200 dark:hover:bg-yellow-800 focus:outline-none focus:ring-2 focus:ring-offset-2 focus:ring-yellow-500 dark:focus:ring-offset-gray-800 transition-colors"
                                        >
                                            Continue Essay
                                            <ArrowRight className="ml-2 -mr-0.5 h-4 w-4" />
                                        </Link>
                                    </div>
                                </div>
                            </div>
                        </div>
                    );

                case "essay_submitted":
                    return (
                        <div className="bg-yellow-50 dark:bg-yellow-900/20 border-l-4 border-yellow-400 dark:border-yellow-500 p-4 mb-6">
                            <div className="flex">
                                <div className="flex-shrink-0">
                                    <Clock className="h-5 w-5 text-yellow-400 dark:text-yellow-300" />
                                </div>
                                <div className="ml-3">
                                    <h3 className="text-sm font-medium text-yellow-800 dark:text-yellow-200">
                                        Essay Under Review
                                    </h3>
                                    <div className="mt-2 text-sm text-yellow-700 dark:text-yellow-300">
                                        <p>
                                            Your essay has been submitted and is
                                            currently under review. We'll notify
                                            you once the review is complete.
                                        </p>
                                        <p className="mt-1">
                                            Submitted on:{" "}
                                            {writerProfile?.essay_completed_at}
                                        </p>
                                    </div>
                                    <div className="mt-4">
                                        <Link
                                            href={route(
                                                "writer.essay.confirmation"
                                            )}
                                            className="inline-flex items-center px-3 py-2 border border-transparent text-sm leading-4 font-medium rounded-md text-yellow-700 bg-yellow-100 dark:text-yellow-100 dark:bg-yellow-800/60 hover:bg-yellow-200 dark:hover:bg-yellow-800 focus:outline-none focus:ring-2 focus:ring-offset-2 focus:ring-yellow-500 dark:focus:ring-offset-gray-800 transition-colors"
                                        >
                                            View Submission Details
                                        </Link>
                                    </div>
                                </div>
                            </div>
                        </div>
                    );

                case "essay_expired":
                    console.log(
                        "Essay expired - daysUntilRetry:",
                        daysUntilRetry
                    );
                    console.log("Essay expired - retryDate:", retryDate);

                    // Default values if daysUntilRetry or retryDate are null
                    const effectiveDaysUntilRetry =
                        daysUntilRetry !== null ? daysUntilRetry : 90;
                    const effectiveRetryDate =
                        retryDate || "3 months from expiration";

                    return (
                        <div className="bg-amber-50 dark:bg-amber-900/20 border-l-4 border-amber-400 dark:border-amber-500 p-4 mb-6">
                            <div className="flex">
                                <div className="flex-shrink-0">
                                    <AlertTriangle className="h-5 w-5 text-amber-400 dark:text-amber-300" />
                                </div>
                                <div className="ml-3">
                                    <h3 className="text-sm font-medium text-amber-800 dark:text-amber-200">
                                        Essay Test Expired
                                    </h3>
                                    <div className="mt-2 text-sm text-amber-700 dark:text-amber-300">
                                        {writerProfile?.has_partial_submission ? (
                                            <p>
                                                Your essay test time expired,
                                                but we received your partial
                                                submission. It has been sent for
                                                review. We'll notify you once
                                                the review is complete.
                                            </p>
                                        ) : (
                                            <p>
                                                Your essay test time expired
                                                without a submission.
                                                {/* Always show the retry date message */}
                                                <span>
                                                    {" "}
                                                    You can retry on{" "}
                                                    <strong>
                                                        {effectiveRetryDate}
                                                    </strong>{" "}
                                                    ({effectiveDaysUntilRetry}{" "}
                                                    days from now).
                                                </span>
                                            </p>
                                        )}
                                        <p className="mt-2">
                                            <span className="inline-flex items-center text-amber-600 dark:text-amber-400">
                                                <Calendar className="h-4 w-4 mr-1" />
                                                Expired on:{" "}
                                                {
                                                    writerProfile?.essay_completed_at
                                                }
                                            </span>
                                        </p>
                                    </div>
                                    <div className="mt-4">
                                        {writerProfile?.has_partial_submission ? (
                                            <Link
                                                href={route(
                                                    "writer.essay.confirmation"
                                                )}
                                                className="inline-flex items-center px-3 py-2 border border-transparent text-sm leading-4 font-medium rounded-md text-amber-700 bg-amber-100 dark:text-amber-100 dark:bg-amber-800/60 hover:bg-amber-200 dark:hover:bg-amber-800 focus:outline-none focus:ring-2 focus:ring-offset-2 focus:ring-amber-500 dark:focus:ring-offset-gray-800 transition-colors"
                                            >
                                                View Submission Details
                                            </Link>
                                        ) : // Only show retry button if days until retry is actually 0 or less
                                        // and is not null or undefined
                                        daysUntilRetry !== null &&
                                          daysUntilRetry <= 0 ? (
                                            <Link
                                                href={route(
                                                    "writer.essay.instructions"
                                                )}
                                                className="inline-flex items-center px-3 py-2 border border-transparent text-sm leading-4 font-medium rounded-md text-blue-700 bg-blue-100 dark:text-blue-100 dark:bg-blue-800/60 hover:bg-blue-200 dark:hover:bg-blue-800 focus:outline-none focus:ring-2 focus:ring-offset-2 focus:ring-blue-500 dark:focus:ring-offset-gray-800 transition-colors"
                                            >
                                                Retry Essay Test
                                                <ArrowRight className="ml-2 -mr-0.5 h-4 w-4" />
                                            </Link>
                                        ) : (
                                            <div className="inline-flex items-center px-3 py-2 text-sm text-amber-600 dark:text-amber-400">
                                                <Clock className="h-4 w-4 mr-1" />
                                                Waiting period active -{" "}
                                                {effectiveDaysUntilRetry} days
                                                remaining
                                            </div>
                                        )}
                                    </div>
                                </div>
                            </div>
                        </div>
                    );

                case "essay_failed":
                    return (
                        <div className="bg-red-50 dark:bg-red-900/20 border-l-4 border-red-400 dark:border-red-500 p-4 mb-6">
                            <div className="flex">
                                <div className="flex-shrink-0">
                                    <XCircle className="h-5 w-5 text-red-400 dark:text-red-300" />
                                </div>
                                <div className="ml-3">
                                    <h3 className="text-sm font-medium text-red-800 dark:text-red-200">
                                        Essay Test Not Passed
                                    </h3>
                                    <div className="mt-2 text-sm text-red-700 dark:text-red-300">
                                        <p>
                                            Unfortunately, your essay submission
                                            did not meet our requirements.
                                            {daysUntilRetry > 0 ? (
                                                <span>
                                                    {" "}
                                                    You can retry on{" "}
                                                    <strong>
                                                        {retryDate}
                                                    </strong>{" "}
                                                    ({daysUntilRetry} days from
                                                    now).
                                                </span>
                                            ) : (
                                                <span>
                                                    {" "}
                                                    You can now retry the essay
                                                    test.
                                                </span>
                                            )}
                                        </p>
                                        {writerProfile?.essay_feedback && (
                                            <div className="mt-2 p-2 bg-red-100 dark:bg-red-900/40 rounded">
                                                <p className="font-medium">
                                                    Feedback:
                                                </p>
                                                <p>
                                                    {
                                                        writerProfile.essay_feedback
                                                    }
                                                </p>
                                            </div>
                                        )}
                                    </div>
                                    <div className="mt-4">
                                        <Link
                                            href={route(
                                                "writer.essay.confirmation"
                                            )}
                                            className="inline-flex items-center px-3 py-2 border border-transparent text-sm leading-4 font-medium rounded-md text-red-700 bg-red-100 dark:text-red-100 dark:bg-red-800/60 hover:bg-red-200 dark:hover:bg-red-800 focus:outline-none focus:ring-2 focus:ring-offset-2 focus:ring-red-500 dark:focus:ring-offset-gray-800 transition-colors mr-3"
                                        >
                                            View Submission Details
                                        </Link>

                                        {daysUntilRetry <= 0 && (
                                            <Link
                                                href={route(
                                                    "writer.essay.instructions"
                                                )}
                                                className="inline-flex items-center px-3 py-2 border border-transparent text-sm leading-4 font-medium rounded-md text-blue-700 bg-blue-100 dark:text-blue-100 dark:bg-blue-800/60 hover:bg-blue-200 dark:hover:bg-blue-800 focus:outline-none focus:ring-2 focus:ring-offset-2 focus:ring-blue-500 dark:focus:ring-offset-gray-800 transition-colors"
                                            >
                                                Retry Essay Test
                                                <ArrowRight className="ml-2 -mr-0.5 h-4 w-4" />
                                            </Link>
                                        )}
                                    </div>
                                </div>
                            </div>
                        </div>
                    );

                case "passed":
                    return (
                        <div className="bg-green-50 dark:bg-green-900/20 border-l-4 border-green-400 dark:border-green-500 p-4 mb-6">
                            <div className="mb-6">
                                <ProfileCompletionCard
                                    percentage={profileCompletionPercentage}
                                    missingFields={missingRequiredFields}
                                    isQualified={isFullyQualified}
                                />
                            </div>
                            <div className="flex">
                                <div className="flex-shrink-0">
                                    <CheckCircle className="h-5 w-5 text-green-400 dark:text-green-300" />
                                </div>
                                <div className="ml-3">
                                    <h3 className="text-sm font-medium text-green-800 dark:text-green-200">
                                        Qualification Complete
                                    </h3>
                                    <div className="mt-2 text-sm text-green-700 dark:text-green-300">
                                        <p>
                                            Congratulations! You have
                                            successfully completed the writer
                                            qualification process. You can now
                                            bid on available orders and start
                                            earning.
                                        </p>
                                    </div>
                                    <div className="mt-4">
                                        <Link
                                            href={route("writer.orders")}
                                            className="inline-flex items-center px-3 py-2 border border-transparent text-sm leading-4 font-medium rounded-md text-green-700 bg-green-100 dark:text-green-100 dark:bg-green-800/60 hover:bg-green-200 dark:hover:bg-green-800 focus:outline-none focus:ring-2 focus:ring-offset-2 focus:ring-green-500 dark:focus:ring-offset-gray-800 transition-colors"
                                        >
                                            Browse Available Orders
                                            <ArrowRight className="ml-2 -mr-0.5 h-4 w-4" />
                                        </Link>
                                    </div>
                                </div>
                            </div>
                        </div>
                    );

                default:
                    return null;
            }
        }
    };

    // Determine if writer is fully approved (passed both tests)
    const isFullyApproved =
        questionsStatus === "passed" && essayStatus === "passed";

    return (
        <WriterLayout>
            <Head title="Writer Dashboard" />

            <div className="py-12">
                <div className="max-w-7xl mx-auto sm:px-6 lg:px-8">
                    <h1 className="text-2xl font-bold text-gray-800 dark:text-gray-100 mb-6">
                        Writer Dashboard
                    </h1>

                    {/* Test Status Section */}
                    {renderTestStatusSection()}

                    {/* Dashboard Content */}
                    <div className="bg-white dark:bg-gray-800 overflow-hidden shadow-sm sm:rounded-lg">
                        {isFullyApproved ? (
                            <div className="p-6">
                                {/* Writer Stats for Approved Writers */}
                                <div className="grid grid-cols-1 md:grid-cols-3 gap-6 mb-6">
                                    <div className="bg-white dark:bg-gray-800 p-4 rounded-lg border border-gray-200 dark:border-gray-700 shadow-sm">
                                        <div className="flex items-center">
                                            <div className="p-3 rounded-full bg-blue-100 dark:bg-blue-900/50 text-blue-500 dark:text-blue-300">
                                                <Briefcase className="h-6 w-6" />
                                            </div>
                                            <div className="ml-4">
                                                <h3 className="text-lg font-medium text-gray-900 dark:text-gray-100">
                                                    Available Orders
                                                </h3>
                                                <p className="text-2xl font-semibold text-gray-800 dark:text-gray-200">
                                                    {availableOrdersCount}
                                                </p>
                                            </div>
                                        </div>
                                    </div>

                                    <div className="bg-white dark:bg-gray-800 p-4 rounded-lg border border-gray-200 dark:border-gray-700 shadow-sm">
                                        <div className="flex items-center">
                                            <div className="p-3 rounded-full bg-green-100 dark:bg-green-900/50 text-green-500 dark:text-green-300">
                                                <BarChart className="h-6 w-6" />
                                            </div>
                                            <div className="ml-4">
                                                <h3 className="text-lg font-medium text-gray-900 dark:text-gray-100">
                                                    Active Bids
                                                </h3>
                                                <p className="text-2xl font-semibold text-gray-800 dark:text-gray-200">
                                                    {recentBids?.filter(
                                                        (bid) =>
                                                            bid.status ===
                                                            "pending"
                                                    ).length || 0}
                                                </p>
                                            </div>
                                        </div>
                                    </div>

                                    <div className="bg-white dark:bg-gray-800 p-4 rounded-lg border border-gray-200 dark:border-gray-700 shadow-sm">
                                        <div className="flex items-center">
                                            <div className="p-3 rounded-full bg-purple-100 dark:bg-purple-900/50 text-purple-500 dark:text-purple-300">
                                                <FileText className="h-6 w-6" />
                                            </div>
                                            <div className="ml-4">
                                                <h3 className="text-lg font-medium text-gray-900 dark:text-gray-100">
                                                    Active Orders
                                                </h3>
                                                <p className="text-2xl font-semibold text-gray-800 dark:text-gray-200">
                                                    {activeOrders?.length || 0}
                                                </p>
                                            </div>
                                        </div>
                                    </div>
                                </div>

                                {/* Recent Bids */}
                                <div className="mb-6">
                                    <h2 className="text-lg font-medium text-gray-900 dark:text-gray-100 mb-4">
                                        Recent Bids
                                    </h2>
                                    {recentBids && recentBids.length > 0 ? (
                                        <div className="bg-white dark:bg-gray-800 overflow-hidden border border-gray-200 dark:border-gray-700 rounded-lg">
                                            <div className="overflow-x-auto">
                                                <table className="min-w-full divide-y divide-gray-200 dark:divide-gray-700">
                                                    <thead className="bg-gray-50 dark:bg-gray-700">
                                                        <tr>
                                                            <th
                                                                scope="col"
                                                                className="px-6 py-3 text-left text-xs font-medium text-gray-500 dark:text-gray-300 uppercase tracking-wider"
                                                            >
                                                                Order
                                                            </th>
                                                            <th
                                                                scope="col"
                                                                className="px-6 py-3 text-left text-xs font-medium text-gray-500 dark:text-gray-300 uppercase tracking-wider"
                                                            >
                                                                Amount
                                                            </th>
                                                            <th
                                                                scope="col"
                                                                className="px-6 py-3 text-left text-xs font-medium text-gray-500 dark:text-gray-300 uppercase tracking-wider"
                                                            >
                                                                Status
                                                            </th>
                                                            <th
                                                                scope="col"
                                                                className="px-6 py-3 text-left text-xs font-medium text-gray-500 dark:text-gray-300 uppercase tracking-wider"
                                                            >
                                                                Date
                                                            </th>
                                                            <th
                                                                scope="col"
                                                                className="px-6 py-3 text-left text-xs font-medium text-gray-500 dark:text-gray-300 uppercase tracking-wider"
                                                            >
                                                                Actions
                                                            </th>
                                                        </tr>
                                                    </thead>
                                                    <tbody className="bg-white dark:bg-gray-800 divide-y divide-gray-200 dark:divide-gray-700">
                                                        {recentBids.map(
                                                            (bid) => (
                                                                <tr
                                                                    key={bid.id}
                                                                >
                                                                    <td className="px-6 py-4 whitespace-nowrap text-sm font-medium text-gray-900 dark:text-gray-100">
                                                                        {bid
                                                                            .order
                                                                            ?.title ||
                                                                            "Order #" +
                                                                                bid.order_id}
                                                                    </td>
                                                                    <td className="px-6 py-4 whitespace-nowrap text-sm text-gray-500 dark:text-gray-400">
                                                                        $
                                                                        {
                                                                            bid.bid_amount
                                                                        }
                                                                    </td>
                                                                    <td className="px-6 py-4 whitespace-nowrap text-sm text-gray-500 dark:text-gray-400">
                                                                        <span
                                                                            className={`px-2 inline-flex text-xs leading-5 font-semibold rounded-full
                                                                        ${
                                                                            bid.status ===
                                                                            "accepted"
                                                                                ? "bg-green-100 text-green-800 dark:bg-green-900/50 dark:text-green-200"
                                                                                : bid.status ===
                                                                                  "rejected"
                                                                                ? "bg-red-100 text-red-800 dark:bg-red-900/50 dark:text-red-200"
                                                                                : "bg-yellow-100 text-yellow-800 dark:bg-yellow-900/50 dark:text-yellow-200"
                                                                        }`}
                                                                        >
                                                                            {bid.status
                                                                                .charAt(
                                                                                    0
                                                                                )
                                                                                .toUpperCase() +
                                                                                bid.status.slice(
                                                                                    1
                                                                                )}
                                                                        </span>
                                                                    </td>
                                                                    <td className="px-6 py-4 whitespace-nowrap text-sm text-gray-500 dark:text-gray-400">
                                                                        {new Date(
                                                                            bid.created_at
                                                                        ).toLocaleDateString()}
                                                                    </td>
                                                                    <td className="px-6 py-4 whitespace-nowrap text-sm text-gray-500 dark:text-gray-400">
                                                                        <Link
                                                                            href={route(
                                                                                "writer.orders.show",
                                                                                bid.order_id
                                                                            )}
                                                                            className="text-indigo-600 hover:text-indigo-900 dark:text-indigo-400 dark:hover:text-indigo-300"
                                                                        >
                                                                            View
                                                                            Order
                                                                        </Link>
                                                                    </td>
                                                                </tr>
                                                            )
                                                        )}
                                                    </tbody>
                                                </table>
                                            </div>
                                        </div>
                                    ) : (
                                        <div className="bg-gray-50 dark:bg-gray-700 p-4 rounded-lg text-center text-gray-500 dark:text-gray-400">
                                            You haven't placed any bids yet.
                                        </div>
                                    )}
                                </div>

                                {/* Active Orders */}
                                <div>
                                    <h2 className="text-lg font-medium text-gray-900 dark:text-gray-100 mb-4">
                                        Active Orders
                                    </h2>
                                    {activeOrders && activeOrders.length > 0 ? (
                                        <div className="bg-white dark:bg-gray-800 overflow-hidden border border-gray-200 dark:border-gray-700 rounded-lg">
                                            <div className="overflow-x-auto">
                                                <table className="min-w-full divide-y divide-gray-200 dark:divide-gray-700">
                                                    <thead className="bg-gray-50 dark:bg-gray-700">
                                                        <tr>
                                                            <th
                                                                scope="col"
                                                                className="px-6 py-3 text-left text-xs font-medium text-gray-500 dark:text-gray-300 uppercase tracking-wider"
                                                            >
                                                                Order
                                                            </th>
                                                            <th
                                                                scope="col"
                                                                className="px-6 py-3 text-left text-xs font-medium text-gray-500 dark:text-gray-300 uppercase tracking-wider"
                                                            >
                                                                Amount
                                                            </th>
                                                            <th
                                                                scope="col"
                                                                className="px-6 py-3 text-left text-xs font-medium text-gray-500 dark:text-gray-300 uppercase tracking-wider"
                                                            >
                                                                Status
                                                            </th>
                                                            <th
                                                                scope="col"
                                                                className="px-6 py-3 text-left text-xs font-medium text-gray-500 dark:text-gray-300 uppercase tracking-wider"
                                                            >
                                                                Deadline
                                                            </th>
                                                            <th
                                                                scope="col"
                                                                className="px-6 py-3 text-left text-xs font-medium text-gray-500 dark:text-gray-300 uppercase tracking-wider"
                                                            >
                                                                Actions
                                                            </th>
                                                        </tr>
                                                    </thead>
                                                    <tbody className="bg-white dark:bg-gray-800 divide-y divide-gray-200 dark:divide-gray-700">
                                                        {activeOrders.map(
                                                            (order) => (
                                                                <tr
                                                                    key={
                                                                        order.id
                                                                    }
                                                                >
                                                                    <td className="px-6 py-4 whitespace-nowrap text-sm font-medium text-gray-900 dark:text-gray-100">
                                                                        {
                                                                            order.title
                                                                        }
                                                                    </td>
                                                                    <td className="px-6 py-4 whitespace-nowrap text-sm text-gray-500 dark:text-gray-400">
                                                                        $
                                                                        {
                                                                            order.writer_amount
                                                                        }
                                                                    </td>
                                                                    <td className="px-6 py-4 whitespace-nowrap text-sm text-gray-500 dark:text-gray-400">
                                                                        <span
                                                                            className={`px-2 inline-flex text-xs leading-5 font-semibold rounded-full
                                                                        ${
                                                                            order.order_status ===
                                                                            "completed"
                                                                                ? "bg-green-100 text-green-800 dark:bg-green-900/50 dark:text-green-200"
                                                                                : order.order_status ===
                                                                                  "in_progress"
                                                                                ? "bg-blue-100 text-blue-800 dark:bg-blue-900/50 dark:text-blue-200"
                                                                                : "bg-yellow-100 text-yellow-800 dark:bg-yellow-900/50 dark:text-yellow-200"
                                                                        }`}
                                                                        >
                                                                            {order.order_status
                                                                                .split(
                                                                                    "_"
                                                                                )
                                                                                .map(
                                                                                    (
                                                                                        word
                                                                                    ) =>
                                                                                        word
                                                                                            .charAt(
                                                                                                0
                                                                                            )
                                                                                            .toUpperCase() +
                                                                                        word.slice(
                                                                                            1
                                                                                        )
                                                                                )
                                                                                .join(
                                                                                    " "
                                                                                )}
                                                                        </span>
                                                                    </td>
                                                                    <td className="px-6 py-4 whitespace-nowrap text-sm text-gray-500 dark:text-gray-400">
                                                                        {new Date(
                                                                            order.writer_deadline
                                                                        ).toLocaleDateString()}
                                                                    </td>
                                                                    <td className="px-6 py-4 whitespace-nowrap text-sm text-gray-500 dark:text-gray-400">
                                                                        <Link
                                                                            href={route(
                                                                                "writer.orders.show",
                                                                                order.id
                                                                            )}
                                                                            className="text-indigo-600 hover:text-indigo-900 dark:text-indigo-400 dark:hover:text-indigo-300"
                                                                        >
                                                                            View
                                                                            Order
                                                                        </Link>
                                                                    </td>
                                                                </tr>
                                                            )
                                                        )}
                                                    </tbody>
                                                </table>
                                            </div>
                                        </div>
                                    ) : (
                                        <div className="bg-gray-50 dark:bg-gray-700 p-4 rounded-lg text-center text-gray-500 dark:text-gray-400">
                                            You don't have any active orders.
                                        </div>
                                    )}
                                </div>
                            </div>
                        ) : (
                            <div className="p-6">
                                <div className="text-center py-8">
                                    <GraduationCap className="mx-auto h-12 w-12 text-gray-400 dark:text-gray-500" />
                                    <h3 className="mt-2 text-sm font-medium text-gray-900 dark:text-gray-100">
                                        Complete Qualification Process
                                    </h3>
                                    <p className="mt-1 text-sm text-gray-500 dark:text-gray-400">
                                        You need to complete the writer
                                        qualification process before you can
                                        access the full dashboard.
                                    </p>
                                </div>
                            </div>
                        )}

                        {/* Urgent Orders Widget */}
                        {isFullyQualified && urgentOrders && Object.values(urgentOrders).some(category => category.length > 0) && (
                            <div className="mt-6">
                                <UrgentOrdersWidget 
                                    urgentOrders={urgentOrders}
                                    urgentOrdersCount={urgentOrdersCount}
                                    userType="writer"
                                />
                            </div>
                        )}
                    </div>
                </div>
            </div>
        </WriterLayout>
    );
}
