import { useState, useEffect, useRef } from "react";
import { Head, useForm } from "@inertiajs/react";
import WriterLayout from "@/Layouts/WriterLayout";
import PrimaryButton from "@/Components/PrimaryButton";
import SecondaryButton from "@/Components/SecondaryButton";
import Modal from "@/Components/Modal";
import axios from "axios";
import {
    Clock,
    Save,
    AlertTriangle,
    Send,
    XCircle,
    CheckCircle,
    Info,
} from "lucide-react";

export default function Write({
    topic,
    timeRemaining,
    testStartedAt,
    testEndAt, // New prop for the end time from database
    draftContent,
    serverTime,
}) {
    const [essayContent, setEssayContent] = useState(draftContent || "");
    const [wordCount, setWordCount] = useState(0);
    const [remainingTime, setRemainingTime] = useState(timeRemaining);
    const [showConfirmation, setShowConfirmation] = useState(false);
    const [lastSaved, setLastSaved] = useState(null);
    const [saving, setSaving] = useState(false);
    const [saveError, setSaveError] = useState(null);
    const timerRef = useRef(null);
    const autoSaveRef = useRef(null);
    const serverSyncRef = useRef(null);
    const editorRef = useRef(null);
    const minWordCount = topic.word_count || 500;
    const lastSyncTimeRef = useRef(Date.now());
    const initialServerTimeRef = useRef(
        serverTime ? serverTime * 1000 : Date.now()
    );
    const initialClientTimeRef = useRef(Date.now());

    const { data, setData, post, processing, errors } = useForm({
        essay_content: essayContent,
        topic_id: topic.id,
    });

    // Initialize timer with server sync
    useEffect(() => {
        // Immediately sync with server on component mount
        syncWithServer();

        // Local timer that decrements every second
        timerRef.current = setInterval(() => {
            setRemainingTime((prevTime) => {
                if (prevTime <= 1) {
                    clearInterval(timerRef.current);
                    handleAutoSubmit();
                    return 0;
                }
                return prevTime - 1;
            });
        }, 1000);

        // Sync with server every 10 seconds to prevent time manipulation
        serverSyncRef.current = setInterval(syncWithServer, 10000);

        // Add event listener for when the page becomes visible again (tab focus)
        document.addEventListener("visibilitychange", handleVisibilityChange);

        // Add listener for when user tries to leave the page
        window.addEventListener("beforeunload", handleBeforeUnload);

        return () => {
            if (timerRef.current) clearInterval(timerRef.current);
            if (autoSaveRef.current) clearInterval(autoSaveRef.current);
            if (serverSyncRef.current) clearInterval(serverSyncRef.current);
            document.removeEventListener(
                "visibilitychange",
                handleVisibilityChange
            );
            window.removeEventListener("beforeunload", handleBeforeUnload);
        };
    }, []);

    // Handle visibility change (tab focus/blur)
    const handleVisibilityChange = () => {
        if (document.visibilityState === "visible") {
            // When tab becomes visible again, immediately sync with server
            syncWithServer();
        }
    };

    // Handle page unload attempt
    const handleBeforeUnload = (e) => {
        // Save draft before leaving
        handleAutoSave();

        // Show a warning
        e.preventDefault();
        e.returnValue =
            "You are in the middle of an essay test. Are you sure you want to leave?";
        return e.returnValue;
    };

    // Function to sync with server
    const syncWithServer = async () => {
        try {
            const response = await axios.get(route("writer.essay.check-time"));

            // Always trust the server's remaining time
            const serverTimeRemaining = response.data.timeRemaining;
            const serverCurrentTime = response.data.serverTime * 1000; // Convert to milliseconds
            const serverTestEndTime = response.data.testEndAt * 1000; // Convert to milliseconds

            console.log("Server time remaining:", serverTimeRemaining);
            console.log("Client time remaining:", remainingTime);
            console.log(
                "Server current time:",
                new Date(serverCurrentTime).toISOString()
            );
            console.log("Client current time:", new Date().toISOString());
            console.log(
                "Server test end time:",
                new Date(serverTestEndTime).toISOString()
            );

            // Always trust the server time
            setRemainingTime(serverTimeRemaining);

            // If time has expired according to server, submit automatically
            if (serverTimeRemaining <= 0) {
                console.log("Time expired according to server");
                clearInterval(timerRef.current);
                clearInterval(serverSyncRef.current);
                handleAutoSubmit();
            }
        } catch (error) {
            console.error("Failed to sync time with server:", error);

            // If we can't reach the server, continue with local timer
            // but force a sync as soon as connection is restored
            const retrySync = async () => {
                try {
                    await axios.get(route("writer.essay.check-time"));
                    // If successful, sync immediately
                    syncWithServer();
                } catch (error) {
                    // If still failing, try again in 5 seconds
                    setTimeout(retrySync, 5000);
                }
            };

            // Start retry process after a short delay
            setTimeout(retrySync, 5000);
        }
    };

    // Initialize auto-save
    useEffect(() => {
        autoSaveRef.current = setInterval(() => {
            if (essayContent.trim() !== "") {
                handleAutoSave();
            }
        }, 30000); // Auto-save every 30 seconds

        return () => {
            if (autoSaveRef.current) clearInterval(autoSaveRef.current);
        };
    }, [essayContent]);

    // Update word count when content changes
    useEffect(() => {
        if (essayContent) {
            // Count words by splitting on whitespace and filtering out empty strings
            const words = essayContent
                .trim()
                .split(/\s+/)
                .filter((word) => word !== "");
            setWordCount(words.length);
        } else {
            setWordCount(0);
        }

        // Update form data
        setData("essay_content", essayContent);
    }, [essayContent]);

    // Format time as MM:SS
    const formatTime = (seconds) => {
        const minutes = Math.floor(seconds / 60);
        const remainingSeconds = seconds % 60;
        return `${minutes.toString().padStart(2, "0")}:${remainingSeconds
            .toString()
            .padStart(2, "0")}`;
    };

    // Format timestamp for last saved indicator
    const formatLastSaved = (timestamp) => {
        if (!timestamp) return "";
        const date = new Date(timestamp);
        return date.toLocaleTimeString([], {
            hour: "2-digit",
            minute: "2-digit",
            second: "2-digit",
        });
    };

    // Handle auto-save
    const handleAutoSave = async () => {
        if (saving || !essayContent.trim()) return;

        setSaving(true);
        setSaveError(null);

        try {
            const response = await axios.post(
                route("writer.essay.save-draft"),
                {
                    essay_content: essayContent,
                }
            );
            setLastSaved(response.data.saved_at);
        } catch (error) {
            console.error("Auto-save failed:", error);
            setSaveError(
                "Failed to save draft. Your work may be lost if you leave this page."
            );
        } finally {
            setSaving(false);
        }
    };

    // Handle manual save
    const handleManualSave = async () => {
        await handleAutoSave();
    };

    // Handle auto-submit when time expires
    const handleAutoSubmit = () => {
        post(route("writer.essay.submit"), {
            preserveScroll: true,
            onSuccess: () => {
                if (autoSaveRef.current) clearInterval(autoSaveRef.current);
                if (timerRef.current) clearInterval(timerRef.current);
                if (serverSyncRef.current) clearInterval(serverSyncRef.current);
            },
        });
    };

    // Handle manual submit
    const handleSubmit = () => {
        if (wordCount < minWordCount) {
            // Show warning but allow submission
            setShowConfirmation(true);
            return;
        }

        submitEssay();
    };

    // Submit the essay
    const submitEssay = () => {
        setShowConfirmation(false);
        post(route("writer.essay.submit"), {
            preserveScroll: true,
            onSuccess: () => {
                if (autoSaveRef.current) clearInterval(autoSaveRef.current);
                if (timerRef.current) clearInterval(timerRef.current);
                if (serverSyncRef.current) clearInterval(serverSyncRef.current);
            },
        });
    };

    // Determine time display color based on remaining time
    const getTimeColor = () => {
        if (remainingTime < 60) return "text-red-600 dark:text-red-400"; // Less than 1 minute
        if (remainingTime < 300) return "text-orange-500 dark:text-orange-400"; // Less than 5 minutes
        return "text-gray-700 dark:text-gray-300";
    };

    // Determine word count color based on minimum requirement
    const getWordCountColor = () => {
        if (wordCount >= minWordCount)
            return "text-green-600 dark:text-green-400";
        if (wordCount >= minWordCount * 0.7)
            return "text-orange-500 dark:text-orange-400";
        return "text-red-600 dark:text-red-400";
    };

    return (
        <WriterLayout>
            <Head title="Essay Test - Writing" />

            <div className="py-6">
                <div className="max-w-7xl mx-auto sm:px-6 lg:px-8">
                    <div className="bg-white dark:bg-gray-800 overflow-hidden shadow-sm sm:rounded-lg">
                        <div className="p-4 sm:p-6 bg-white dark:bg-gray-800 border-b border-gray-200 dark:border-gray-700">
                            {/* Header with Timer and Controls */}
                            <div className="flex flex-col sm:flex-row justify-between items-start sm:items-center mb-4 space-y-2 sm:space-y-0">
                                <div>
                                    <h1 className="text-xl font-bold text-gray-800 dark:text-gray-100">
                                        {topic.title}
                                    </h1>
                                    <p className="text-sm text-gray-600 dark:text-gray-400 mt-1">
                                        {minWordCount} words minimum required
                                    </p>
                                </div>

                                <div className="flex items-center space-x-4">
                                    {/* Last saved indicator */}
                                    {lastSaved && (
                                        <div className="text-xs text-gray-500 dark:text-gray-400 flex items-center">
                                            {saving ? (
                                                <span className="flex items-center">
                                                    <Save className="h-3 w-3 mr-1 animate-pulse" />
                                                    Saving...
                                                </span>
                                            ) : (
                                                <span className="flex items-center">
                                                    <Save className="h-3 w-3 mr-1" />
                                                    Saved at{" "}
                                                    {formatLastSaved(lastSaved)}
                                                </span>
                                            )}
                                        </div>
                                    )}

                                    {/* Timer */}
                                    <div
                                        className={`flex items-center font-medium ${getTimeColor()}`}
                                    >
                                        <Clock className="h-5 w-5 mr-1" />
                                        <span className="text-lg">
                                            {formatTime(remainingTime)}
                                        </span>
                                    </div>
                                </div>
                            </div>

                            {/* Save error alert */}
                            {saveError && (
                                <div className="mb-4 p-3 bg-red-50 dark:bg-red-900/20 border-l-4 border-red-500 text-red-700 dark:text-red-300">
                                    <div className="flex">
                                        <AlertTriangle className="h-5 w-5 mr-2" />
                                        <span>{saveError}</span>
                                    </div>
                                </div>
                            )}

                            {/* Topic description */}
                            <div className="mb-4 p-4 bg-gray-50 dark:bg-gray-700/50 rounded-lg border border-gray-200 dark:border-gray-600">
                                <div className="flex items-start">
                                    <Info className="h-5 w-5 text-blue-500 dark:text-blue-400 mt-0.5 mr-2 flex-shrink-0" />
                                    <div>
                                        <h2 className="text-sm font-medium text-gray-800 dark:text-gray-200">
                                            Topic Description
                                        </h2>
                                        <p className="mt-1 text-sm text-gray-600 dark:text-gray-400">
                                            {topic.description}
                                        </p>
                                    </div>
                                </div>
                            </div>

                            {/* Essay Editor */}
                            <div className="mb-4">
                                <div className="border border-gray-300 dark:border-gray-600 rounded-lg overflow-hidden">
                                    <div className="bg-gray-50 dark:bg-gray-700 px-3 py-2 border-b border-gray-300 dark:border-gray-600 flex justify-between items-center">
                                        <span className="text-sm font-medium text-gray-700 dark:text-gray-300">
                                            Essay Content
                                        </span>
                                        <span
                                            className={`text-sm font-medium ${getWordCountColor()}`}
                                        >
                                            {wordCount} / {minWordCount} words
                                        </span>
                                    </div>
                                    <textarea
                                        ref={editorRef}
                                        value={essayContent}
                                        onChange={(e) =>
                                            setEssayContent(e.target.value)
                                        }
                                        className="w-full p-3 min-h-[400px] focus:ring-blue-500 focus:border-blue-500 block border-0 focus:outline-none resize-y bg-white dark:bg-gray-800 text-gray-900 dark:text-gray-100"
                                        placeholder="Start writing your essay here..."
                                        autoFocus
                                    ></textarea>
                                </div>
                                {errors.essay_content && (
                                    <div className="text-sm text-red-600 dark:text-red-400 mt-1">
                                        {errors.essay_content}
                                    </div>
                                )}
                            </div>

                            {/* Action Buttons */}
                            <div className="flex justify-between items-center mt-6">
                                <SecondaryButton
                                    onClick={handleManualSave}
                                    disabled={saving || processing}
                                    className="flex items-center"
                                >
                                    <Save className="h-4 w-4 mr-1" />
                                    Save Draft
                                </SecondaryButton>

                                <PrimaryButton
                                    onClick={handleSubmit}
                                    disabled={processing || saving}
                                    className="flex items-center"
                                >
                                    <Send className="h-4 w-4 mr-1" />
                                    Submit Essay
                                </PrimaryButton>
                            </div>
                        </div>
                    </div>
                </div>
            </div>

            {/* Confirmation Modal */}
            <Modal
                show={showConfirmation}
                onClose={() => setShowConfirmation(false)}
            >
                <div className="p-6">
                    <div className="flex items-center mb-4">
                        <AlertTriangle className="h-6 w-6 text-yellow-500 dark:text-yellow-400 mr-2" />
                        <h2 className="text-lg font-medium text-gray-900 dark:text-gray-100">
                            Word Count Below Minimum
                        </h2>
                    </div>

                    <p className="mb-4 text-gray-600 dark:text-gray-400">
                        Your essay currently has {wordCount} words, which is
                        below the minimum requirement of {minWordCount} words.
                        Essays below the minimum word count may receive lower
                        scores or be rejected.
                    </p>

                    <p className="mb-4 text-gray-600 dark:text-gray-400">
                        Are you sure you want to submit your essay now?
                    </p>

                    <div className="mt-6 flex justify-end space-x-2">
                        <SecondaryButton
                            onClick={() => setShowConfirmation(false)}
                            className="flex items-center"
                        >
                            <XCircle className="h-4 w-4 mr-1" />
                            Continue Writing
                        </SecondaryButton>
                        <PrimaryButton
                            onClick={submitEssay}
                            className="flex items-center"
                        >
                            <CheckCircle className="h-4 w-4 mr-1" />
                            Submit Anyway
                        </PrimaryButton>
                    </div>
                </div>
            </Modal>
        </WriterLayout>
    );
}
