import { useState } from 'react';
import { Head, Link } from '@inertiajs/react';
import WriterLayout from '@/Layouts/WriterLayout';
import { 
    Clock, 
    FileText, 
    AlertTriangle, 
    CheckCircle, 
    XCircle, 
    Edit3,
    Download,
    Eye,
    Calendar,
    User,
    BookOpen,
    AlertCircle,
    Search,
    Filter,
    Info
} from 'lucide-react';
import { formatCurrency, formatDeadline } from '@/lib/formatters';

export default function OrderHistory({ orders }) {
    const [searchTerm, setSearchTerm] = useState('');
    const [statusFilter, setStatusFilter] = useState('all');
    const [showInfractions, setShowInfractions] = useState({});

    const getStatusDisplayName = (status) => {
        const statusMap = {
            'assigned': 'Assigned',
            'in_progress': 'In Progress',
            'submitted': 'Submitted',
            'under_review': 'Under Review',
            'revision_requested': 'Revision Requested',
            'approved': 'Approved',
            'cancelled': 'Cancelled'
        };
        return statusMap[status] || status;
    };

    const getStatusIcon = (status) => {
        switch (status) {
            case 'assigned':
                return <BookOpen size={16} className="text-blue-500" />;
            case 'in_progress':
                return <Edit3 size={16} className="text-yellow-500" />;
            case 'submitted':
                return <CheckCircle size={16} className="text-green-500" />;
            case 'under_review':
                return <Eye size={16} className="text-purple-500" />;
            case 'revision_requested':
                return <AlertCircle size={16} className="text-orange-500" />;
            case 'approved':
                return <CheckCircle size={16} className="text-green-600" />;
            case 'cancelled':
                return <XCircle size={16} className="text-red-500" />;
            default:
                return <FileText size={16} className="text-gray-500" />;
        }
    };

    const getStatusColor = (status) => {
        switch (status) {
            case 'assigned':
                return 'bg-blue-100 text-blue-800 border-blue-200';
            case 'in_progress':
                return 'bg-yellow-100 text-yellow-800 border-yellow-200';
            case 'submitted':
                return 'bg-green-100 text-green-800 border-green-200';
            case 'under_review':
                return 'bg-purple-100 text-purple-800 border-purple-200';
            case 'revision_requested':
                return 'bg-orange-100 text-orange-800 border-orange-200';
            case 'approved':
                return 'bg-green-100 text-green-800 border-green-200';
            case 'cancelled':
                return 'bg-red-100 text-red-800 border-red-200';
            default:
                return 'bg-gray-100 text-gray-800 border-gray-200';
        }
    };

    const isOrderExpired = (order) => {
        if (order.order_status === 'approved' || order.order_status === 'cancelled') {
            return false;
        }
        return new Date(order.writer_deadline) <= new Date();
    };

    const hasInfractions = (order) => {
        return order.writer_infractions && order.writer_infractions.length > 0;
    };

    const getInfractionSeverityColor = (severity) => {
        switch (severity) {
            case 'warning':
                return 'bg-yellow-100 text-yellow-800 border-yellow-200';
            case 'minor':
                return 'bg-blue-100 text-blue-800 border-blue-200';
            case 'major':
                return 'bg-orange-100 text-orange-800 border-orange-200';
            case 'critical':
                return 'bg-red-100 text-red-800 border-red-200';
            default:
                return 'bg-gray-100 text-gray-800 border-gray-200';
        }
    };

    const getInfractionTypeLabel = (type) => {
        const typeMap = {
            'deadline_violation': 'Deadline Violation',
            'quality_issues': 'Quality Issues',
            'plagiarism': 'Plagiarism',
            'communication_failure': 'Communication Failure',
            'policy_violation': 'Policy Violation',
            'unprofessional_behavior': 'Unprofessional Behavior',
            'client_complaint': 'Client Complaint',
            'other': 'Other'
        };
        return typeMap[type] || type;
    };

    const filteredOrders = orders.data.filter(order => {
        const matchesSearch = order.title.toLowerCase().includes(searchTerm.toLowerCase()) ||
                            order.order_number.toLowerCase().includes(searchTerm.toLowerCase());
        const matchesStatus = statusFilter === 'all' || order.order_status === statusFilter;
        return matchesSearch && matchesStatus;
    });

    const renderOrderCard = (order) => {
        const expired = isOrderExpired(order);
        const hasInfraction = hasInfractions(order);

        return (
            <div key={order.id} className={`bg-white rounded-lg shadow-md border-l-4 ${
                expired ? 'border-red-500' : 
                order.order_status === 'approved' ? 'border-green-500' :
                order.order_status === 'cancelled' ? 'border-gray-500' :
                'border-blue-500'
            }`}>
                <div className="p-6">
                    <div className="flex items-start justify-between mb-4">
                        <div className="flex-1">
                            <div className="flex items-center gap-3 mb-2">
                                <h3 className="text-lg font-semibold text-gray-900">
                                    {order.title}
                                </h3>
                                <span className={`inline-flex items-center px-2.5 py-0.5 rounded-full text-xs font-medium border ${getStatusColor(order.order_status)}`}>
                                    {getStatusIcon(order.order_status)}
                                    <span className="ml-1">{getStatusDisplayName(order.order_status)}</span>
                                </span>
                                {expired && (
                                    <span className="inline-flex items-center px-2.5 py-0.5 rounded-full text-xs font-medium bg-red-100 text-red-800 border-red-200">
                                        <AlertTriangle size={12} className="mr-1" />
                                        Expired
                                    </span>
                                )}
                                {hasInfraction && (
                                    <span className="inline-flex items-center px-2.5 py-0.5 rounded-full text-xs font-medium bg-orange-100 text-orange-800 border-orange-200">
                                        <AlertTriangle size={12} className="mr-1" />
                                        Has Infractions
                                    </span>
                                )}
                                {order.submission_was_late && (
                                    <span className="inline-flex items-center px-2.5 py-0.5 rounded-full text-xs font-medium bg-red-100 text-red-800 border-red-200">
                                        <Clock size={12} className="mr-1" />
                                        Late Submission
                                    </span>
                                )}
                            </div>
                            
                            <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-4 gap-4 text-sm text-gray-600">
                                <div className="flex items-center gap-2">
                                    <BookOpen size={16} />
                                    <span>{order.type_of_paper}</span>
                                </div>
                                <div className="flex items-center gap-2">
                                    <User size={16} />
                                    <span>{order.academic_level}</span>
                                </div>
                                <div className="flex items-center gap-2">
                                    <FileText size={16} />
                                    <span>{order.pages} pages</span>
                                </div>
                                <div className="flex items-center gap-2">
                                    <span className="font-medium">{formatCurrency(order.writer_amount)}</span>
                                </div>
                            </div>
                        </div>
                    </div>

                    <div className="border-t pt-4">
                        <div className="flex items-center justify-between text-sm">
                            <div className="flex items-center gap-4">
                                <div className="flex items-center gap-2">
                                    <Calendar size={16} className="text-gray-500" />
                                    <span className="text-gray-600">
                                        Assigned: {order.writer_assigned_at ? new Date(order.writer_assigned_at).toLocaleDateString() : 'N/A'}
                                    </span>
                                </div>
                                <div className="flex items-center gap-2">
                                    <Clock size={16} className="text-gray-500" />
                                    <span className={`font-medium ${
                                        expired ? 'text-red-600' : 'text-gray-900'
                                    }`}>
                                        Deadline: {formatDeadline(order.writer_deadline)}
                                    </span>
                                </div>
                                {order.work_submitted_at && (
                                    <div className="flex items-center gap-2">
                                        <CheckCircle size={16} className="text-gray-500" />
                                        <span className={`font-medium ${
                                            order.submission_was_late ? 'text-orange-600' : 'text-green-600'
                                        }`}>
                                            Submitted: {new Date(order.work_submitted_at).toLocaleDateString()}
                                            {order.submission_was_late && (
                                                <span className="ml-1 text-xs bg-orange-100 text-orange-800 px-2 py-1 rounded-full">
                                                    {order.submission_late_by_hours}h late
                                                </span>
                                            )}
                                        </span>
                                    </div>
                                )}
                            </div>
                            
                            <div className="flex items-center gap-2">
                                <Link
                                    href={route('writer.orders.show', order.id)}
                                    className="inline-flex items-center px-3 py-2 border border-gray-300 text-sm leading-4 font-medium rounded-md text-gray-700 bg-white hover:bg-gray-50 focus:outline-none focus:ring-2 focus:ring-offset-2 focus:ring-indigo-500"
                                >
                                    View Details
                                </Link>
                                {hasInfraction && (
                                    <button
                                        onClick={() => setShowInfractions(prev => ({ ...prev, [order.id]: !prev[order.id] }))}
                                        className="inline-flex items-center px-3 py-2 border border-orange-300 text-sm leading-4 font-medium rounded-md text-orange-700 bg-orange-50 hover:bg-orange-100 focus:outline-none focus:ring-2 focus:ring-offset-2 focus:ring-orange-500"
                                    >
                                        {showInfractions[order.id] ? 'Hide' : 'Show'} Infractions
                                    </button>
                                )}
                            </div>
                        </div>

                        {/* Infractions Section */}
                        {hasInfraction && showInfractions[order.id] && (
                            <div className="mt-4 p-4 bg-orange-50 border border-orange-200 rounded-md">
                                <h4 className="text-sm font-medium text-orange-800 mb-3 flex items-center gap-2">
                                    <AlertTriangle size={16} />
                                    Infractions ({order.writer_infractions.length})
                                </h4>
                                <div className="space-y-3">
                                    {order.writer_infractions.map((infraction, index) => (
                                        <div key={index} className="bg-white p-3 rounded border border-orange-200">
                                            <div className="flex items-center justify-between mb-2">
                                                <span className={`inline-flex items-center px-2 py-1 rounded-full text-xs font-medium border ${getInfractionSeverityColor(infraction.severity_level)}`}>
                                                    {infraction.severity_level.charAt(0).toUpperCase() + infraction.severity_level.slice(1)}
                                                </span>
                                                <span className="text-xs text-gray-500">
                                                    {new Date(infraction.reported_at).toLocaleDateString()}
                                                </span>
                                            </div>
                                            <div className="text-sm font-medium text-gray-900 mb-1">
                                                {getInfractionTypeLabel(infraction.infraction_type)}
                                            </div>
                                            <div className="text-sm text-gray-600 mb-2">
                                                {infraction.description}
                                            </div>
                                            <div className="text-xs text-gray-500">
                                                Status: {infraction.status.replace('_', ' ').replace(/\b\w/g, l => l.toUpperCase())}
                                            </div>
                                        </div>
                                    ))}
                                </div>
                            </div>
                        )}
                    </div>
                </div>
            </div>
        );
    };

    return (
        <WriterLayout>
            <Head title="Order History" />
            
            <div className="py-6">
                <div className="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8">
                    <div className="mb-8">
                        <h1 className="text-3xl font-bold text-gray-900">Order History</h1>
                        <p className="mt-2 text-gray-600">
                            View all your assigned orders, their current status, and any infractions
                        </p>
                    </div>

                    {/* Search and Filter Controls */}
                    <div className="mb-6 flex flex-col sm:flex-row gap-4">
                        <div className="flex-1">
                            <div className="relative">
                                <Search size={20} className="absolute left-3 top-1/2 transform -translate-y-1/2 text-gray-400" />
                                <input
                                    type="text"
                                    placeholder="Search by title or order number..."
                                    value={searchTerm}
                                    onChange={(e) => setSearchTerm(e.target.value)}
                                    className="w-full pl-10 pr-4 py-2 border border-gray-300 rounded-md focus:ring-2 focus:ring-indigo-500 focus:border-indigo-500"
                                />
                            </div>
                        </div>
                        <div className="sm:w-48">
                            <select
                                value={statusFilter}
                                onChange={(e) => setStatusFilter(e.target.value)}
                                className="w-full px-3 py-2 border border-gray-300 rounded-md focus:ring-2 focus:ring-indigo-500 focus:border-indigo-500"
                            >
                                <option value="all">All Statuses</option>
                                <option value="assigned">Assigned</option>
                                <option value="in_progress">In Progress</option>
                                <option value="submitted">Submitted</option>
                                <option value="under_review">Under Review</option>
                                <option value="revision_requested">Revision Requested</option>
                                <option value="approved">Approved</option>
                                <option value="cancelled">Cancelled</option>
                            </select>
                        </div>
                    </div>

                    {/* Orders List */}
                    <div className="space-y-6">
                        {filteredOrders.length === 0 ? (
                            <div className="text-center py-12">
                                <FileText size={48} className="mx-auto text-gray-400 mb-4" />
                                <h3 className="text-lg font-medium text-gray-900 mb-2">No Orders Found</h3>
                                <p className="text-gray-600">
                                    {searchTerm || statusFilter !== 'all' 
                                        ? 'Try adjusting your search or filter criteria.'
                                        : 'You haven\'t been assigned any orders yet.'
                                    }
                                </p>
                            </div>
                        ) : (
                            <>
                                {filteredOrders.map(order => renderOrderCard(order))}
                                
                                {/* Pagination */}
                                {orders.links && orders.links.length > 3 && (
                                    <div className="mt-8">
                                        <nav className="flex items-center justify-between">
                                            <div className="flex-1 flex justify-between sm:hidden">
                                                {orders.prev_page_url && (
                                                    <Link
                                                        href={orders.prev_page_url}
                                                        className="relative inline-flex items-center px-4 py-2 border border-gray-300 text-sm font-medium rounded-md text-gray-700 bg-white hover:bg-gray-50"
                                                    >
                                                        Previous
                                                    </Link>
                                                )}
                                                {orders.next_page_url && (
                                                    <Link
                                                        href={orders.next_page_url}
                                                        className="ml-3 relative inline-flex items-center px-4 py-2 border border-gray-300 text-sm font-medium rounded-md text-gray-700 bg-white hover:bg-gray-50"
                                                    >
                                                        Next
                                                    </Link>
                                                )}
                                            </div>
                                            <div className="hidden sm:flex-1 sm:flex sm:items-center sm:justify-between">
                                                <div>
                                                    <p className="text-sm text-gray-700">
                                                        Showing{' '}
                                                        <span className="font-medium">{orders.from}</span>
                                                        {' '}to{' '}
                                                        <span className="font-medium">{orders.to}</span>
                                                        {' '}of{' '}
                                                        <span className="font-medium">{orders.total}</span>
                                                        {' '}results
                                                    </p>
                                                </div>
                                                <div>
                                                    <nav className="relative z-0 inline-flex rounded-md shadow-sm -space-x-px">
                                                        {orders.links.map((link, index) => (
                                                            <Link
                                                                key={index}
                                                                href={link.url}
                                                                className={`relative inline-flex items-center px-4 py-2 text-sm font-medium ${
                                                                    link.url === null
                                                                        ? 'text-gray-400 cursor-not-allowed'
                                                                        : link.active
                                                                        ? 'z-10 bg-indigo-50 border-indigo-500 text-indigo-600'
                                                                        : 'bg-white border-gray-300 text-gray-500 hover:bg-gray-50'
                                                                } border`}
                                                                dangerouslySetInnerHTML={{ __html: link.label }}
                                                            />
                                                        ))}
                                                    </nav>
                                                </div>
                                            </div>
                                        </nav>
                                    </div>
                                )}
                            </>
                        )}
                    </div>
                </div>
            </div>
        </WriterLayout>
    );
} 