import React, { useState } from 'react';
import { Head, Link } from '@inertiajs/react';
import WriterLayout from '@/Layouts/WriterLayout';
import WithdrawalRequestForm from '@/Components/Writer/WithdrawalRequestForm';
import { 
    Wallet, 
    FileText, 
    DollarSign, 
    Download,
    Calendar,
    User
} from 'lucide-react';

export default function AvailableOrders({ orders, summary }) {
    const [showWithdrawalForm, setShowWithdrawalForm] = useState(false);

    const formatCurrency = (amount) => {
        return new Intl.NumberFormat('en-US', {
            style: 'currency',
            currency: 'USD'
        }).format(amount);
    };

    const formatDate = (dateString) => {
        if (!dateString) return 'N/A';
        return new Date(dateString).toLocaleDateString('en-US', {
            year: 'numeric',
            month: 'short',
            day: 'numeric'
        });
    };

    const formatDateTime = (dateString) => {
        if (!dateString) return 'N/A';
        return new Date(dateString).toLocaleString('en-US', {
            year: 'numeric',
            month: 'short',
            day: 'numeric',
            hour: '2-digit',
            minute: '2-digit'
        });
    };

    return (
        <WriterLayout>
            <Head title="Available Payments" />
            
            <div className="p-6 max-w-7xl mx-auto">
                {/* Header */}
                <div className="mb-8">
                    <div className="flex items-center justify-between">
                        <div>
                            <h1 className="text-3xl font-bold text-gray-900 mb-2">Available Payments</h1>
                            <p className="text-gray-600">
                                Orders ready for withdrawal - {summary.available_orders_count} orders available
                            </p>
                        </div>
                        
                        {summary.available_amount > 0 && (
                            <button
                                onClick={() => setShowWithdrawalForm(true)}
                                className="flex items-center gap-2 bg-green-600 text-white px-4 py-2 rounded-lg hover:bg-green-700 transition-colors"
                            >
                                <Download size={20} />
                                <span>Request Withdrawal</span>
                            </button>
                        )}
                    </div>

                    {/* Summary Cards */}
                    <div className="grid grid-cols-1 md:grid-cols-3 gap-4 mb-6">
                        <div className="bg-white rounded-lg border p-4">
                            <div className="flex items-center justify-between">
                                <div>
                                    <p className="text-sm text-gray-600">Available Amount</p>
                                    <p className="text-2xl font-bold text-green-600">
                                        {formatCurrency(summary.available_amount)}
                                    </p>
                                </div>
                                <Wallet size={24} className="text-green-600" />
                            </div>
                        </div>
                        
                        <div className="bg-white rounded-lg border p-4">
                            <div className="flex items-center justify-between">
                                <div>
                                    <p className="text-sm text-gray-600">Available Orders</p>
                                    <p className="text-2xl font-bold text-gray-900">
                                        {summary.available_orders_count}
                                    </p>
                                </div>
                                <FileText size={24} className="text-gray-600" />
                            </div>
                        </div>
                        
                        <div className="bg-white rounded-lg border p-4">
                            <div className="flex items-center justify-between">
                                <div>
                                    <p className="text-sm text-gray-600">Total Withdrawn</p>
                                    <p className="text-2xl font-bold text-blue-600">
                                        {formatCurrency(summary.withdrawn_amount)}
                                    </p>
                                </div>
                                <DollarSign size={24} className="text-blue-600" />
                            </div>
                        </div>
                    </div>
                </div>

                {/* Orders Table */}
                <div className="bg-white rounded-lg border overflow-hidden">
                    <div className="px-6 py-4 border-b">
                        <h2 className="text-lg font-semibold text-gray-900">Available Orders</h2>
                    </div>
                    
                    {orders.length === 0 ? (
                        <div className="p-8 text-center">
                            <Wallet size={48} className="mx-auto text-gray-400 mb-4" />
                            <h3 className="text-lg font-medium text-gray-900 mb-2">
                                No available payments
                            </h3>
                            <p className="text-gray-600">
                                You don't have any payments available for withdrawal at the moment.
                            </p>
                            <Link
                                href="/writer/orders"
                                className="inline-flex items-center gap-2 mt-4 text-blue-600 hover:text-blue-800"
                            >
                                <FileText size={16} />
                                View All Orders
                            </Link>
                        </div>
                    ) : (
                        <>
                            {/* Desktop Table */}
                            <div className="hidden md:block overflow-x-auto">
                                <table className="min-w-full divide-y divide-gray-200">
                                    <thead className="bg-gray-50">
                                        <tr>
                                            <th className="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">
                                                Order
                                            </th>
                                            <th className="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">
                                                Client
                                            </th>
                                            <th className="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">
                                                Amount
                                            </th>
                                            <th className="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">
                                                Approved Date
                                            </th>
                                            <th className="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">
                                                Available Since
                                            </th>
                                            <th className="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">
                                                Actions
                                            </th>
                                        </tr>
                                    </thead>
                                    <tbody className="bg-white divide-y divide-gray-200">
                                        {orders.map((order) => (
                                            <tr key={order.id} className="hover:bg-gray-50">
                                                <td className="px-6 py-4 whitespace-nowrap">
                                                    <div>
                                                        <div className="text-sm font-medium text-gray-900">
                                                            Order #{order.order_number}
                                                        </div>
                                                        <div className="text-sm text-gray-500">
                                                            {order.title}
                                                        </div>
                                                    </div>
                                                </td>
                                                <td className="px-6 py-4 whitespace-nowrap">
                                                    <div className="flex items-center">
                                                        <div className="flex-shrink-0 h-8 w-8">
                                                            <div className="h-8 w-8 rounded-full bg-gray-300 flex items-center justify-center">
                                                                <User size={16} className="text-gray-600" />
                                                            </div>
                                                        </div>
                                                        <div className="ml-3">
                                                            <div className="text-sm font-medium text-gray-900">
                                                                {order.user?.name || 'Unknown Client'}
                                                            </div>
                                                        </div>
                                                    </div>
                                                </td>
                                                <td className="px-6 py-4 whitespace-nowrap">
                                                    <div className="text-sm font-medium text-green-600">
                                                        {formatCurrency(order.writer_amount)}
                                                    </div>
                                                </td>
                                                <td className="px-6 py-4 whitespace-nowrap text-sm text-gray-900">
                                                    {order.writer_payment_released_at ? formatDateTime(order.writer_payment_released_at) : 'N/A'}
                                                </td>
                                                <td className="px-6 py-4 whitespace-nowrap text-sm text-gray-900">
                                                    {order.writer_payment_released_at ? formatDate(order.writer_payment_released_at) : 'N/A'}
                                                </td>
                                                <td className="px-6 py-4 whitespace-nowrap text-sm font-medium">
                                                    <Link
                                                        href={`/writer/orders/${order.id}`}
                                                        className="text-blue-600 hover:text-blue-900"
                                                    >
                                                        View Details
                                                    </Link>
                                                </td>
                                            </tr>
                                        ))}
                                    </tbody>
                                </table>
                            </div>

                            {/* Mobile Cards */}
                            <div className="md:hidden">
                                <div className="p-4 space-y-4">
                                    {orders.map((order) => (
                                        <div key={order.id} className="bg-gray-50 rounded-lg p-4">
                                            <div className="flex items-center justify-between mb-3">
                                                <div className="text-sm font-medium text-gray-900">
                                                    Order #{order.order_number}
                                                </div>
                                                <div className="text-sm font-medium text-green-600">
                                                    {formatCurrency(order.writer_amount)}
                                                </div>
                                            </div>
                                            <div className="text-sm text-gray-600 mb-2">
                                                {order.title}
                                            </div>
                                            <div className="text-xs text-gray-500 mb-3">
                                                Client: {order.user?.name || 'Unknown'}
                                            </div>
                                            <div className="flex items-center justify-between text-xs text-gray-500">
                                                <span>
                                                    Available: {order.writer_payment_released_at ? formatDate(order.writer_payment_released_at) : 'N/A'}
                                                </span>
                                                <Link
                                                    href={`/writer/orders/${order.id}`}
                                                    className="text-blue-600 hover:text-blue-800"
                                                >
                                                    View Details
                                                </Link>
                                            </div>
                                        </div>
                                    ))}
                                </div>
                            </div>
                        </>
                    )}
                </div>

                {/* Withdrawal Information */}
                {summary.available_amount > 0 && (
                    <div className="mt-8 bg-blue-50 border border-blue-200 rounded-lg p-6">
                        <div className="flex items-start">
                            <div className="flex-shrink-0">
                                <Download size={24} className="text-blue-600" />
                            </div>
                            <div className="ml-3">
                                <h3 className="text-lg font-medium text-blue-900 mb-2">
                                    Ready to Withdraw?
                                </h3>
                                <p className="text-blue-800 mb-4">
                                    You have {formatCurrency(summary.available_amount)} available for withdrawal. 
                                    Click "Request Withdrawal" to submit your withdrawal request for the next processing date.
                                </p>
                                <button
                                    onClick={() => setShowWithdrawalForm(true)}
                                    className="bg-blue-600 text-white px-4 py-2 rounded-md hover:bg-blue-700 transition-colors"
                                >
                                    Request Withdrawal
                                </button>
                            </div>
                        </div>
                    </div>
                )}
            </div>

            {/* Withdrawal Request Form Modal */}
            <WithdrawalRequestForm
                isOpen={showWithdrawalForm}
                onClose={() => setShowWithdrawalForm(false)}
                eligibility={{
                    can_withdraw: summary.available_amount > 0,
                    minimum_withdrawal: 5.00,
                    available_amount: summary.available_amount
                }}
                availableAmount={summary.available_amount}
            />
        </WriterLayout>
    );
}