import React, { useState, useEffect } from 'react';
import { Head, Link } from '@inertiajs/react';
import { 
    Clock, 
    DollarSign, 
    TrendingUp, 
    Wallet, 
    Calendar,
    FileText,
    CheckCircle,
    AlertCircle,
    ChevronLeft,
    Eye,
    Settings
} from 'lucide-react';
import AuthenticatedLayout from '@/Layouts/AuthenticatedLayout';
import { formatCurrency, formatDate } from '@/lib/formatters';

export default function PaymentHistory({ auth, summary, paymentRecords, recentWithdrawals }) {
    const [activeTab, setActiveTab] = useState('payments');
    const [searchTerm, setSearchTerm] = useState('');
    const [filterStatus, setFilterStatus] = useState('all');

    const filteredPayments = paymentRecords?.data?.filter(payment => {
        const matchesSearch = (payment.notes?.toLowerCase().includes(searchTerm.toLowerCase()) ||
                            payment.order_id?.toString().includes(searchTerm));
        const matchesStatus = filterStatus === 'all' || payment.status === filterStatus;
        return matchesSearch && matchesStatus;
    }) || [];

    const getStatusColor = (status) => {
        switch (status) {
            case 'completed':
                return 'text-green-600 bg-green-100';
            case 'pending':
                return 'text-yellow-600 bg-yellow-100';
            case 'failed':
                return 'text-red-600 bg-red-100';
            default:
                return 'text-gray-600 bg-gray-100';
        }
    };

    const getStatusIcon = (status) => {
        switch (status) {
            case 'completed':
                return <CheckCircle size={16} className="text-green-600" />;
            case 'pending':
                return <Clock size={16} className="text-yellow-600" />;
            case 'failed':
                return <AlertCircle size={16} className="text-red-600" />;
            default:
                return <Clock size={16} className="text-gray-600" />;
        }
    };

    return (
        <AuthenticatedLayout
            user={auth.user}
            header={
                <div className="flex items-center gap-4">
                    <Link
                        href={route('writer.dashboard')}
                        className="flex items-center gap-2 text-gray-600 hover:text-gray-900 transition-colors"
                    >
                        <ChevronLeft size={20} />
                        <span>Back to Dashboard</span>
                    </Link>
                    <h2 className="text-xl font-semibold text-gray-900">Payment History</h2>
                </div>
            }
        >
            <Head title="Payment History" />

            <div className="py-6">
                <div className="max-w-7xl mx-auto sm:px-6 lg:px-8">
                    {/* Summary Cards */}
                    <div className="grid grid-cols-1 md:grid-cols-4 gap-6 mb-8">
                        <div className="bg-white overflow-hidden shadow-sm sm:rounded-lg border border-gray-200">
                            <div className="p-6">
                                <div className="flex items-center">
                                    <div className="flex-shrink-0">
                                        <Wallet className="h-8 w-8 text-green-600" />
                                    </div>
                                    <div className="ml-4">
                                        <p className="text-sm font-medium text-gray-500">Total Earnings</p>
                                        <p className="text-2xl font-semibold text-gray-900">
                                            {formatCurrency(summary?.totalEarnings || 0)}
                                        </p>
                                    </div>
                                </div>
                            </div>
                        </div>

                        <div className="bg-white overflow-hidden shadow-sm sm:rounded-lg border border-gray-200">
                            <div className="p-6">
                                <div className="flex items-center">
                                    <div className="flex-shrink-0">
                                        <TrendingUp className="h-8 w-8 text-blue-600" />
                                    </div>
                                    <div className="ml-4">
                                        <p className="text-sm font-medium text-gray-500">Available for Withdrawal</p>
                                        <p className="text-2xl font-semibold text-gray-900">
                                            {formatCurrency(summary?.availableForWithdrawal || 0)}
                                        </p>
                                    </div>
                                </div>
                            </div>
                        </div>

                        <div className="bg-white overflow-hidden shadow-sm sm:rounded-lg border border-gray-200">
                            <div className="p-6">
                                <div className="flex items-center">
                                    <div className="flex-shrink-0">
                                        <Clock className="h-8 w-8 text-yellow-600" />
                                    </div>
                                    <div className="ml-4">
                                        <p className="text-sm font-medium text-gray-500">Pending Payments</p>
                                        <p className="text-2xl font-semibold text-gray-900">
                                            {formatCurrency(summary?.pendingPayments || 0)}
                                        </p>
                                    </div>
                                </div>
                            </div>
                        </div>

                        <div className="bg-white overflow-hidden shadow-sm sm:rounded-lg border border-gray-200">
                            <div className="p-6">
                                <div className="flex items-center">
                                    <div className="flex-shrink-0">
                                        <DollarSign className="h-8 w-8 text-purple-600" />
                                    </div>
                                    <div className="ml-4">
                                        <p className="text-sm font-medium text-gray-500">This Month</p>
                                        <p className="text-2xl font-semibold text-gray-900">
                                            {formatCurrency(summary?.monthlyEarnings || 0)}
                                        </p>
                                    </div>
                                </div>
                            </div>
                        </div>
                    </div>

                    {/* Tabs */}
                    <div className="bg-white shadow-sm sm:rounded-lg border border-gray-200 mb-6">
                        <div className="border-b border-gray-200">
                            <nav className="-mb-px flex space-x-8 px-6">
                                <button
                                    onClick={() => setActiveTab('payments')}
                                    className={`py-4 px-1 border-b-2 font-medium text-sm ${
                                        activeTab === 'payments'
                                            ? 'border-green-500 text-green-600'
                                            : 'border-transparent text-gray-500 hover:text-gray-700 hover:border-gray-300'
                                    }`}
                                >
                                    Payment Records
                                </button>
                                <button
                                    onClick={() => setActiveTab('withdrawals')}
                                    className={`py-4 px-1 border-b-2 font-medium text-sm ${
                                        activeTab === 'withdrawals'
                                            ? 'border-green-500 text-green-600'
                                            : 'border-transparent text-gray-500 hover:text-gray-700 hover:border-gray-300'
                                    }`}
                                >
                                    Withdrawal Requests
                                </button>
                            </nav>
                        </div>

                        <div className="p-6">
                            {activeTab === 'payments' && (
                                <div>
                                    {/* Filters */}
                                    <div className="flex flex-col sm:flex-row gap-4 mb-6">
                                        <div className="flex-1">
                                            <input
                                                type="text"
                                                placeholder="Search payments..."
                                                value={searchTerm}
                                                onChange={(e) => setSearchTerm(e.target.value)}
                                                className="w-full px-3 py-2 border border-gray-300 rounded-md shadow-sm focus:outline-none focus:ring-green-500 focus:border-green-500"
                                            />
                                        </div>
                                        <select
                                            value={filterStatus}
                                            onChange={(e) => setFilterStatus(e.target.value)}
                                            className="px-3 py-2 border border-gray-300 rounded-md shadow-sm focus:outline-none focus:ring-green-500 focus:border-green-500"
                                        >
                                            <option value="all">All Statuses</option>
                                            <option value="completed">Completed</option>
                                            <option value="pending">Pending</option>
                                            <option value="failed">Failed</option>
                                        </select>
                                    </div>

                                    {/* Payment Records Table */}
                                    <div className="overflow-x-auto">
                                        <table className="min-w-full divide-y divide-gray-200">
                                            <thead className="bg-gray-50">
                                                <tr>
                                                    <th className="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">
                                                        Date
                                                    </th>
                                                    <th className="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">
                                                        Description
                                                    </th>
                                                    <th className="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">
                                                        Order ID
                                                    </th>
                                                    <th className="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">
                                                        Amount
                                                    </th>
                                                    <th className="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">
                                                        Status
                                                    </th>
                                                    <th className="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">
                                                        Actions
                                                    </th>
                                                </tr>
                                            </thead>
                                            <tbody className="bg-white divide-y divide-gray-200">
                                                {filteredPayments.length > 0 ? (
                                                    filteredPayments.map((payment) => (
                                                        <tr key={payment.id} className="hover:bg-gray-50">
                                                            <td className="px-6 py-4 whitespace-nowrap text-sm text-gray-900">
                                                                {formatDate(payment.created_at)}
                                                            </td>
                                                            <td className="px-6 py-4 whitespace-nowrap text-sm text-gray-900">
                                                                {payment.notes || 'Payment for order'}
                                                            </td>
                                                            <td className="px-6 py-4 whitespace-nowrap text-sm text-gray-900">
                                                                #{payment.order_id}
                                                            </td>
                                                            <td className="px-6 py-4 whitespace-nowrap text-sm font-medium text-gray-900">
                                                                {formatCurrency(payment.payment_amount)}
                                                            </td>
                                                            <td className="px-6 py-4 whitespace-nowrap">
                                                                <span className={`inline-flex items-center px-2.5 py-0.5 rounded-full text-xs font-medium ${getStatusColor(payment.status)}`}>
                                                                    {getStatusIcon(payment.status)}
                                                                    <span className="ml-1 capitalize">{payment.status}</span>
                                                                </span>
                                                            </td>
                                                            <td className="px-6 py-4 whitespace-nowrap text-sm text-gray-500">
                                                                <Link
                                                                    href={route('writer.orders.show', payment.order_id)}
                                                                    className="text-green-600 hover:text-green-900 flex items-center gap-1"
                                                                >
                                                                    <Eye size={16} />
                                                                    View Order
                                                                </Link>
                                                            </td>
                                                        </tr>
                                                    ))
                                                ) : (
                                                    <tr>
                                                        <td colSpan="6" className="px-6 py-4 text-center text-sm text-gray-500">
                                                            No payment records found.
                                                        </td>
                                                    </tr>
                                                )}
                                            </tbody>
                                        </table>
                                    </div>

                                    {/* Pagination */}
                                    {paymentRecords?.links && (
                                        <div className="mt-6">
                                            {/* Add pagination component here if needed */}
                                        </div>
                                    )}
                                </div>
                            )}

                            {activeTab === 'withdrawals' && (
                                <div>
                                    <div className="space-y-4">
                                        {recentWithdrawals.length > 0 ? (
                                            recentWithdrawals.map((withdrawal) => (
                                                <div key={withdrawal.id} className="border border-gray-200 rounded-lg p-4">
                                                    <div className="flex items-center justify-between">
                                                        <div className="flex items-center gap-3">
                                                            <div className="flex-shrink-0">
                                                                <Wallet className="h-6 w-6 text-green-600" />
                                                            </div>
                                                            <div>
                                                                <p className="text-sm font-medium text-gray-900">
                                                                    Withdrawal Request
                                                                </p>
                                                                <p className="text-sm text-gray-500">
                                                                    {formatDate(withdrawal.created_at)}
                                                                </p>
                                                            </div>
                                                        </div>
                                                        <div className="text-right">
                                                            <p className="text-lg font-semibold text-gray-900">
                                                                {formatCurrency(withdrawal.amount)}
                                                            </p>
                                                            <span className={`inline-flex items-center px-2.5 py-0.5 rounded-full text-xs font-medium ${getStatusColor(withdrawal.status)}`}>
                                                                {getStatusIcon(withdrawal.status)}
                                                                <span className="ml-1 capitalize">{withdrawal.status}</span>
                                                            </span>
                                                        </div>
                                                    </div>
                                                    {withdrawal.notes && (
                                                        <div className="mt-3 p-3 bg-gray-50 rounded-md">
                                                            <p className="text-sm text-gray-600">
                                                                <strong>Notes:</strong> {withdrawal.notes}
                                                            </p>
                                                        </div>
                                                    )}
                                                </div>
                                            ))
                                        ) : (
                                            <div className="text-center py-8">
                                                <FileText className="mx-auto h-12 w-12 text-gray-400" />
                                                <h3 className="mt-2 text-sm font-medium text-gray-900">No withdrawal requests</h3>
                                                <p className="mt-1 text-sm text-gray-500">
                                                    You haven't made any withdrawal requests yet.
                                                </p>
                                            </div>
                                        )}
                                    </div>
                                </div>
                            )}
                        </div>
                    </div>

                    {/* Quick Actions */}
                    <div className="bg-white shadow-sm sm:rounded-lg border border-gray-200">
                        <div className="p-6">
                            <h3 className="text-lg font-medium text-gray-900 mb-4">Quick Actions</h3>
                            <div className="flex flex-wrap gap-4">
                                <Link
                                    href={route('writer.payment-methods')}
                                    className="inline-flex items-center px-4 py-2 border border-transparent text-sm font-medium rounded-md text-white bg-green-600 hover:bg-green-700 focus:outline-none focus:ring-2 focus:ring-offset-2 focus:ring-green-500"
                                >
                                    <Settings className="h-4 w-4 mr-2" />
                                    Manage Payment Methods
                                </Link>
                                <Link
                                    href={route('writer.earnings')}
                                    className="inline-flex items-center px-4 py-2 border border-gray-300 text-sm font-medium rounded-md text-gray-700 bg-white hover:bg-gray-50 focus:outline-none focus:ring-2 focus:ring-offset-2 focus:ring-green-500"
                                >
                                    <TrendingUp className="h-4 w-4 mr-2" />
                                    View Earnings Overview
                                </Link>
                            </div>
                        </div>
                    </div>
                </div>
            </div>
        </AuthenticatedLayout>
    );
} 