import React, { useState, useEffect } from 'react';
import { Head, Link, useForm } from '@inertiajs/react';
import { 
    CreditCard, 
    DollarSign, 
    TrendingUp, 
    Wallet, 
    Plus,
    Edit,
    Trash2,
    Star,
    StarOff,
    CheckCircle,
    AlertCircle,
    ChevronLeft,
    Settings,
    Shield,
    Clock
} from 'lucide-react';
import AuthenticatedLayout from '@/Layouts/AuthenticatedLayout';
import { formatCurrency } from '@/lib/formatters';

export default function PaymentMethods({ auth, summary, availableMethods, paymentAccounts }) {
    const [showAddForm, setShowAddForm] = useState(false);
    const [editingAccount, setEditingAccount] = useState(null);
    const [selectedMethod, setSelectedMethod] = useState(null);

    const { data, setData, post, put, delete: destroy, processing, errors, reset } = useForm({
        payment_method_id: '',
        account_details: {},
        is_primary: false
    });

    const handleAddMethod = () => {
        setShowAddForm(true);
        setEditingAccount(null);
        reset();
    };

    const handleEditMethod = (account) => {
        setEditingAccount(account);
        setShowAddForm(true);
        setData({
            payment_method_id: account.payment_method_id,
            account_details: account.account_details,
            is_primary: account.is_primary
        });
    };

    const handleSubmit = (e) => {
        e.preventDefault();
        
        if (editingAccount) {
            put(route('writer.payment-methods.update', editingAccount.id), {
                onSuccess: () => {
                    setShowAddForm(false);
                    setEditingAccount(null);
                    reset();
                }
            });
        } else {
            post(route('writer.payment-methods.store'), {
                onSuccess: () => {
                    setShowAddForm(false);
                    reset();
                }
            });
        }
    };

    const handleDelete = (accountId) => {
        if (confirm('Are you sure you want to delete this payment method? This action cannot be undone.')) {
            destroy(route('writer.payment-methods.delete', accountId));
        }
    };

    const handleSetPrimary = (accountId) => {
        // This would typically make an API call to set the account as primary
        // For now, we'll just show a confirmation
        if (confirm('Set this as your primary payment method?')) {
            // Make API call here
            console.log('Setting primary:', accountId);
        }
    };

    const getMethodIcon = (methodName) => {
        switch (methodName.toLowerCase()) {
            case 'paypal':
                return <CreditCard className="h-6 w-6 text-blue-600" />;
            case 'bank transfer':
                return <CreditCard className="h-6 w-6 text-green-600" />;
            case 'stripe':
                return <CreditCard className="h-6 w-6 text-purple-600" />;
            default:
                return <CreditCard className="h-6 w-6 text-gray-600" />;
        }
    };

    const renderAccountDetails = (details, methodName) => {
        if (methodName.toLowerCase() === 'paypal') {
            return (
                <div className="text-sm text-gray-600">
                    <p><strong>Email:</strong> {details.email || 'N/A'}</p>
                </div>
            );
        } else if (methodName.toLowerCase() === 'bank transfer') {
            return (
                <div className="text-sm text-gray-600">
                    <p><strong>Account Name:</strong> {details.account_name || 'N/A'}</p>
                    <p><strong>Account Number:</strong> {details.account_number ? `****${details.account_number.slice(-4)}` : 'N/A'}</p>
                    <p><strong>Bank:</strong> {details.bank_name || 'N/A'}</p>
                </div>
            );
        }
        return <div className="text-sm text-gray-600">Account details available</div>;
    };

    return (
        <AuthenticatedLayout
            user={auth.user}
            header={
                <div className="flex items-center gap-4">
                    <Link
                        href={route('writer.dashboard')}
                        className="flex items-center gap-2 text-gray-600 hover:text-gray-900 transition-colors"
                    >
                        <ChevronLeft size={20} />
                        <span>Back to Dashboard</span>
                    </Link>
                    <h2 className="text-xl font-semibold text-gray-900">Payment Methods</h2>
                </div>
            }
        >
            <Head title="Payment Methods" />

            <div className="py-6">
                <div className="max-w-7xl mx-auto sm:px-6 lg:px-8">
                    {/* Summary Cards */}
                    <div className="grid grid-cols-1 md:grid-cols-3 gap-6 mb-8">
                        <div className="bg-white overflow-hidden shadow-sm sm:rounded-lg border border-gray-200">
                            <div className="p-6">
                                <div className="flex items-center">
                                    <div className="flex-shrink-0">
                                        <Wallet className="h-8 w-8 text-green-600" />
                                    </div>
                                    <div className="ml-4">
                                        <p className="text-sm font-medium text-gray-500">Available for Withdrawal</p>
                                        <p className="text-2xl font-semibold text-gray-900">
                                            {formatCurrency(summary?.availableForWithdrawal || 0)}
                                        </p>
                                    </div>
                                </div>
                            </div>
                        </div>

                        <div className="bg-white overflow-hidden shadow-sm sm:rounded-lg border border-gray-200">
                            <div className="p-6">
                                <div className="flex items-center">
                                    <div className="flex-shrink-0">
                                        <CreditCard className="h-8 w-8 text-blue-600" />
                                    </div>
                                    <div className="ml-4">
                                        <p className="text-sm font-medium text-gray-500">Payment Methods</p>
                                        <p className="text-2xl font-semibold text-gray-900">
                                            {paymentAccounts?.length || 0}
                                        </p>
                                    </div>
                                </div>
                            </div>
                        </div>

                        <div className="bg-white overflow-hidden shadow-sm sm:rounded-lg border border-gray-200">
                            <div className="p-6">
                                <div className="flex items-center">
                                    <div className="flex-shrink-0">
                                        <CheckCircle className="h-8 w-8 text-green-600" />
                                    </div>
                                    <div className="ml-4">
                                        <p className="text-sm font-medium text-gray-500">Verified Methods</p>
                                        <p className="text-2xl font-semibold text-gray-900">
                                            {paymentAccounts?.filter(acc => acc.is_verified)?.length || 0}
                                        </p>
                                    </div>
                                </div>
                            </div>
                        </div>
                    </div>

                    {/* Current Payment Methods */}
                    <div className="bg-white shadow-sm sm:rounded-lg border border-gray-200 mb-6">
                        <div className="px-6 py-4 border-b border-gray-200">
                            <div className="flex items-center justify-between">
                                <h3 className="text-lg font-medium text-gray-900">Your Payment Methods</h3>
                                <button
                                    onClick={handleAddMethod}
                                    className="inline-flex items-center px-4 py-2 border border-transparent text-sm font-medium rounded-md text-white bg-green-600 hover:bg-green-700 focus:outline-none focus:ring-2 focus:ring-offset-2 focus:ring-green-500"
                                >
                                    <Plus className="h-4 w-4 mr-2" />
                                    Add Payment Method
                                </button>
                            </div>
                        </div>

                        <div className="p-6">
                            {paymentAccounts && paymentAccounts.length > 0 ? (
                                <div className="space-y-4">
                                    {paymentAccounts.map((account) => (
                                        <div key={account.id} className="border border-gray-200 rounded-lg p-4">
                                            <div className="flex items-center justify-between">
                                                <div className="flex items-center gap-4">
                                                    <div className="flex-shrink-0">
                                                        {getMethodIcon(account.payment_method?.name)}
                                                    </div>
                                                    <div className="flex-1">
                                                        <div className="flex items-center gap-2 mb-2">
                                                            <h4 className="text-lg font-medium text-gray-900">
                                                                {account.payment_method?.name}
                                                            </h4>
                                                            {account.is_primary && (
                                                                <span className="inline-flex items-center px-2.5 py-0.5 rounded-full text-xs font-medium bg-green-100 text-green-800">
                                                                    <Star className="h-3 w-3 mr-1" />
                                                                    Primary
                                                                </span>
                                                            )}
                                                            {account.is_verified ? (
                                                                <span className="inline-flex items-center px-2.5 py-0.5 rounded-full text-xs font-medium bg-blue-100 text-blue-800">
                                                                    <CheckCircle className="h-3 w-3 mr-1" />
                                                                    Verified
                                                                </span>
                                                            ) : (
                                                                <span className="inline-flex items-center px-2.5 py-0.5 rounded-full text-xs font-medium bg-yellow-100 text-yellow-800">
                                                                    <AlertCircle className="h-3 w-3 mr-1" />
                                                                    Pending Verification
                                                                </span>
                                                            )}
                                                        </div>
                                                        {renderAccountDetails(account.account_details, account.payment_method?.name)}
                                                    </div>
                                                </div>
                                                <div className="flex items-center gap-2">
                                                    {!account.is_primary && (
                                                        <button
                                                            onClick={() => handleSetPrimary(account.id)}
                                                            className="inline-flex items-center px-3 py-1.5 border border-gray-300 text-xs font-medium rounded text-gray-700 bg-white hover:bg-gray-50 focus:outline-none focus:ring-2 focus:ring-offset-2 focus:ring-green-500"
                                                        >
                                                            <StarOff className="h-3 w-3 mr-1" />
                                                            Set Primary
                                                        </button>
                                                    )}
                                                    <button
                                                        onClick={() => handleEditMethod(account)}
                                                        className="inline-flex items-center px-3 py-1.5 border border-gray-300 text-xs font-medium rounded text-gray-700 bg-white hover:bg-gray-50 focus:outline-none focus:ring-2 focus:ring-offset-2 focus:ring-green-500"
                                                    >
                                                        <Edit className="h-3 w-3 mr-1" />
                                                        Edit
                                                    </button>
                                                    <button
                                                        onClick={() => handleDelete(account.id)}
                                                        className="inline-flex items-center px-3 py-1.5 border border-red-300 text-xs font-medium rounded text-red-700 bg-white hover:bg-red-50 focus:outline-none focus:ring-2 focus:ring-offset-2 focus:ring-red-500"
                                                    >
                                                        <Trash2 className="h-3 w-3 mr-1" />
                                                        Delete
                                                    </button>
                                                </div>
                                            </div>
                                        </div>
                                    ))}
                                </div>
                            ) : (
                                <div className="text-center py-8">
                                    <CreditCard className="mx-auto h-12 w-12 text-gray-400" />
                                    <h3 className="mt-2 text-sm font-medium text-gray-900">No payment methods</h3>
                                    <p className="mt-1 text-sm text-gray-500">
                                        Get started by adding your first payment method.
                                    </p>
                                    <div className="mt-6">
                                        <button
                                            onClick={handleAddMethod}
                                            className="inline-flex items-center px-4 py-2 border border-transparent text-sm font-medium rounded-md text-white bg-green-600 hover:bg-green-700 focus:outline-none focus:ring-2 focus:ring-offset-2 focus:ring-green-500"
                                        >
                                            <Plus className="h-4 w-4 mr-2" />
                                            Add Payment Method
                                        </button>
                                    </div>
                                </div>
                            )}
                        </div>
                    </div>

                    {/* Add/Edit Payment Method Form */}
                    {showAddForm && (
                        <div className="bg-white shadow-sm sm:rounded-lg border border-gray-200 mb-6">
                            <div className="px-6 py-4 border-b border-gray-200">
                                <h3 className="text-lg font-medium text-gray-900">
                                    {editingAccount ? 'Edit Payment Method' : 'Add Payment Method'}
                                </h3>
                            </div>

                            <form onSubmit={handleSubmit} className="p-6 space-y-6">
                                <div>
                                    <label className="block text-sm font-medium text-gray-700 mb-2">
                                        Payment Method Type
                                    </label>
                                    <select
                                        value={data.payment_method_id}
                                        onChange={(e) => {
                                            setData('payment_method_id', e.target.value);
                                            setSelectedMethod(availableMethods.find(m => m.id == e.target.value));
                                        }}
                                        className="w-full px-3 py-2 border border-gray-300 rounded-md shadow-sm focus:outline-none focus:ring-green-500 focus:border-green-500"
                                        required
                                    >
                                        <option value="">Select a payment method</option>
                                        {availableMethods.map((method) => (
                                            <option key={method.id} value={method.id}>
                                                {method.name}
                                            </option>
                                        ))}
                                    </select>
                                    {errors.payment_method_id && (
                                        <p className="mt-1 text-sm text-red-600">{errors.payment_method_id}</p>
                                    )}
                                </div>

                                {selectedMethod && (
                                    <div>
                                        <label className="block text-sm font-medium text-gray-700 mb-2">
                                            Account Details
                                        </label>
                                        {selectedMethod.name.toLowerCase() === 'paypal' && (
                                            <div className="space-y-4">
                                                <div>
                                                    <label className="block text-sm font-medium text-gray-700 mb-1">
                                                        PayPal Email
                                                    </label>
                                                    <input
                                                        type="email"
                                                        value={data.account_details.email || ''}
                                                        onChange={(e) => setData('account_details', { ...data.account_details, email: e.target.value })}
                                                        className="w-full px-3 py-2 border border-gray-300 rounded-md shadow-sm focus:outline-none focus:ring-green-500 focus:border-green-500"
                                                        placeholder="your-email@example.com"
                                                        required
                                                    />
                                                </div>
                                            </div>
                                        )}

                                        {selectedMethod.name.toLowerCase() === 'bank transfer' && (
                                            <div className="space-y-4">
                                                <div className="grid grid-cols-1 md:grid-cols-2 gap-4">
                                                    <div>
                                                        <label className="block text-sm font-medium text-gray-700 mb-1">
                                                            Account Holder Name
                                                        </label>
                                                        <input
                                                            type="text"
                                                            value={data.account_details.account_name || ''}
                                                            onChange={(e) => setData('account_details', { ...data.account_details, account_name: e.target.value })}
                                                            className="w-full px-3 py-2 border border-gray-300 rounded-md shadow-sm focus:outline-none focus:ring-green-500 focus:border-green-500"
                                                            placeholder="John Doe"
                                                            required
                                                        />
                                                    </div>
                                                    <div>
                                                        <label className="block text-sm font-medium text-gray-700 mb-1">
                                                            Account Number
                                                        </label>
                                                        <input
                                                            type="text"
                                                            value={data.account_details.account_number || ''}
                                                            onChange={(e) => setData('account_details', { ...data.account_details, account_number: e.target.value })}
                                                            className="w-full px-3 py-2 border border-gray-300 rounded-md shadow-sm focus:outline-none focus:ring-green-500 focus:border-green-500"
                                                            placeholder="12345678"
                                                            required
                                                        />
                                                    </div>
                                                </div>
                                                <div>
                                                    <label className="block text-sm font-medium text-gray-700 mb-1">
                                                        Bank Name
                                                    </label>
                                                    <input
                                                        type="text"
                                                        value={data.account_details.bank_name || ''}
                                                        onChange={(e) => setData('account_details', { ...data.account_details, bank_name: e.target.value })}
                                                        className="w-full px-3 py-2 border border-gray-300 rounded-md shadow-sm focus:outline-none focus:ring-green-500 focus:border-green-500"
                                                        placeholder="Bank of America"
                                                        required
                                                    />
                                                </div>
                                            </div>
                                        )}
                                    </div>
                                )}

                                <div className="flex items-center">
                                    <input
                                        id="is_primary"
                                        type="checkbox"
                                        checked={data.is_primary}
                                        onChange={(e) => setData('is_primary', e.target.checked)}
                                        className="h-4 w-4 text-green-600 focus:ring-green-500 border-gray-300 rounded"
                                    />
                                    <label htmlFor="is_primary" className="ml-2 block text-sm text-gray-900">
                                        Set as primary payment method
                                    </label>
                                </div>

                                <div className="flex items-center justify-end space-x-3">
                                    <button
                                        type="button"
                                        onClick={() => {
                                            setShowAddForm(false);
                                            setEditingAccount(null);
                                            reset();
                                        }}
                                        className="px-4 py-2 border border-gray-300 text-sm font-medium rounded-md text-gray-700 bg-white hover:bg-gray-50 focus:outline-none focus:ring-2 focus:ring-offset-2 focus:ring-green-500"
                                    >
                                        Cancel
                                    </button>
                                    <button
                                        type="submit"
                                        disabled={processing}
                                        className="px-4 py-2 border border-transparent text-sm font-medium rounded-md text-white bg-green-600 hover:bg-green-700 focus:outline-none focus:ring-2 focus:ring-offset-2 focus:ring-green-500 disabled:opacity-50"
                                    >
                                        {processing ? 'Saving...' : (editingAccount ? 'Update Method' : 'Add Method')}
                                    </button>
                                </div>
                            </form>
                        </div>
                    )}

                    {/* Security Notice */}
                    <div className="bg-blue-50 border border-blue-200 rounded-lg p-4">
                        <div className="flex">
                            <div className="flex-shrink-0">
                                <Shield className="h-5 w-5 text-blue-400" />
                            </div>
                            <div className="ml-3">
                                <h3 className="text-sm font-medium text-blue-800">Security & Privacy</h3>
                                <div className="mt-2 text-sm text-blue-700">
                                    <p>
                                        Your payment information is encrypted and securely stored. We never share your 
                                        financial details with third parties. All transactions are processed through 
                                        secure, PCI-compliant payment gateways.
                                    </p>
                                </div>
                            </div>
                        </div>
                    </div>

                    {/* Quick Actions */}
                    <div className="bg-white shadow-sm sm:rounded-lg border border-gray-200 mt-6">
                        <div className="p-6">
                            <h3 className="text-lg font-medium text-gray-900 mb-4">Quick Actions</h3>
                            <div className="flex flex-wrap gap-4">
                                <Link
                                    href={route('writer.payments')}
                                    className="inline-flex items-center px-4 py-2 border border-gray-300 text-sm font-medium rounded-md text-gray-700 bg-white hover:bg-gray-50 focus:outline-none focus:ring-2 focus:ring-offset-2 focus:ring-green-500"
                                >
                                    <Clock className="h-4 w-4 mr-2" />
                                    View Payment History
                                </Link>
                                <Link
                                    href={route('writer.earnings')}
                                    className="inline-flex items-center px-4 py-2 border border-gray-300 text-sm font-medium rounded-md text-gray-700 bg-white hover:bg-gray-50 focus:outline-none focus:ring-2 focus:ring-offset-2 focus:ring-green-500"
                                >
                                    <TrendingUp className="h-4 w-4 mr-2" />
                                    View Earnings Overview
                                </Link>
                            </div>
                        </div>
                    </div>
                </div>
            </div>
        </AuthenticatedLayout>
    );
} 