import React from 'react';
import { Head, Link } from '@inertiajs/react';
import WriterLayout from '@/Layouts/WriterLayout';
import { Clock, ArrowLeft, FileText, DollarSign, Calendar } from 'lucide-react';

export default function PendingOrders({ orders, summary }) {
    const formatCurrency = (amount) => {
        return new Intl.NumberFormat('en-US', {
            style: 'currency',
            currency: 'USD'
        }).format(amount);
    };

    const formatDate = (dateString) => {
        return new Date(dateString).toLocaleDateString('en-US', {
            year: 'numeric',
            month: 'short',
            day: 'numeric'
        });
    };

    const getStatusColor = (status) => {
        const colors = {
            'assigned': 'bg-blue-100 text-blue-800',
            'in_progress': 'bg-yellow-100 text-yellow-800',
            'submitted': 'bg-purple-100 text-purple-800',
            'under_review': 'bg-orange-100 text-orange-800',
            'revision_requested': 'bg-red-100 text-red-800'
        };
        return colors[status] || 'bg-gray-100 text-gray-800';
    };

    const getStatusLabel = (status) => {
        const labels = {
            'assigned': 'Assigned',
            'in_progress': 'In Progress',
            'submitted': 'Submitted',
            'under_review': 'Under Review',
            'revision_requested': 'Revision Requested'
        };
        return labels[status] || status;
    };

    return (
        <WriterLayout>
            <Head title="Pending Payments" />
            
            <div className="p-6 max-w-7xl mx-auto">
                {/* Header */}
                <div className="mb-6">
                    <div className="flex items-center gap-4 mb-4">
                        <Link
                            href="/writer/payment/dashboard"
                            className="flex items-center gap-2 text-gray-600 hover:text-gray-800 transition-colors"
                        >
                            <ArrowLeft size={20} />
                            <span>Back to Payment Dashboard</span>
                        </Link>
                    </div>
                    
                    <div className="flex items-center gap-3 mb-6">
                        <div className="p-3 bg-yellow-100 rounded-full">
                            <Clock size={24} className="text-yellow-600" />
                        </div>
                        <div>
                            <h1 className="text-2xl font-bold text-gray-900">Pending Payments</h1>
                            <p className="text-gray-600">
                                Orders in progress - payments will be available after approval
                            </p>
                        </div>
                    </div>

                    {/* Summary Cards */}
                    <div className="grid grid-cols-1 md:grid-cols-3 gap-4 mb-6">
                        <div className="bg-white rounded-lg border p-4">
                            <div className="flex items-center justify-between">
                                <div>
                                    <p className="text-sm text-gray-600">Total Pending</p>
                                    <p className="text-2xl font-bold text-yellow-600">
                                        {formatCurrency(summary.pending_amount)}
                                    </p>
                                </div>
                                <Clock size={24} className="text-yellow-600" />
                            </div>
                        </div>
                        
                        <div className="bg-white rounded-lg border p-4">
                            <div className="flex items-center justify-between">
                                <div>
                                    <p className="text-sm text-gray-600">Pending Orders</p>
                                    <p className="text-2xl font-bold text-gray-900">
                                        {summary.pending_orders_count}
                                    </p>
                                </div>
                                <FileText size={24} className="text-gray-600" />
                            </div>
                        </div>
                        
                        <div className="bg-white rounded-lg border p-4">
                            <div className="flex items-center justify-between">
                                <div>
                                    <p className="text-sm text-gray-600">Available Amount</p>
                                    <p className="text-2xl font-bold text-green-600">
                                        {formatCurrency(summary.available_amount)}
                                    </p>
                                </div>
                                <DollarSign size={24} className="text-green-600" />
                            </div>
                        </div>
                    </div>
                </div>

                {/* Orders Table */}
                <div className="bg-white rounded-lg border overflow-hidden">
                    <div className="px-6 py-4 border-b">
                        <h2 className="text-lg font-semibold text-gray-900">Pending Orders</h2>
                    </div>
                    
                    {orders.length === 0 ? (
                        <div className="p-8 text-center">
                            <Clock size={48} className="mx-auto text-gray-400 mb-4" />
                            <h3 className="text-lg font-medium text-gray-900 mb-2">
                                No pending payments
                            </h3>
                            <p className="text-gray-600">
                                You don't have any orders with pending payments at the moment.
                            </p>
                        </div>
                    ) : (
                        <>
                            {/* Desktop Table */}
                            <div className="hidden md:block overflow-x-auto">
                                <table className="min-w-full divide-y divide-gray-200">
                                    <thead className="bg-gray-50">
                                        <tr>
                                            <th className="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">
                                                Order
                                            </th>
                                            <th className="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">
                                                Client
                                            </th>
                                            <th className="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">
                                                Status
                                            </th>
                                            <th className="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">
                                                Amount
                                            </th>
                                            <th className="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">
                                                Deadline
                                            </th>
                                            <th className="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">
                                                Actions
                                            </th>
                                        </tr>
                                    </thead>
                                    <tbody className="bg-white divide-y divide-gray-200">
                                        {orders.map((order) => (
                                            <tr key={order.id} className="hover:bg-gray-50">
                                                <td className="px-6 py-4 whitespace-nowrap">
                                                    <div>
                                                        <div className="text-sm font-medium text-gray-900">
                                                            #{order.order_number}
                                                        </div>
                                                        <div className="text-sm text-gray-500 truncate max-w-xs">
                                                            {order.title}
                                                        </div>
                                                    </div>
                                                </td>
                                                <td className="px-6 py-4 whitespace-nowrap">
                                                    <div className="text-sm text-gray-900">
                                                        {order.user?.name || 'N/A'}
                                                    </div>
                                                </td>
                                                <td className="px-6 py-4 whitespace-nowrap">
                                                    <span className={`inline-flex px-2 py-1 text-xs font-semibold rounded-full ${getStatusColor(order.order_status)}`}>
                                                        {getStatusLabel(order.order_status)}
                                                    </span>
                                                </td>
                                                <td className="px-6 py-4 whitespace-nowrap">
                                                    <div className="text-sm font-medium text-gray-900">
                                                        {formatCurrency(order.writer_amount)}
                                                    </div>
                                                </td>
                                                <td className="px-6 py-4 whitespace-nowrap">
                                                    <div className="text-sm text-gray-900">
                                                        {formatDate(order.writer_deadline)}
                                                    </div>
                                                </td>
                                                <td className="px-6 py-4 whitespace-nowrap text-sm font-medium">
                                                    <Link
                                                        href={`/writer/orders/${order.id}`}
                                                        className="text-blue-600 hover:text-blue-900"
                                                    >
                                                        View Order
                                                    </Link>
                                                </td>
                                            </tr>
                                        ))}
                                    </tbody>
                                </table>
                            </div>

                            {/* Mobile Cards */}
                            <div className="md:hidden">
                                {orders.map((order) => (
                                    <div key={order.id} className="p-4 border-b last:border-b-0">
                                        <div className="flex justify-between items-start mb-2">
                                            <div>
                                                <h3 className="font-medium text-gray-900">
                                                    #{order.order_number}
                                                </h3>
                                                <p className="text-sm text-gray-600 mt-1">
                                                    {order.title}
                                                </p>
                                            </div>
                                            <span className={`inline-flex px-2 py-1 text-xs font-semibold rounded-full ${getStatusColor(order.order_status)}`}>
                                                {getStatusLabel(order.order_status)}
                                            </span>
                                        </div>
                                        
                                        <div className="grid grid-cols-2 gap-4 mt-3 text-sm">
                                            <div>
                                                <p className="text-gray-600">Client</p>
                                                <p className="font-medium">{order.user?.name || 'N/A'}</p>
                                            </div>
                                            <div>
                                                <p className="text-gray-600">Amount</p>
                                                <p className="font-medium">{formatCurrency(order.writer_amount)}</p>
                                            </div>
                                            <div>
                                                <p className="text-gray-600">Deadline</p>
                                                <p className="font-medium">{formatDate(order.writer_deadline)}</p>
                                            </div>
                                            <div className="flex items-end">
                                                <Link
                                                    href={`/writer/orders/${order.id}`}
                                                    className="text-blue-600 hover:text-blue-900 text-sm font-medium"
                                                >
                                                    View Order
                                                </Link>
                                            </div>
                                        </div>
                                    </div>
                                ))}
                            </div>
                        </>
                    )}
                </div>
            </div>
        </WriterLayout>
    );
}