import React, { useState, useRef } from "react";
import { Head, useForm } from "@inertiajs/react";
import WriterLayout from "@/Layouts/WriterLayout";
import {
    User,
    Mail,
    Phone,
    MapPin,
    GraduationCap,
    Briefcase,
    Award,
    Clock,
    DollarSign,
    FileText,
    Camera,
    Save,
    AlertCircle,
    CheckCircle,
    Upload,
    X,
    Eye,
    EyeOff,
    CreditCard,
} from "lucide-react";
import { toast } from "react-hot-toast";

export default function EditProfile({
    profile,
    user,
    qualificationStatus,
    isFullyQualified,
    educationLevels,
    subjects,
    languages,
    citationStyles,
    writerLevels,
    specializations,
    paymentMethods,
    maxSubjects = 5,
    maxLanguages = 3,
    maxCitationStyles = 5,
}) {
    // Safety check for profile
    if (!profile) {
        return (
            <WriterLayout>
                <div className="flex items-center justify-center min-h-screen">
                    <div className="animate-spin rounded-full h-12 w-12 border-b-2 border-blue-500"></div>
                </div>
            </WriterLayout>
        );
    }

    const [imagePreview, setImagePreview] = useState(
        profile.profile_image ? `/storage/${profile.profile_image}` : null
    );
    const [isUploading, setIsUploading] = useState(false);
    const [showPaymentDetails, setShowPaymentDetails] = useState(false);
    const fileInputRef = useRef(null);

    // Initialize form with existing data
    const { data, setData, put, patch, processing, errors, reset, isDirty } =
        useForm({
            name: user?.name || "",
            bio: profile?.bio || "",
            education_level_id: profile?.education_level_id || "",
            specialization_id: profile?.specialization_id || "",
            writer_level_id: profile?.writer_level_id || "",
            subjects: profile?.subjects?.map((subject) => subject.id) || [],
            languages: profile?.languages?.map((language) => language.id) || [],
            citation_styles:
                profile?.citationStyles?.map((style) => style.id) || [],
            hourly_rate: profile?.hourly_rate || "",
            is_available: profile?.is_available || false,
            payment_method_id:
                profile?.paymentAccounts?.[0]?.payment_method_id || "",
            payment_account_details:
                profile?.paymentAccounts?.[0]?.account_details || "",
            experience_years: profile?.experience_years || "",
        });

    const [isSubmitting, setIsSubmitting] = useState(false);

    // Form submit handler
    const handleSubmit = (e) => {
        e.preventDefault();
        setIsSubmitting(true);

        patch(route("writer.profile.update"), {
            onFinish: () => setIsSubmitting(false),
            onSuccess: () => {
                toast.success("Profile updated successfully!");
            },
            onError: (errors) => {
                toast.error(
                    "Failed to update profile. Please check the form for errors."
                );
                console.error("Form errors:", errors);
            },
        });
    };

    // Handle profile image upload
    const handleImageUpload = async (event) => {
        const file = event.target.files[0];
        if (!file) return;

        // Validate file
        if (!file.type.startsWith("image/")) {
            toast.error("Please select a valid image file.");
            return;
        }

        if (file.size > 2 * 1024 * 1024) {
            // 2MB limit
            toast.error("Image size must be less than 2MB.");
            return;
        }

        setIsUploading(true);

        const formData = new FormData();
        formData.append("profile_image", file);

        try {
            const response = await axios.post(
                route("writer.profile.upload-image"),
                formData,
                {
                    headers: {
                        "Content-Type": "multipart/form-data",
                    },
                }
            );

            if (response.data.success) {
                setImagePreview(response.data.image_url);
                toast.success("Profile image updated successfully!");
            }
        } catch (error) {
            console.error("Upload error:", error);
            toast.error("Failed to upload image. Please try again.");
        } finally {
            setIsUploading(false);
        }
    };

    // Handle array field changes
    const handleArrayChange = (field, value, maxItems) => {
        const currentArray = data[field] || [];
        let newArray;

        if (currentArray.includes(value)) {
            newArray = currentArray.filter((item) => item !== value);
        } else {
            if (currentArray.length >= maxItems) {
                toast.error(`You can select maximum ${maxItems} items.`);
                return;
            }
            newArray = [...currentArray, value];
        }

        setData(field, newArray);
    };

    // Get qualification status badge
    const getQualificationBadge = () => {
        if (isFullyQualified) {
            return (
                <span className="inline-flex items-center px-3 py-1 rounded-full text-sm font-medium bg-green-100 text-green-800 dark:bg-green-900 dark:text-green-200">
                    <CheckCircle className="w-4 h-4 mr-1" />
                    Fully Qualified
                </span>
            );
        }
        return (
            <span className="inline-flex items-center px-3 py-1 rounded-full text-sm font-medium bg-yellow-100 text-yellow-800 dark:bg-yellow-900 dark:text-yellow-200">
                <AlertCircle className="w-4 h-4 mr-1" />
                Qualification Pending
            </span>
        );
    };

    return (
        <WriterLayout>
            <Head title="Edit Profile" />

            <div className="max-w-4xl mx-auto px-4 sm:px-6 lg:px-8 py-8">
                {/* Header */}
                <div className="mb-8">
                    <div className="flex items-center justify-between">
                        <div>
                            <h1 className="text-3xl font-bold text-gray-900 dark:text-white">
                                Edit Profile
                            </h1>
                            <p className="mt-2 text-gray-600 dark:text-gray-400">
                                Update your profile information to attract more
                                clients
                            </p>
                        </div>
                        {getQualificationBadge()}
                    </div>
                </div>

                {/* Qualification Notice */}
                {!isFullyQualified && (
                    <div className="mb-6 bg-yellow-50 dark:bg-yellow-900/20 border border-yellow-200 dark:border-yellow-800 rounded-lg p-4">
                        <div className="flex">
                            <AlertCircle className="h-5 w-5 text-yellow-400 mr-3 flex-shrink-0" />
                            <div>
                                <h3 className="text-sm font-medium text-yellow-800 dark:text-yellow-200">
                                    Limited Profile Access
                                </h3>
                                <p className="mt-1 text-sm text-yellow-700 dark:text-yellow-300">
                                    Complete your qualification tests to unlock
                                    all profile features and start bidding on
                                    orders.
                                </p>
                            </div>
                        </div>
                    </div>
                )}

                <form onSubmit={handleSubmit} className="space-y-8">
                    {/* Profile Image Section */}
                    <div className="bg-white dark:bg-gray-800 shadow rounded-lg p-6">
                        <h2 className="text-lg font-medium text-gray-900 dark:text-white mb-4 flex items-center">
                            <Camera className="w-5 h-5 mr-2" />
                            Profile Image
                        </h2>

                        <div className="flex items-center space-x-6">
                            <div className="relative">
                                <div className="w-24 h-24 rounded-full overflow-hidden bg-gray-100 dark:bg-gray-700">
                                    {imagePreview ? (
                                        <img
                                            src={imagePreview}
                                            alt="Profile"
                                            className="w-full h-full object-cover"
                                        />
                                    ) : (
                                        <div className="w-full h-full flex items-center justify-center">
                                            <User className="w-8 h-8 text-gray-400" />
                                        </div>
                                    )}
                                </div>
                                {isUploading && (
                                    <div className="absolute inset-0 bg-black bg-opacity-50 rounded-full flex items-center justify-center">
                                        <div className="animate-spin rounded-full h-6 w-6 border-b-2 border-white"></div>
                                    </div>
                                )}
                            </div>

                            <div>
                                <button
                                    type="button"
                                    onClick={() =>
                                        fileInputRef.current?.click()
                                    }
                                    disabled={isUploading}
                                    className="inline-flex items-center px-4 py-2 border border-gray-300 dark:border-gray-600 rounded-md shadow-sm text-sm font-medium text-gray-700 dark:text-gray-300 bg-white dark:bg-gray-700 hover:bg-gray-50 dark:hover:bg-gray-600 focus:outline-none focus:ring-2 focus:ring-offset-2 focus:ring-indigo-500 disabled:opacity-50"
                                >
                                    <Upload className="w-4 h-4 mr-2" />
                                    {isUploading
                                        ? "Uploading..."
                                        : "Upload Photo"}
                                </button>
                                <input
                                    ref={fileInputRef}
                                    type="file"
                                    accept="image/*"
                                    onChange={handleImageUpload}
                                    className="hidden"
                                />
                                <p className="mt-2 text-xs text-gray-500 dark:text-gray-400">
                                    JPG, PNG, GIF up to 2MB
                                </p>
                            </div>
                        </div>
                    </div>

                    {/* Basic Information */}
                    <div className="bg-white dark:bg-gray-800 shadow rounded-lg p-6">
                        <h2 className="text-lg font-medium text-gray-900 dark:text-white mb-4 flex items-center">
                            <User className="w-5 h-5 mr-2" />
                            Basic Information
                        </h2>

                        <div className="grid grid-cols-1 md:grid-cols-2 gap-6">
                            <div>
                                <label className="block text-sm font-medium text-gray-700 dark:text-gray-300 mb-2">
                                    Full Name
                                </label>
                                <input
                                    type="text"
                                    value={data.name}
                                    onChange={(e) =>
                                        setData("name", e.target.value)
                                    }
                                    className="w-full px-3 py-2 border border-gray-300 dark:border-gray-600 rounded-md shadow-sm focus:outline-none focus:ring-indigo-500 focus:border-indigo-500 dark:bg-gray-700 dark:text-white"
                                    disabled
                                />
                                <p className="mt-1 text-xs text-gray-500 dark:text-gray-400">
                                    Contact support to change your name
                                </p>
                            </div>



                            <div className="md:col-span-2">
                                <label className="block text-sm font-medium text-gray-700 dark:text-gray-300 mb-2">
                                    Bio *
                                </label>
                                <textarea
                                    value={data.bio}
                                    onChange={(e) =>
                                        setData("bio", e.target.value)
                                    }
                                    rows={4}
                                    className="w-full px-3 py-2 border border-gray-300 dark:border-gray-600 rounded-md shadow-sm focus:outline-none focus:ring-indigo-500 focus:border-indigo-500 dark:bg-gray-700 dark:text-white"
                                    placeholder="Tell clients about yourself, your experience, and what makes you unique..."
                                />
                                <div className="mt-1 flex justify-between">
                                    {errors.bio && (
                                        <p className="text-sm text-red-600 dark:text-red-400">
                                            {errors.bio}
                                        </p>
                                    )}
                                    <p className="text-xs text-gray-500 dark:text-gray-400">
                                        {data.bio.length}/1000 characters
                                    </p>
                                </div>
                            </div>

                            <div>
                                <label className="block text-sm font-medium text-gray-700 dark:text-gray-300 mb-2">
                                    Years of Experience *
                                </label>
                                <input
                                    type="number"
                                    min="0"
                                    max="50"
                                    value={data.experience_years}
                                    onChange={(e) =>
                                        setData(
                                            "experience_years",
                                            parseInt(e.target.value) || 0
                                        )
                                    }
                                    className="w-full px-3 py-2 border border-gray-300 dark:border-gray-600 rounded-md shadow-sm focus:outline-none focus:ring-indigo-500 focus:border-indigo-500 dark:bg-gray-700 dark:text-white"
                                />
                                {errors.experience_years && (
                                    <p className="mt-1 text-sm text-red-600 dark:text-red-400">
                                        {errors.experience_years}
                                    </p>
                                )}
                            </div>
                        </div>
                    </div>

                    {/* Professional Information - Only for qualified writers */}
                    {isFullyQualified && (
                        <div className="bg-white dark:bg-gray-800 shadow rounded-lg p-6">
                            <h2 className="text-lg font-medium text-gray-900 dark:text-white mb-4 flex items-center">
                                <Briefcase className="w-5 h-5 mr-2" />
                                Professional Information
                            </h2>

                            <div className="grid grid-cols-1 md:grid-cols-2 gap-6">
                                <div>
                                    <label className="block text-sm font-medium text-gray-700 dark:text-gray-300 mb-2">
                                        Education Level *
                                    </label>
                                    <select
                                        value={data.education_level_id}
                                        onChange={(e) =>
                                            setData(
                                                "education_level_id",
                                                e.target.value
                                            )
                                        }
                                        className="w-full px-3 py-2 border border-gray-300 dark:border-gray-600 rounded-md shadow-sm focus:outline-none focus:ring-indigo-500 focus:border-indigo-500 dark:bg-gray-700 dark:text-white"
                                    >
                                        <option value="">
                                            Select Education Level
                                        </option>
                                        {educationLevels.map((level) => (
                                            <option
                                                key={level.id}
                                                value={level.id}
                                            >
                                                {level.name}
                                            </option>
                                        ))}
                                    </select>
                                    {errors.education_level_id && (
                                        <p className="mt-1 text-sm text-red-600 dark:text-red-400">
                                            {errors.education_level_id}
                                        </p>
                                    )}
                                </div>

                                <div>
                                    <label className="block text-sm font-medium text-gray-700 dark:text-gray-300 mb-2">
                                        Specialization *
                                    </label>
                                    <select
                                        value={data.specialization_id}
                                        onChange={(e) =>
                                            setData(
                                                "specialization_id",
                                                e.target.value
                                            )
                                        }
                                        className="w-full px-3 py-2 border border-gray-300 dark:border-gray-600 rounded-md shadow-sm focus:outline-none focus:ring-indigo-500 focus:border-indigo-500 dark:bg-gray-700 dark:text-white"
                                    >
                                        <option value="">
                                            Select Specialization
                                        </option>
                                        {specializations.map((spec) => (
                                            <option
                                                key={spec.id}
                                                value={spec.id}
                                            >
                                                {spec.name}
                                            </option>
                                        ))}
                                    </select>
                                    {errors.specialization_id && (
                                        <p className="mt-1 text-sm text-red-600 dark:text-red-400">
                                            {errors.specialization_id}
                                        </p>
                                    )}
                                </div>

                                <div>
                                    <label className="block text-sm font-medium text-gray-700 dark:text-gray-300 mb-2">
                                        Writer Level
                                    </label>
                                    <select
                                        value={data.writer_level_id}
                                        onChange={(e) =>
                                            setData(
                                                "writer_level_id",
                                                e.target.value
                                            )
                                        }
                                        className="w-full px-3 py-2 border border-gray-300 dark:border-gray-600 rounded-md shadow-sm focus:outline-none focus:ring-indigo-500 focus:border-indigo-500 dark:bg-gray-700 dark:text-white"
                                    >
                                        <option value="">
                                            Select Writer Level
                                        </option>
                                        {writerLevels.map((level) => (
                                            <option
                                                key={level.id}
                                                value={level.id}
                                            >
                                                {level.name}
                                            </option>
                                        ))}
                                    </select>
                                    {errors.writer_level_id && (
                                        <p className="mt-1 text-sm text-red-600 dark:text-red-400">
                                            {errors.writer_level_id}
                                        </p>
                                    )}
                                </div>

                                <div>
                                    <label className="block text-sm font-medium text-gray-700 dark:text-gray-300 mb-2">
                                        Hourly Rate (USD)
                                    </label>
                                    <div className="relative">
                                        <DollarSign className="absolute left-3 top-1/2 transform -translate-y-1/2 text-gray-400 w-4 h-4" />
                                        <input
                                            type="number"
                                            min="0"
                                            max="200"
                                            step="0.01"
                                            value={data.hourly_rate}
                                            onChange={(e) =>
                                                setData(
                                                    "hourly_rate",
                                                    e.target.value
                                                )
                                            }
                                            className="w-full pl-10 pr-3 py-2 border border-gray-300 dark:border-gray-600 rounded-md shadow-sm focus:outline-none focus:ring-indigo-500 focus:border-indigo-500 dark:bg-gray-700 dark:text-white"
                                            placeholder="25.00"
                                        />
                                    </div>
                                    {errors.hourly_rate && (
                                        <p className="mt-1 text-sm text-red-600 dark:text-red-400">
                                            {errors.hourly_rate}
                                        </p>
                                    )}
                                </div>
                            </div>

                            {/* Availability Toggle */}
                            <div className="mt-6">
                                <div className="flex items-center">
                                    <input
                                        type="checkbox"
                                        id="is_available"
                                        checked={data.is_available}
                                        onChange={(e) =>
                                            setData(
                                                "is_available",
                                                e.target.checked
                                            )
                                        }
                                        className="h-4 w-4 text-indigo-600 focus:ring-indigo-500 border-gray-300 rounded"
                                    />
                                    <label
                                        htmlFor="is_available"
                                        className="ml-2 block text-sm text-gray-900 dark:text-white"
                                    >
                                        Available for new orders
                                    </label>
                                </div>
                                <p className="mt-1 text-xs text-gray-500 dark:text-gray-400">
                                    When checked, you'll be visible to clients
                                    and can receive new order invitations
                                </p>
                            </div>
                        </div>
                    )}

                    {/* Subjects - Only for qualified writers */}
                    {isFullyQualified && (
                        <div className="bg-white dark:bg-gray-800 shadow rounded-lg p-6">
                            <h2 className="text-lg font-medium text-gray-900 dark:text-white mb-4 flex items-center">
                                <GraduationCap className="w-5 h-5 mr-2" />
                                Subjects ({data.subjects.length}/{maxSubjects})
                            </h2>

                            <div className="grid grid-cols-2 md:grid-cols-3 lg:grid-cols-4 gap-3">
                                {subjects.map((subject) => (
                                    <label
                                        key={subject.id}
                                        className={`relative flex items-center p-3 rounded-lg border cursor-pointer hover:bg-gray-50 dark:hover:bg-gray-700 transition-colors ${
                                            data.subjects.includes(subject.id)
                                                ? "border-indigo-500 bg-indigo-50 dark:bg-indigo-900/20"
                                                : "border-gray-300 dark:border-gray-600"
                                        }`}
                                    >
                                        <input
                                            type="checkbox"
                                            checked={data.subjects.includes(
                                                subject.id
                                            )}
                                            onChange={() =>
                                                handleArrayChange(
                                                    "subjects",
                                                    subject.id,
                                                    maxSubjects
                                                )
                                            }
                                            className="sr-only"
                                        />
                                        <span className="text-sm font-medium text-gray-900 dark:text-white">
                                            {subject.name}
                                        </span>
                                        {data.subjects.includes(subject.id) && (
                                            <CheckCircle className="ml-auto w-4 h-4 text-indigo-500" />
                                        )}
                                    </label>
                                ))}
                            </div>
                            {errors.subjects && (
                                <p className="mt-2 text-sm text-red-600 dark:text-red-400">
                                    {errors.subjects}
                                </p>
                            )}
                        </div>
                    )}

                    {/* Languages - Only for qualified writers */}
                    {isFullyQualified && (
                        <div className="bg-white dark:bg-gray-800 shadow rounded-lg p-6">
                            <h2 className="text-lg font-medium text-gray-900 dark:text-white mb-4 flex items-center">
                                <MapPin className="w-5 h-5 mr-2" />
                                Languages ({data.languages.length}/
                                {maxLanguages})
                            </h2>

                            <div className="grid grid-cols-2 md:grid-cols-3 lg:grid-cols-4 gap-3">
                                {languages.map((language) => (
                                    <label
                                        key={language.id}
                                        className={`relative flex items-center p-3 rounded-lg border cursor-pointer hover:bg-gray-50 dark:hover:bg-gray-700 transition-colors ${
                                            data.languages.includes(language.id)
                                                ? "border-indigo-500 bg-indigo-50 dark:bg-indigo-900/20"
                                                : "border-gray-300 dark:border-gray-600"
                                        }`}
                                    >
                                        <input
                                            type="checkbox"
                                            checked={data.languages.includes(
                                                language.id
                                            )}
                                            onChange={() =>
                                                handleArrayChange(
                                                    "languages",
                                                    language.id,
                                                    maxLanguages
                                                )
                                            }
                                            className="sr-only"
                                        />
                                        <span className="text-sm font-medium text-gray-900 dark:text-white">
                                            {language.name}
                                        </span>
                                        {data.languages.includes(
                                            language.id
                                        ) && (
                                            <CheckCircle className="ml-auto w-4 h-4 text-indigo-500" />
                                        )}
                                    </label>
                                ))}
                            </div>
                            {errors.languages && (
                                <p className="mt-2 text-sm text-red-600 dark:text-red-400">
                                    {errors.languages}
                                </p>
                            )}
                        </div>
                    )}

                    {/* Citation Styles - Only for qualified writers */}
                    {isFullyQualified && (
                        <div className="bg-white dark:bg-gray-800 shadow rounded-lg p-6">
                            <h2 className="text-lg font-medium text-gray-900 dark:text-white mb-4 flex items-center">
                                <FileText className="w-5 h-5 mr-2" />
                                Citation Styles ({data.citation_styles.length}/
                                {maxCitationStyles})
                            </h2>

                            <div className="grid grid-cols-2 md:grid-cols-3 lg:grid-cols-4 gap-3">
                                {citationStyles.map((style) => (
                                    <label
                                        key={style.id}
                                        className={`relative flex items-center p-3 rounded-lg border cursor-pointer hover:bg-gray-50 dark:hover:bg-gray-700 transition-colors ${
                                            data.citation_styles.includes(
                                                style.id
                                            )
                                                ? "border-indigo-500 bg-indigo-50 dark:bg-indigo-900/20"
                                                : "border-gray-300 dark:border-gray-600"
                                        }`}
                                    >
                                        <input
                                            type="checkbox"
                                            checked={data.citation_styles.includes(
                                                style.id
                                            )}
                                            onChange={() =>
                                                handleArrayChange(
                                                    "citation_styles",
                                                    style.id,
                                                    maxCitationStyles
                                                )
                                            }
                                            className="sr-only"
                                        />
                                        <span className="text-sm font-medium text-gray-900 dark:text-white">
                                            {style.name}
                                        </span>
                                        {data.citation_styles.includes(
                                            style.id
                                        ) && (
                                            <CheckCircle className="ml-auto w-4 h-4 text-indigo-500" />
                                        )}
                                    </label>
                                ))}
                            </div>
                            {errors.citation_styles && (
                                <p className="mt-2 text-sm text-red-600 dark:text-red-400">
                                    {errors.citation_styles}
                                </p>
                            )}
                        </div>
                    )}

                    {/* Payment Information - Only for qualified writers */}
                    {isFullyQualified && (
                        <div className="bg-white dark:bg-gray-800 shadow rounded-lg p-6">
                            <h2 className="text-lg font-medium text-gray-900 dark:text-white mb-4 flex items-center">
                                <CreditCard className="w-5 h-5 mr-2" />
                                Payment Information
                            </h2>

                            <div className="grid grid-cols-1 md:grid-cols-2 gap-6">
                                <div>
                                    <label className="block text-sm font-medium text-gray-700 dark:text-gray-300 mb-2">
                                        Payment Method
                                    </label>
                                    <select
                                        value={data.payment_method_id}
                                        onChange={(e) =>
                                            setData(
                                                "payment_method_id",
                                                e.target.value
                                            )
                                        }
                                        className="w-full px-3 py-2 border border-gray-300 dark:border-gray-600 rounded-md shadow-sm focus:outline-none focus:ring-indigo-500 focus:border-indigo-500 dark:bg-gray-700 dark:text-white"
                                    >
                                        <option value="">
                                            Select Payment Method
                                        </option>
                                        {paymentMethods.map((method) => (
                                            <option
                                                key={method.id}
                                                value={method.id}
                                            >
                                                {method.name}
                                            </option>
                                        ))}
                                    </select>
                                    {errors.payment_method_id && (
                                        <p className="mt-1 text-sm text-red-600 dark:text-red-400">
                                            {errors.payment_method_id}
                                        </p>
                                    )}
                                </div>

                                <div>
                                    <label className="block text-sm font-medium text-gray-700 dark:text-gray-300 mb-2">
                                        Account Details
                                    </label>
                                    <div className="relative">
                                        <input
                                            type={
                                                showPaymentDetails
                                                    ? "text"
                                                    : "password"
                                            }
                                            value={data.payment_account_details}
                                            onChange={(e) =>
                                                setData(
                                                    "payment_account_details",
                                                    e.target.value
                                                )
                                            }
                                            className="w-full px-3 py-2 pr-10 border border-gray-300 dark:border-gray-600 rounded-md shadow-sm focus:outline-none focus:ring-indigo-500 focus:border-indigo-500 dark:bg-gray-700 dark:text-white"
                                            placeholder="Enter your payment account details"
                                        />
                                        <button
                                            type="button"
                                            onClick={() =>
                                                setShowPaymentDetails(
                                                    !showPaymentDetails
                                                )
                                            }
                                            className="absolute inset-y-0 right-0 pr-3 flex items-center"
                                        >
                                            {showPaymentDetails ? (
                                                <EyeOff className="h-4 w-4 text-gray-400" />
                                            ) : (
                                                <Eye className="h-4 w-4 text-gray-400" />
                                            )}
                                        </button>
                                    </div>
                                    {errors.payment_account_details && (
                                        <p className="mt-1 text-sm text-red-600 dark:text-red-400">
                                            {errors.payment_account_details}
                                        </p>
                                    )}
                                    <p className="mt-1 text-xs text-gray-500 dark:text-gray-400">
                                        Your payment information is encrypted
                                        and secure
                                    </p>
                                </div>
                            </div>

                            {/* Payment Security Notice */}
                            <div className="mt-4 bg-blue-50 dark:bg-blue-900/20 border border-blue-200 dark:border-blue-800 rounded-lg p-4">
                                <div className="flex">
                                    <AlertCircle className="h-5 w-5 text-blue-400 mr-3 flex-shrink-0" />
                                    <div>
                                        <h3 className="text-sm font-medium text-blue-800 dark:text-blue-200">
                                            Payment Security
                                        </h3>
                                        <p className="mt-1 text-sm text-blue-700 dark:text-blue-300">
                                            Your payment details are securely
                                            stored and will be verified by our
                                            admin team before activation.
                                        </p>
                                    </div>
                                </div>
                            </div>
                        </div>
                    )}

                    {/* Form Actions */}
                    <div className="bg-white dark:bg-gray-800 shadow rounded-lg p-6">
                        <div className="flex items-center justify-between">
                            <div className="flex items-center space-x-4">
                                {isDirty && (
                                    <div className="flex items-center text-amber-600 dark:text-amber-400">
                                        <AlertCircle className="w-4 h-4 mr-1" />
                                        <span className="text-sm">
                                            You have unsaved changes
                                        </span>
                                    </div>
                                )}
                            </div>

                            <div className="flex items-center space-x-3">
                                <button
                                    type="button"
                                    onClick={() => reset()}
                                    disabled={!isDirty || isSubmitting}
                                    className="inline-flex items-center px-4 py-2 border border-gray-300 dark:border-gray-600 rounded-md shadow-sm text-sm font-medium text-gray-700 dark:text-gray-300 bg-white dark:bg-gray-700 hover:bg-gray-50 dark:hover:bg-gray-600 focus:outline-none focus:ring-2 focus:ring-offset-2 focus:ring-indigo-500 disabled:opacity-50 disabled:cursor-not-allowed"
                                >
                                    <X className="w-4 h-4 mr-2" />
                                    Reset Changes
                                </button>

                                <button
                                    type="submit"
                                    disabled={isSubmitting || !isDirty}
                                    className="inline-flex items-center px-4 py-2 bg-indigo-600 border border-transparent rounded-md font-semibold text-xs text-white uppercase tracking-widest hover:bg-indigo-700 focus:bg-indigo-700 active:bg-indigo-900 focus:outline-none focus:ring-2 focus:ring-indigo-500 focus:ring-offset-2 dark:focus:ring-offset-gray-800 transition ease-in-out duration-150 disabled:opacity-50 disabled:cursor-not-allowed"
                                >
                                    {isSubmitting ? (
                                        <svg
                                            className="animate-spin -ml-1 mr-2 h-4 w-4 text-white"
                                            xmlns="http://www.w3.org/2000/svg"
                                            fill="none"
                                            viewBox="0 0 24 24"
                                        >
                                            <circle
                                                className="opacity-25"
                                                cx="12"
                                                cy="12"
                                                r="10"
                                                stroke="currentColor"
                                                strokeWidth="4"
                                            ></circle>
                                            <path
                                                className="opacity-75"
                                                fill="currentColor"
                                                d="M4 12a8 8 0 018-8V0C5.373 0 0 5.373 0 12h4zm2 5.291A7.962 7.962 0 014 12H0c0 3.042 1.135 5.824 3 7.938l3-2.647z"
                                            ></path>
                                        </svg>
                                    ) : (
                                        <Save className="w-4 h-4 mr-2" />
                                    )}
                                    {isSubmitting
                                        ? "Updating..."
                                        : "Save Changes"}
                                </button>
                            </div>
                        </div>
                    </div>

                    {/* Profile Completion Tips */}
                    <div className="bg-gradient-to-r from-indigo-50 to-blue-50 dark:from-indigo-900/20 dark:to-blue-900/20 border border-indigo-200 dark:border-indigo-800 rounded-lg p-6">
                        <h3 className="text-lg font-medium text-indigo-900 dark:text-indigo-200 mb-4">
                            💡 Profile Tips
                        </h3>
                        <ul className="space-y-2 text-sm text-indigo-800 dark:text-indigo-300">
                            <li className="flex items-start">
                                <CheckCircle className="w-4 h-4 mr-2 mt-0.5 text-green-500" />
                                Write a compelling bio that highlights your
                                expertise and experience
                            </li>
                            <li className="flex items-start">
                                <CheckCircle className="w-4 h-4 mr-2 mt-0.5 text-green-500" />
                                Upload a professional profile photo to build
                                trust with clients
                            </li>
                            <li className="flex items-start">
                                <CheckCircle className="w-4 h-4 mr-2 mt-0.5 text-green-500" />
                                Select subjects you're genuinely passionate
                                about and skilled in
                            </li>
                            <li className="flex items-start">
                                <CheckCircle className="w-4 h-4 mr-2 mt-0.5 text-green-500" />
                                Set competitive hourly rates based on your
                                experience level
                            </li>
                            <li className="flex items-start">
                                <CheckCircle className="w-4 h-4 mr-2 mt-0.5 text-green-500" />
                                Keep your availability status updated to receive
                                relevant orders
                            </li>
                        </ul>
                    </div>
                </form>
            </div>
        </WriterLayout>
    );
}
