import PayPalButton from "@/Components/PayPalButton";
import React, { useState, useEffect } from "react";
import { usePage, Head } from "@inertiajs/react";
import WriterLayout from "@/Layouts/WriterLayout";
import {
    FileText,
    Clock,
    User,
    Download,
    Upload,
    Notebook,
    Timer,
    MoveLeft,
    CircleStop,
} from "lucide-react";
import OrderPostedTime from "@/Components/OrderPostedTime";

import ClearDeadlineCountdown from "@/Components/ClearDeadlineCountdown";
import WriterFileUpload from "@/Components/WriterFileUpload";
import FileManagement from "@/Components/FileManagement";
import { Link, useForm } from "@inertiajs/react";

export default function Show({ order, isAdmin, existingBid }) {
    const { auth } = usePage().props;
    const { flash } = usePage().props;
    const [flashMsg, setFlashMsg] = useState(flash.message);
    const [currentOrder, setCurrentOrder] = useState(order);

    const { delete: destroy } = useForm();

    useEffect(() => {
        if (flashMsg) {
            const timer = setTimeout(() => {
                setFlashMsg("");
            }, 3000);
            return () => clearTimeout(timer);
        }
    }, [flashMsg]);

    function submit(e) {
        e.preventDefault();
        if (confirm("Are you sure you want to delete this job?")) {
            destroy(route("orders.destroy", order.id));
        }
    }

    const capitalize = (string) => {
        return string.charAt(0).toUpperCase() + string.slice(1);
    };

    const [isBiddingModalOpen, setIsBiddingModalOpen] = useState(false);
    const [bidAmount, setBidAmount] = useState(order.writer_amount); // Default to writer_amount
    const [completionTime, setCompletionTime] = useState(order.writer_deadline); // Default to writer_deadline
    const [comments, setComments] = useState("");

    const openBiddingModal = () => {
        setIsBiddingModalOpen(true);
    };

    const closeBiddingModal = () => {
        setIsBiddingModalOpen(false);
        setBidAmount(order.writer_amount); // Reset to default
        setCompletionTime(order.writer_deadline); // Reset to default
        setComments("");
    };

    const handleFileUploaded = (file) => {
        // Update the order files list
        setCurrentOrder(prev => ({
            ...prev,
            order_files: [...(prev.order_files || []), file]
        }));
    };

    const handleOrderStatusUpdate = (newStatus) => {
        // Update the order status
        setCurrentOrder(prev => ({
            ...prev,
            order_status: newStatus
        }));
    };

    const handlePlaceBid = async () => {
        // Validate bid amount
        if (bidAmount <= 0 || bidAmount > order.writer_amount) {
            alert(
                `Bid amount must be greater than 0 and cannot exceed ${order.writer_amount}`
            );
            return;
        }

        // Validate completion time
        if (
            new Date(completionTime) > new Date(order.writer_deadline) ||
            new Date(completionTime) < new Date(order.dateposted)
        ) {
            alert(
                `Completion time must be between the order's posted date (${new Date(
                    order.dateposted
                ).toLocaleString()}) and the writer deadline (${new Date(
                    order.writer_deadline
                ).toLocaleString()}).`
            );
            return;
        }

        try {
            // Change this line to match your route
            const response = await axios.post(
                `/writer/orders/${order.id}/bid`, // Changed from /writer/orders/${order.id}/bids
                {
                    bid_amount: bidAmount,
                    completion_time: completionTime,
                    comments,
                    writer_amount: order.writer_amount,
                    writer_deadline: order.writer_deadline,
                }
            );

            alert(response.data.message);
            window.location.href = "/writer/bids/"; // Redirect to "My Bids" page
        } catch (error) {
            alert(
                error.response?.data?.error ||
                    "An error occurred while placing the bid."
            );
        }
    };

    return (
        <WriterLayout>
            <Head title={order.title} />
            <div className="w-full mx-auto px-6 sm:px-6 lg:px-6 py-2 space-y-6 dark:bg-gray-900 overflow-x-hidden">
                <div className="flex justify-end items-center mb-6">
                    <Link
                        href="/writer/orders"
                        className="flex items-center px-3 py-1.5 sm:px-4 sm:py-2 text-indigo-600 dark:text-gray-100 bg-indigo-100 dark:bg-indigo-900 hover:bg-indigo-100 dark:hover:bg-indigo-800 rounded-full shadow-sm border border-indigo-200 dark:border-indigo-700 transition-colors duration-200 text-sm"
                    >
                        <MoveLeft className="mr-1 h-3 w-3 sm:h-4 sm:w-4" />
                        <span className="font-medium">Available Orders</span>
                    </Link>
                </div>
                <div className="flex flex-wrap gap-6 items-start">
                    {/* Header Section */}
                    <div className="flex-1 space-y-6">
                        <div className="bg-gradient-to-b from-orange-300 to-cyan-300 dark:bg-none  dark:bg-gray-800 rounded-lg shadow-sm p-4 sm:p-6">
                            <div className="flex flex-col sm:flex-row justify-between items-start sm:items-center space-y-4 sm:space-y-0">
                                {/* Order Title and Status */}
                                <div className="flex flex-col sm:flex-row sm:items-center gap-2 sm:gap-4  ">
                                    <h1 className="text-lg sm:text-lg md:text-lg font-bold dark:text-white">
                                        Order #{order.order_number}
                                    </h1>
                                    <div className="flex flex-wrap gap-2">
                                        {/* Order Status */}
                                        <span
                                            className={`px-3 py-1 rounded-full text-xs sm:text-sm font-medium flex items-center ${
                                                currentOrder.order_status ===
                                                "completed"
                                                    ? "bg-blue-100 text-blue-800 dark:bg-blue-800 dark:text-gray-100"
                                                    : currentOrder.order_status ===
                                                      "assigned"
                                                    ? "bg-green-500 text-gray-100 dark:bg-green-800 dark:text-gray-50"
                                                    : currentOrder.order_status ===
                                                      "pending"
                                                    ? "bg-yellow-100 text-yellow-800 dark:bg-yellow-800 dark:text-yellow-100"
                                                    : currentOrder.order_status ===
                                                      "bidding"
                                                    ? "bg-pink-600 text-gray-100 dark:bg-pink-800 dark:text-gray-100"
                                                    : currentOrder.order_status ===
                                                      "in_progress"
                                                    ? "bg-green-500 text-gray-100 dark:bg-green-800 dark:text-gray-50"
                                                    : currentOrder.order_status ===
                                                      "under_review"
                                                    ? "bg-yellow-100 text-yellow-800 dark:bg-yellow-800 dark:text-yellow-100"
                                                    : "bg-yellow-100 text-yellow-800 dark:bg-yellow-800 dark:text-yellow-100"
                                            }`}
                                        >
                                            <Clock className="w-4 h-4 mr-1" />
                                            <span>
                                                {capitalize(currentOrder.order_status)}
                                            </span>
                                        </span>
                                    </div>
                                </div>
                            </div>
                        </div>

                        <div className="grid grid-cols-1 lg:grid-cols-2 gap-4 sm:gap-6">
                            <div className="bg-gradient-to-b from-orange-300 to-cyan-300 dark:bg-none   dark:bg-gray-800 rounded-lg shadow-sm p-4 sm:p-6">
                                <h2 className="text-lg sm:text-lg font-semibold mb-4 flex items-center dark:text-white">
                                    <Timer className="w-4 h-5 mr-1 " />
                                    Remaining Time
                                </h2>
                                <div className="space-y-3">
                                    <div className="flex items-center gap-2">
                                        <div>
                                            <ClearDeadlineCountdown
                                                deadline={order.writer_deadline}
                                            />
                                        </div>
                                        {/* <button className="px-2 py-1 bg-blue-600 text-white rounded-md hover:bg-blue-700 dark:hover:bg-blue-800 transition-colors flex items-center justify-center">
                                            Adjust Time
                                        </button> */}
                                    </div>
                                    <div className="pt-3 border-t dark:border-gray-700">
                                        <div className="grid grid-cols-2 gap-4">
                                            <div>
                                                <div className="text-sm text-gray-500 dark:text-gray-400">
                                                    Posted
                                                </div>
                                                <div className="font-medium dark:text-white">
                                                    <OrderPostedTime
                                                        postedDate={
                                                            order.dateposted
                                                        }
                                                        className="text-blue-500"
                                                    />
                                                </div>
                                            </div>
                                            <div>
                                                <div className="text-sm text-gray-100 dark:text-gray-400">
                                                    Deadline
                                                </div>
                                                <div className="font-medium dark:text-white break-words">
                                                    {new Date(
                                                        order.writer_deadline
                                                    ).toLocaleString("en-US", {
                                                        day: "2-digit",
                                                        month: "short",
                                                        year: "numeric",
                                                        hour: "2-digit",
                                                        minute: "2-digit",
                                                    })}
                                                </div>
                                            </div>
                                        </div>
                                    </div>
                                </div>
                            </div>

                            {/* Bidding Section */}
                            <div className="bg-gradient-to-br from-orange-300 to-cyan-300 dark:from-gray-800 dark:to-gray-900 rounded-2xl shadow-lg p-6 sm:p-8 transition-all duration-300">
                                <h2 className="text-xl sm:text-2xl font-bold mb-6 flex items-center text-gray-900 dark:text-white">
                                    <Notebook className="w-6 h-6 mr-3 text-orange-600 dark:text-orange-400" />
                                    Bidding Details
                                </h2>

                                <div className="space-y-6">
                                    {/* If a bid already exists, show message and View Bid button */}
                                    {existingBid ? (
                                        <div className="space-y-4">
                                            <div className="bg-blue-100 dark:bg-blue-900 text-blue-800 dark:text-blue-200 border border-blue-300 dark:border-blue-700 rounded-lg p-4 text-sm sm:text-base">
                                                <p className="mb-2">
                                                    You’ve already placed a bid
                                                    on this order.
                                                </p>
                                                <p className="text-sm text-gray-700 dark:text-gray-300">
                                                    <span className="font-semibold">
                                                        Placed on:
                                                    </span>{" "}
                                                    {new Date(
                                                        existingBid.created_at
                                                    ).toLocaleString(
                                                        undefined,
                                                        {
                                                            year: "numeric",
                                                            month: "long",
                                                            day: "numeric",
                                                            hour: "2-digit",
                                                            minute: "2-digit",
                                                        }
                                                    )}
                                                </p>

                                                <p className="text-sm text-gray-700 dark:text-gray-300">
                                                    <span className="font-semibold">
                                                        Bid Amount:
                                                    </span>{" "}
                                                    ${order.writer_amount}
                                                </p>

                                                <p className="text-sm text-gray-700 dark:text-gray-300">
                                                    <span className="font-semibold">
                                                        Bid Status:
                                                    </span>{" "}
                                                    {existingBid.status}
                                                </p>
                                            </div>

                                            <Link
                                                href={`/writer/bids/${existingBid.id}`}
                                                className="w-full bg-gradient-to-r from-blue-900 to-indigo-900 hover:from-blue-700 hover:to-indigo-800 text-white font-semibold py-3 px-5 rounded-xl shadow-md hover:shadow-xl transition duration-200 flex items-center justify-center focus:outline-none focus:ring-2 focus:ring-offset-2 focus:ring-indigo-500"
                                            >
                                                <span className="text-base sm:text-lg">
                                                    View Bid
                                                </span>
                                            </Link>
                                        </div>
                                    ) : (
                                        <button
                                            className="w-full bg-gradient-to-r from-blue-600 to-indigo-700 hover:from-blue-700 hover:to-indigo-800 text-white font-semibold py-3 px-5 rounded-xl shadow-md hover:shadow-xl transition duration-200 flex items-center justify-center focus:outline-none focus:ring-2 focus:ring-offset-2 focus:ring-indigo-500"
                                            onClick={openBiddingModal}
                                        >
                                            <span className="text-base sm:text-lg">
                                                Place Bid
                                            </span>
                                        </button>
                                    )}
                                </div>
                            </div>
                        </div>

                        {/* Title Section */}
                        <div className="bg-gradient-to-r from-orange-300 to-cyan-300 dark:bg-none  text-black  dark:bg-gray-800 rounded-lg shadow-sm p-4 sm:p-6">
                            <div className="flex flex-col sm:flex-row justify-between items-start sm:items-center space-y-4 sm:space-y-0">
                                <div className="flex flex-col sm:flex-row sm:items-center gap-2 sm:gap-4">
                                    <h3 className="text-lg sm:text-lg md:text-lg font-bold dark:text-white">
                                        Title:
                                    </h3>

                                    <p className="text-md sm:text-md font-bold dark:text-white">
                                        {order.title}
                                    </p>
                                </div>
                            </div>
                        </div>

                        {/* Order Details Section */}
                        <div className="lg:col-span-2 bg-gradient-to-r from-orange-300 to-cyan-300 dark:bg-none   dark:bg-gray-800 rounded-lg shadow-sm p-4 sm:p-6">
                            <h2 className="text-lg sm:text-lg font-semibold mb-4 flex items-center dark:text-white">
                                <FileText className="w-5 h-5 mr-2" />
                                Order Details
                            </h2>
                            <div className="grid grid-cols-1 sm:grid-cols-2 gap-4">
                                <div className="space-y-3">
                                    <div className="flex flex-col">
                                        <span className="text-gray-600 dark:text-gray-400 mb-1">
                                            Academic level
                                        </span>
                                        <span className="font-base dark:text-white">
                                            {order.academic_level}
                                        </span>
                                    </div>
                                    <div className="flex flex-col">
                                        <span className="text-gray-600 dark:text-gray-400 mb-1">
                                            Type of Paper:
                                        </span>
                                        <span className="font-base dark:text-white">
                                            {order.type_of_paper}
                                        </span>
                                    </div>
                                    <div className="flex flex-col">
                                        <span className="text-gray-600 dark:text-gray-400 mb-1">
                                            Subject:
                                        </span>
                                        <span className="font-medium dark:text-white">
                                            {order.subject}
                                        </span>
                                    </div>
                                    <div className="flex flex-col">
                                        <span className="text-gray-600 dark:text-gray-400 mb-1">
                                            Pages:
                                        </span>
                                        <span className="font-medium dark:text-white">
                                            {order.pages}
                                        </span>
                                    </div>
                                    <div className="flex flex-col">
                                        <span className="text-gray-600 dark:text-gray-400 mb-1">
                                            Spacing
                                        </span>
                                        <span className="font-medium dark:text-white">
                                            {order.spacing}
                                        </span>
                                    </div>
                                </div>
                                <div className="space-y-3">
                                    <div className="flex flex-col">
                                        <span className="text-gray-600 dark:text-gray-400 mb-1">
                                            Amount:
                                        </span>
                                        <span className="font-medium dark:text-white">
                                            ${order.writer_amount}
                                        </span>
                                    </div>
                                    <div className="flex flex-col">
                                        <span className="text-gray-600 dark:text-gray-400 mb-1">
                                            Deadline:
                                        </span>
                                        <span className="font-medium dark:text-white">
                                            {order.writer_deadline}
                                        </span>
                                    </div>
                                    <div className="flex flex-col">
                                        <span className="text-gray-600 dark:text-gray-400 mb-1">
                                            Writing Style:
                                        </span>
                                        <span className="font-medium dark:text-white">
                                            {order.styles}
                                        </span>
                                    </div>

                                    <div className="flex flex-col">
                                        <span className="text-gray-600 dark:text-gray-400 mb-1">
                                            Sources/References:
                                        </span>
                                        <span className="font-medium dark:text-white">
                                            {order.sources}
                                        </span>
                                    </div>
                                </div>
                            </div>
                        </div>

                        {/* Instructions Section */}
                        <div className="bg-gradient-to-b from-orange-300 to-cyan-300 dark:bg-none   dark:bg-gray-800 rounded-lg shadow-sm p-4 sm:p-6">
                            <div className="flex flex-col space-y-4">
                                <h3 className="text-lg sm:text-lg font-bold dark:text-white">
                                    Instructions/Description
                                </h3>
                                <p className="text-md dark:text-white break-words whitespace-normal">
                                    {order.instructions}
                                </p>
                            </div>
                        </div>

                        {/* File Management Section */}
                        <div className="bg-gradient-to-r from-orange-300 to-cyan-300 dark:bg-none   dark:bg-gray-800 rounded-lg shadow-sm p-4 sm:p-6">
                            <h2 className="text-lg sm:text-lg font-semibold mb-4 flex items-center dark:text-white">
                                <FileText className="w-5 h-5 mr-2" />
                                File Management
                            </h2>

                            <WriterFileUpload 
                                orderId={order.id} 
                                onFileUploaded={handleFileUploaded}
                                onStatusUpdate={handleOrderStatusUpdate}
                            />
                            <FileManagement
                                files={currentOrder.order_files}
                                isAdmin={isAdmin}
                            />
                        </div>
                    </div>

                    {/* sidebar */}
                    {/* <div className="w-full md:w-auto">
                        <div className="relative overflow-hidden rounded-xl shadow-lg dark:shadow-slate-700/40 bg-gradient-to-br from-orange-200 to-cyan-200 dark:from-slate-800 dark:to-slate-900 border border-gray-100 dark:border-slate-700 p-6 transition-all duration-300">

                            <h3 className="text-lg font-semibold mb-4 text-gray-800 dark:text-white">
                                Bidding Details
                            </h3>


                            <div className="absolute inset-0 opacity-10 bg-grid-pattern pointer-events-none"></div>
                        </div>
                    </div> */}

                    {/* Bidding Modal */}
                    {isBiddingModalOpen && (
                        <div className="fixed inset-0 bg-black bg-opacity-50 flex items-center justify-center z-50 overflow-y-auto">
                            <div className="bg-gradient-to-r from-orange-200 to-cyan-200 dark:bg-none dark:bg-gray-800 rounded-lg p-6 w-full max-w-md mx-4">
                                <h3 className="text-lg font-semibold mb-4 dark:text-white">
                                    Place Your Bid
                                </h3>

                                <div className="mb-4">
                                    <label className="block text-sm font-medium text-gray-700 dark:text-gray-300 mb-1">
                                        Bid Amount (USD)
                                    </label>
                                    <input
                                        type="text"
                                        value={bidAmount}
                                        onChange={(e) =>
                                            setBidAmount(e.target.value)
                                        }
                                        className="w-full p-2 border border-gray-300 dark:border-gray-600 rounded-md dark:bg-gray-700 dark:text-white"
                                        placeholder="Enter your bid amount"
                                        disabled
                                    />
                                    <p className="text-xs font-semibold text-red-500 mt-1"></p>
                                </div>

                                <div className="mb-4">
                                    <label className="block text-sm font-medium text-gray-700 dark:text-gray-300 mb-1">
                                        Completion Time
                                    </label>
                                    <input
                                        type="datetime-local"
                                        value={completionTime}
                                        onChange={(e) =>
                                            setCompletionTime(e.target.value)
                                        }
                                        className="w-full p-2 border border-gray-300 dark:border-gray-600 rounded-md dark:bg-gray-700 dark:text-white"
                                    />
                                    <p className="text-xs font-semibold text-red-500 mt-1">
                                        Must be between{" "}
                                        {new Date(
                                            order.dateposted
                                        ).toLocaleString()}{" "}
                                        and{" "}
                                        {new Date(
                                            order.writer_deadline
                                        ).toLocaleString()}
                                        .
                                    </p>
                                </div>

                                <div className="mb-4">
                                    <label className="block text-sm font-medium text-gray-700 dark:text-gray-300 mb-1">
                                        Comments (Optional)
                                    </label>
                                    <textarea
                                        value={comments}
                                        onChange={(e) =>
                                            setComments(e.target.value)
                                        }
                                        className="w-full p-2 border border-gray-300 dark:border-gray-600 rounded-md dark:bg-gray-700 dark:text-white"
                                        placeholder="Add any additional comments"
                                    ></textarea>
                                </div>

                                <div className="flex justify-end space-x-3">
                                    <button
                                        onClick={closeBiddingModal}
                                        className="px-4 py-2 bg-gray-200 text-gray-800 rounded-md hover:bg-gray-300 dark:bg-gray-700 dark:text-gray-200 dark:hover:bg-gray-600"
                                    >
                                        Cancel
                                    </button>
                                    <button
                                        onClick={handlePlaceBid}
                                        className="px-4 py-2 bg-blue-600 text-white rounded-md hover:bg-blue-700 dark:hover:bg-blue-800"
                                    >
                                        Place Bid
                                    </button>
                                </div>
                            </div>
                        </div>
                    )}
                </div>
            </div>
        </WriterLayout>
    );
}
