import { useState } from "react";
import { Head, Link } from "@inertiajs/react";
import WriterLayout from "@/Layouts/WriterLayout";
import {
    CheckCircle,
    XCircle,
    AlertTriangle,
    Calendar,
    Clock,
    ArrowRight,
    BarChart3,
    BookOpen,
    AlertCircle,
    FileText,
    Award,
} from "lucide-react";

export default function Results({
    score,
    categorizedResults,
    canRetryAfter,
    daysUntilRetry,
    timeViolation,
    timeViolationNotes,
    autoProcessed,
}) {
    const [showTimeViolationDetails, setShowTimeViolationDetails] =
        useState(false);

    // Format percentage to 1 decimal place
    const formatPercentage = (value) => {
        return value.toFixed(1) + "%";
    };

    return (
        <WriterLayout>
            <Head title="Test Results" />

            <div className="py-12">
                <div className="max-w-7xl mx-auto sm:px-6 lg:px-8">
                    <div className="bg-white dark:bg-gray-800 overflow-hidden shadow-sm sm:rounded-lg">
                        <div className="p-6 bg-white dark:bg-gray-800 border-b border-gray-200 dark:border-gray-700">
                            <h1 className="text-2xl font-bold text-gray-800 dark:text-gray-100 mb-6">
                                Writer Qualification Test Results
                            </h1>

                            {/* Score overview */}
                            <div className="mb-8">
                                <div
                                    className={`p-6 rounded-lg ${
                                        score.passed
                                            ? "bg-green-50 dark:bg-green-900/20 border border-green-200 dark:border-green-800"
                                            : "bg-red-50 dark:bg-red-900/20 border border-red-200 dark:border-red-800"
                                    }`}
                                >
                                    <div className="flex items-center mb-4">
                                        {score.passed ? (
                                            <CheckCircle className="h-8 w-8 text-green-500 dark:text-green-400 mr-3" />
                                        ) : (
                                            <XCircle className="h-8 w-8 text-red-500 dark:text-red-400 mr-3" />
                                        )}
                                        <h2 className="text-xl font-semibold">
                                            {score.passed
                                                ? "Congratulations! You passed the test."
                                                : "You did not pass the test."}
                                        </h2>
                                    </div>

                                    <div className="grid grid-cols-1 md:grid-cols-3 gap-4 mb-4">
                                        <div className="bg-white dark:bg-gray-800 p-4 rounded-md shadow-sm">
                                            <p className="text-sm text-gray-500 dark:text-gray-400 mb-1">
                                                Your Score
                                            </p>
                                            <p
                                                className={`text-2xl font-bold ${
                                                    score.passed
                                                        ? "text-green-600 dark:text-green-400"
                                                        : "text-red-600 dark:text-red-400"
                                                }`}
                                            >
                                                {formatPercentage(
                                                    score.percentage
                                                )}
                                            </p>
                                        </div>
                                        <div className="bg-white dark:bg-gray-800 p-4 rounded-md shadow-sm">
                                            <p className="text-sm text-gray-500 dark:text-gray-400 mb-1">
                                                Correct Answers
                                            </p>
                                            <p className="text-2xl font-bold text-gray-800 dark:text-gray-200">
                                                {score.correct} / {score.total}
                                            </p>
                                        </div>
                                        <div className="bg-white dark:bg-gray-800 p-4 rounded-md shadow-sm">
                                            <p className="text-sm text-gray-500 dark:text-gray-400 mb-1">
                                                Passing Score
                                            </p>
                                            <p className="text-2xl font-bold text-gray-800 dark:text-gray-200">
                                                90%
                                            </p>
                                        </div>
                                    </div>

                                    {/* Auto-processed notification */}
                                    {autoProcessed && (
                                        <div className="mt-4 p-3 bg-amber-50 dark:bg-amber-900/20 border border-amber-200 dark:border-amber-800 rounded-md">
                                            <div className="flex items-start">
                                                <AlertCircle className="h-5 w-5 text-amber-500 dark:text-amber-400 mr-2 flex-shrink-0 mt-0.5" />
                                                <div>
                                                    <h3 className="text-sm font-medium text-amber-800 dark:text-amber-300">
                                                        Auto-Processed Test
                                                    </h3>
                                                    <p className="mt-1 text-sm text-amber-700 dark:text-amber-400">
                                                        Your test was
                                                        automatically processed
                                                        when the time expired.
                                                        The system has scored
                                                        the answers you provided
                                                        before the time limit.
                                                    </p>
                                                </div>
                                            </div>
                                        </div>
                                    )}

                                    {/* Time violation warning */}
                                    {timeViolation && (
                                        <div className="mt-4 p-3 bg-amber-50 dark:bg-amber-900/20 border border-amber-200 dark:border-amber-800 rounded-md">
                                            <div className="flex items-start">
                                                <AlertTriangle className="h-5 w-5 text-amber-500 dark:text-amber-400 mr-2 flex-shrink-0 mt-0.5" />
                                                <div>
                                                    <h3 className="text-sm font-medium text-amber-800 dark:text-amber-300">
                                                        Time Irregularities
                                                        Detected
                                                    </h3>
                                                    <p className="mt-1 text-sm text-amber-700 dark:text-amber-400">
                                                        Our system detected some
                                                        timing irregularities
                                                        during your test.
                                                    </p>
                                                    {timeViolationNotes && (
                                                        <div className="mt-2">
                                                            <button
                                                                type="button"
                                                                onClick={() =>
                                                                    setShowTimeViolationDetails(
                                                                        !showTimeViolationDetails
                                                                    )
                                                                }
                                                                className="text-sm text-amber-600 dark:text-amber-300 underline"
                                                            >
                                                                {showTimeViolationDetails
                                                                    ? "Hide details"
                                                                    : "Show details"}
                                                            </button>
                                                            {showTimeViolationDetails && (
                                                                <p className="mt-2 text-sm text-amber-700 dark:text-amber-400 bg-amber-100 dark:bg-amber-900/40 p-2 rounded">
                                                                    {
                                                                        timeViolationNotes
                                                                    }
                                                                </p>
                                                            )}
                                                        </div>
                                                    )}
                                                </div>
                                            </div>
                                        </div>
                                    )}

                                    {!score.passed && canRetryAfter && (
                                        <div className="mt-4 flex items-start">
                                            <Calendar className="h-5 w-5 text-gray-500 dark:text-gray-400 mr-2 flex-shrink-0 mt-0.5" />
                                            <div>
                                                <p className="text-sm text-gray-600 dark:text-gray-400">
                                                    You can retry the test after{" "}
                                                    <span className="font-medium">
                                                        {canRetryAfter}
                                                    </span>{" "}
                                                    ({daysUntilRetry} days from
                                                    now).
                                                </p>
                                            </div>
                                        </div>
                                    )}

                                    {score.passed && (
                                        <div className="mt-4 flex items-start">
                                            <Award className="h-5 w-5 text-green-500 dark:text-green-400 mr-2 flex-shrink-0 mt-0.5" />
                                            <div>
                                                <p className="text-sm text-gray-600 dark:text-gray-400">
                                                    You have passed the
                                                    multiple-choice test! The
                                                    next step is to complete the
                                                    essay test.
                                                </p>
                                            </div>
                                        </div>
                                    )}
                                </div>
                            </div>

                            {/* Category breakdown */}
                            <div className="mb-8">
                                <h2 className="text-lg font-semibold text-gray-700 dark:text-gray-200 mb-4 flex items-center">
                                    <BarChart3 className="h-5 w-5 mr-2 text-blue-500 dark:text-blue-400" />
                                    Performance by Category
                                </h2>

                                <div className="space-y-4">
                                    {Object.entries(categorizedResults).map(
                                        ([category, result]) => (
                                            <div
                                                key={category}
                                                className="bg-gray-50 dark:bg-gray-700/50 p-4 rounded-lg"
                                            >
                                                <div className="flex justify-between items-center mb-2">
                                                    <h3 className="font-medium text-gray-700 dark:text-gray-200">
                                                        {category}
                                                    </h3>
                                                    <span
                                                        className={`px-2 py-1 text-xs font-medium rounded-full ${
                                                            result.percentage >=
                                                            90
                                                                ? "bg-green-100 dark:bg-green-900/30 text-green-800 dark:text-green-300"
                                                                : result.percentage >=
                                                                  70
                                                                ? "bg-yellow-100 dark:bg-yellow-900/30 text-yellow-800 dark:text-yellow-300"
                                                                : "bg-red-100 dark:bg-red-900/30 text-red-800 dark:text-red-300"
                                                        }`}
                                                    >
                                                        {formatPercentage(
                                                            result.percentage
                                                        )}
                                                    </span>
                                                </div>
                                                <p className="text-sm text-gray-500 dark:text-gray-400 mb-2">
                                                    {result.correct} correct out
                                                    of {result.total} questions
                                                </p>
                                                <div className="w-full bg-gray-200 dark:bg-gray-600 rounded-full h-2.5">
                                                    <div
                                                        className={`h-2.5 rounded-full ${
                                                            result.percentage >=
                                                            90
                                                                ? "bg-green-500"
                                                                : result.percentage >=
                                                                  70
                                                                ? "bg-yellow-500"
                                                                : "bg-red-500"
                                                        }`}
                                                        style={{
                                                            width: `${result.percentage}%`,
                                                        }}
                                                    ></div>
                                                </div>
                                            </div>
                                        )
                                    )}
                                </div>
                            </div>

                            {/* Next steps */}
                            <div className="mb-8">
                                <h2 className="text-lg font-semibold text-gray-700 dark:text-gray-200 mb-4 flex items-center">
                                    <FileText className="h-5 w-5 mr-2 text-blue-500 dark:text-blue-400" />
                                    Next Steps
                                </h2>

                                <div className="bg-blue-50 dark:bg-blue-900/20 p-5 rounded-lg border border-blue-200 dark:border-blue-800">
                                    {score.passed ? (
                                        <div>
                                            <p className="text-blue-700 dark:text-blue-300 mb-4">
                                                Congratulations on passing the
                                                multiple-choice test! Here's
                                                what comes next:
                                            </p>
                                            <ol className="list-decimal list-inside space-y-2 text-blue-600 dark:text-blue-400">
                                                <li>
                                                    Proceed to the essay test to
                                                    demonstrate your writing
                                                    skills
                                                </li>
                                                <li>
                                                    The essay test will evaluate
                                                    your ability to create
                                                    well-structured, coherent
                                                    content
                                                </li>
                                                <li>
                                                    After passing both tests,
                                                    you'll be able to bid on
                                                    available orders
                                                </li>
                                            </ol>
                                            <div className="mt-6">
                                                <Link
                                                    href={route(
                                                        "writer.essay.instructions"
                                                    )}
                                                    className="inline-flex items-center px-4 py-2 bg-blue-600 border border-transparent rounded-md font-semibold text-xs text-white uppercase tracking-widest hover:bg-blue-700 active:bg-blue-800 focus:outline-none focus:border-blue-800 focus:ring ring-blue-300 disabled:opacity-25 transition"
                                                >
                                                    Continue to Essay Test
                                                    <ArrowRight className="ml-2 h-4 w-4" />
                                                </Link>
                                            </div>
                                        </div>
                                    ) : (
                                        <div>
                                            <p className="text-blue-700 dark:text-blue-300 mb-4">
                                                You didn't pass the test this
                                                time, but don't worry. Here's
                                                what you can do:
                                            </p>
                                            <ul className="list-disc list-inside space-y-2 text-blue-600 dark:text-blue-400">
                                                <li>
                                                    Review the categories where
                                                    you scored lower to improve
                                                    your knowledge
                                                </li>
                                                <li>
                                                    Study academic writing
                                                    principles and citation
                                                    styles
                                                </li>
                                                <li>
                                                    You can retry the test after
                                                    the waiting period (
                                                    {daysUntilRetry} days)
                                                </li>
                                            </ul>
                                            <div className="mt-6">
                                                <Link
                                                    href={route(
                                                        "writer.dashboard"
                                                    )}
                                                    className="inline-flex items-center px-4 py-2 bg-blue-600 border border-transparent rounded-md font-semibold text-xs text-white uppercase tracking-widest hover:bg-blue-700 active:bg-blue-800 focus:outline-none focus:border-blue-800 focus:ring ring-blue-300 disabled:opacity-25 transition"
                                                >
                                                    Return to Dashboard
                                                    <ArrowRight className="ml-2 h-4 w-4" />
                                                </Link>
                                            </div>
                                        </div>
                                    )}
                                </div>
                            </div>

                            {/* Study resources */}
                            {!score.passed && (
                                <div>
                                    <h2 className="text-lg font-semibold text-gray-700 dark:text-gray-200 mb-4 flex items-center">
                                        <BookOpen className="h-5 w-5 mr-2 text-blue-500 dark:text-blue-400" />
                                        Recommended Study Resources
                                    </h2>

                                    <div className="bg-gray-50 dark:bg-gray-700/50 p-5 rounded-lg">
                                        <p className="text-gray-600 dark:text-gray-400 mb-4">
                                            Here are some resources to help you
                                            improve in the areas where you need
                                            more practice:
                                        </p>
                                        <ul className="space-y-3">
                                            <li className="flex items-start">
                                                <div className="flex-shrink-0">
                                                    <div className="w-8 h-8 rounded-full bg-purple-100 dark:bg-purple-900/30 flex items-center justify-center text-purple-600 dark:text-purple-400">
                                                        1
                                                    </div>
                                                </div>
                                                <div className="ml-3">
                                                    <h3 className="text-sm font-medium text-gray-700 dark:text-gray-200">
                                                        Academic Writing
                                                    </h3>
                                                    <p className="mt-1 text-sm text-gray-500 dark:text-gray-400">
                                                        Purdue OWL Writing
                                                        Resources, Academic
                                                        Writing for Graduate
                                                        Students
                                                    </p>
                                                </div>
                                            </li>
                                            <li className="flex items-start">
                                                <div className="flex-shrink-0">
                                                    <div className="w-8 h-8 rounded-full bg-green-100 dark:bg-green-900/30 flex items-center justify-center text-green-600 dark:text-green-400">
                                                        2
                                                    </div>
                                                </div>
                                                <div className="ml-3">
                                                    <h3 className="text-sm font-medium text-gray-700 dark:text-gray-200">
                                                        Citation Styles
                                                    </h3>
                                                    <p className="mt-1 text-sm text-gray-500 dark:text-gray-400">
                                                        APA, MLA, Chicago, and
                                                        Harvard style guides
                                                    </p>
                                                </div>
                                            </li>
                                            <li className="flex items-start">
                                                <div className="flex-shrink-0">
                                                    <div className="w-8 h-8 rounded-full bg-blue-100 dark:bg-blue-900/30 flex items-center justify-center text-blue-600 dark:text-blue-400">
                                                        3
                                                    </div>
                                                </div>
                                                <div className="ml-3">
                                                    <h3 className="text-sm font-medium text-gray-700 dark:text-gray-200">
                                                        Research Methods
                                                    </h3>
                                                    <p className="mt-1 text-sm text-gray-500 dark:text-gray-400">
                                                        JSTOR, Google Scholar,
                                                        and academic databases
                                                    </p>
                                                </div>
                                            </li>
                                        </ul>
                                    </div>
                                </div>
                            )}
                        </div>
                    </div>
                </div>
            </div>
        </WriterLayout>
    );
}
