import { useState } from 'react';
import { Head, useForm, router } from '@inertiajs/react';
import WriterLayout from '@/Layouts/WriterLayout';
import { 
    Upload, 
    FileText, 
    AlertTriangle, 
    Clock, 
    CheckCircle,
    XCircle,
    Calendar,
    User,
    BookOpen,
    AlertCircle,
    Edit3,
    Eye
} from 'lucide-react';
import { formatCurrency, formatDeadline } from '@/lib/formatters';

export default function WorkSubmission({ order, isExpired, deadline, currentTime }) {
    const [uploadedFiles, setUploadedFiles] = useState([]);
    const [isSubmitting, setIsSubmitting] = useState(false);
    const [submissionResult, setSubmissionResult] = useState(null);

    const { data, setData, post, processing, errors, reset } = useForm({
        submission_notes: '',
        files: [],
        submission_type: 'complete' // 'draft' or 'complete'
    });

    // Submission type constants
    const SUBMISSION_TYPES = {
        DRAFT: 'draft',
        COMPLETE: 'complete'
    };

    const handleFileChange = (e) => {
        const files = Array.from(e.target.files);
        setData('files', files);
        setUploadedFiles(files);
    };

    const removeFile = (index) => {
        const newFiles = uploadedFiles.filter((_, i) => i !== index);
        setUploadedFiles(newFiles);
        setData('files', newFiles);
    };

    const handleSubmit = async (e) => {
        e.preventDefault();
        setIsSubmitting(true);
        setSubmissionResult(null);

        try {
            // Use Inertia router.post for proper form submission
            await router.post(route('writer.orders.submit-work.store', order.id), {
                submission_notes: data.submission_notes,
                submission_type: data.submission_type,
                files: uploadedFiles
            }, {
                forceFormData: true, // Ensure files are properly handled
                onSuccess: () => {
                    // Success will be handled by flash messages
                    reset();
                    setUploadedFiles([]);
                    // Redirect to assignments page after successful submission
                    window.location.href = route('writer.assignments');
                },
                onError: (errors) => {
                    console.error('Submission errors:', errors);
                    setSubmissionResult({
                        type: 'error',
                        message: 'Please check the form for errors and try again.'
                    });
                },
                onFinish: () => {
                    setIsSubmitting(false);
                }
            });
        } catch (error) {
            console.error('Submission error:', error);
            setSubmissionResult({
                type: 'error',
                message: 'An error occurred while submitting work'
            });
            setIsSubmitting(false);
        }
    };

    const getStatusDisplayName = (status) => {
        const statusMap = {
            'assigned': 'Assigned',
            'in_progress': 'In Progress',
            'submitted': 'Submitted',
            'under_review': 'Under Review',
            'revision_requested': 'Revision Requested'
        };
        return statusMap[status] || status;
    };

    const getStatusIcon = (status) => {
        switch (status) {
            case 'assigned':
                return <BookOpen size={16} className="text-blue-500" />;
            case 'in_progress':
                return <Edit3 size={16} className="text-yellow-500" />;
            case 'submitted':
                return <CheckCircle size={16} className="text-green-500" />;
            case 'under_review':
                return <Eye size={16} className="text-purple-500" />;
            case 'revision_requested':
                return <AlertCircle size={16} className="text-orange-500" />;
            default:
                return <FileText size={16} className="text-gray-500" />;
        }
    };

    const getStatusColor = (status) => {
        switch (status) {
            case 'assigned':
                return 'bg-blue-100 text-blue-800 border-blue-200';
            case 'in_progress':
                return 'bg-yellow-100 text-yellow-800 border-yellow-200';
            case 'submitted':
                return 'bg-green-100 text-green-800 border-green-200';
            case 'under_review':
                return 'bg-purple-100 text-purple-800 border-purple-200';
            case 'revision_requested':
                return 'bg-orange-100 text-orange-800 border-orange-200';
            default:
                return 'bg-gray-100 text-gray-800 border-gray-200';
        }
    };

    const formatTimeDifference = (deadline, currentTime) => {
        const deadlineDate = new Date(deadline);
        const currentDate = new Date(currentTime);
        const diffInHours = Math.floor((currentDate - deadlineDate) / (1000 * 60 * 60));
        
        if (diffInHours < 0) {
            return `${Math.abs(diffInHours)} hours remaining`;
        } else if (diffInHours === 0) {
            return 'Deadline is now';
        } else {
            return `${diffInHours} hours overdue`;
        }
    };

    return (
        <WriterLayout>
            <Head title="Submit Work" />
            
            <div className="py-6">
                <div className="max-w-4xl mx-auto px-4 sm:px-6 lg:px-8">
                    {/* Header */}
                    <div className="mb-8">
                        <h1 className="text-3xl font-bold text-gray-900">Submit Work</h1>
                        <p className="mt-2 text-gray-600">
                            Submit your completed work for review
                        </p>
                    </div>

                    {/* Order Information */}
                    <div className="bg-white rounded-lg shadow-md border-l-4 border-blue-500 mb-8">
                        <div className="p-6">
                            <div className="flex items-start justify-between mb-4">
                                <div className="flex-1">
                                    <div className="flex items-center gap-3 mb-2">
                                        <h2 className="text-xl font-semibold text-gray-900">
                                            {order.title}
                                        </h2>
                                        <span className={`inline-flex items-center px-2.5 py-0.5 rounded-full text-xs font-medium border ${getStatusColor(order.order_status)}`}>
                                            {getStatusIcon(order.order_status)}
                                            <span className="ml-1">{getStatusDisplayName(order.order_status)}</span>
                                        </span>
                                    </div>
                                    
                                    <div className="grid grid-cols-1 md:grid-cols-3 gap-4 text-sm text-gray-600 mb-4">
                                        <div className="flex items-center gap-2">
                                            <BookOpen size={16} />
                                            <span>{order.type_of_paper}</span>
                                        </div>
                                        <div className="flex items-center gap-2">
                                            <User size={16} />
                                            <span>{order.academic_level}</span>
                                        </div>
                                        <div className="flex items-center gap-2">
                                            <FileText size={16} />
                                            <span>{order.pages} pages</span>
                                        </div>
                                    </div>
                                </div>
                                
                                <div className="text-right">
                                    <div className="text-sm text-gray-500">Writer Amount</div>
                                    <div className="text-lg font-semibold text-gray-900">
                                        {formatCurrency(order.writer_amount)}
                                    </div>
                                </div>
                            </div>

                            {/* Deadline Information */}
                            <div className="border-t pt-4">
                                <div className="flex items-center justify-between">
                                    <div className="flex items-center gap-4">
                                        <div className="flex items-center gap-2">
                                            <Calendar size={16} className="text-gray-500" />
                                            <span className="text-gray-600">
                                                Deadline: {formatDeadline(deadline)}
                                            </span>
                                        </div>
                                        <div className="flex items-center gap-2">
                                            <Clock size={16} className="text-gray-500" />
                                            <span className={`font-medium ${
                                                isExpired ? 'text-red-600' : 'text-gray-900'
                                            }`}>
                                                {formatTimeDifference(deadline, currentTime)}
                                            </span>
                                        </div>
                                    </div>
                                    
                                    {isExpired && (
                                        <div className="flex items-center gap-2 text-red-600">
                                            <AlertTriangle size={20} />
                                            <span className="text-sm font-medium">Deadline Expired</span>
                                        </div>
                                    )}
                                </div>
                            </div>
                        </div>
                    </div>

                    {/* Late Submission Warning */}
                    {isExpired && (
                        <div className="bg-yellow-50 border border-yellow-200 rounded-md p-4 mb-8">
                            <div className="flex">
                                <AlertTriangle size={20} className="text-yellow-400" />
                                <div className="ml-3">
                                    <h3 className="text-sm font-medium text-yellow-800">
                                        Late Submission Notice
                                    </h3>
                                    <div className="mt-2 text-sm text-yellow-700">
                                        <p>
                                            This order has passed its deadline. You can still submit your work, 
                                            but it will be marked as a late submission. The admin will review 
                                            your submission and may assess penalties according to the infractions system.
                                        </p>
                                    </div>
                                </div>
                            </div>
                        </div>
                    )}

                    {/* Draft Submission Notice */}
                    {data.submission_type === SUBMISSION_TYPES.DRAFT && (
                        <div className="bg-blue-50 border border-blue-200 rounded-md p-4 mb-8">
                            <div className="flex">
                                <Edit3 size={20} className="text-blue-400" />
                                <div className="ml-3">
                                    <h3 className="text-sm font-medium text-blue-800">
                                        Draft Submission Notice
                                    </h3>
                                    <div className="mt-2 text-sm text-blue-700">
                                        <p>
                                            You are submitting a draft. This will save your work and files but will 
                                            <strong> NOT </strong> change the order status or timeline. Use this option 
                                            when you want to save progress without finalizing the submission.
                                        </p>
                                    </div>
                                </div>
                            </div>
                        </div>
                    )}

                    {/* Submission Form */}
                    <div className="bg-white rounded-lg shadow-md">
                        <div className="p-6">
                            <div className="flex items-center justify-between mb-6">
                                <h3 className="text-lg font-medium text-gray-900">Submit Your Work</h3>
                                <div className={`px-3 py-1 rounded-full text-xs font-medium ${
                                    data.submission_type === SUBMISSION_TYPES.DRAFT 
                                        ? 'bg-blue-100 text-blue-800' 
                                        : 'bg-green-100 text-green-800'
                                }`}>
                                    {data.submission_type === SUBMISSION_TYPES.DRAFT ? 'Draft Mode' : 'Complete Mode'}
                                </div>
                            </div>
                            
                            <form onSubmit={handleSubmit}>
                                {/* Submission Type Selection */}
                                <div className="mb-6">
                                    <label className="block text-sm font-medium text-gray-700 mb-3">
                                        Submission Type *
                                    </label>
                                    <div className="space-y-3">
                                        <label className="flex items-center">
                                            <input
                                                type="radio"
                                                name="submission_type"
                                                value={SUBMISSION_TYPES.COMPLETE}
                                                checked={data.submission_type === SUBMISSION_TYPES.COMPLETE}
                                                onChange={(e) => setData('submission_type', e.target.value)}
                                                className="h-4 w-4 text-blue-600 focus:ring-blue-500 border-gray-300"
                                            />
                                            <span className="ml-3 text-sm text-gray-700">
                                                <span className="font-medium text-green-600">Complete Work</span>
                                                <span className="text-gray-500"> - Final submission that changes order status and timeline</span>
                                            </span>
                                        </label>
                                        <label className="flex items-center">
                                            <input
                                                type="radio"
                                                name="submission_type"
                                                value={SUBMISSION_TYPES.DRAFT}
                                                checked={data.submission_type === SUBMISSION_TYPES.DRAFT}
                                                onChange={(e) => setData('submission_type', e.target.value)}
                                                className="h-4 w-4 text-blue-600 focus:ring-blue-500 border-gray-300"
                                            />
                                            <span className="ml-3 text-sm text-gray-700">
                                                <span className="font-medium text-blue-600">Draft</span>
                                                <span className="text-gray-500"> - Work in progress, no status change</span>
                                            </span>
                                        </label>
                                    </div>
                                    {errors.submission_type && (
                                        <p className="mt-1 text-sm text-red-600">{errors.submission_type}</p>
                                    )}
                                    
                                    <div className="mt-3 p-3 bg-gray-50 rounded-md">
                                        <p className="text-xs text-gray-600">
                                            <strong>Complete Work:</strong> Use this when you're finished with the assignment. 
                                            This will change the order status to "Submitted" and notify admins for review.
                                        </p>
                                        <p className="text-xs text-gray-600 mt-1">
                                            <strong>Draft:</strong> Use this to save your progress without finalizing. 
                                            The order status remains unchanged, and no notifications are sent.
                                        </p>
                                    </div>
                                </div>

                                {/* File Upload */}
                                <div className="mb-6">
                                    <label className="block text-sm font-medium text-gray-700 mb-2">
                                        Upload Files *
                                    </label>
                                    <div className="border-2 border-dashed border-gray-300 rounded-lg p-6 text-center">
                                        <Upload size={24} className="mx-auto text-gray-400 mb-2" />
                                        <div className="text-sm text-gray-600 mb-4">
                                            <label htmlFor="file-upload" className="cursor-pointer">
                                                <span className="font-medium text-blue-600 hover:text-blue-500">
                                                    Click to upload
                                                </span>
                                                {' '}or drag and drop
                                            </label>
                                            <input
                                                id="file-upload"
                                                name="files[]"
                                                type="file"
                                                multiple
                                                accept=".pdf,.doc,.docx,.txt"
                                                className="sr-only"
                                                onChange={handleFileChange}
                                                required
                                            />
                                        </div>
                                        <p className="text-xs text-gray-500">
                                            PDF, DOC, DOCX, TXT up to 10MB each
                                        </p>
                                    </div>
                                    
                                    {/* File List */}
                                    {uploadedFiles.length > 0 && (
                                        <div className="mt-4">
                                            <h4 className="text-sm font-medium text-gray-700 mb-2">Selected Files:</h4>
                                            <div className="space-y-2">
                                                {uploadedFiles.map((file, index) => (
                                                    <div key={index} className="flex items-center justify-between bg-gray-50 p-3 rounded-md">
                                                        <div className="flex items-center gap-2">
                                                            <FileText size={16} className="text-gray-500" />
                                                            <span className="text-sm text-gray-700">{file.name}</span>
                                                            <span className="text-xs text-gray-500">
                                                                ({(file.size / 1024 / 1024).toFixed(2)} MB)
                                                            </span>
                                                        </div>
                                                        <button
                                                            type="button"
                                                            onClick={() => removeFile(index)}
                                                            className="text-red-500 hover:text-red-700"
                                                        >
                                                            <XCircle size={16} />
                                                        </button>
                                                    </div>
                                                ))}
                                            </div>
                                        </div>
                                    )}
                                    
                                    {errors.files && (
                                        <p className="mt-1 text-sm text-red-600">{errors.files}</p>
                                    )}
                                </div>

                                {/* Submission Notes */}
                                <div className="mb-6">
                                    <label htmlFor="submission_notes" className="block text-sm font-medium text-gray-700 mb-2">
                                        Submission Notes (Optional)
                                    </label>
                                    <textarea
                                        id="submission_notes"
                                        name="submission_notes"
                                        rows={4}
                                        className="w-full px-3 py-2 border border-gray-300 rounded-md focus:ring-2 focus:ring-indigo-500 focus:border-indigo-500"
                                        placeholder="Add any notes about your submission, special instructions, or clarifications..."
                                        value={data.submission_notes}
                                        onChange={(e) => setData('submission_notes', e.target.value)}
                                    />
                                    {errors.submission_notes && (
                                        <p className="mt-1 text-sm text-red-600">{errors.submission_notes}</p>
                                    )}
                                </div>

                                {/* Submit Button */}
                                <div className="flex justify-end">
                                    <button
                                        type="submit"
                                        disabled={processing || isSubmitting || uploadedFiles.length === 0}
                                        className={`inline-flex items-center px-4 py-2 border border-transparent text-sm font-medium rounded-md text-white ${
                                            processing || isSubmitting || uploadedFiles.length === 0
                                                ? 'bg-gray-400 cursor-not-allowed'
                                                : 'bg-blue-600 hover:bg-blue-700 focus:outline-none focus:ring-2 focus:ring-offset-2 focus:ring-blue-500'
                                        }`}
                                    >
                                        {isSubmitting ? (
                                            <>
                                                <div className="animate-spin rounded-full h-4 w-4 border-b-2 border-white mr-2"></div>
                                                Submitting...
                                            </>
                                        ) : (
                                            <>
                                                <Upload size={16} className="mr-2" />
                                                {data.submission_type === SUBMISSION_TYPES.DRAFT ? 'Save Draft' : 'Submit Complete Work'}
                                            </>
                                        )}
                                    </button>
                                </div>
                            </form>
                        </div>
                    </div>

                    {/* Submission Result */}
                    {submissionResult && (
                        <div className={`mt-6 p-4 rounded-md ${
                            submissionResult.type === 'success' 
                                ? 'bg-green-50 border border-green-200' 
                                : 'bg-red-50 border border-red-200'
                        }`}>
                            <div className="flex">
                                {submissionResult.type === 'success' ? (
                                    <CheckCircle size={20} className="text-green-400" />
                                ) : (
                                    <XCircle size={20} className="text-red-400" />
                                )}
                                <div className="ml-3">
                                    <h3 className={`text-sm font-medium ${
                                        submissionResult.type === 'success' ? 'text-green-800' : 'text-red-800'
                                    }`}>
                                        {submissionResult.type === 'success' ? 'Submission Successful' : 'Submission Failed'}
                                    </h3>
                                    <div className={`mt-2 text-sm ${
                                        submissionResult.type === 'success' ? 'text-green-700' : 'text-red-700'
                                    }`}>
                                        <p>{submissionResult.message}</p>
                                        {submissionResult.isLateSubmission && (
                                            <p className="mt-1 font-medium">
                                                ⚠️ This submission was made after the deadline and will be reviewed for potential penalties.
                                            </p>
                                        )}
                                    </div>
                                </div>
                            </div>
                        </div>
                    )}
                </div>
            </div>
        </WriterLayout>
    );
} 