import { Head } from "@inertiajs/react";
import { useMemo } from "react";
import MainLayout from "@/Layouts/MainLayout";
import WriterHero from "@/Components/Writers/WriterHero";
import SelectionProcess from "@/Components/Writers/SelectionProcess";
import WriterSuperpowers from "@/Components/Writers/WriterSuperpowers";
import QualityControl from "@/Components/Writers/QualityControl";
import ClientBenefits from "@/Components/Writers/ClientBenefits";
import FeaturedWriters from "@/Components/Writers/FeaturedWriters";
import WriterExpertise from "@/Components/Writers/WriterExpertise";
import ServiceFAQs from "@/Components/Services/ServiceFAQs";
import PriceCalculatorSection from "@/Components/CMS/PriceCalculatorSection";
import ScrollableContentSection from "@/Components/Services/ScrollableContentSection";

export default function Writers(props) {
    const { seoData, contentBlocks, structuredData, isCMS } = props;

    // Sort content blocks by order field for dynamic rendering
    const sortedBlocks = useMemo(() => {
        if (!isCMS || !contentBlocks) return [];
        return [...contentBlocks].sort((a, b) => (a.order || 0) - (b.order || 0));
    }, [contentBlocks, isCMS]);

    // Render individual block based on block_type
    const renderBlock = (block) => {
        const content = block.content;
        
        switch (block.block_type) {
            case 'writers_hero':
                return <WriterHero key={block.id} hero={content || fallbackHero} stats={content?.stats || fallbackHero.stats} />;
            case 'selection_process':
                return <SelectionProcess key={block.id} title={content?.title} description={content?.description} processes={content?.processes || []} />;
            case 'writer_superpowers':
                return <WriterSuperpowers key={block.id} title={content?.title} description={content?.description} superpowers={content?.superpowers || []} />;
            case 'quality_control':
                return <QualityControl key={block.id} title={content?.title} description={content?.description} features={content?.features || []} stats={content?.stats || []} />;
            case 'client_benefits':
                return <ClientBenefits key={block.id} title={content?.title} description={content?.description} benefits={content?.benefits || []} />;
            case 'featured_writers':
                return <FeaturedWriters key={block.id} title={content?.title} description={content?.description} writers={content?.writers || []} />;
            case 'writer_expertise':
                return <WriterExpertise key={block.id} title={content?.title} description={content?.description} expertise={content?.expertise || []} />;
            case 'writers_faqs':
                return <ServiceFAQs key={block.id} title={content?.title} description={content?.description} faqs={content?.faqs || []} />;
            case 'writers_cta':
                // CTA is handled separately, skip here
                return null;
            case 'price_calculator':
                return <PriceCalculatorSection key={block.id} content={content} />;
            case 'service_scrollable':
                return content?.sections ? <ScrollableContentSection key={block.id} sections={content.sections} /> : null;
            default:
                return null;
        }
    };

    // Extract content from CMS blocks (for fallback compatibility)
    const getContentBlock = (blockKey) => {
        if (!isCMS || !contentBlocks) return null;
        const block = contentBlocks.find((b) => b.block_key === blockKey);
        return block ? block.content : null;
    };

    // Get all content blocks from CMS
    const hero = getContentBlock('writers_hero');
    const selectionProcess = getContentBlock('selection_process');
    const superpowers = getContentBlock('writer_superpowers');
    const qualityControl = getContentBlock('quality_control');
    const clientBenefits = getContentBlock('client_benefits');
    const featuredWriters = getContentBlock('featured_writers');
    const expertise = getContentBlock('writer_expertise');
    const faqs = getContentBlock('writers_faqs');
    const cta = getContentBlock('writers_cta');

    // FALLBACK DATA (exact copy from original for SEO if CMS not available)
    const fallbackHero = {
        title: 'Professional Essay Writers for Hire',
        description: 'Hire expert academic writers with PhD and Masters degrees. Our 847+ qualified writers deliver custom essays, research papers, dissertations, and assignments with guaranteed quality.',
        stats: [
            { value: '847', label: 'Expert Writers' },
            { value: '312', label: 'PhD Writers' },
            { value: '8.5+', label: 'Years Experience' },
            { value: '98.7%', label: 'Satisfaction Rate' },
            { value: '45,678', label: 'Papers Completed' },
            { value: '99.2%', label: 'On-Time Delivery' },
            { value: '100+', label: 'Academic Disciplines' },
        ]
    };

    const fallbackCta = {
        title: 'Ready to Hire a Professional Essay Writer?',
        description: 'Join thousands of satisfied students who have achieved academic success with our expert writers. Get matched with the perfect writer for your needs today.',
        primaryButtonText: 'Hire a Writer Now',
        primaryButtonLink: '/place-order',
        secondaryButtonText: 'View Pricing',
        secondaryButtonLink: '/pricing',
        trustBadges: [
            '847+ Expert Writers',
            'PhD & Masters Degrees',
            '99.2% On-Time Delivery',
            'Unlimited Free Revisions',
        ]
    };

    return (
        <MainLayout>
            <Head>
                <title>{seoData?.title || 'Professional Essay Writers for Hire | 847+ Expert Academic Writers | Academic Scribe'}</title>
                <meta name="description" content={seoData?.description || 'Hire professional essay writers with PhD and Masters degrees. 847+ qualified academic writers available for custom essays, research papers, dissertations, and assignments. Expert writers for students worldwide.'} />
                <meta name="keywords" content={seoData?.keywords || 'essay writers for hire, professional academic writers, expert writers, PhD writers, research paper writers, dissertation writers, assignment help writers, qualified essay writers, academic writing experts, hire essay writer, professional writing service, custom essay writers'} />
                <link rel="canonical" href={seoData?.canonicalUrl || 'https://academicscribe.com/writers'} />
                <meta property="og:title" content={seoData?.ogTitle || 'Professional Essay Writers for Hire | 847+ Expert Academic Writers'} />
                <meta property="og:description" content={seoData?.ogDescription || 'Hire qualified essay writers with advanced degrees. Expert academic writers available for custom papers, research, and assignments.'} />
                <meta property="og:type" content="website" />
                <meta property="og:url" content={seoData?.ogUrl || 'https://academicscribe.com/writers'} />
                <meta property="og:image" content={seoData?.ogImage || '/images/writers-og.jpg'} />
                <meta name="twitter:card" content="summary_large_image" />
                <meta name="twitter:title" content={seoData?.twitterTitle || 'Professional Essay Writers for Hire | Expert Academic Writers'} />
                <meta name="twitter:description" content={seoData?.twitterDescription || 'Hire qualified essay writers with PhD and Masters degrees. 847+ expert academic writers available for custom papers and assignments.'} />
                <meta name="twitter:image" content={seoData?.twitterImage || '/images/writers-twitter.jpg'} />
                
                {/* Structured Data */}
                {structuredData && (
                    <script type="application/ld+json">
                        {JSON.stringify(structuredData)}
                    </script>
                )}
            </Head>

            <div className="writers-page bg-white dark:bg-gray-900">
                {isCMS && sortedBlocks.length > 0 ? (
                    sortedBlocks.map((block) => renderBlock(block))
                ) : (
                    <>
                        {/* Fallback: Static sections */}
                        <WriterHero 
                            hero={fallbackHero}
                            stats={fallbackHero.stats}
                        />
                    </>
                )}

                {/* Final CTA Section */}
                <section className="py-16 bg-gradient-to-r from-blue-600 to-purple-600 dark:from-blue-800 dark:to-purple-900">
                    <div className="max-w-4xl mx-auto px-4 sm:px-6 lg:px-8 text-center">
                        <h2 className="text-3xl md:text-4xl font-bold text-white mb-4">
                            {cta?.title || fallbackCta.title}
                        </h2>
                        <p className="text-xl text-blue-100 dark:text-blue-200 mb-8 max-w-2xl mx-auto">
                            {cta?.description || fallbackCta.description}
                        </p>

                        <div className="flex flex-col sm:flex-row gap-4 justify-center mb-8">
                            <a
                                href={cta?.primaryButtonLink || fallbackCta.primaryButtonLink}
                                className="bg-white text-blue-600 hover:bg-gray-100 dark:bg-gray-100 dark:hover:bg-white px-8 py-4 rounded-lg font-bold text-lg transition-colors"
                            >
                                {cta?.primaryButtonText || fallbackCta.primaryButtonText}
                            </a>
                            <a
                                href={cta?.secondaryButtonLink || fallbackCta.secondaryButtonLink}
                                className="border-2 border-white text-white hover:bg-white hover:text-blue-600 dark:hover:bg-gray-100 px-8 py-4 rounded-lg font-bold text-lg transition-colors"
                            >
                                {cta?.secondaryButtonText || fallbackCta.secondaryButtonText}
                            </a>
                        </div>

                        {/* Trust Indicators */}
                        <div className="flex flex-wrap justify-center items-center gap-8 text-blue-100 dark:text-blue-200">
                            {(cta?.trustBadges || fallbackCta.trustBadges).map((badge, index) => (
                                <div key={index} className="flex items-center">
                                    <svg
                                        className="w-5 h-5 mr-2"
                                        fill="currentColor"
                                        viewBox="0 0 20 20"
                                    >
                                        <path
                                            fillRule="evenodd"
                                            d="M16.707 5.293a1 1 0 010 1.414l-8 8a1 1 0 01-1.414 0l-4-4a1 1 0 011.414-1.414L8 12.586l7.293-7.293a1 1 0 011.414 0z"
                                            clipRule="evenodd"
                                        />
                                    </svg>
                                    <span className="text-sm">{badge}</span>
                                </div>
                            ))}
                        </div>
                    </div>
                </section>
            </div>
        </MainLayout>
    );
}
