import axios from 'axios';
window.axios = axios;

// Ensure axios relies on the XSRF cookie instead of any stale header
const headerBuckets = ['common', 'delete', 'get', 'head', 'post', 'put', 'patch'];
headerBuckets.forEach((bucket) => {
    if (window.axios.defaults.headers[bucket]) {
        delete window.axios.defaults.headers[bucket]['X-CSRF-TOKEN'];
    }
});
window.axios.defaults.headers.common['X-Requested-With'] = 'XMLHttpRequest';

// Laravel Echo setup for real-time messaging (only if broadcasting is enabled)
const reverbKey = import.meta.env.VITE_REVERB_APP_KEY;
const pusherKey = import.meta.env.VITE_PUSHER_APP_KEY;
const hasValidKey = (reverbKey && reverbKey !== 'your-app-key') || (pusherKey && pusherKey !== 'your-app-key');

if (hasValidKey) {
    // Dynamic imports for conditional loading
    Promise.all([
        import('laravel-echo'),
        import('pusher-js')
    ]).then(([Echo, Pusher]) => {
        window.Pusher = Pusher.default;

        // Determine configuration based on available keys
        const config = {
            broadcaster: 'pusher',
            key: pusherKey || reverbKey,
            cluster: import.meta.env.VITE_PUSHER_APP_CLUSTER || import.meta.env.VITE_REVERB_APP_CLUSTER || 'mt1',
            wsHost: import.meta.env.VITE_PUSHER_HOST || import.meta.env.VITE_REVERB_HOST || window.location.hostname,
            wsPort: import.meta.env.VITE_PUSHER_PORT || import.meta.env.VITE_REVERB_PORT || 8080,
            wssPort: import.meta.env.VITE_PUSHER_PORT || import.meta.env.VITE_REVERB_PORT || 8080,
            forceTLS: (import.meta.env.VITE_PUSHER_SCHEME || import.meta.env.VITE_REVERB_SCHEME || 'http') === 'https',
            disableStats: true,
            enabledTransports: ['ws', 'wss'],
            authEndpoint: '/broadcasting/auth',
            auth: {
                headers: {
                    'X-CSRF-TOKEN': document.head.querySelector('meta[name="csrf-token"]')?.content || '',
                    'X-Requested-With': 'XMLHttpRequest',
                    'Accept': 'application/json',
                },
            },
            withCredentials: true,
        };

        // If using Pusher.com (external service), remove host/port configs
        if (pusherKey && pusherKey !== 'your-app-key') {
            delete config.wsHost;
            delete config.wsPort;
            delete config.wssPort;
        }

        window.Echo = new Echo.default(config);
        
        // Only log in development
        if (import.meta.env.DEV) {
            console.log('Echo initialized successfully with', pusherKey ? 'Pusher.com' : 'Reverb');
        }
    }).catch((error) => {
        console.warn('Failed to initialize Echo:', error);
    });
} else {
    // Only log in development
    if (import.meta.env.DEV) {
        console.log('Broadcasting disabled - Echo not initialized');
    }
}



// Set CSRF token for all requests
const token = document.head.querySelector('meta[name="csrf-token"]');
if (token) {
    // window.axios.defaults.headers.common['X-CSRF-TOKEN'] = token.content;
} else {
    console.error('CSRF token not found: https://laravel.com/docs/csrf#csrf-x-csrf-token');
}

// Simplified CSRF Token Management System
// Following Laravel + Inertia.js best practices: Let the framework handle CSRF automatically
class CSRFTokenManager {
    constructor() {
        this.isRefreshing = false;
        this.refreshPromise = null;
        this.failedQueue = [];
    }

    // Get current token from meta tag
    getCurrentToken() {
        const metaTag = document.head.querySelector('meta[name="csrf-token"]');
        return metaTag ? metaTag.getAttribute('content') : null;
    }

    // Update CSRF token globally (only when explicitly needed)
    updateToken(newToken) {
        // Update meta tag
        const metaTag = document.head.querySelector('meta[name="csrf-token"]');
        if (metaTag) {
            metaTag.setAttribute('content', newToken);
        }
        
        // Update axios default headers
        // window.axios.defaults.headers.common['X-CSRF-TOKEN'] = newToken;
    }

    // Refresh CSRF token from server (only on actual 419 errors)
    async refreshToken() {
        // Prevent multiple simultaneous refresh requests
        if (this.isRefreshing) {
            return this.refreshPromise;
        }

        this.isRefreshing = true;
        this.refreshPromise = this.performTokenRefresh();

        try {
            const newToken = await this.refreshPromise;
            this.updateToken(newToken);
            return newToken;
        } finally {
            this.isRefreshing = false;
            this.refreshPromise = null;
        }
    }

    // Perform the actual token refresh
    async performTokenRefresh() {
        try {
            // Use session()->token() endpoint to avoid unnecessary token rotation
            const response = await fetch('/csrf-token', {
                method: 'GET',
                headers: {
                    'Accept': 'application/json',
                },
                credentials: 'same-origin' // Ensure cookies are sent
            });
            
            if (response.ok) {
                const data = await response.json();
                return data.token;
            } else {
                throw new Error('Failed to refresh CSRF token');
            }
        } catch (error) {
            console.error('Failed to refresh CSRF token:', error);
            throw error;
        }
    }

    // Handle failed requests and retry with fresh token (defensive only)
    async handleFailedRequest(error) {
        if (error.response && error.response.status === 419) {
            const originalRequest = error.config;
            
            // Prevent infinite retry loops - only retry once
            if (originalRequest._csrfRetryCount && originalRequest._csrfRetryCount >= 1) {
                console.error('CSRF token refresh failed after retry. Session may have expired.');
                this.showSessionExpiredToast();
                return Promise.reject(error);
            }
            
            // Mark this request as being retried
            originalRequest._csrfRetryCount = (originalRequest._csrfRetryCount || 0) + 1;
            
            try {
                // Refresh token once
                await this.refreshToken();
                
                // Ensure we do not send a stale manual header
                if (originalRequest.headers) {
                    delete originalRequest.headers['X-CSRF-TOKEN'];
                }
                
                // Retry the original request
                return axios(originalRequest);
            } catch (refreshError) {
                console.error('Failed to refresh token for retry:', refreshError);
                this.showSessionExpiredToast();
                throw error; // Re-throw original error
            }
        }
        
        return Promise.reject(error);
    }

    // Show session expired toast notification
    showSessionExpiredToast() {
        // Check if toast already exists
        if (document.getElementById('csrf-session-expired-toast')) {
            return;
        }

        // Create toast element
        const toast = document.createElement('div');
        toast.id = 'csrf-session-expired-toast';
        toast.style.cssText = `
            position: fixed;
            bottom: 2rem;
            right: 2rem;
            z-index: 9999;
            background: linear-gradient(135deg, #ef4444 0%, #dc2626 100%);
            color: white;
            padding: 1.25rem 1.5rem;
            border-radius: 0.75rem;
            box-shadow: 0 20px 25px -5px rgba(0, 0, 0, 0.1), 0 10px 10px -5px rgba(0, 0, 0, 0.04);
            display: flex;
            align-items: center;
            gap: 1rem;
            min-width: 350px;
            max-width: 450px;
            animation: slideInFromBottom 0.3s ease-out;
        `;

        toast.innerHTML = `
            <style>
                @keyframes slideInFromBottom {
                    from {
                        transform: translateY(100%);
                        opacity: 0;
                    }
                    to {
                        transform: translateY(0);
                        opacity: 1;
                    }
                }
                @keyframes slideOutToBottom {
                    from {
                        transform: translateY(0);
                        opacity: 1;
                    }
                    to {
                        transform: translateY(100%);
                        opacity: 0;
                    }
                }
            </style>
            <svg width="24" height="24" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round" style="flex-shrink: 0;">
                <circle cx="12" cy="12" r="10"></circle>
                <line x1="12" y1="8" x2="12" y2="12"></line>
                <line x1="12" y1="16" x2="12.01" y2="16"></line>
            </svg>
            <div style="flex: 1;">
                <p style="font-weight: 600; margin: 0 0 0.25rem 0; font-size: 0.95rem;">Session Expired</p>
                <p style="font-size: 0.875rem; margin: 0; opacity: 0.95;">
                    Your session has expired. Please <button onclick="window.location.reload()" style="background: none; border: none; color: white; text-decoration: underline; cursor: pointer; font-size: 0.875rem; padding: 0; font-weight: 600;">refresh the page</button> to continue.
                </p>
            </div>
            <button onclick="this.parentElement.remove()" style="background: rgba(255, 255, 255, 0.2); border: none; color: white; width: 28px; height: 28px; border-radius: 0.375rem; cursor: pointer; display: flex; align-items: center; justify-content: center; flex-shrink: 0; transition: background 0.2s; hover: background: rgba(255, 255, 255, 0.3);" onmouseover="this.style.background='rgba(255, 255, 255, 0.3)'" onmouseout="this.style.background='rgba(255, 255, 255, 0.2)'">
                <svg width="16" height="16" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round">
                    <line x1="18" y1="6" x2="6" y2="18"></line>
                    <line x1="6" y1="6" x2="18" y2="18"></line>
                </svg>
            </button>
        `;

        document.body.appendChild(toast);

        // Auto-remove after 30 seconds
        setTimeout(() => {
            if (toast.parentElement) {
                toast.style.animation = 'slideOutToBottom 0.3s ease-in forwards';
                setTimeout(() => toast.remove(), 300);
            }
        }, 30000);
    }
}

// Create global token manager instance
window.csrfTokenManager = new CSRFTokenManager();

// Create a function to update CSRF token globally (backward compatibility)
window.updateCSRFToken = (newToken) => {
    window.csrfTokenManager.updateToken(newToken);
};

// Add response interceptor to handle CSRF token failures
axios.interceptors.response.use(
    (response) => response,
    async (error) => {
        return window.csrfTokenManager.handleFailedRequest(error);
    }
);

// Utility functions for debugging and monitoring
window.CSRFDebug = {
    // Get current token info
    getTokenInfo() {
        const token = window.csrfTokenManager?.getCurrentToken();
        return {
            token: token ? token.substring(0, 10) + '...' : null,
            length: token?.length || 0,
            timestamp: new Date().toISOString()
        };
    },

    // Force token refresh (use sparingly - for debugging only)
    async forceRefresh() {
        if (window.csrfTokenManager) {
            console.warn('Manual token refresh triggered - this should only be used for debugging');
            return await window.csrfTokenManager.refreshToken();
        }
        return null;
    },

    // Log current state
    logState() {
        console.log('CSRF Token State:', {
            tokenInfo: this.getTokenInfo(),
            managerAvailable: !!window.csrfTokenManager
        });
    }
};
