/**
 * A simple confirmation dialog utility
 * This uses the browser's native confirm() by default,
 * but can be replaced with a more custom implementation
 *
 * @param {Object} options Configuration options for the dialog
 * @param {string} options.title Dialog title (default: 'Confirm')
 * @param {string} options.message Confirmation message to display
 * @param {string} options.confirmText Text for confirm button (default: 'Confirm')
 * @param {string} options.cancelText Text for cancel button (default: 'Cancel')
 * @param {boolean} options.dangerous Whether this is a dangerous action (default: false)
 * @returns {Promise<boolean>} Resolves to true if confirmed, false if cancelled
 */
export function confirmDialog({
    title = "Confirm",
    message = "Are you sure you want to continue?",
    confirmText = "Confirm",
    cancelText = "Cancel",
    dangerous = false,
} = {}) {
    // Currently using browser's native confirm dialog
    // This can be replaced with a custom modal implementation later
    return new Promise((resolve) => {
        const confirmed = window.confirm(message);
        resolve(confirmed);
    });
}

/**
 * Convenience method for deletion confirmation
 *
 * @param {string} itemName Name of the item being deleted
 * @returns {Promise<boolean>} Resolves to true if confirmed, false if cancelled
 */
export function confirmDelete(itemName = "item") {
    return confirmDialog({
        title: "Confirm Deletion",
        message: `Are you sure you want to delete this ${itemName}? This action cannot be undone.`,
        confirmText: "Delete",
        cancelText: "Cancel",
        dangerous: true,
    });
}

/**
 * Convenience method for bulk deletion confirmation
 *
 * @param {number} count Number of items being deleted
 * @param {string} itemName Name of the items being deleted
 * @returns {Promise<boolean>} Resolves to true if confirmed, false if cancelled
 */
export function confirmBulkDelete(count, itemName = "items") {
    return confirmDialog({
        title: "Confirm Bulk Deletion",
        message: `Are you sure you want to delete ${count} ${itemName}? This action cannot be undone.`,
        confirmText: "Delete All",
        cancelText: "Cancel",
        dangerous: true,
    });
}

/**
 * Convenience method for status change confirmation
 *
 * @param {string} action The action being performed (e.g., 'activate', 'deactivate')
 * @param {string} itemName Name of the item being modified
 * @returns {Promise<boolean>} Resolves to true if confirmed, false if cancelled
 */
export function confirmStatusChange(action, itemName = "item") {
    return confirmDialog({
        title: `Confirm ${action}`,
        message: `Are you sure you want to ${action} this ${itemName}?`,
        confirmText: "Yes, Continue",
        cancelText: "Cancel",
    });
}

/**
 * Convenience method for bulk status change confirmation
 *
 * @param {string} action The action being performed (e.g., 'activate', 'deactivate')
 * @param {number} count Number of items being modified
 * @param {string} itemName Name of the items being modified
 * @returns {Promise<boolean>} Resolves to true if confirmed, false if cancelled
 */
export function confirmBulkStatusChange(action, count, itemName = "items") {
    return confirmDialog({
        title: `Confirm Bulk ${action}`,
        message: `Are you sure you want to ${action} ${count} ${itemName}?`,
        confirmText: "Yes, Continue",
        cancelText: "Cancel",
    });
}

export default {
    confirm: confirmDialog,
    delete: confirmDelete,
    bulkDelete: confirmBulkDelete,
    statusChange: confirmStatusChange,
    bulkStatusChange: confirmBulkStatusChange,
};
