/**
 * Utility functions for formatting data consistently throughout the application
 */

/**
 * Format a date string or object into a human-readable format
 *
 * @param {string|Date} date - The date to format
 * @param {string} format - Optional format specification (default: 'medium')
 * @returns {string} Formatted date string
 */
export const formatDate = (date, format = "medium") => {
    if (!date) return "N/A";

    try {
        const dateObj = typeof date === "string" ? new Date(date) : date;

        // Check if date is valid
        if (isNaN(dateObj.getTime())) {
            return "Invalid date";
        }

        // Different format options
        switch (format) {
            case "short":
                return dateObj.toLocaleDateString("en-US", {
                    month: "short",
                    day: "numeric",
                    year: "numeric",
                });
            case "long":
                return dateObj.toLocaleDateString("en-US", {
                    weekday: "long",
                    month: "long",
                    day: "numeric",
                    year: "numeric",
                });
            case "time":
                return dateObj.toLocaleTimeString("en-US", {
                    hour: "2-digit",
                    minute: "2-digit",
                });
            case "full":
                return dateObj.toLocaleDateString("en-US", {
                    weekday: "long",
                    month: "long",
                    day: "numeric",
                    year: "numeric",
                    hour: "2-digit",
                    minute: "2-digit",
                });
            case "medium":
            default:
                return dateObj.toLocaleDateString("en-US", {
                    month: "long",
                    day: "numeric",
                    year: "numeric",
                });
        }
    } catch (error) {
        console.error("Error formatting date:", error);
        return "Error formatting date";
    }
};

/**
 * Format a number as currency
 *
 * @param {number} amount - The amount to format
 * @param {string} currency - Currency code (default: 'USD')
 * @returns {string} Formatted currency string
 */
export const formatCurrency = (amount, currency = "USD") => {
    if (amount === null || amount === undefined) return "N/A";

    return new Intl.NumberFormat("en-US", {
        style: "currency",
        currency: currency,
        minimumFractionDigits: 2,
    }).format(amount);
};

/**
 * Format a number with thousand separators
 *
 * @param {number} num - The number to format
 * @param {number} decimals - Number of decimal places (default: 0)
 * @returns {string} Formatted number string
 */
export const formatNumber = (num, decimals = 0) => {
    if (num === null || num === undefined) return "N/A";

    return new Intl.NumberFormat("en-US", {
        minimumFractionDigits: decimals,
        maximumFractionDigits: decimals,
    }).format(num);
};

/**
 * Format a percentage value
 *
 * @param {number} value - The percentage value (0-100)
 * @param {number} decimals - Number of decimal places (default: 1)
 * @returns {string} Formatted percentage string
 */
export const formatPercent = (value, decimals = 1) => {
    if (value === null || value === undefined) return "N/A";

    return `${parseFloat(value).toFixed(decimals)}%`;
};

/**
 * Format a deadline with relative time
 *
 * @param {string|Date} deadline - The deadline to format
 * @returns {string} Formatted deadline string
 */
export const formatDeadline = (deadline) => {
    if (!deadline) return "N/A";

    try {
        const deadlineDate = new Date(deadline);
        const now = new Date();
        const diffTime = deadlineDate.getTime() - now.getTime();
        const diffDays = Math.ceil(diffTime / (1000 * 60 * 60 * 24));

        if (diffDays < 0) {
            return `Expired ${Math.abs(diffDays)} day${Math.abs(diffDays) !== 1 ? 's' : ''} ago`;
        } else if (diffDays === 0) {
            return "Due today";
        } else if (diffDays === 1) {
            return "Due tomorrow";
        } else if (diffDays <= 7) {
            return `Due in ${diffDays} days`;
        } else {
            return deadlineDate.toLocaleDateString("en-US", {
                month: "short",
                day: "numeric",
                year: "numeric",
            });
        }
    } catch (error) {
        console.error("Error formatting deadline:", error);
        return "Error formatting deadline";
    }
};

/**
 * Get status color classes for different order statuses
 *
 * @param {string} status - The order status
 * @returns {string} CSS classes for the status
 */
export const getStatusColor = (status) => {
    const statusColors = {
        'assigned': 'bg-blue-100 text-blue-800 border-blue-200',
        'in_progress': 'bg-yellow-100 text-yellow-800 border-yellow-200',
        'submitted': 'bg-green-100 text-green-800 border-green-200',
        'under_review': 'bg-purple-100 text-purple-800 border-purple-200',
        'revision_requested': 'bg-orange-100 text-orange-800 border-orange-200',
        'approved': 'bg-green-100 text-green-800 border-green-200',
        'cancelled': 'bg-red-100 text-red-800 border-red-200'
    };
    return statusColors[status] || 'bg-gray-100 text-gray-800 border-gray-200';
};

/**
 * Get status icon for different order statuses
 *
 * @param {string} status - The order status
 * @returns {string} Icon name or component
 */
export const getStatusIcon = (status) => {
    const statusIcons = {
        'assigned': 'BookOpen',
        'in_progress': 'Edit3',
        'submitted': 'CheckCircle',
        'under_review': 'Eye',
        'revision_requested': 'AlertCircle',
        'approved': 'CheckCircle',
        'cancelled': 'XCircle'
    };
    return statusIcons[status] || 'FileText';
};
