/**
 * Blog Image Utilities
 * Centralized functions for managing blog post images and alt text
 */

/**
 * Get the appropriate image path for a blog post
 * @param {Object} post - The blog post object
 * @param {string} size - Image size ('thumbnail', 'medium', 'large', 'featured')
 * @param {string} format - Image format preference ('webp', 'jpg', 'png')
 * @returns {string} - The image path
 */
export const getBlogPostImage = (post, size = "featured", format = "webp") => {
    // If post has a specific featured image, use it
    if (post?.featuredImage) {
        return post.featuredImage;
    }

    // If post has a slug, try to generate image path from slug
    if (post?.slug) {
        const baseImagePath = `/images/blog/${post.slug}`;

        // Size variations
        const sizeMap = {
            thumbnail: "-thumb",
            medium: "-medium",
            large: "-large",
            featured: "",
        };

        const sizeSuffix = sizeMap[size] || "";

        // Try WebP first, then fallback to JPG
        if (format === "webp") {
            return `${baseImagePath}${sizeSuffix}.webp`;
        }

        return `${baseImagePath}${sizeSuffix}.jpg`;
    }

    // Category-based fallback images
    if (post?.category) {
        const categoryImages = {
            "essay-writing": "/images/blog/defaults/essay-writing-default",
            "research-papers": "/images/blog/defaults/research-papers-default",
            "academic-writing":
                "/images/blog/defaults/academic-writing-default",
            "study-tips": "/images/blog/defaults/study-tips-default",
            "citation-guides": "/images/blog/defaults/citation-guides-default",
            "writing-tips": "/images/blog/defaults/writing-tips-default",
        };

        const categoryImage = categoryImages[post.category];
        if (categoryImage) {
            const extension = format === "webp" ? ".webp" : ".jpg";
            return `${categoryImage}${extension}`;
        }
    }

    // Ultimate fallback - generic blog image
    const extension = format === "webp" ? ".webp" : ".jpg";
    return `/images/blog/defaults/blog-post-default${extension}`;
};

/**
 * Get optimized image sources for responsive images
 * @param {Object} post - The blog post object
 * @returns {Object} - Object with different image sources and sizes
 */
export const getBlogPostImageSources = (post) => {
    const baseSlug = post?.slug || "default";
    const basePath = post?.featuredImage
        ? post.featuredImage.replace(/\.[^/.]+$/, "")
        : `/images/blog/${baseSlug}`;

    return {
        // WebP sources
        webp: {
            thumbnail: `${basePath}-thumb.webp`,
            medium: `${basePath}-medium.webp`,
            large: `${basePath}-large.webp`,
            featured: `${basePath}.webp`,
        },
        // Fallback sources
        fallback: {
            thumbnail: `${basePath}-thumb.jpg`,
            medium: `${basePath}-medium.jpg`,
            large: `${basePath}-large.jpg`,
            featured: `${basePath}.jpg`,
        },
        // Responsive sizes
        sizes: {
            thumbnail: "150px",
            medium: "400px",
            large: "800px",
            featured: "1200px",
        },
    };
};

/**
 * Generate appropriate alt text for blog post images
 * @param {Object} post - The blog post object
 * @param {string} context - Context where image is used ('featured', 'thumbnail', 'inline')
 * @returns {string} - The alt text
 */
export const getBlogImageAlt = (post, context = "featured") => {
    // If post already has specific alt text, use it
    if (post?.featuredImageAlt) {
        return post.featuredImageAlt;
    }

    // If post has alt text in meta, use it
    if (post?.meta?.imageAlt) {
        return post.meta.imageAlt;
    }

    // Generate alt text based on post content
    if (post?.title) {
        const categoryName = getCategoryDisplayName(post.category);

        // Context-specific alt text
        switch (context) {
            case "thumbnail":
                return `${post.title} - ${categoryName} thumbnail`;
            case "featured":
                return `${post.title} - ${categoryName} guide and tips`;
            case "inline":
                return `Illustration for ${post.title}`;
            case "card":
                return `${
                    post.title
                } - Read our ${categoryName.toLowerCase()} guide`;
            default:
                return `${post.title} - ${categoryName}`;
        }
    }

    // Category-based fallback alt text
    if (post?.category) {
        const categoryName = getCategoryDisplayName(post.category);
        return `${categoryName} - Academic writing tips and guides`;
    }

    // Ultimate fallback
    return "Academic writing blog post - Tips and guides for students";
};

/**
 * Get category display name for alt text
 * @param {string} categorySlug - The category slug
 * @returns {string} - Human-readable category name
 */
const getCategoryDisplayName = (categorySlug) => {
    const categoryNames = {
        "essay-writing": "Essay Writing",
        "research-papers": "Research Papers",
        "academic-writing": "Academic Writing",
        "study-tips": "Study Tips",
        "citation-guides": "Citation Guides",
        "writing-tips": "Writing Tips",
        "thesis-writing": "Thesis Writing",
        "dissertation-help": "Dissertation Help",
        "literature-review": "Literature Review",
    };

    return categoryNames[categorySlug] || "Academic Writing";
};

/**
 * Check if an image exists (for fallback handling)
 * @param {string} imagePath - Path to the image
 * @returns {Promise<boolean>} - Whether the image exists
 */
export const checkImageExists = async (imagePath) => {
    try {
        const response = await fetch(imagePath, { method: "HEAD" });
        return response.ok;
    } catch (error) {
        return false;
    }
};

/**
 * Get the best available image format for the browser
 * @returns {string} - Preferred image format ('webp' or 'jpg')
 */
export const getPreferredImageFormat = () => {
    // Check if browser supports WebP
    if (typeof window !== "undefined") {
        const canvas = document.createElement("canvas");
        canvas.width = 1;
        canvas.height = 1;

        // Check WebP support
        const webpSupported =
            canvas.toDataURL("image/webp").indexOf("data:image/webp") === 0;
        return webpSupported ? "webp" : "jpg";
    }

    // Default to WebP for modern environments
    return "webp";
};

/**
 * Generate srcSet for responsive images
 * @param {Object} post - The blog post object
 * @param {string} format - Image format ('webp' or 'jpg')
 * @returns {string} - srcSet string for responsive images
 */
export const generateImageSrcSet = (post, format = "webp") => {
    const sources = getBlogPostImageSources(post);
    const imageSet = format === "webp" ? sources.webp : sources.fallback;

    return [
        `${imageSet.thumbnail} 150w`,
        `${imageSet.medium} 400w`,
        `${imageSet.large} 800w`,
        `${imageSet.featured} 1200w`,
    ].join(", ");
};

/**
 * Get image dimensions for layout optimization
 * @param {string} size - Image size ('thumbnail', 'medium', 'large', 'featured')
 * @returns {Object} - Width and height dimensions
 */
export const getImageDimensions = (size = "featured") => {
    const dimensions = {
        thumbnail: { width: 150, height: 100 },
        medium: { width: 400, height: 250 },
        large: { width: 800, height: 450 },
        featured: { width: 1200, height: 630 },
    };

    return dimensions[size] || dimensions.featured;
};

/**
 * Generate Open Graph image URL for blog posts
 * @param {Object} post - The blog post object
 * @returns {string} - Open Graph optimized image URL
 */
export const getBlogPostOGImage = (post) => {
    // Use featured image if available
    if (post?.featuredImage) {
        return post.featuredImage;
    }

    // Generate OG image path
    if (post?.slug) {
        return `/images/blog/og/${post.slug}-og.jpg`;
    }

    // Category-based OG image
    if (post?.category) {
        return `/images/blog/og/${post.category}-og.jpg`;
    }

    // Default OG image
    return "/images/blog/og/blog-default-og.jpg";
};

/**
 * Preload critical blog images
 * @param {Array} posts - Array of blog posts
 * @param {number} limit - Number of images to preload
 */
export const preloadBlogImages = (posts, limit = 3) => {
    if (typeof window === "undefined" || !posts?.length) return;

    const format = getPreferredImageFormat();

    posts.slice(0, limit).forEach((post) => {
        const imagePath = getBlogPostImage(post, "featured", format);

        // Create preload link
        const link = document.createElement("link");
        link.rel = "preload";
        link.as = "image";
        link.href = imagePath;
        link.type = `image/${format}`;

        document.head.appendChild(link);

        // Clean up after 10 seconds
        setTimeout(() => {
            if (link.parentNode) {
                link.parentNode.removeChild(link);
            }
        }, 10000);
    });
};

/**
 * Generate structured data for blog post images
 * @param {Object} post - The blog post object
 * @returns {Object} - Structured data for the image
 */
export const getBlogImageStructuredData = (post) => {
    const imagePath = getBlogPostImage(post, "featured", "jpg");
    const altText = getBlogImageAlt(post, "featured");
    const dimensions = getImageDimensions("featured");

    return {
        "@type": "ImageObject",
        url: `${window.location.origin}${imagePath}`,
        width: dimensions.width,
        height: dimensions.height,
        caption: altText,
        description: post?.excerpt || altText,
    };
};
