/**
 * Order State Tester - Comprehensive testing utility for order state management
 */

import { OrderStateManager } from './orderStateManager';
import { URLStateManager } from './urlStateManager';

export class OrderStateTester {
    static testResults = {
        localStorage: {},
        urlState: {},
        integration: {},
        performance: {}
    };

    /**
     * Test localStorage functionality
     */
    static testLocalStorage() {
        console.group('🧪 Testing LocalStorage State Management');
        
        try {
            // Test 1: Save and load basic data
            const testData = {
                title: 'Test Essay',
                subject: 'Computer Science',
                pages: 5,
                currentStep: 2
            };
            
            OrderStateManager.saveOrderState(testData);
            const loadedData = OrderStateManager.loadOrderState();
            
            this.testResults.localStorage.basicSaveLoad = 
                loadedData && loadedData.title === testData.title;
            
            console.log('✅ Basic save/load:', this.testResults.localStorage.basicSaveLoad);

            // Test 2: Data validation
            const invalidData = { invalid: 'data' };
            OrderStateManager.saveOrderState(invalidData);
            const validatedData = OrderStateManager.loadOrderState();
            
            this.testResults.localStorage.validation = 
                validatedData && validatedData.version;
            
            console.log('✅ Data validation:', this.testResults.localStorage.validation);

            // Test 3: Expiration handling
            const expiredData = {
                ...testData,
                timestamp: Date.now() - (25 * 60 * 60 * 1000) // 25 hours ago
            };
            OrderStateManager.saveOrderState(expiredData);
            const expiredLoaded = OrderStateManager.loadOrderState();
            
            this.testResults.localStorage.expiration = !expiredLoaded;
            
            console.log('✅ Expiration handling:', this.testResults.localStorage.expiration);

            // Test 4: Clear functionality
            OrderStateManager.clearOrderState();
            const clearedData = OrderStateManager.loadOrderState();
            
            this.testResults.localStorage.clear = !clearedData;
            
            console.log('✅ Clear functionality:', this.testResults.localStorage.clear);

        } catch (error) {
            console.error('❌ LocalStorage test failed:', error);
            this.testResults.localStorage.error = error.message;
        }
        
        console.groupEnd();
    }

    /**
     * Test URL state functionality
     */
    static testURLState() {
        console.group('🧪 Testing URL State Management');
        
        try {
            // Test 1: Encode and decode
            const testData = {
                title: 'Test Essay',
                subject: 'Computer Science',
                pages: 5
            };
            const testStep = 3;
            
            const encoded = URLStateManager.encodeOrderState(testData, testStep);
            const decoded = URLStateManager.decodeOrderState();
            
            this.testResults.urlState.encodeDecode = 
                encoded && encoded.includes('title') && encoded.includes('step');
            
            console.log('✅ Encode/decode:', this.testResults.urlState.encodeDecode);

            // Test 2: URL validation
            const isValid = URLStateManager.validateURLParams();
            this.testResults.urlState.validation = isValid;
            
            console.log('✅ URL validation:', this.testResults.urlState.validation);

            // Test 3: URL update
            const updateSuccess = URLStateManager.updateURL(testData, testStep);
            this.testResults.urlState.update = updateSuccess;
            
            console.log('✅ URL update:', this.testResults.urlState.update);

            // Test 4: Clear URL state
            const clearSuccess = URLStateManager.clearURLState();
            this.testResults.urlState.clear = clearSuccess;
            
            console.log('✅ URL clear:', this.testResults.urlState.clear);

        } catch (error) {
            console.error('❌ URL state test failed:', error);
            this.testResults.urlState.error = error.message;
        }
        
        console.groupEnd();
    }

    /**
     * Test integration between systems
     */
    static testIntegration() {
        console.group('🧪 Testing System Integration');
        
        try {
            // Test 1: Priority system (URL over localStorage)
            const localStorageData = { title: 'Local Data', step: 2 };
            const urlData = { title: 'URL Data', step: 3 };
            
            OrderStateManager.saveOrderState(localStorageData);
            URLStateManager.updateURL(urlData, 3);
            
            // Simulate URL state restoration
            const urlState = URLStateManager.decodeOrderState();
            const localState = OrderStateManager.loadOrderState();
            
            this.testResults.integration.priority = 
                urlState && urlState.orderData.title === 'URL Data';
            
            console.log('✅ Priority system:', this.testResults.integration.priority);

            // Test 2: State synchronization
            const syncData = { title: 'Sync Test', subject: 'Math' };
            OrderStateManager.saveOrderState(syncData);
            URLStateManager.updateURL(syncData, 2);
            
            this.testResults.integration.sync = 
                URLStateManager.hasOrderState() && 
                OrderStateManager.getStateSummary().hasData;
            
            console.log('✅ State synchronization:', this.testResults.integration.sync);

            // Test 3: Error recovery
            try {
                // Simulate corrupted data
                localStorage.setItem('orderFormData', 'invalid json');
                const recovered = OrderStateManager.loadOrderState();
                
                this.testResults.integration.recovery = !recovered;
                
                console.log('✅ Error recovery:', this.testResults.integration.recovery);
            } catch (error) {
                this.testResults.integration.recovery = true;
                console.log('✅ Error recovery:', true);
            }

        } catch (error) {
            console.error('❌ Integration test failed:', error);
            this.testResults.integration.error = error.message;
        }
        
        console.groupEnd();
    }

    /**
     * Test performance characteristics
     */
    static testPerformance() {
        console.group('🧪 Testing Performance');
        
        try {
            // Test 1: Save performance
            const startTime = performance.now();
            const largeData = {
                title: 'Performance Test',
                instructions: 'A'.repeat(1000), // Large data
                currentStep: 4
            };
            
            OrderStateManager.saveOrderState(largeData);
            const saveTime = performance.now() - startTime;
            
            this.testResults.performance.saveTime = saveTime;
            this.testResults.performance.saveAcceptable = saveTime < 50; // 50ms threshold
            
            console.log('✅ Save performance:', `${saveTime.toFixed(2)}ms`);

            // Test 2: Load performance
            const loadStart = performance.now();
            OrderStateManager.loadOrderState();
            const loadTime = performance.now() - loadStart;
            
            this.testResults.performance.loadTime = loadTime;
            this.testResults.performance.loadAcceptable = loadTime < 30; // 30ms threshold
            
            console.log('✅ Load performance:', `${loadTime.toFixed(2)}ms`);

            // Test 3: URL encoding performance
            const urlStart = performance.now();
            URLStateManager.encodeOrderState(largeData, 4);
            const urlTime = performance.now() - urlStart;
            
            this.testResults.performance.urlTime = urlTime;
            this.testResults.performance.urlAcceptable = urlTime < 20; // 20ms threshold
            
            console.log('✅ URL encoding performance:', `${urlTime.toFixed(2)}ms`);

        } catch (error) {
            console.error('❌ Performance test failed:', error);
            this.testResults.performance.error = error.message;
        }
        
        console.groupEnd();
    }

    /**
     * Run all tests
     */
    static runAllTests() {
        console.log('🚀 Starting Order State Management Tests...');
        
        this.testLocalStorage();
        this.testURLState();
        this.testIntegration();
        this.testPerformance();
        
        this.generateReport();
    }

    /**
     * Generate test report
     */
    static generateReport() {
        console.group('📊 Test Report');
        
        const allTests = {
            localStorage: Object.values(this.testResults.localStorage).filter(v => typeof v === 'boolean'),
            urlState: Object.values(this.testResults.urlState).filter(v => typeof v === 'boolean'),
            integration: Object.values(this.testResults.integration).filter(v => typeof v === 'boolean'),
            performance: Object.values(this.testResults.performance).filter(v => typeof v === 'boolean')
        };
        
        const totalTests = allTests.localStorage.length + allTests.urlState.length + 
                          allTests.integration.length + allTests.performance.length;
        const passedTests = allTests.localStorage.filter(Boolean).length + 
                           allTests.urlState.filter(Boolean).length + 
                           allTests.integration.filter(Boolean).length + 
                           allTests.performance.filter(Boolean).length;
        
        console.log(`📈 Overall: ${passedTests}/${totalTests} tests passed`);
        
        // Performance summary
        if (this.testResults.performance.saveTime) {
            console.log(`⚡ Performance: Save=${this.testResults.performance.saveTime.toFixed(2)}ms, Load=${this.testResults.performance.loadTime.toFixed(2)}ms, URL=${this.testResults.performance.urlTime.toFixed(2)}ms`);
        }
        
        // Recommendations
        const recommendations = [];
        if (!this.testResults.performance.saveAcceptable) {
            recommendations.push('Optimize save performance');
        }
        if (!this.testResults.performance.loadAcceptable) {
            recommendations.push('Optimize load performance');
        }
        if (Object.values(this.testResults.localStorage).some(v => v === false)) {
            recommendations.push('Fix localStorage issues');
        }
        if (Object.values(this.testResults.urlState).some(v => v === false)) {
            recommendations.push('Fix URL state issues');
        }
        
        if (recommendations.length > 0) {
            console.log('🔧 Recommendations:', recommendations.join(', '));
        } else {
            console.log('✅ All systems working optimally!');
        }
        
        console.groupEnd();
        
        return {
            passed: passedTests,
            total: totalTests,
            performance: this.testResults.performance,
            recommendations
        };
    }

    /**
     * Clean up test data
     */
    static cleanup() {
        OrderStateManager.clearOrderState();
        URLStateManager.clearURLState();
        console.log('🧹 Test cleanup completed');
    }
} 