/**
 * Performance Monitor - Tracks and optimizes order form operations
 */

export class PerformanceMonitor {
    static metrics = {
        saveOperations: [],
        loadOperations: [],
        urlOperations: [],
        authOperations: [],
        renderOperations: []
    };

    static thresholds = {
        save: 50,    // 50ms
        load: 30,    // 30ms
        url: 50,     // 50ms - increased to reduce warnings
        auth: 300,   // 300ms - increased for network operations
        render: 100  // 100ms
    };

    /**
     * Start timing an operation
     */
    static startTimer(operation) {
        return {
            operation,
            startTime: performance.now(),
            start: () => this.startTimer(operation)
        };
    }

    /**
     * End timing and record metrics
     */
    static endTimer(timer, success = true) {
        const duration = performance.now() - timer.startTime;
        const metric = {
            operation: timer.operation,
            duration,
            timestamp: Date.now(),
            success,
            threshold: this.thresholds[timer.operation] || 100
        };

        // Store metric
        if (this.metrics[`${timer.operation}Operations`]) {
            this.metrics[`${timer.operation}Operations`].push(metric);
        }

        // Keep only last 100 operations
        if (this.metrics[`${timer.operation}Operations`].length > 100) {
            this.metrics[`${timer.operation}Operations`] = 
                this.metrics[`${timer.operation}Operations`].slice(-100);
        }

        // Log slow operations
        if (duration > metric.threshold) {
            console.warn(`⚠️ Slow ${timer.operation} operation: ${duration.toFixed(2)}ms (threshold: ${metric.threshold}ms)`);
        }

        return metric;
    }

    /**
     * Monitor a function execution
     */
    static monitor(operation, fn) {
        const timer = this.startTimer(operation);
        try {
            const result = fn();
            this.endTimer(timer, true);
            return result;
        } catch (error) {
            this.endTimer(timer, false);
            throw error;
        }
    }

    /**
     * Monitor async function execution
     */
    static async monitorAsync(operation, fn) {
        const timer = this.startTimer(operation);
        try {
            const result = await fn();
            this.endTimer(timer, true);
            return result;
        } catch (error) {
            this.endTimer(timer, false);
            throw error;
        }
    }

    /**
     * Get performance statistics
     */
    static getStats(operation = null) {
        const operations = operation ? [operation] : Object.keys(this.metrics);
        const stats = {};

        operations.forEach(op => {
            const metrics = this.metrics[`${op}Operations`] || [];
            if (metrics.length === 0) return;

            const durations = metrics.map(m => m.duration);
            const successful = metrics.filter(m => m.success);
            const failed = metrics.filter(m => !m.success);

            stats[op] = {
                total: metrics.length,
                successful: successful.length,
                failed: failed.length,
                avgDuration: durations.reduce((a, b) => a + b, 0) / durations.length,
                minDuration: Math.min(...durations),
                maxDuration: Math.max(...durations),
                successRate: (successful.length / metrics.length) * 100,
                slowOperations: metrics.filter(m => m.duration > m.threshold).length
            };
        });

        return stats;
    }

    /**
     * Get performance recommendations
     */
    static getRecommendations() {
        const stats = this.getStats();
        const recommendations = [];

        Object.entries(stats).forEach(([operation, stat]) => {
            const threshold = this.thresholds[operation];
            
            if (stat.avgDuration > threshold * 0.8) {
                recommendations.push({
                    operation,
                    type: 'performance',
                    message: `${operation} operations are slow (avg: ${stat.avgDuration.toFixed(2)}ms)`,
                    priority: stat.avgDuration > threshold ? 'high' : 'medium'
                });
            }

            if (stat.successRate < 95) {
                recommendations.push({
                    operation,
                    type: 'reliability',
                    message: `${operation} has low success rate (${stat.successRate.toFixed(1)}%)`,
                    priority: stat.successRate < 90 ? 'high' : 'medium'
                });
            }

            if (stat.slowOperations > stat.total * 0.1) {
                recommendations.push({
                    operation,
                    type: 'consistency',
                    message: `${operation} has many slow operations (${stat.slowOperations}/${stat.total})`,
                    priority: 'medium'
                });
            }
        });

        return recommendations.sort((a, b) => {
            const priorityOrder = { high: 3, medium: 2, low: 1 };
            return priorityOrder[b.priority] - priorityOrder[a.priority];
        });
    }

    /**
     * Generate performance report
     */
    static generateReport() {
        const stats = this.getStats();
        const recommendations = this.getRecommendations();

        console.group('📊 Performance Report');
        
        Object.entries(stats).forEach(([operation, stat]) => {
            const threshold = this.thresholds[operation];
            const status = stat.avgDuration <= threshold ? '✅' : '⚠️';
            
            console.log(`${status} ${operation}:`);
            console.log(`  - Avg: ${stat.avgDuration.toFixed(2)}ms (threshold: ${threshold}ms)`);
            console.log(`  - Success: ${stat.successRate.toFixed(1)}%`);
            console.log(`  - Slow ops: ${stat.slowOperations}/${stat.total}`);
        });

        if (recommendations.length > 0) {
            console.log('\n🔧 Recommendations:');
            recommendations.forEach(rec => {
                const icon = rec.priority === 'high' ? '🔴' : rec.priority === 'medium' ? '🟡' : '🟢';
                console.log(`${icon} ${rec.message}`);
            });
        } else {
            console.log('\n✅ All operations performing well!');
        }

        console.groupEnd();

        return { stats, recommendations };
    }

    /**
     * Clear all metrics
     */
    static clearMetrics() {
        Object.keys(this.metrics).forEach(key => {
            this.metrics[key] = [];
        });
        console.log('🧹 Performance metrics cleared');
    }

    /**
     * Export metrics for analysis
     */
    static exportMetrics() {
        return {
            metrics: this.metrics,
            stats: this.getStats(),
            recommendations: this.getRecommendations(),
            timestamp: Date.now()
        };
    }
} 