/**
 * Smart image preloading - only preload what's actually needed
 */

/**
 * Preload critical images based on current page
 */
export const preloadCriticalImages = () => {
    if (typeof window === "undefined") return;

    // Always preload logo (appears on all pages)
    preloadImage("/images/logo.webp", "webp");
    preloadImage("/images/logo.jpg", "jpeg");

    // Get current path to determine what to preload
    const currentPath = window.location.pathname;

    // Home page - preload hero image only
    if (currentPath === "/" || currentPath === "/home") {
        preloadImage("/images/hero-academic-writers.webp", "webp");
        preloadImage("/images/hero-academic-writers.jpg", "jpeg");
    }

    // About page - preload about image
    if (currentPath === "/about") {
        preloadImage("/images/about-story.webp", "webp");
        preloadImage("/images/about-story.jpg", "jpeg");
    }

    // Pricing page - preload pricing images
    if (currentPath === "/pricing-page") {
        preloadImage("/images/Pricing Page Image.webp", "webp");
        preloadImage("/images/Pricing Page Image.jpg", "jpeg");
        preloadImage("/images/academic-writing-pricing.webp", "webp");
        preloadImage("/images/academic-writing-pricing.jpg", "jpeg");
    }

    // Service pages - preload specific service images
    if (currentPath.startsWith("/services/")) {
        const serviceName = currentPath.replace("/services/", "");
        preloadServiceImage(serviceName);
    }

    // Blog pages - preload blog images
    if (currentPath.startsWith("/blog")) {
        preloadBlogImages(currentPath);
    }
};

/**
 * Preload specific service image
 */
export const preloadServiceImage = (serviceName) => {
    if (typeof window === "undefined") return;

    const serviceImages = {
        "research-papers": "/images/research-paper-benefits",
        "dissertation-writing": "/images/dissertation-writing-benefits",
        "essay-writing": "/images/essay-writing-benefits",
        assignments: "/images/assignment-writing-benefits",
        "case-studies": "/images/case-study-writing-benefits",
        "literature-reviews": "/images/literature-review-benefits",
        "thesis-proposals": "/images/thesis-statement-writing-benefits",
        "thesis-statements": "/images/thesis-statement-writing-benefits",
    };

    const imagePath = serviceImages[serviceName];
    if (imagePath) {
        preloadImage(`${imagePath}.webp`, "webp");
        preloadImage(`${imagePath}.jpg`, "jpeg");
    }
};

/**
 * Preload blog-specific images based on page type
 */
export const preloadBlogImages = (currentPath) => {
    if (typeof window === "undefined") return;

    // Blog index page
    if (currentPath === "/blog") {
        // Preload available blog featured images
        preloadImage("/images/blog/essay-writing-tips-2025.webp", "webp");
        preloadImage("/images/blog/essay-writing-tips-2025.jpg", "jpeg");
        preloadImage(
            "/images/blog/essay-structure-guide-featured.webp",
            "webp"
        );
        preloadImage("/images/blog/essay-structure-guide-featured.jpg", "jpeg");
        preloadImage(
            "/images/blog/academic-citation-guide-featured.webp",
            "webp"
        );
        preloadImage(
            "/images/blog/academic-citation-guide-featured.jpg",
            "jpeg"
        );
    }

    // Blog category pages
    if (currentPath.startsWith("/blog/category/")) {
        const category = currentPath.replace("/blog/category/", "");
        preloadBlogCategoryImages(category);
    }

    // Individual blog posts
    if (
        currentPath.startsWith("/blog/") &&
        !currentPath.includes("/category/")
    ) {
        const slug = currentPath.replace("/blog/", "");
        preloadBlogPostImages(slug);
    }
};

/**
 * Preload blog category images
 */
export const preloadBlogCategoryImages = (category) => {
    if (typeof window === "undefined") return;

    // Since we don't have category-specific images yet, preload common blog images
    preloadImage("/images/blog/essay-writing-tips-2025.webp", "webp");
    preloadImage("/images/blog/essay-structure-guide-featured.webp", "webp");
    preloadImage("/images/blog/academic-citation-guide-featured.webp", "webp");
    preloadImage("/images/blog/essay-topic-ideas-featured.webp", "webp");

    // Preload category-specific post images
    preloadCategoryPostImages(category);
};

/**
 * Preload blog post images based on actual available images
 */
export const preloadBlogPostImages = (slug) => {
    if (typeof window === "undefined") return;

    // Map actual available blog images
    const availablePostImages = {
        "8-essay-writing-tips-for-students-2025": {
            featured: "/images/blog/essay-writing-tips-2025",
        },
        "essay-structure-guide": {
            featured: "/images/blog/essay-structure-guide-featured",
        },
        "academic-citation-guide": {
            featured: "/images/blog/academic-citation-guide-featured",
        },
        "essay-topic-ideas": {
            featured: "/images/blog/essay-topic-ideas-featured",
        },
    };

    const postImages = availablePostImages[slug];
    if (postImages && postImages.featured) {
        preloadImage(`${postImages.featured}.webp`, "webp");
        preloadImage(`${postImages.featured}.jpg`, "jpeg");
    }

    // Preload related post images (other available posts)
    Object.keys(availablePostImages).forEach((otherSlug) => {
        if (otherSlug !== slug && availablePostImages[otherSlug].featured) {
            preloadImage(
                `${availablePostImages[otherSlug].featured}.webp`,
                "webp"
            );
        }
    });
};

/**
 * Preload category-specific post images
 */
export const preloadCategoryPostImages = (category) => {
    if (typeof window === "undefined") return;

    const categoryPosts = {
        "essay-writing": [
            "/images/blog/essay-writing-tips-2025",
            "/images/blog/essay-structure-guide-featured",
            "/images/blog/essay-topic-ideas-featured",
        ],
        "research-papers": ["/images/blog/academic-citation-guide-featured"],
        "academic-writing": [
            "/images/blog/academic-citation-guide-featured",
            "/images/blog/essay-structure-guide-featured",
        ],
        "study-tips": ["/images/blog/essay-writing-tips-2025"],
    };

    const posts = categoryPosts[category] || [];
    posts.forEach((imagePath) => {
        preloadImage(`${imagePath}.webp`, "webp");
        preloadImage(`${imagePath}.jpg`, "jpeg");
    });
};

/**
 * Preload author images
 */
export const preloadAuthorImages = () => {
    if (typeof window === "undefined") return;

    const authorImages = [
        "/images/authors/dr-sarah-johnson",
        "/images/authors/prof-david-martinez",
        "/images/authors/prof-michael-thompson",
    ];

    authorImages.forEach((imagePath) => {
        preloadImage(`${imagePath}.webp`, "webp");
        preloadImage(`${imagePath}.jpg`, "jpeg");
    });
};

/**
 * Preload images for home page only (called from Home.jsx)
 */
export const preloadHomePageImages = () => {
    if (typeof window === "undefined") return;

    // Hero image (critical)
    preloadImage("/images/hero-academic-writers.webp", "webp");
    preloadImage("/images/hero-academic-writers.jpg", "jpeg");

    // About section image (below the fold but commonly viewed)
    setTimeout(() => {
        preloadImage("/images/about-story.webp", "webp");
        preloadImage("/images/about-story.jpg", "jpeg");
    }, 1000); // Delay non-critical preloads
};

/**
 * Preload a single image with better error handling
 */
export const preloadImage = (src, type = "webp") => {
    if (typeof window === "undefined") return;

    // Check if already preloaded
    const existingLink = document.querySelector(`link[href="${src}"]`);
    if (existingLink) return;

    // Check if image actually exists before preloading
    const testImage = new Image();
    testImage.onload = () => {
        // Image exists, create preload link
        const link = document.createElement("link");
        link.rel = "preload";
        link.as = "image";
        link.href = src;
        link.type = `image/${type}`;

        // Remove preload link after 10 seconds to avoid warnings
        setTimeout(() => {
            if (link.parentNode) {
                link.parentNode.removeChild(link);
            }
        }, 10000);

        document.head.appendChild(link);
    };

    testImage.onerror = () => {
        // Image doesn't exist, don't preload
        console.warn(`Image not found for preloading: ${src}`);
    };

    testImage.src = src;
};

/**
 * Preload images when they're about to be needed (intersection observer)
 */
export const preloadOnScroll = (imageSrc, threshold = 0.1) => {
    if (typeof window === "undefined" || !("IntersectionObserver" in window)) {
        return;
    }

    const observer = new IntersectionObserver(
        (entries) => {
            entries.forEach((entry) => {
                if (entry.isIntersecting) {
                    preloadImage(imageSrc, "webp");
                    observer.disconnect();
                }
            });
        },
        { threshold }
    );

    // Observe the viewport
    const trigger = document.createElement("div");
    trigger.style.position = "absolute";
    trigger.style.top = "50vh";
    trigger.style.height = "1px";
    trigger.style.width = "1px";
    trigger.style.opacity = "0";
    document.body.appendChild(trigger);

    observer.observe(trigger);
};

/**
 * Preload images for specific layout types
 */
export const preloadLayoutImages = (layoutType) => {
    if (typeof window === "undefined") return;

    const layoutImages = {
        "service-hero": [
            "/images/hero-academic-writers.webp",
            "/images/hero-academic-writers.jpg",
        ],
        "about-section": [
            "/images/about-story.webp",
            "/images/about-story.jpg",
        ],
        "pricing-section": [
            "/images/Pricing Page Image.webp",
            "/images/Pricing Page Image.jpg",
            "/images/academic-writing-pricing.webp",
            "/images/academic-writing-pricing.jpg",
        ],
    };

    const images = layoutImages[layoutType] || [];
    images.forEach((imageSrc) => {
        const extension = imageSrc.split(".").pop();
        preloadImage(imageSrc, extension === "webp" ? "webp" : "jpeg");
    });
};

/**
 * Batch preload multiple images with priority
 */
export const batchPreloadImages = (imageList, priority = "low") => {
    if (typeof window === "undefined") return;

    const delay = priority === "high" ? 0 : 100;

    imageList.forEach((imageSrc, index) => {
        setTimeout(() => {
            const extension = imageSrc.split(".").pop();
            preloadImage(imageSrc, extension === "webp" ? "webp" : "jpeg");
        }, delay * index);
    });
};

/**
 * Smart preloading based on user behavior
 */
export const smartPreload = () => {
    if (typeof window === "undefined") return;

    // Preload on hover for navigation links
    document.addEventListener(
        "mouseenter",
        (e) => {
            if (e.target.tagName === "A" && e.target.href) {
                const href = e.target.href;

                // Service pages
                if (href.includes("/services/")) {
                    const serviceName = href.split("/services/")[1];
                    preloadServiceImage(serviceName);
                }

                // Blog pages
                if (href.includes("/blog/")) {
                    preloadBlogImages(new URL(href).pathname);
                }

                // About page
                if (href.includes("/about")) {
                    preloadImage("/images/about-story.webp", "webp");
                }

                // Authors page
                if (href.includes("/authors") || href.includes("/writers")) {
                    preloadAuthorImages();
                }
            }
        },
        true
    );

    // Preload on scroll near bottom (for pagination)
    let ticking = false;
    window.addEventListener("scroll", () => {
        if (!ticking) {
            requestAnimationFrame(() => {
                const scrollPercent =
                    (window.scrollY /
                        (document.body.scrollHeight - window.innerHeight)) *
                    100;

                if (scrollPercent > 80) {
                    // User is near bottom, preload next page content
                    const currentPath = window.location.pathname;
                    if (currentPath.includes("/blog")) {
                        // Preload more blog content
                        preloadBlogImages(currentPath);
                    }
                }

                ticking = false;
            });
            ticking = true;
        }
    });
};

/**
 * Initialize smart preloading
 */
export const initializePreloading = () => {
    if (typeof window === "undefined") return;

    // Preload critical images immediately
    preloadCriticalImages();

    // Initialize smart preloading after page load
    window.addEventListener("load", () => {
        setTimeout(() => {
            smartPreload();
        }, 1000);
    });
};

/**
 * Get actual available blog post images
 */
export const getAvailableBlogImages = () => {
    return {
        "8-essay-writing-tips-for-students-2025": {
            featured: "/images/blog/essay-writing-tips-2025",
        },
        "essay-structure-guide": {
            featured: "/images/blog/essay-structure-guide-featured",
        },
        "academic-citation-guide": {
            featured: "/images/blog/academic-citation-guide-featured",
        },
        "essay-topic-ideas": {
            featured: "/images/blog/essay-topic-ideas-featured",
        },
    };
};
