/**
 * URL State Manager - Handles URL parameters for order state and deep linking
 */

export class URLStateManager {
    /**
     * Encode order state in URL parameters
     */
    static encodeOrderState(orderData, currentStep) {
        try {
            const params = new URLSearchParams();
            
            // Basic order information
            if (orderData.title) params.set('title', encodeURIComponent(orderData.title));
            if (orderData.subject) params.set('subject', encodeURIComponent(orderData.subject));
            if (orderData.aclevel) params.set('level', encodeURIComponent(orderData.aclevel));
            if (orderData.papertype) params.set('type', encodeURIComponent(orderData.papertype));
            if (orderData.pages) params.set('pages', orderData.pages);
            if (orderData.deadline) params.set('deadline', encodeURIComponent(orderData.deadline));
            if (orderData.spacing) params.set('spacing', encodeURIComponent(orderData.spacing));
            if (orderData.currency) params.set('currency', orderData.currency);
            
            // Step information
            params.set('step', currentStep);
            
            // Timestamp for validation
            params.set('t', Date.now());
            
            return params.toString();
        } catch (error) {
            console.error('Failed to encode order state in URL:', error);
            return '';
        }
    }

    /**
     * Decode order state from URL parameters
     */
    static decodeOrderState() {
        try {
            const params = new URLSearchParams(window.location.search);
            const orderData = {};
            
            // Basic order information
            if (params.get('title')) orderData.title = decodeURIComponent(params.get('title'));
            if (params.get('subject')) orderData.subject = decodeURIComponent(params.get('subject'));
            if (params.get('level')) orderData.aclevel = decodeURIComponent(params.get('level'));
            if (params.get('type')) orderData.papertype = decodeURIComponent(params.get('type'));
            if (params.get('pages')) orderData.pages = parseInt(params.get('pages'));
            if (params.get('deadline')) orderData.deadline = decodeURIComponent(params.get('deadline'));
            if (params.get('spacing')) orderData.spacing = decodeURIComponent(params.get('spacing'));
            if (params.get('currency')) orderData.currency = params.get('currency');
            
            const currentStep = parseInt(params.get('step')) || 1;
            const timestamp = parseInt(params.get('t')) || 0;
            
            // Validate timestamp (URL should be fresh)
            const maxAge = 120 * 60 * 1000; // 2 hours - more reasonable for order forms
            if (Date.now() - timestamp > maxAge) {
                // Silently clear old URL state - localStorage fallback will handle restoration
                this.clearURLState();
                return null;
            }
            
            // Only return state if we have at least some order data
            if (!orderData || Object.keys(orderData).length === 0) {
                return null;
            }
            
            return {
                orderData,
                currentStep,
                timestamp
            };
        } catch (error) {
            console.error('Failed to decode order state from URL:', error);
            return null;
        }
    }

    /**
     * Update URL with current order state
     */
    static updateURL(orderData, currentStep) {
        try {
            const params = this.encodeOrderState(orderData, currentStep);
            const newURL = `${window.location.pathname}?${params}`;
            
            // Use history API to update URL without page reload
            window.history.replaceState(
                { orderState: { orderData, currentStep } },
                '',
                newURL
            );
            
            return true;
        } catch (error) {
            console.error('Failed to update URL:', error);
            return false;
        }
    }

    /**
     * Clear URL state
     */
    static clearURLState() {
        try {
            const newURL = window.location.pathname;
            window.history.replaceState({}, '', newURL);
            return true;
        } catch (error) {
            console.error('Failed to clear URL state:', error);
            return false;
        }
    }

    /**
     * Check if URL contains order state
     */
    static hasOrderState() {
        const params = new URLSearchParams(window.location.search);
        return params.has('step') && params.has('t');
    }

    /**
     * Validate URL parameters
     */
    static validateURLParams() {
        const params = new URLSearchParams(window.location.search);
        const step = parseInt(params.get('step'));
        const timestamp = parseInt(params.get('t'));
        
        // Basic validation
        if (!step || step < 1 || step > 5) {
            return false;
        }
        
        // Use same max age as decodeOrderState (2 hours)
        const maxAge = 120 * 60 * 1000; // 2 hours
        if (!timestamp || Date.now() - timestamp > maxAge) {
            return false;
        }
        
        return true;
    }

    /**
     * Get URL state summary for debugging
     */
    static getURLStateSummary() {
        try {
            const params = new URLSearchParams(window.location.search);
            return {
                hasParams: params.toString().length > 0,
                step: params.get('step'),
                timestamp: params.get('t'),
                age: params.get('t') ? Date.now() - parseInt(params.get('t')) : null,
                isValid: this.validateURLParams()
            };
        } catch (error) {
            return { error: error.message };
        }
    }

    /**
     * Handle browser back/forward navigation
     */
    static handleNavigation(callback) {
        const handlePopState = (event) => {
            if (event.state && event.state.orderState) {
                callback(event.state.orderState);
            } else {
                // Handle direct URL access
                const urlState = this.decodeOrderState();
                if (urlState) {
                    callback(urlState);
                }
            }
        };

        window.addEventListener('popstate', handlePopState);
        
        // Return cleanup function
        return () => {
            window.removeEventListener('popstate', handlePopState);
        };
    }

    /**
     * Create shareable link with order state
     */
    static createShareableLink(orderData, currentStep) {
        try {
            const params = this.encodeOrderState(orderData, currentStep);
            return `${window.location.origin}${window.location.pathname}?${params}`;
        } catch (error) {
            console.error('Failed to create shareable link:', error);
            return window.location.href;
        }
    }
} 