#!/bin/bash

# Migration Dry-Run Test Script
# Tests all migrations on a fresh test database

echo "🔍 Testing Migrations on Fresh Database"
echo "========================================"
echo ""

# Colors for output
GREEN='\033[0;32m'
RED='\033[0;31m'
YELLOW='\033[1;33m'
NC='\033[0m' # No Color

# Backup current .env
cp .env .env.backup

# Create test database config
echo "${YELLOW}📝 Creating test database configuration...${NC}"

# Copy current .env and modify for testing
cp .env .env.testing

# Update database settings for test
sed -i 's/DB_DATABASE=.*/DB_DATABASE=academic_scribe_test/' .env.testing

# Ensure Pusher has dummy values (to avoid errors during migration)
if ! grep -q "PUSHER_APP_ID" .env.testing; then
    echo "PUSHER_APP_ID=dummy_test_id" >> .env.testing
    echo "PUSHER_APP_KEY=dummy_test_key" >> .env.testing
    echo "PUSHER_APP_SECRET=dummy_test_secret" >> .env.testing
fi

# Create test database
echo "${YELLOW}🗄️  Creating test database...${NC}"
mysql -u root -e "DROP DATABASE IF EXISTS academic_scribe_test;"
mysql -u root -e "CREATE DATABASE academic_scribe_test CHARACTER SET utf8mb4 COLLATE utf8mb4_unicode_ci;"

if [ $? -eq 0 ]; then
    echo "${GREEN}✅ Test database created successfully${NC}"
else
    echo "${RED}❌ Failed to create test database${NC}"
    exit 1
fi

# Temporarily use test config
cp .env.testing .env

# Clear caches
echo "${YELLOW}🧹 Clearing caches...${NC}"
php artisan config:clear > /dev/null 2>&1
php artisan cache:clear > /dev/null 2>&1

# Run migrations
echo ""
echo "${YELLOW}🚀 Running fresh migrations...${NC}"
echo "================================"
php artisan migrate:fresh --force

MIGRATION_EXIT_CODE=$?

# Restore original .env
cp .env.backup .env
rm .env.testing .env.backup

# Clear caches again
php artisan config:clear > /dev/null 2>&1

# Drop test database
echo ""
echo "${YELLOW}🗑️  Cleaning up test database...${NC}"
mysql -u root -e "DROP DATABASE IF EXISTS academic_scribe_test;"

# Report results
echo ""
echo "========================================"
if [ $MIGRATION_EXIT_CODE -eq 0 ]; then
    echo "${GREEN}✅ ALL MIGRATIONS PASSED!${NC}"
    echo "${GREEN}✅ Your migrations are ready for fresh deployment!${NC}"
    exit 0
else
    echo "${RED}❌ MIGRATIONS FAILED!${NC}"
    echo "${RED}❌ Please check the errors above and fix the problematic migrations${NC}"
    exit 1
fi

