<?php

namespace Tests\Feature;

use Tests\TestCase;
use App\Models\User;
use App\Models\Order;
use App\Models\Bid;
use App\Notifications\ClientBidPlacedNotification;
use App\Notifications\AdminBidPlacedNotification;
use Illuminate\Foundation\Testing\RefreshDatabase;
use Illuminate\Support\Facades\Notification;

class BidNotificationTest extends TestCase
{
    use RefreshDatabase;

    protected $client;
    protected $writer;
    protected $admin;
    protected $order;

    protected function setUp(): void
    {
        parent::setUp();
        
        // Create test users
        $this->client = User::factory()->create([
            'user_type' => 'client',
            'name' => 'Test Client',
            'email' => 'client@test.com'
        ]);

        $this->writer = User::factory()->create([
            'user_type' => 'writer',
            'name' => 'Test Writer',
            'email' => 'writer@test.com'
        ]);

        // Create writer profile for the writer
        $writerProfile = $this->writer->writerProfile()->create([
            'is_approved' => true,
            'profile_completion_percentage' => 100,
            'experience_years' => 5,
            'nickname' => 'TestWriter'
        ]);

        // Create qualification test with passed status
        $writerProfile->qualificationTest()->create([
            'questions_test_status' => 'passed',
            'questions_completed_at' => now(),
            'score' => 85,
            'total_questions' => 20
        ]);

        // Create essay test with passed status
        $writerProfile->essayTest()->create([
            'essay_test_status' => 'passed',
            'essay_completed_at' => now(),
            'essay_submitted_at' => now(),
            'essay_reviewed_at' => now()
        ]);

        $this->admin = User::factory()->create([
            'user_type' => 'admin',
            'name' => 'Test Admin',
            'email' => 'admin@test.com'
        ]);

        // Create a test order
        $this->order = Order::factory()->create([
            'user_id' => $this->client->id,
            'order_status' => 'bidding',
            'payment_status' => 'paid',
            'order_number' => 'TEST-001',
            'title' => 'Test Order',
            'subject' => 'Test Subject',
            'pages' => 5,
            'writer_amount' => 100.00,
            'writer_deadline' => now()->addDays(7)
        ]);

        // Mock notifications
        Notification::fake();
    }

    /** @test */
    public function client_receives_notification_when_bid_is_placed()
    {
        $bid = Bid::create([
            'order_id' => $this->order->id,
            'writer_id' => $this->writer->id,
            'bid_amount' => 85.00,
            'completion_time' => now()->addDays(5),
            'comments' => 'I can complete this order efficiently.',
            'status' => 'pending'
        ]);

        // Send notification manually to test
        $this->client->notify(new ClientBidPlacedNotification($this->order, $bid));

        // Assert notification was sent to client
        Notification::assertSentTo(
            $this->client,
            ClientBidPlacedNotification::class
        );
    }

    /** @test */
    public function admin_receives_notification_when_bid_is_placed()
    {
        $bid = Bid::create([
            'order_id' => $this->order->id,
            'writer_id' => $this->writer->id,
            'bid_amount' => 90.00,
            'completion_time' => now()->addDays(6),
            'comments' => 'Experienced writer ready to help.',
            'status' => 'pending'
        ]);

        // Send notification manually to test
        $this->admin->notify(new AdminBidPlacedNotification($this->order, $bid));

        // Assert notification was sent to admin
        Notification::assertSentTo(
            $this->admin,
            AdminBidPlacedNotification::class
        );
    }

    /** @test */
    public function bid_creation_triggers_notifications_automatically()
    {
        // Act as the writer
        $this->actingAs($this->writer);

        // Place a bid through the controller
        $response = $this->postJson("/writer/orders/{$this->order->id}/bid", [
            'bid_amount' => 80.00,
            'completion_time' => now()->addDays(4)->toDateTimeString(),
            'comments' => 'Quick turnaround guaranteed.',
            'writer_amount' => $this->order->writer_amount,
            'writer_deadline' => $this->order->writer_deadline
        ]);



        $response->assertStatus(201);

        // Assert notifications were sent
        Notification::assertSentTo(
            $this->client,
            ClientBidPlacedNotification::class
        );

        Notification::assertSentTo(
            $this->admin,
            AdminBidPlacedNotification::class
        );
    }

    /** @test */
    public function notification_contains_correct_bid_information()
    {
        $bid = Bid::create([
            'order_id' => $this->order->id,
            'writer_id' => $this->writer->id,
            'bid_amount' => 75.00,
            'completion_time' => now()->addDays(3),
            'comments' => 'Quality work at competitive price.',
            'status' => 'pending'
        ]);

        // Send notification to client
        $this->client->notify(new ClientBidPlacedNotification($this->order, $bid));

        // Assert notification was sent
        Notification::assertSentTo(
            $this->client,
            ClientBidPlacedNotification::class
        );
    }

    /** @test */
    public function notification_handles_missing_writer_profile_gracefully()
    {
        // Create writer without profile
        $writerWithoutProfile = User::factory()->create([
            'user_type' => 'writer',
            'name' => 'New Writer',
            'email' => 'new@test.com'
        ]);

        $bid = Bid::create([
            'order_id' => $this->order->id,
            'writer_id' => $writerWithoutProfile->id,
            'bid_amount' => 70.00,
            'completion_time' => now()->addDays(2),
            'comments' => null,
            'status' => 'pending'
        ]);

        // Should not throw an error
        $this->client->notify(new ClientBidPlacedNotification($this->order, $bid));

        // Assert notification was sent successfully
        Notification::assertSentTo(
            $this->client,
            ClientBidPlacedNotification::class
        );
    }
} 