<?php

namespace Tests\Feature;

use Tests\TestCase;
use App\Models\User;
use App\Models\Order;
use App\Notifications\NewOrderAvailableForBiddingNotification;
use App\Services\WriterNotificationService;
use Illuminate\Foundation\Testing\RefreshDatabase;
use Illuminate\Support\Facades\Notification;

class NewOrderAvailableForBiddingNotificationTest extends TestCase
{
    use RefreshDatabase;

    protected $client;
    protected $qualifiedWriter;
    protected $unqualifiedWriter;
    protected $order;
    protected $writerNotificationService;

    protected function setUp(): void
    {
        parent::setUp();
        
        // Create test users
        $this->client = User::factory()->create([
            'user_type' => 'client',
            'name' => 'Test Client',
            'email' => 'client@test.com'
        ]);

        $this->qualifiedWriter = User::factory()->create([
            'user_type' => 'writer',
            'name' => 'Qualified Writer',
            'email' => 'qualified@test.com'
        ]);

        $this->unqualifiedWriter = User::factory()->create([
            'user_type' => 'writer',
            'name' => 'Unqualified Writer',
            'email' => 'unqualified@test.com'
        ]);

        // Create qualified writer profile with passed tests
        $writerProfile = $this->qualifiedWriter->writerProfile()->create([
            'is_approved' => true,
            'profile_completion_percentage' => 100,
            'experience_years' => 5,
            'nickname' => 'QualifiedWriter'
        ]);

        // Create qualification test with passed status
        $writerProfile->qualificationTest()->create([
            'questions_test_status' => 'passed',
            'questions_completed_at' => now(),
            'score' => 85,
            'total_questions' => 20
        ]);

        // Create essay test with passed status
        $writerProfile->essayTest()->create([
            'essay_test_status' => 'passed',
            'essay_completed_at' => now(),
            'essay_submitted_at' => now(),
            'essay_reviewed_at' => now()
        ]);

        // Create unqualified writer profile (no tests passed)
        $this->unqualifiedWriter->writerProfile()->create([
            'is_approved' => false,
            'profile_completion_percentage' => 50,
            'experience_years' => 2,
            'nickname' => 'UnqualifiedWriter'
        ]);

        // Create a test order
        $this->order = Order::factory()->create([
            'user_id' => $this->client->id,
            'order_status' => 'bidding',
            'payment_status' => 'paid',
            'order_number' => 'TEST-001',
            'title' => 'Test Order',
            'subject' => 'Test Subject',
            'academic_level' => 'Undergraduate',
            'type_of_paper' => 'Essay',
            'urgency' => 'Normal',
            'pages' => 5,
            'writer_amount' => 100.00,
            'writer_deadline' => now()->addDays(7),
            'deadline' => now()->addDays(14)
        ]);

        // Initialize the service
        $this->writerNotificationService = new WriterNotificationService();

        // Mock notifications
        Notification::fake();
    }

    /** @test */
    public function qualified_writers_receive_notification_when_order_becomes_available_for_bidding()
    {
        // Send notification to qualified writers
        $this->writerNotificationService->notifyQualifiedWriters($this->order);

        // Assert notification was sent to qualified writer
        Notification::assertSentTo(
            $this->qualifiedWriter,
            NewOrderAvailableForBiddingNotification::class
        );

        // Assert notification was NOT sent to unqualified writer
        Notification::assertNotSentTo(
            $this->unqualifiedWriter,
            NewOrderAvailableForBiddingNotification::class
        );
    }

    /** @test */
    public function notification_contains_correct_order_information()
    {
        // Send notification to qualified writers
        $this->writerNotificationService->notifyQualifiedWriters($this->order);

        // Assert notification was sent
        Notification::assertSentTo(
            $this->qualifiedWriter,
            NewOrderAvailableForBiddingNotification::class
        );
    }

    /** @test */
    public function unqualified_writers_do_not_receive_notifications()
    {
        // Send notification to qualified writers
        $this->writerNotificationService->notifyQualifiedWriters($this->order);

        // Assert notification was NOT sent to unqualified writer
        Notification::assertNotSentTo(
            $this->unqualifiedWriter,
            NewOrderAvailableForBiddingNotification::class
        );
    }

    /** @test */
    public function service_correctly_identifies_qualified_writers()
    {
        // Test the isWriterQualified method
        $this->assertTrue($this->writerNotificationService->isWriterQualified($this->qualifiedWriter));
        $this->assertFalse($this->writerNotificationService->isWriterQualified($this->unqualifiedWriter));
        $this->assertFalse($this->writerNotificationService->isWriterQualified($this->client));
    }

    /** @test */
    public function service_returns_correct_qualified_writers_count()
    {
        $count = $this->writerNotificationService->getQualifiedWritersCount();
        $this->assertEquals(1, $count);
    }

    /** @test */
    public function notification_handles_empty_qualified_writers_list()
    {
        // Create an order but no qualified writers exist
        $order = Order::factory()->create([
            'user_id' => $this->client->id,
            'order_status' => 'bidding',
            'payment_status' => 'paid',
            'order_number' => 'TEST-002',
            'title' => 'Another Test Order',
            'pages' => 3
        ]);

        // Delete all qualified writers
        $this->qualifiedWriter->delete();

        // This should not throw an error
        $this->writerNotificationService->notifyQualifiedWriters($order);

        // No notifications should be sent
        Notification::assertNothingSent();
    }

    /** @test */
    public function notification_handles_missing_writer_profiles()
    {
        // Create a writer without a profile
        $writerWithoutProfile = User::factory()->create([
            'user_type' => 'writer',
            'name' => 'No Profile Writer',
            'email' => 'noprofile@test.com'
        ]);

        // Send notification
        $this->writerNotificationService->notifyQualifiedWriters($this->order);

        // Should not throw an error and should not send notification to writer without profile
        Notification::assertNotSentTo(
            $writerWithoutProfile,
            NewOrderAvailableForBiddingNotification::class
        );
    }
} 