<?php

namespace Tests\Feature;

use Tests\TestCase;
use App\Models\User;
use App\Models\Order;
use App\Notifications\WorkSubmittedNotification;
use App\Notifications\WorkSubmittedAdminNotification;
use App\Notifications\OrderApprovedNotification;
use App\Notifications\RevisionRequestedNotification;
use App\Notifications\OrderApprovalReminderNotification;
use App\Notifications\WorkReadyForReviewNotification;
use App\Notifications\WriterStartedWorkingNotification;
use App\Notifications\WriterStartedWorkingAdminNotification;
use App\Notifications\WriterWorkSubmittedNotification;
use App\Services\AdminNotificationService;
use Illuminate\Foundation\Testing\RefreshDatabase;
use Illuminate\Foundation\Testing\WithFaker;
use Illuminate\Support\Facades\Notification;

class NotificationSystemTest extends TestCase
{
    use RefreshDatabase, WithFaker;

    protected function setUp(): void
    {
        parent::setUp();
        Notification::fake();
    }

    /** @test */
    public function it_sends_work_submitted_notification_to_client()
    {
        $client = User::factory()->create(['user_type' => 'client']);
        $writer = User::factory()->create(['user_type' => 'writer']);
        
        $order = Order::factory()->create([
            'user_id' => $client->id,
            'writer_id' => $writer->id,
            'order_status' => Order::ORDER_STATUS_SUBMITTED,
            'payment_status' => 'paid'
        ]);

        $client->notify(new WorkSubmittedNotification($order));

        Notification::assertSentTo(
            $client,
            WorkSubmittedNotification::class
        );
    }

    /** @test */
    public function it_sends_work_submitted_notification_to_admins()
    {
        $client = User::factory()->create(['user_type' => 'client']);
        $writer = User::factory()->create(['user_type' => 'writer']);
        $admin = User::factory()->create(['user_type' => 'admin']);
        
        $order = Order::factory()->create([
            'user_id' => $client->id,
            'writer_id' => $writer->id,
            'order_status' => Order::ORDER_STATUS_SUBMITTED,
            'payment_status' => 'paid'
        ]);

        AdminNotificationService::notifyAllAdmins(new WorkSubmittedAdminNotification($order));

        Notification::assertSentTo(
            $admin,
            WorkSubmittedAdminNotification::class
        );
    }

    /** @test */
    public function it_sends_order_approved_notification_to_writer()
    {
        $client = User::factory()->create(['user_type' => 'client']);
        $writer = User::factory()->create(['user_type' => 'writer']);
        
        $order = Order::factory()->create([
            'user_id' => $client->id,
            'writer_id' => $writer->id,
            'order_status' => Order::ORDER_STATUS_APPROVED,
            'payment_status' => 'paid'
        ]);

        $writer->notify(new OrderApprovedNotification($order));

        Notification::assertSentTo(
            $writer,
            OrderApprovedNotification::class
        );
    }

    /** @test */
    public function it_sends_revision_requested_notification_to_writer()
    {
        $client = User::factory()->create(['user_type' => 'client']);
        $writer = User::factory()->create(['user_type' => 'writer']);
        
        $order = Order::factory()->create([
            'user_id' => $client->id,
            'writer_id' => $writer->id,
            'order_status' => Order::ORDER_STATUS_REVISION_REQUESTED,
            'payment_status' => 'paid',
            'revision_notes' => 'Need more research'
        ]);

        $writer->notify(new RevisionRequestedNotification($order, 'Need more research'));

        Notification::assertSentTo(
            $writer,
            RevisionRequestedNotification::class
        );
    }

    /** @test */
    public function it_sends_reminder_notification_to_client()
    {
        $client = User::factory()->create(['user_type' => 'client']);
        $writer = User::factory()->create(['user_type' => 'writer']);
        
        $order = Order::factory()->create([
            'user_id' => $client->id,
            'writer_id' => $writer->id,
            'order_status' => Order::ORDER_STATUS_UNDER_REVIEW,
            'payment_status' => 'paid',
            'client_review_started_at' => now()->subHours(25)
        ]);

        $client->notify(new OrderApprovalReminderNotification($order, 23));

        Notification::assertSentTo(
            $client,
            OrderApprovalReminderNotification::class
        );
    }

    /** @test */
    public function it_sends_work_ready_for_review_notification_to_client()
    {
        $client = User::factory()->create(['user_type' => 'client']);
        $writer = User::factory()->create(['user_type' => 'writer']);
        
        $order = Order::factory()->create([
            'user_id' => $client->id,
            'writer_id' => $writer->id,
            'order_status' => Order::ORDER_STATUS_UNDER_REVIEW,
            'payment_status' => 'paid',
            'client_review_started_at' => now()
        ]);

        $client->notify(new WorkReadyForReviewNotification($order));

        Notification::assertSentTo(
            $client,
            WorkReadyForReviewNotification::class
        );
    }

    /** @test */
    public function it_sends_writer_started_working_notification_to_writer()
    {
        $writer = User::factory()->create(['user_type' => 'writer']);
        
        $order = Order::factory()->create([
            'writer_id' => $writer->id,
            'order_status' => Order::ORDER_STATUS_IN_PROGRESS,
            'payment_status' => 'paid',
            'work_started_at' => now()
        ]);

        $writer->notify(new WriterStartedWorkingNotification($order));

        Notification::assertSentTo(
            $writer,
            WriterStartedWorkingNotification::class
        );
    }

    /** @test */
    public function it_sends_writer_started_working_notification_to_admins()
    {
        $writer = User::factory()->create(['user_type' => 'writer']);
        $admin = User::factory()->create(['user_type' => 'admin']);
        
        $order = Order::factory()->create([
            'writer_id' => $writer->id,
            'order_status' => Order::ORDER_STATUS_IN_PROGRESS,
            'payment_status' => 'paid',
            'work_started_at' => now()
        ]);

        AdminNotificationService::notifyAllAdmins(new WriterStartedWorkingAdminNotification($order));

        Notification::assertSentTo(
            $admin,
            WriterStartedWorkingAdminNotification::class
        );
    }

    /** @test */
    public function it_sends_writer_work_submitted_notification_to_writer()
    {
        $writer = User::factory()->create(['user_type' => 'writer']);
        
        $order = Order::factory()->create([
            'writer_id' => $writer->id,
            'order_status' => Order::ORDER_STATUS_SUBMITTED,
            'payment_status' => 'paid',
            'work_submitted_at' => now()
        ]);

        $writer->notify(new WriterWorkSubmittedNotification($order, [], false));

        Notification::assertSentTo(
            $writer,
            WriterWorkSubmittedNotification::class
        );
    }

    /** @test */
    public function it_sends_writer_work_submitted_notification_to_writer_for_late_submission()
    {
        $writer = User::factory()->create(['user_type' => 'writer']);
        
        $order = Order::factory()->create([
            'writer_id' => $writer->id,
            'order_status' => Order::ORDER_STATUS_SUBMITTED,
            'payment_status' => 'paid',
            'work_submitted_at' => now(),
            'writer_deadline' => now()->subHours(2) // 2 hours late
        ]);

        $writer->notify(new WriterWorkSubmittedNotification($order, [], true));

        Notification::assertSentTo(
            $writer,
            WriterWorkSubmittedNotification::class
        );
    }

    /** @test */
    public function it_handles_notification_preferences()
    {
        $client = User::factory()->create(['user_type' => 'client']);
        
        // Create notification preferences
        $client->notificationPreferences()->create([
            'notification_type' => 'work_submitted',
            'email_enabled' => true,
            'in_app_enabled' => true,
            'push_enabled' => false
        ]);

        $order = Order::factory()->create([
            'user_id' => $client->id,
            'order_status' => Order::ORDER_STATUS_SUBMITTED,
            'payment_status' => 'paid'
        ]);

        $client->notify(new WorkSubmittedNotification($order));

        Notification::assertSentTo(
            $client,
            WorkSubmittedNotification::class
        );
    }

    /** @test */
    public function it_handles_notification_with_files()
    {
        $client = User::factory()->create(['user_type' => 'client']);
        $writer = User::factory()->create(['user_type' => 'writer']);
        
        $order = Order::factory()->create([
            'user_id' => $client->id,
            'writer_id' => $writer->id,
            'order_status' => Order::ORDER_STATUS_SUBMITTED,
            'payment_status' => 'paid'
        ]);

        $uploadedFiles = [
            (object) [
                'file_name' => 'document.docx',
                'file_size' => 1024000,
                'file_type' => 'application/vnd.openxmlformats-officedocument.wordprocessingml.document'
            ]
        ];

        $client->notify(new WorkSubmittedNotification($order, $uploadedFiles));

        Notification::assertSentTo(
            $client,
            WorkSubmittedNotification::class
        );
    }

    /** @test */
    public function it_handles_auto_approval_notification()
    {
        $client = User::factory()->create(['user_type' => 'client']);
        $writer = User::factory()->create(['user_type' => 'writer']);
        
        $order = Order::factory()->create([
            'user_id' => $client->id,
            'writer_id' => $writer->id,
            'order_status' => Order::ORDER_STATUS_UNDER_REVIEW,
            'payment_status' => 'paid',
            'client_review_started_at' => now()->subHours(49)
        ]);

        // Simulate auto-approval
        $order->update([
            'order_status' => Order::ORDER_STATUS_APPROVED,
            'auto_approval_at' => now()
        ]);

        $writer->notify(new OrderApprovedNotification($order));

        Notification::assertSentTo(
            $writer,
            OrderApprovedNotification::class
        );
    }

    /** @test */
    public function it_handles_multiple_notification_channels()
    {
        $client = User::factory()->create(['user_type' => 'client']);
        
        $order = Order::factory()->create([
            'user_id' => $client->id,
            'order_status' => Order::ORDER_STATUS_SUBMITTED,
            'payment_status' => 'paid'
        ]);

        $notification = new WorkSubmittedNotification($order);
        
        $this->assertContains('mail', $notification->via($client));
        $this->assertContains('database', $notification->via($client));
    }

    /** @test */
    public function it_handles_notification_without_preferences()
    {
        $client = User::factory()->create(['user_type' => 'client']);
        
        $order = Order::factory()->create([
            'user_id' => $client->id,
            'order_status' => Order::ORDER_STATUS_SUBMITTED,
            'payment_status' => 'paid'
        ]);

        $notification = new WorkSubmittedNotification($order);
        
        // Should use default channels when no preferences exist
        $this->assertContains('mail', $notification->via($client));
        $this->assertContains('database', $notification->via($client));
    }
} 