<?php

namespace Tests\Feature;

use Tests\TestCase;
use App\Models\User;
use App\Services\OrderContextService;
use Illuminate\Foundation\Testing\RefreshDatabase;
use Illuminate\Foundation\Testing\WithFaker;

class OrderFormIntegrationTest extends TestCase
{
    use RefreshDatabase, WithFaker;

    public function test_order_context_storage_and_retrieval()
    {
        $orderData = [
            'title' => 'Test Essay',
            'subject' => 'Computer Science',
            'pages' => 5,
            'deadline' => '2024-12-31',
            'instructions' => 'Test instructions'
        ];

        $response = $this->postJson('/store-order-context', [
            'orderData' => $orderData,
            'currentStep' => 3
        ]);

        $response->assertStatus(200)
                ->assertJson(['success' => true]);

        // Verify context is stored in session
        $this->assertTrue(OrderContextService::hasValidOrderContext());
        
        $context = OrderContextService::getOrderContext();
        $this->assertEquals($orderData, $context['data']);
        $this->assertEquals(3, $context['step']);
    }

    public function test_order_login_redirects_back_to_order_form()
    {
        $user = User::factory()->create([
            'email' => 'test@example.com',
            'password' => bcrypt('password')
        ]);

        // Store order context first
        $this->postJson('/store-order-context', [
            'orderData' => ['title' => 'Test Essay'],
            'currentStep' => 2
        ]);

        $response = $this->post('/order/login', [
            'email' => 'test@example.com',
            'password' => 'password'
        ]);

        $response->assertRedirect('/place-order');
        
        // Check that session has restoration flags
        $this->assertTrue(session()->has('order_context_restored'));
        $this->assertEquals(2, session('order_step'));
    }

    public function test_order_register_creates_user_and_redirects()
    {
        $userData = [
            'name' => 'Test User',
            'email' => 'newuser@example.com',
            'password' => 'password',
            'password_confirmation' => 'password'
        ];

        // Store order context first
        $this->postJson('/store-order-context', [
            'orderData' => ['title' => 'Test Essay'],
            'currentStep' => 2
        ]);

        $response = $this->post('/order/register', $userData);

        $response->assertRedirect('/place-order');
        
        // Verify user was created
        $this->assertDatabaseHas('users', [
            'email' => 'newuser@example.com',
            'user_type' => 'client'
        ]);

        // Check that user is verified (bypass email verification)
        $user = User::where('email', 'newuser@example.com')->first();
        $this->assertNotNull($user->email_verified_at);
    }

    public function test_order_context_expiration()
    {
        // Store context
        $this->postJson('/store-order-context', [
            'orderData' => ['title' => 'Test'],
            'currentStep' => 1
        ]);

        // Verify context exists
        $this->assertTrue(OrderContextService::hasValidOrderContext());

        // Simulate time passing (older than 24 hours)
        $context = OrderContextService::getOrderContext();
        $context['timestamp'] = now()->subHours(25)->timestamp;
        session(['order_context' => $context]);

        // Verify context is now invalid
        $this->assertFalse(OrderContextService::hasValidOrderContext());
    }

    public function test_order_form_with_restored_context()
    {
        $user = User::factory()->create();

        // Simulate restored context
        session([
            'order_context_restored' => true,
            'order_step' => 3,
            'order_data' => [
                'title' => 'Restored Essay',
                'subject' => 'Mathematics',
                'pages' => 10
            ]
        ]);

        $response = $this->actingAs($user)->get('/place-order');

        $response->assertStatus(200);
        $response->assertInertia(fn ($page) => 
            $page->component('Order/Create')
                  ->has('restoredOrderData')
                  ->has('restoredStep', 3)
        );
    }

    public function test_invalid_order_context_handling()
    {
        $response = $this->postJson('/store-order-context', [
            'orderData' => 'invalid_data', // Should be array
            'currentStep' => 'invalid_step' // Should be integer
        ]);

        $response->assertStatus(422);
    }

    public function test_order_context_clear_after_successful_auth()
    {
        // Store context
        $this->postJson('/store-order-context', [
            'orderData' => ['title' => 'Test'],
            'currentStep' => 2
        ]);

        $user = User::factory()->create([
            'email' => 'test@example.com',
            'password' => bcrypt('password')
        ]);

        // Login should clear context
        $this->post('/order/login', [
            'email' => 'test@example.com',
            'password' => 'password'
        ]);

        // Context should be cleared from session
        $this->assertFalse(session()->has('order_context'));
    }

    public function test_order_form_step_validation()
    {
        $user = User::factory()->create();

        // Test that authenticated users can access all steps
        $response = $this->actingAs($user)->get('/place-order');
        $response->assertStatus(200);

        // Test step validation logic
        $response = $this->actingAs($user)->postJson('/validate-step', [
            'step' => 4,
            'hasQualifiedDiscounts' => true
        ]);

        $response->assertStatus(200);
    }

    public function test_performance_monitoring_endpoints()
    {
        // Test that performance monitoring doesn't break normal operations
        $response = $this->get('/place-order');
        $response->assertStatus(200);

        // Test with performance monitoring enabled
        $response = $this->withHeaders([
            'X-Performance-Monitoring' => 'true'
        ])->get('/place-order');
        
        $response->assertStatus(200);
    }

    public function test_error_recovery_mechanisms()
    {
        // Test that the system handles corrupted state gracefully
        session(['order_context' => 'invalid_json_data']);

        $response = $this->get('/place-order');
        $response->assertStatus(200); // Should not crash

        // Test localStorage fallback
        $response = $this->withHeaders([
            'X-Test-LocalStorage' => 'corrupted'
        ])->get('/place-order');
        
        $response->assertStatus(200);
    }

    public function test_deep_linking_functionality()
    {
        // Test URL state encoding/decoding
        $orderData = ['title' => 'Deep Link Test'];
        $step = 3;

        // Simulate URL state
        $urlState = [
            'orderData' => $orderData,
            'currentStep' => $step,
            'timestamp' => now()->timestamp
        ];

        // Test that URL state takes priority
        session(['url_state' => $urlState]);

        $response = $this->get('/place-order');
        $response->assertStatus(200);
    }

    public function test_csrf_protection()
    {
        // Test that CSRF protection is active
        $response = $this->postJson('/store-order-context', [
            'orderData' => ['title' => 'Test'],
            'currentStep' => 1
        ]);

        // Should work with proper CSRF token
        $response->assertStatus(200);
    }

    protected function setUp(): void
    {
        parent::setUp();
        
        // Clear any existing order context
        OrderContextService::clearOrderContext();
    }

    protected function tearDown(): void
    {
        // Clean up after tests
        OrderContextService::clearOrderContext();
        
        parent::tearDown();
    }
} 