<?php

namespace Tests\Feature;

use Tests\TestCase;
use App\Models\User;
use App\Models\Coupon;
use App\Models\Pricing;
use App\Models\Subject;
use App\Models\PaperType;
use App\Models\AcademicLevel;
use App\Models\AdminSetting;
use Illuminate\Foundation\Testing\DatabaseTransactions;
use Illuminate\Foundation\Testing\WithFaker;

class OrderPricingTest extends TestCase
{
    use DatabaseTransactions, WithFaker;

    protected function setUp(): void
    {
        parent::setUp();
        
        // Use existing records from the database instead of creating new ones
        $this->academicLevel = AcademicLevel::first();
        $this->subject = Subject::first();
        $this->paperType = PaperType::first();
        $this->deadline = Pricing::first();
        
        // Ensure we have the required data
        if (!$this->academicLevel || !$this->subject || !$this->paperType || !$this->deadline) {
            $this->markTestSkipped('Required test data not available in database');
        }
    }

    /** @test */
    public function it_calculates_basic_pricing_correctly()
    {
        $response = $this->postJson('/calculate-order-pricing', [
            'academic_level_id' => $this->academicLevel->id,
            'paper_type_id' => $this->paperType->id,
            'deadline_id' => $this->deadline->id,
            'pages' => 5,
            'slides' => 2,
            'subject_id' => $this->subject->id,
            'sources' => 3,
            'currency_code' => 'USD',
            'spacing' => 'double',
        ]);

        $response->assertStatus(200)
            ->assertJsonStructure([
                'success',
                'pricing' => [
                    'base_price',
                    'slides_cost',
                    'total_before_discount',
                    'discount_amount',
                    'discount_percentage',
                    'final_price',
                    'currency',
                    'currency_rate',
                    'breakdown'
                ],
                'is_authenticated',
                'available_coupons'
            ]);

        $pricing = $response->json('pricing');
        
        // Verify basic calculation: pages * academic_level_value * deadline_value * subject_value * paper_type_value
        $expectedBasePrice = 5 * $this->academicLevel->value * $this->deadline->value * $this->subject->value * $this->paperType->value;
        $this->assertEquals($expectedBasePrice, $pricing['base_price'], '', 0.01);
        
        // Verify slides cost
        $expectedSlidesCost = 2 * 4.00; // Using the actual price_per_slide from AdminSetting
        $this->assertEquals($expectedSlidesCost, $pricing['slides_cost'], '', 0.01);
        
        // Verify total before discount
        $expectedTotal = $expectedBasePrice + $expectedSlidesCost;
        $this->assertEquals($expectedTotal, $pricing['total_before_discount'], '', 0.01);
    }

    /** @test */
    public function it_validates_required_fields()
    {
        $response = $this->postJson('/calculate-order-pricing', [
            'pages' => 5,
            'slides' => 2,
        ]);

        $response->assertStatus(422)
            ->assertJsonValidationErrors([
                'academic_level_id',
                'paper_type_id',
                'deadline_id',
                'subject_id'
            ]);
    }

    /** @test */
    public function it_enforces_reasonable_limits()
    {
        $response = $this->postJson('/calculate-order-pricing', [
            'academic_level_id' => $this->academicLevel->id,
            'paper_type_id' => $this->paperType->id,
            'deadline_id' => $this->deadline->id,
            'pages' => 200, // Exceeds max limit
            'slides' => 100, // Exceeds max limit
            'subject_id' => $this->subject->id,
            'sources' => 50, // Exceeds max limit
            'currency_code' => 'USD',
            'spacing' => 'double',
        ]);

        $response->assertStatus(422)
            ->assertJsonValidationErrors(['pages', 'slides', 'sources']);
    }

    /** @test */
    public function it_prevents_manipulation_of_pricing_data()
    {
        // Test that the backend calculation is used, not frontend values
        $response = $this->postJson('/calculate-order-pricing', [
            'academic_level_id' => $this->academicLevel->id,
            'paper_type_id' => $this->paperType->id,
            'deadline_id' => $this->deadline->id,
            'pages' => 5,
            'slides' => 2,
            'subject_id' => $this->subject->id,
            'sources' => 3,
            'currency_code' => 'USD',
            'spacing' => 'double',
        ]);

        $response->assertStatus(200);
        
        $pricing = $response->json('pricing');
        
        // Verify the calculation is done server-side
        $expectedBasePrice = 5 * $this->academicLevel->value * $this->deadline->value * $this->subject->value * $this->paperType->value;
        $this->assertEquals($expectedBasePrice, $pricing['base_price'], '', 0.01);
        
        // Even if someone tries to manipulate the request, the server calculates correctly
        $manipulatedResponse = $this->postJson('/calculate-order-pricing', [
            'academic_level_id' => $this->academicLevel->id,
            'paper_type_id' => $this->paperType->id,
            'deadline_id' => $this->deadline->id,
            'pages' => 5,
            'slides' => 2,
            'subject_id' => $this->subject->id,
            'sources' => 3,
            'currency_code' => 'USD',
            'spacing' => 'double',
            // These would be ignored in a real attack
            'totalAmount' => 999.99,
            'netAmount' => 1.00,
        ]);

        $manipulatedResponse->assertStatus(200);
        
        $manipulatedPricing = $manipulatedResponse->json('pricing');
        
        // Should still calculate correctly, ignoring manipulated values
        $this->assertEquals($expectedBasePrice, $manipulatedPricing['base_price'], '', 0.01);
    }
}
