<?php

namespace Tests\Feature;

use Tests\TestCase;
use App\Models\User;
use App\Models\Order;
use App\Models\OrderStatusHistory;
use App\Services\OrderStatusService;
use Illuminate\Foundation\Testing\WithFaker;
use Illuminate\Foundation\Testing\DatabaseTransactions;

class OrderTimelineTest extends TestCase
{
    use DatabaseTransactions, WithFaker;

    protected $orderStatusService;

    protected function setUp(): void
    {
        parent::setUp();
        $this->orderStatusService = app(OrderStatusService::class);
    }

    /** @test */
    public function it_can_create_order_with_initial_pending_status()
    {
        $user = User::factory()->create(['user_type' => 'client']);
        
        $order = Order::factory()->create([
            'user_id' => $user->id,
            'order_status' => Order::ORDER_STATUS_PENDING,
            'payment_status' => 'unpaid'
        ]);

        $this->assertEquals(Order::ORDER_STATUS_PENDING, $order->order_status);
        $this->assertTrue($order->isActive());
        $this->assertFalse($order->isCompleted());
    }

    /** @test */
    public function it_can_transition_from_pending_to_bidding_after_payment()
    {
        $user = User::factory()->create(['user_type' => 'client']);
        
        $order = Order::factory()->create([
            'user_id' => $user->id,
            'order_status' => Order::ORDER_STATUS_PENDING,
            'payment_status' => 'unpaid'
        ]);

        // Simulate payment completion
        $order->payment_status = 'paid';
        $order->save();

        $this->orderStatusService->handlePaymentCompleted($order);

        $this->assertEquals(Order::ORDER_STATUS_BIDDING, $order->fresh()->order_status);
        $this->assertTrue($order->fresh()->isBidding());
    }

    /** @test */
    public function it_can_transition_from_bidding_to_assigned_when_bid_accepted()
    {
        $user = User::factory()->create(['user_type' => 'client']);
        $writer = User::factory()->create(['user_type' => 'writer']);
        
        $order = Order::factory()->create([
            'user_id' => $user->id,
            'order_status' => Order::ORDER_STATUS_BIDDING,
            'payment_status' => 'paid'
        ]);

        $this->orderStatusService->handleBidAccepted($order, $writer->id, 1);

        $this->assertEquals(Order::ORDER_STATUS_ASSIGNED, $order->fresh()->order_status);
        $this->assertEquals($writer->id, $order->fresh()->writer_id);
        $this->assertNotNull($order->fresh()->writer_assigned_at);
    }

    /** @test */
    public function it_can_transition_from_assigned_to_in_progress()
    {
        $user = User::factory()->create(['user_type' => 'client']);
        $writer = User::factory()->create(['user_type' => 'writer']);
        
        $order = Order::factory()->create([
            'user_id' => $user->id,
            'writer_id' => $writer->id,
            'order_status' => Order::ORDER_STATUS_ASSIGNED,
            'payment_status' => 'paid'
        ]);

        $this->orderStatusService->changeStatus($order, Order::ORDER_STATUS_IN_PROGRESS, 'Writer started working');

        $this->assertEquals(Order::ORDER_STATUS_IN_PROGRESS, $order->fresh()->order_status);
        $this->assertNotNull($order->fresh()->work_started_at);
    }

    /** @test */
    public function it_can_transition_from_in_progress_to_submitted()
    {
        $user = User::factory()->create(['user_type' => 'client']);
        $writer = User::factory()->create(['user_type' => 'writer']);
        
        $order = Order::factory()->create([
            'user_id' => $user->id,
            'writer_id' => $writer->id,
            'order_status' => Order::ORDER_STATUS_IN_PROGRESS,
            'payment_status' => 'paid'
        ]);

        $this->orderStatusService->handleWorkSubmitted($order);

        $this->assertEquals(Order::ORDER_STATUS_SUBMITTED, $order->fresh()->order_status);
        $this->assertNotNull($order->fresh()->work_submitted_at);
    }

    /** @test */
    public function it_can_transition_from_submitted_to_under_review()
    {
        $user = User::factory()->create(['user_type' => 'client']);
        $writer = User::factory()->create(['user_type' => 'writer']);
        
        $order = Order::factory()->create([
            'user_id' => $user->id,
            'writer_id' => $writer->id,
            'order_status' => Order::ORDER_STATUS_SUBMITTED,
            'payment_status' => 'paid'
        ]);

        $this->orderStatusService->handleAdminReviewCompleted($order);

        $this->assertEquals(Order::ORDER_STATUS_UNDER_REVIEW, $order->fresh()->order_status);
        $this->assertNotNull($order->fresh()->admin_reviewed_at);
        $this->assertNotNull($order->fresh()->client_review_started_at);
    }

    /** @test */
    public function it_can_transition_from_under_review_to_approved()
    {
        $user = User::factory()->create(['user_type' => 'client']);
        $writer = User::factory()->create(['user_type' => 'writer']);
        
        $order = Order::factory()->create([
            'user_id' => $user->id,
            'writer_id' => $writer->id,
            'order_status' => Order::ORDER_STATUS_UNDER_REVIEW,
            'payment_status' => 'paid',
            'client_review_started_at' => now()
        ]);

        $this->orderStatusService->handleClientApproval($order);

        $this->assertEquals(Order::ORDER_STATUS_APPROVED, $order->fresh()->order_status);
        $this->assertNotNull($order->fresh()->client_approved_at);
        $this->assertTrue($order->fresh()->isCompleted());
    }

    /** @test */
    public function it_can_handle_revision_request()
    {
        $user = User::factory()->create(['user_type' => 'client']);
        $writer = User::factory()->create(['user_type' => 'writer']);
        
        $order = Order::factory()->create([
            'user_id' => $user->id,
            'writer_id' => $writer->id,
            'order_status' => Order::ORDER_STATUS_UNDER_REVIEW,
            'payment_status' => 'paid',
            'client_review_started_at' => now()
        ]);

        $this->orderStatusService->handleClientRejection($order, 'Need more research');

        $this->assertEquals(Order::ORDER_STATUS_REVISION_REQUESTED, $order->fresh()->order_status);
        $this->assertEquals('Need more research', $order->fresh()->revision_notes);
        $this->assertEquals(1, $order->fresh()->revision_count);
    }

    /** @test */
    public function it_can_handle_auto_approval_after_48_hours()
    {
        $user = User::factory()->create(['user_type' => 'client']);
        $writer = User::factory()->create(['user_type' => 'writer']);
        
        $order = Order::factory()->create([
            'user_id' => $user->id,
            'writer_id' => $writer->id,
            'order_status' => Order::ORDER_STATUS_UNDER_REVIEW,
            'payment_status' => 'paid',
            'client_review_started_at' => now()->subHours(49)
        ]);

        $ordersNeedingAutoApproval = $this->orderStatusService->getOrdersNeedingAutoApproval();
        
        $this->assertTrue($ordersNeedingAutoApproval->contains($order->id));
    }

    /** @test */
    public function it_can_send_reminder_after_24_hours()
    {
        $user = User::factory()->create(['user_type' => 'client']);
        $writer = User::factory()->create(['user_type' => 'writer']);
        
        $order = Order::factory()->create([
            'user_id' => $user->id,
            'writer_id' => $writer->id,
            'order_status' => Order::ORDER_STATUS_UNDER_REVIEW,
            'payment_status' => 'paid',
            'client_review_started_at' => now()->subHours(25)
        ]);

        $ordersNeedingReminder = $this->orderStatusService->getOrdersNeedingReminder();
        
        $this->assertTrue($ordersNeedingReminder->contains($order->id));
    }

    /** @test */
    public function it_maintains_status_history()
    {
        $user = User::factory()->create(['user_type' => 'client']);
        
        $order = Order::factory()->create([
            'user_id' => $user->id,
            'order_status' => Order::ORDER_STATUS_PENDING,
            'payment_status' => 'unpaid'
        ]);

        // Simulate status transitions
        $this->orderStatusService->changeStatus($order, Order::ORDER_STATUS_BIDDING, 'Payment completed');
        $this->orderStatusService->changeStatus($order, Order::ORDER_STATUS_ASSIGNED, 'Bid accepted');

        $statusHistory = $order->statusHistory()->get();
        
        $this->assertCount(2, $statusHistory);
        $this->assertEquals(Order::ORDER_STATUS_BIDDING, $statusHistory->first()->to_status);
        $this->assertEquals(Order::ORDER_STATUS_ASSIGNED, $statusHistory->last()->to_status);
    }

    /** @test */
    public function it_validates_status_transitions()
    {
        $user = User::factory()->create(['user_type' => 'client']);
        
        $order = Order::factory()->create([
            'user_id' => $user->id,
            'order_status' => Order::ORDER_STATUS_PENDING,
            'payment_status' => 'unpaid'
        ]);

        // Should not be able to go directly from pending to approved
        $this->assertFalse($order->canTransitionTo(Order::ORDER_STATUS_APPROVED));
        
        // Should be able to go from pending to bidding
        $this->assertTrue($order->canTransitionTo(Order::ORDER_STATUS_BIDDING));
    }

    /** @test */
    public function it_handles_invalid_status_transitions()
    {
        $user = User::factory()->create(['user_type' => 'client']);
        
        $order = Order::factory()->create([
            'user_id' => $user->id,
            'order_status' => Order::ORDER_STATUS_PENDING,
            'payment_status' => 'unpaid'
        ]);

        $this->expectException(\InvalidArgumentException::class);
        
        $this->orderStatusService->changeStatus($order, Order::ORDER_STATUS_APPROVED, 'Invalid transition');
    }

    /** @test */
    public function it_can_get_next_possible_statuses()
    {
        $user = User::factory()->create(['user_type' => 'client']);
        
        $order = Order::factory()->create([
            'user_id' => $user->id,
            'order_status' => Order::ORDER_STATUS_PENDING,
            'payment_status' => 'unpaid'
        ]);

        $nextStatuses = $order->getNextPossibleStatuses();
        
        $this->assertContains(Order::ORDER_STATUS_BIDDING, $nextStatuses);
        $this->assertContains(Order::ORDER_STATUS_CANCELLED, $nextStatuses);
        $this->assertNotContains(Order::ORDER_STATUS_APPROVED, $nextStatuses);
    }

    /** @test */
    public function it_can_get_status_display_information()
    {
        $user = User::factory()->create(['user_type' => 'client']);
        
        $order = Order::factory()->create([
            'user_id' => $user->id,
            'order_status' => Order::ORDER_STATUS_PENDING,
            'payment_status' => 'unpaid'
        ]);

        $this->assertEquals('Pending', $order->getStatusDisplayName());
        $this->assertEquals('gray', $order->getStatusColor());
        $this->assertEquals('clock', $order->getStatusIcon());
    }

    /** @test */
    public function it_can_handle_cancellation()
    {
        $user = User::factory()->create(['user_type' => 'client']);
        
        $order = Order::factory()->create([
            'user_id' => $user->id,
            'order_status' => Order::ORDER_STATUS_PENDING,
            'payment_status' => 'unpaid'
        ]);

        $this->orderStatusService->handleCancellation($order, 'Client requested cancellation');

        $this->assertEquals(Order::ORDER_STATUS_CANCELLED, $order->fresh()->order_status);
        $this->assertTrue($order->fresh()->isCompleted());
    }
} 