<?php

namespace Tests\Feature;

use Tests\TestCase;
use App\Models\User;
use App\Models\Order;
use App\Models\OrderStatusHistory;
use Illuminate\Foundation\Testing\RefreshDatabase;
use Illuminate\Foundation\Testing\WithFaker;
use Inertia\Testing\AssertableInertia as Assert;

class UIComponentsTest extends TestCase
{
    use RefreshDatabase, WithFaker;

    /** @test */
    public function it_displays_order_timeline_component()
    {
        $user = User::factory()->create(['user_type' => 'client']);
        $this->actingAs($user);

        $order = Order::factory()->create([
            'user_id' => $user->id,
            'order_status' => Order::ORDER_STATUS_BIDDING,
            'payment_status' => 'paid'
        ]);

        $response = $this->get(route('orders.show', $order->id));

        $response->assertStatus(200);
        $response->assertInertia(fn (Assert $page) => $page
            ->component('Order/Show')
            ->has('order')
        );
    }

    /** @test */
    public function it_displays_status_management_for_admins()
    {
        $admin = User::factory()->create(['user_type' => 'admin']);
        $this->actingAs($admin);

        $order = Order::factory()->create([
            'order_status' => Order::ORDER_STATUS_BIDDING,
            'payment_status' => 'paid'
        ]);

        $response = $this->get(route('admin.orders.show', $order->id));

        $response->assertStatus(200);
        $response->assertInertia(fn (Assert $page) => $page
            ->component('Admin/Orders/Show')
            ->has('order')
        );
    }

    /** @test */
    public function it_displays_client_approval_interface_when_ready()
    {
        $user = User::factory()->create(['user_type' => 'client']);
        $this->actingAs($user);

        $order = Order::factory()->create([
            'user_id' => $user->id,
            'order_status' => Order::ORDER_STATUS_UNDER_REVIEW,
            'payment_status' => 'paid',
            'client_review_started_at' => now()
        ]);

        $response = $this->get(route('orders.show', $order->id));

        $response->assertStatus(200);
        $response->assertInertia(fn (Assert $page) => $page
            ->component('Order/Show')
            ->has('order')
        );
    }

    /** @test */
    public function it_handles_deadline_updates()
    {
        $admin = User::factory()->create(['user_type' => 'admin']);
        $this->actingAs($admin);

        $order = Order::factory()->create([
            'deadline' => now()->addDays(7)
        ]);

        $newDeadline = now()->addDays(10)->format('Y-m-d H:i:s');

        $response = $this->post(route('admin.orders.updateDeadline', $order->id), [
            'deadline' => $newDeadline
        ]);

        $response->assertStatus(200);
        $this->assertEquals($newDeadline, $order->fresh()->deadline);
    }

    /** @test */
    public function it_handles_writer_deadline_updates()
    {
        $admin = User::factory()->create(['user_type' => 'admin']);
        $this->actingAs($admin);

        $order = Order::factory()->create([
            'writer_deadline' => now()->addDays(5)
        ]);

        $newWriterDeadline = now()->addDays(6)->format('Y-m-d H:i:s');

        $response = $this->post(route('admin.orders.updateWriterDeadline', $order->id), [
            'writer_deadline' => $newWriterDeadline
        ]);

        $response->assertStatus(200);
        $this->assertEquals($newWriterDeadline, $order->fresh()->writer_deadline);
    }

    /** @test */
    public function it_handles_payment_status_updates()
    {
        $admin = User::factory()->create(['user_type' => 'admin']);
        $this->actingAs($admin);

        $order = Order::factory()->create([
            'payment_status' => 'unpaid'
        ]);

        $response = $this->post(route('admin.orders.update-payment-status', $order->id), [
            'payment_status' => 'paid'
        ]);

        $response->assertStatus(200);
        $this->assertEquals('paid', $order->fresh()->payment_status);
    }

    /** @test */
    public function it_validates_deadline_updates()
    {
        $admin = User::factory()->create(['user_type' => 'admin']);
        $this->actingAs($admin);

        $order = Order::factory()->create([
            'dateposted' => now(),
            'deadline' => now()->addDays(7)
        ]);

        // Try to set deadline before posted date
        $invalidDeadline = now()->subDays(1)->format('Y-m-d H:i:s');

        $response = $this->post(route('admin.orders.updateDeadline', $order->id), [
            'deadline' => $invalidDeadline
        ]);

        $response->assertStatus(422);
    }

    /** @test */
    public function it_handles_status_changes_via_api()
    {
        $admin = User::factory()->create(['user_type' => 'admin']);
        $this->actingAs($admin);

        $order = Order::factory()->create([
            'order_status' => Order::ORDER_STATUS_BIDDING
        ]);

        $response = $this->put(route('admin.orders.status.change', $order->id), [
            'status' => Order::ORDER_STATUS_ASSIGNED,
            'reason' => 'Manual assignment',
            'metadata' => [
                'changed_by' => 'admin',
                'previous_status' => Order::ORDER_STATUS_BIDDING
            ]
        ]);

        $response->assertStatus(200);
        $this->assertEquals(Order::ORDER_STATUS_ASSIGNED, $order->fresh()->order_status);
    }

    /** @test */
    public function it_handles_client_approval()
    {
        $user = User::factory()->create(['user_type' => 'client']);
        $this->actingAs($user);

        $order = Order::factory()->create([
            'user_id' => $user->id,
            'order_status' => Order::ORDER_STATUS_UNDER_REVIEW,
            'payment_status' => 'paid',
            'client_review_started_at' => now()
        ]);

        $response = $this->post(route('orders.approve', $order->id), [
            'action' => 'approve'
        ]);

        $response->assertStatus(200);
        $this->assertEquals(Order::ORDER_STATUS_APPROVED, $order->fresh()->order_status);
    }

    /** @test */
    public function it_handles_client_revision_request()
    {
        $user = User::factory()->create(['user_type' => 'client']);
        $this->actingAs($user);

        $order = Order::factory()->create([
            'user_id' => $user->id,
            'order_status' => Order::ORDER_STATUS_UNDER_REVIEW,
            'payment_status' => 'paid',
            'client_review_started_at' => now()
        ]);

        $response = $this->post(route('orders.revision', $order->id), [
            'action' => 'revision',
            'revision_notes' => 'Need more research'
        ]);

        $response->assertStatus(200);
        $this->assertEquals(Order::ORDER_STATUS_REVISION_REQUESTED, $order->fresh()->order_status);
        $this->assertEquals('Need more research', $order->fresh()->revision_notes);
    }

    /** @test */
    public function it_displays_status_history()
    {
        $user = User::factory()->create(['user_type' => 'client']);
        $this->actingAs($user);

        $order = Order::factory()->create([
            'user_id' => $user->id,
            'order_status' => Order::ORDER_STATUS_BIDDING
        ]);

        // Create some status history
        OrderStatusHistory::create([
            'order_id' => $order->id,
            'from_status' => Order::ORDER_STATUS_PENDING,
            'to_status' => Order::ORDER_STATUS_BIDDING,
            'changed_by' => 'system',
            'reason' => 'Payment completed'
        ]);

        $response = $this->get(route('orders.show', $order->id));

        $response->assertStatus(200);
        $response->assertInertia(fn (Assert $page) => $page
            ->component('Order/Show')
            ->has('order')
        );
    }

    /** @test */
    public function it_handles_file_upload_interface()
    {
        $user = User::factory()->create(['user_type' => 'client']);
        $this->actingAs($user);

        $order = Order::factory()->create([
            'user_id' => $user->id
        ]);

        $response = $this->get(route('orders.show', $order->id));

        $response->assertStatus(200);
        $response->assertInertia(fn (Assert $page) => $page
            ->component('Order/Show')
            ->has('order')
        );
    }

    /** @test */
    public function it_handles_mobile_responsive_design()
    {
        $user = User::factory()->create(['user_type' => 'client']);
        $this->actingAs($user);

        $order = Order::factory()->create([
            'user_id' => $user->id
        ]);

        $response = $this->get(route('orders.show', $order->id));

        $response->assertStatus(200);
        // The response should contain mobile-responsive classes
        $response->assertSee('sm:');
        $response->assertSee('lg:');
    }

    /** @test */
    public function it_handles_dark_mode_classes()
    {
        $user = User::factory()->create(['user_type' => 'client']);
        $this->actingAs($user);

        $order = Order::factory()->create([
            'user_id' => $user->id
        ]);

        $response = $this->get(route('orders.show', $order->id));

        $response->assertStatus(200);
        // The response should contain dark mode classes
        $response->assertSee('dark:');
    }

    /** @test */
    public function it_handles_gradient_styling()
    {
        $user = User::factory()->create(['user_type' => 'client']);
        $this->actingAs($user);

        $order = Order::factory()->create([
            'user_id' => $user->id
        ]);

        $response = $this->get(route('orders.show', $order->id));

        $response->assertStatus(200);
        // The response should contain gradient classes
        $response->assertSee('bg-gradient-to-r');
        $response->assertSee('from-blue-100');
        $response->assertSee('to-cyan-100');
    }
} 