<?php

namespace Tests\Feature;

use Tests\TestCase;
use App\Models\Order;
use App\Models\User;
use App\Services\UrgentOrderNotificationService;
use Carbon\Carbon;
use Illuminate\Foundation\Testing\RefreshDatabase;

class UrgentOrderNotificationTest extends TestCase
{
    use RefreshDatabase;

    protected UrgentOrderNotificationService $urgentOrderService;

    protected function setUp(): void
    {
        parent::setUp();
        $this->urgentOrderService = app(UrgentOrderNotificationService::class);
    }

    /** @test */
    public function it_identifies_orders_due_within_12_hours_as_critical()
    {
        $order = Order::factory()->create([
            'deadline' => Carbon::now()->addHours(10),
            'order_status' => Order::ORDER_STATUS_IN_PROGRESS
        ]);

        $urgentOrders = $this->urgentOrderService->getUrgentOrders();
        
        $this->assertArrayHasKey('critical_12h', $urgentOrders);
        $this->assertCount(1, $urgentOrders['critical_12h']);
        $this->assertEquals($order->id, $urgentOrders['critical_12h'][0]['id']);
        $this->assertEquals('critical', $urgentOrders['critical_12h'][0]['urgency_level']);
    }

    /** @test */
    public function it_identifies_orders_due_within_24_hours_as_urgent()
    {
        $order = Order::factory()->create([
            'deadline' => Carbon::now()->addHours(20),
            'order_status' => Order::ORDER_STATUS_ASSIGNED
        ]);

        $urgentOrders = $this->urgentOrderService->getUrgentOrders();
        
        $this->assertArrayHasKey('urgent_24h', $urgentOrders);
        $this->assertCount(1, $urgentOrders['urgent_24h']);
        $this->assertEquals($order->id, $urgentOrders['urgent_24h'][0]['id']);
        $this->assertEquals('urgent', $urgentOrders['urgent_24h'][0]['urgency_level']);
    }

    /** @test */
    public function it_identifies_orders_due_within_48_hours_as_warning()
    {
        $order = Order::factory()->create([
            'deadline' => Carbon::now()->addHours(36),
            'order_status' => Order::ORDER_STATUS_SUBMITTED
        ]);

        $urgentOrders = $this->urgentOrderService->getUrgentOrders();
        
        $this->assertArrayHasKey('warning_48h', $urgentOrders);
        $this->assertCount(1, $urgentOrders['warning_48h']);
        $this->assertEquals($order->id, $urgentOrders['warning_48h'][0]['id']);
        $this->assertEquals('warning', $urgentOrders['warning_48h'][0]['urgency_level']);
    }

    /** @test */
    public function it_does_not_include_completed_orders()
    {
        Order::factory()->create([
            'deadline' => Carbon::now()->addHours(10),
            'order_status' => Order::ORDER_STATUS_APPROVED
        ]);

        $urgentOrders = $this->urgentOrderService->getUrgentOrders();
        
        $this->assertCount(0, $urgentOrders['critical_12h']);
    }

    /** @test */
    public function it_includes_overdue_orders_in_overdue_category()
    {
        Order::factory()->create([
            'deadline' => Carbon::now()->subHours(5),
            'order_status' => Order::ORDER_STATUS_IN_PROGRESS
        ]);

        $urgentOrders = $this->urgentOrderService->getUrgentOrders();
        
        $this->assertArrayHasKey('overdue', $urgentOrders);
        $this->assertCount(1, $urgentOrders['overdue']);
        $this->assertEquals('overdue', $urgentOrders['overdue'][0]['urgency_level']);
    }

    /** @test */
    public function it_calculates_hours_remaining_correctly()
    {
        $deadline = Carbon::now()->addHours(15);
        $order = Order::factory()->create([
            'deadline' => $deadline,
            'order_status' => Order::ORDER_STATUS_IN_PROGRESS
        ]);

        $urgentOrders = $this->urgentOrderService->getUrgentOrders();
        
        $this->assertArrayHasKey('urgent_24h', $urgentOrders);
        $this->assertCount(1, $urgentOrders['urgent_24h']);
        
        $hoursRemaining = $urgentOrders['urgent_24h'][0]['hours_remaining'];
        $this->assertGreaterThan(14, $hoursRemaining);
        $this->assertLessThan(16, $hoursRemaining);
    }

    /** @test */
    public function it_returns_correct_urgency_colors()
    {
        // Test that urgency colors are properly set on orders
        $order = Order::factory()->create([
            'deadline' => Carbon::now()->addHours(10),
            'order_status' => Order::ORDER_STATUS_IN_PROGRESS
        ]);

        $urgentOrders = $this->urgentOrderService->getUrgentOrders();
        
        $this->assertArrayHasKey('critical_12h', $urgentOrders);
        $this->assertCount(1, $urgentOrders['critical_12h']);
        $this->assertEquals('#dc2626', $urgentOrders['critical_12h'][0]['urgency_color']); // Red for critical
    }
} 