<?php

namespace Tests\Feature;

use Tests\TestCase;
use App\Models\User;
use App\Models\Order;
use App\Notifications\WriterStartedWorkingNotification;
use App\Notifications\WriterStartedWorkingAdminNotification;
use App\Notifications\WriterWorkSubmittedNotification;
use App\Services\AdminNotificationService;
use Illuminate\Foundation\Testing\RefreshDatabase;
use Illuminate\Foundation\Testing\WithFaker;
use Illuminate\Support\Facades\Notification;

class WriterNotificationTest extends TestCase
{
    use RefreshDatabase, WithFaker;

    protected function setUp(): void
    {
        parent::setUp();
        Notification::fake();
    }

    /** @test */
    public function it_sends_writer_started_working_notification_to_writer()
    {
        $writer = User::factory()->create(['user_type' => 'writer']);
        
        $order = Order::factory()->create([
            'writer_id' => $writer->id,
            'order_status' => Order::ORDER_STATUS_IN_PROGRESS,
            'payment_status' => 'paid',
            'work_started_at' => now()
        ]);

        $writer->notify(new WriterStartedWorkingNotification($order));

        Notification::assertSentTo(
            $writer,
            WriterStartedWorkingNotification::class
        );
    }

    /** @test */
    public function it_sends_writer_started_working_notification_to_admins()
    {
        $writer = User::factory()->create(['user_type' => 'writer']);
        $admin = User::factory()->create(['user_type' => 'admin']);
        
        $order = Order::factory()->create([
            'writer_id' => $writer->id,
            'order_status' => Order::ORDER_STATUS_IN_PROGRESS,
            'payment_status' => 'paid',
            'work_started_at' => now()
        ]);

        AdminNotificationService::notifyAllAdmins(new WriterStartedWorkingAdminNotification($order));

        Notification::assertSentTo(
            $admin,
            WriterStartedWorkingAdminNotification::class
        );
    }

    /** @test */
    public function it_sends_writer_work_submitted_notification_to_writer()
    {
        $writer = User::factory()->create(['user_type' => 'writer']);
        
        $order = Order::factory()->create([
            'writer_id' => $writer->id,
            'order_status' => Order::ORDER_STATUS_SUBMITTED,
            'payment_status' => 'paid',
            'work_submitted_at' => now()
        ]);

        $writer->notify(new WriterWorkSubmittedNotification($order, [], false));

        Notification::assertSentTo(
            $writer,
            WriterWorkSubmittedNotification::class
        );
    }

    /** @test */
    public function it_sends_writer_work_submitted_notification_to_writer_for_late_submission()
    {
        $writer = User::factory()->create(['user_type' => 'writer']);
        
        $order = Order::factory()->create([
            'writer_id' => $writer->id,
            'order_status' => Order::ORDER_STATUS_SUBMITTED,
            'payment_status' => 'paid',
            'work_submitted_at' => now(),
            'writer_deadline' => now()->subHours(2) // 2 hours late
        ]);

        $writer->notify(new WriterWorkSubmittedNotification($order, [], true));

        Notification::assertSentTo(
            $writer,
            WriterWorkSubmittedNotification::class
        );
    }

    /** @test */
    public function writer_started_working_notification_contains_correct_order_information()
    {
        $writer = User::factory()->create(['user_type' => 'writer']);
        
        $order = Order::factory()->create([
            'writer_id' => $writer->id,
            'order_status' => Order::ORDER_STATUS_IN_PROGRESS,
            'payment_status' => 'paid',
            'work_started_at' => now(),
            'order_number' => 'ORD-12345',
            'title' => 'Test Assignment',
            'pages' => 5,
            'academic_level' => 'Undergraduate',
            'subject' => 'English'
        ]);

        $notification = new WriterStartedWorkingNotification($order);
        $arrayData = $notification->toArray($writer);

        $this->assertEquals('Work Started Successfully', $arrayData['title']);
        $this->assertStringContainsString('ORD-12345', $arrayData['message']);
        $this->assertStringContainsString('Test Assignment', $arrayData['message']);
        $this->assertEquals('order', $arrayData['category']);
        $this->assertEquals('success', $arrayData['type']);
    }

    /** @test */
    public function writer_started_working_admin_notification_contains_correct_order_information()
    {
        $admin = User::factory()->create(['user_type' => 'admin']);
        
        $order = Order::factory()->create([
            'order_status' => Order::ORDER_STATUS_IN_PROGRESS,
            'payment_status' => 'paid',
            'work_started_at' => now(),
            'order_number' => 'ORD-12345',
            'title' => 'Test Assignment',
            'pages' => 5,
            'academic_level' => 'Undergraduate',
            'subject' => 'English'
        ]);

        $notification = new WriterStartedWorkingAdminNotification($order);
        $arrayData = $notification->toArray($admin);

        $this->assertEquals('Writer Started Working', $arrayData['title']);
        $this->assertStringContainsString('ORD-12345', $arrayData['message']);
        $this->assertStringContainsString('Test Assignment', $arrayData['message']);
        $this->assertEquals('order', $arrayData['category']);
        $this->assertEquals('info', $arrayData['type']);
    }
} 