<?php

namespace Tests\Unit;

use Tests\TestCase;
use App\Services\RevisionService;
use App\Models\Order;
use App\Models\OrderRevision;
use App\Models\User;
use Illuminate\Foundation\Testing\RefreshDatabase;
use Illuminate\Foundation\Testing\WithFaker;
use Illuminate\Support\Facades\Auth;
use Mockery;

class RevisionServiceTest extends TestCase
{
    use RefreshDatabase, WithFaker;

    protected $revisionService;
    protected $order;
    protected $user;
    protected $writer;

    protected function setUp(): void
    {
        parent::setUp();
        
        $this->revisionService = new RevisionService();
        
        // Create test user (admin)
        $this->user = User::factory()->create([
            'user_type' => 'admin',
        ]);
        
        // Create test writer
        $this->writer = User::factory()->create([
            'user_type' => 'writer',
        ]);
        
        // Create test order
        $this->order = Order::factory()->create([
            'order_status' => 'submitted',
            'writer_id' => $this->writer->id,
            'deadline' => now()->addDays(7),
            'writer_deadline' => now()->addDays(5),
        ]);
        
        // Mock Auth facade
        Auth::shouldReceive('id')->andReturn($this->user->id);
    }

    protected function tearDown(): void
    {
        Mockery::close();
        parent::tearDown();
    }

    /** @test */
    public function it_can_create_a_revision()
    {
        $revisionData = [
            'revision_type' => OrderRevision::REVISION_TYPE_CONTENT,
            'revision_priority' => OrderRevision::PRIORITY_MEDIUM,
            'revision_section' => 'Introduction',
            'revision_reason' => 'Content needs improvement',
            'specific_instructions' => 'Make introduction more engaging',
            'assigned_to' => $this->writer->id,
            'revision_admin_notes' => 'Internal note for admin team',
        ];

        $revision = $this->revisionService->createRevision($this->order, $revisionData);

        $this->assertInstanceOf(OrderRevision::class, $revision);
        $this->assertEquals($this->order->id, $revision->order_id);
        $this->assertEquals($this->user->id, $revision->requested_by);
        $this->assertEquals(OrderRevision::REVISION_TYPE_CONTENT, $revision->revision_type);
        $this->assertEquals(OrderRevision::PRIORITY_MEDIUM, $revision->revision_priority);
        $this->assertEquals('Introduction', $revision->revision_section);
        $this->assertEquals('Content needs improvement', $revision->revision_reason);
        $this->assertEquals(OrderRevision::STATUS_REQUESTED, $revision->revision_status);
        $this->assertEquals(1, $revision->revision_number);
        $this->assertNotNull($revision->revision_deadline);
        $this->assertNotNull($revision->requested_at);
    }

    /** @test */
    public function it_updates_order_status_when_revision_is_created()
    {
        $revisionData = [
            'revision_type' => OrderRevision::REVISION_TYPE_CONTENT,
            'revision_priority' => OrderRevision::PRIORITY_MEDIUM,
            'revision_reason' => 'Content needs improvement',
        ];

        $this->revisionService->createRevision($this->order, $revisionData);

        $this->order->refresh();
        $this->assertEquals('revision_requested', $this->order->order_status);
        $this->assertEquals(1, $this->order->revision_count);
        $this->assertEquals('Content needs improvement', $this->order->revision_notes);
    }

    /** @test */
    public function it_calculates_deadline_based_on_priority()
    {
        // Test high priority
        $deadline = $this->revisionService->calculateRevisionDeadline($this->order, OrderRevision::PRIORITY_HIGH);
        $expectedDeadline = now()->addHours(4);
        $this->assertEquals($expectedDeadline->hour, $deadline->hour);
        $this->assertEquals($expectedDeadline->day, $deadline->day);

        // Test medium priority
        $deadline = $this->revisionService->calculateRevisionDeadline($this->order, OrderRevision::PRIORITY_MEDIUM);
        $expectedDeadline = now()->addHours(8);
        $this->assertEquals($expectedDeadline->hour, $deadline->hour);
        $this->assertEquals($expectedDeadline->day, $deadline->day);

        // Test low priority
        $deadline = $this->revisionService->calculateRevisionDeadline($this->order, OrderRevision::PRIORITY_LOW);
        $expectedDeadline = now()->addHours(24);
        $this->assertEquals($expectedDeadline->day, $deadline->day);
    }

    /** @test */
    public function it_adjusts_deadline_based_on_order_deadlines()
    {
        // Create order with tight deadlines
        $tightOrder = Order::factory()->create([
            'deadline' => now()->addHours(24),
            'writer_deadline' => now()->addHours(16),
        ]);

        $deadline = $this->revisionService->calculateRevisionDeadline($tightOrder, OrderRevision::PRIORITY_MEDIUM);
        
        // Should not exceed writer deadline
        $this->assertTrue($deadline->isBefore($tightOrder->writer_deadline));
        $this->assertTrue($deadline->isBefore($tightOrder->deadline));
    }

    /** @test */
    public function it_can_update_revision_status()
    {
        $revision = OrderRevision::factory()->create([
            'order_id' => $this->order->id,
            'revision_status' => OrderRevision::STATUS_REQUESTED,
            'revision_type' => OrderRevision::REVISION_TYPE_CONTENT,
            'revision_priority' => OrderRevision::PRIORITY_MEDIUM,
        ]);

        $success = $this->revisionService->updateRevisionStatus(
            $revision, 
            OrderRevision::STATUS_ACKNOWLEDGED
        );

        $this->assertTrue($success);
        $revision->refresh();
        $this->assertEquals(OrderRevision::STATUS_ACKNOWLEDGED, $revision->revision_status);
        $this->assertNotNull($revision->revision_acknowledged_at);
    }

    /** @test */
    public function it_prevents_invalid_status_transitions()
    {
        $revision = OrderRevision::factory()->create([
            'order_id' => $this->order->id,
            'revision_status' => OrderRevision::STATUS_REQUESTED,
        ]);

        $this->expectException(\InvalidArgumentException::class);
        
        $this->revisionService->updateRevisionStatus(
            $revision, 
            OrderRevision::STATUS_REVIEWED // Cannot go directly from requested to reviewed
        );
    }

    /** @test */
    public function it_can_assign_revision_to_writer()
    {
        $revision = OrderRevision::factory()->create([
            'order_id' => $this->order->id,
            'revision_status' => OrderRevision::STATUS_REQUESTED,
            'assigned_to' => null,
        ]);

        $success = $this->revisionService->assignRevisionToWriter($revision, $this->writer);

        $this->assertTrue($success);
        $revision->refresh();
        $this->assertEquals($this->writer->id, $revision->assigned_to);
        $this->assertEquals(OrderRevision::STATUS_ACKNOWLEDGED, $revision->revision_status);
        $this->assertNotNull($revision->revision_acknowledged_at);
    }

    /** @test */
    public function it_prevents_assignment_to_unqualified_writer()
    {
        $revision = OrderRevision::factory()->create([
            'order_id' => $this->order->id,
            'revision_status' => OrderRevision::STATUS_REQUESTED,
        ]);

        // Create user without writer role
        $nonWriter = User::factory()->create(['user_type' => 'client']);

        $this->expectException(\InvalidArgumentException::class);
        
        $this->revisionService->assignRevisionToWriter($revision, $nonWriter);
    }

    /** @test */
    public function it_prevents_assignment_to_overloaded_writer()
    {
        // Create a writer with 3 active revisions
        $overloadedWriter = User::factory()->create(['user_type' => 'writer']);
        
        // Create 3 active revisions for this writer
        for ($i = 0; $i < 3; $i++) {
            OrderRevision::factory()->create([
                'assigned_to' => $overloadedWriter->id,
                'revision_status' => OrderRevision::STATUS_IN_PROGRESS,
            ]);
        }

        $revision = OrderRevision::factory()->create([
            'order_id' => $this->order->id,
            'revision_status' => OrderRevision::STATUS_REQUESTED,
        ]);

        $this->expectException(\InvalidArgumentException::class);
        
        $this->revisionService->assignRevisionToWriter($revision, $overloadedWriter);
    }

    /** @test */
    public function it_handles_revision_completion_correctly()
    {
        // Create order with multiple revisions using the service
        $revision1 = $this->revisionService->createRevision($this->order, [
            'revision_type' => OrderRevision::REVISION_TYPE_CONTENT,
            'revision_priority' => OrderRevision::PRIORITY_MEDIUM,
            'revision_reason' => 'First revision needed',
        ]);
        
        $revision2 = $this->revisionService->createRevision($this->order, [
            'revision_type' => OrderRevision::REVISION_TYPE_FORMATTING,
            'revision_priority' => OrderRevision::PRIORITY_LOW,
            'revision_reason' => 'Second revision needed',
        ]);
        
        // Manually set the second revision to in_progress status
        $revision2->update(['revision_status' => OrderRevision::STATUS_IN_PROGRESS]);

        // Complete one revision by going through the proper status flow
        $this->revisionService->updateRevisionStatus($revision1, OrderRevision::STATUS_ACKNOWLEDGED);
        $this->revisionService->updateRevisionStatus($revision1, OrderRevision::STATUS_IN_PROGRESS);
        $this->revisionService->updateRevisionStatus($revision1, OrderRevision::STATUS_SUBMITTED);
        $this->revisionService->updateRevisionStatus($revision1, OrderRevision::STATUS_REVIEWED);

        // Refresh both revisions and order to ensure we have the latest state
        $revision1->refresh();
        $revision2->refresh();
        $this->order->refresh();
        
        // Debug: Check the current status of both revisions
        $this->assertEquals(OrderRevision::STATUS_REVIEWED, $revision1->revision_status);
        $this->assertEquals(OrderRevision::STATUS_IN_PROGRESS, $revision2->revision_status);
        
        // Order should still be in revision_requested status since there's another active revision
        $this->assertEquals('revision_requested', $this->order->order_status);

        // Complete the second revision by going through the proper status flow
        // Since revision2 is already in_progress, we can go directly to submitted
        $this->revisionService->updateRevisionStatus($revision2, OrderRevision::STATUS_SUBMITTED);
        $this->revisionService->updateRevisionStatus($revision2, OrderRevision::STATUS_REVIEWED);

        // Order should now be back to submitted status
        $this->order->refresh();
        $this->assertEquals('submitted', $this->order->order_status);
    }

    /** @test */
    public function it_can_get_overdue_revisions()
    {
        // Create overdue revision
        $overdueRevision = OrderRevision::factory()->create([
            'order_id' => $this->order->id,
            'revision_deadline' => now()->subHours(2),
            'revision_status' => OrderRevision::STATUS_REQUESTED,
        ]);

        // Create on-time revision
        $onTimeRevision = OrderRevision::factory()->create([
            'order_id' => $this->order->id,
            'revision_deadline' => now()->addHours(2),
            'revision_status' => OrderRevision::STATUS_REQUESTED,
        ]);

        $overdueRevisions = $this->revisionService->getOverdueRevisions($this->order);

        $this->assertCount(1, $overdueRevisions);
        $this->assertEquals($overdueRevision->id, $overdueRevisions->first()->id);
    }

    /** @test */
    public function it_can_get_high_priority_revisions()
    {
        // Create high priority revision
        $highPriorityRevision = OrderRevision::factory()->create([
            'order_id' => $this->order->id,
            'revision_priority' => OrderRevision::PRIORITY_HIGH,
        ]);

        // Create medium priority revision
        $mediumPriorityRevision = OrderRevision::factory()->create([
            'order_id' => $this->order->id,
            'revision_priority' => OrderRevision::PRIORITY_MEDIUM,
        ]);

        $highPriorityRevisions = $this->revisionService->getHighPriorityRevisions($this->order);

        $this->assertCount(1, $highPriorityRevisions);
        $this->assertEquals($highPriorityRevision->id, $highPriorityRevisions->first()->id);
    }

    /** @test */
    public function it_can_get_revision_statistics()
    {
        // Create revisions in different states
        OrderRevision::factory()->create([
            'order_id' => $this->order->id,
            'revision_status' => OrderRevision::STATUS_REQUESTED,
            'revision_priority' => OrderRevision::PRIORITY_HIGH,
        ]);
        
        OrderRevision::factory()->create([
            'order_id' => $this->order->id,
            'revision_status' => OrderRevision::STATUS_IN_PROGRESS,
            'revision_priority' => OrderRevision::PRIORITY_MEDIUM,
        ]);
        
        OrderRevision::factory()->create([
            'order_id' => $this->order->id,
            'revision_status' => OrderRevision::STATUS_REVIEWED,
            'revision_priority' => OrderRevision::PRIORITY_LOW,
        ]);

        $stats = $this->revisionService->getRevisionStatistics($this->order);

        $this->assertEquals(3, $stats['total_revisions']);
        $this->assertEquals(2, $stats['active_revisions']);
        $this->assertEquals(1, $stats['completed_revisions']);
        $this->assertEquals(1, $stats['high_priority_revisions']);
        // Check that we have some revision types
        $this->assertGreaterThan(0, count($stats['revision_types']));
        // Check that we have at least one revision of any type
        $this->assertGreaterThan(0, $stats['revision_types']->sum());
    }

    /** @test */
    public function it_calculates_average_completion_time()
    {
        // Create completed revisions with known timestamps
        $revision1 = OrderRevision::factory()->create([
            'order_id' => $this->order->id,
            'revision_status' => OrderRevision::STATUS_REVIEWED,
            'requested_at' => now()->subHours(6),
            'revision_reviewed_at' => now()->subHours(2),
        ]);
        
        $revision2 = OrderRevision::factory()->create([
            'order_id' => $this->order->id,
            'revision_status' => OrderRevision::STATUS_REVIEWED,
            'requested_at' => now()->subHours(8),
            'revision_reviewed_at' => now()->subHours(4),
        ]);

        $stats = $this->revisionService->getRevisionStatistics($this->order);

        // Average should be (4 + 4) / 2 = 4 hours
        $this->assertEquals(4.0, $stats['average_completion_time']);
    }

    /** @test */
    public function it_returns_null_average_completion_time_for_no_completed_revisions()
    {
        $stats = $this->revisionService->getRevisionStatistics($this->order);

        $this->assertNull($stats['average_completion_time']);
    }
}
